package freenet.fs.acct;

import freenet.fs.*;
import freenet.crypt.Digest;
import freenet.support.Fields;
import java.io.*;

/**
 * An AccountingTable treats a set of storage fragments as a collection of
 * atomic, fixed-size, randomly addressable "accounting blocks," each of which
 * is checksummed so that incompletely written blocks can be caught.
 * @author tavin
 */
public class AccountingTable {

    private static final class FragmentIndex {
        final Fragment range;
        final int blo, bhi;
        FragmentIndex(Fragment range, int blo, int bhi) {
            this.range = range;
            this.blo = blo;
            this.bhi = bhi;
        }
    }
    

    /** for I/O access to the storage */
    private final LockGrantor lg;
    
    /** storage regions holding the blocks */
    private final FragmentIndex[] fragmentIndices;
    
    /** for check-summing the blocks */
    private final Digest ctx;


    /** byte length of checksum */
    private final int checkWidth;

    /** length of data segment */
    private final int dataWidth;

    /** total byte length of each block */
    private final int blockWidth;


    /** number of block slots */
    private int blockCount;


    private final byte[] zeroes;
    

    
    /**
     * @param lg          I/O access layer
     * @param ranges      storage fragments
     * @param ctx         checksum algorithm
     * @param blockWidth  total byte size of each block
     */
    public AccountingTable(LockGrantor lg, Fragment[] ranges,
                           Digest ctx, int blockWidth) {

        this.lg = lg;
        this.ctx = ctx;
        this.blockWidth = blockWidth;
        
        checkWidth = ctx.digestSize() >> 3;
        dataWidth = blockWidth - checkWidth;
        
        zeroes = new byte[checkWidth];

        if (ranges == null)
            ranges = new Fragment[0];

        blockCount = 0;
        fragmentIndices = new FragmentIndex[ranges.length];
        for (int i=0; i<ranges.length; ++i) {
            fragmentIndices[i] =
                new FragmentIndex(ranges[i],
                                  blockCount,
                                  (blockCount += ranges[i].size() / blockWidth) - 1);
        }
    }

    /**
     * Clone an existing AccountingTable, but extend
     * the available storage fragments.
     * TODO: optionally copy the blocks over
     *       (will be needed for defragging)
     */
    public AccountingTable(AccountingTable ancestor, Fragment[] ranges) {
        this(ancestor.lg, ranges, ancestor.ctx, ancestor.blockWidth);
    }

    /**
     * @return  the storage fragments occupied by the table
     */
    public final Fragment[] ranges() {
        Fragment[] ret = new Fragment[fragmentIndices.length];
        for (int i=0; i<ret.length; ++i)
            ret[i] = fragmentIndices[i].range;
        return ret;
    }
    
    /**
     * @return  number of block slots in the table
     */
    public final int getBlockCount() {
        return blockCount;
    }

    /**
     * @return  byte size of the data portion of the blocks
     */
    public final int getDataWidth() {
        return dataWidth;
    }

    /**
     * @return  total byte size of the blocks
     */
    public final int getBlockWidth() {
        return blockWidth;
    }
    
    /**
     * Retrieves the data from the numbered slot.
     * @return  the data bytes, or null if the block is destroyed
     */
    public byte[] getBlock(int bnum) throws IOException {
        InputStream in = getInputStream(bnum);
        try {
            byte[] check = new byte[checkWidth];
            byte[] data = new byte[dataWidth];
            DataInputStream din = new DataInputStream(in);
            din.readFully(check);
            din.readFully(data);
            ctx.update(data);
            return Fields.byteArrayEqual(ctx.digest(), check) ? data : null;
        }
        finally {
            in.close();
        }
    }

    public final DataInput readBlock(int bnum) throws IOException {
        byte[] block = getBlock(bnum);
        return block == null ? null
                             : new DataInputStream(new ByteArrayInputStream(block));
    }

    /**
     * Stores the data in the numbered slot.
     */
    public void putBlock(int bnum, byte[] data) throws IOException {
        OutputStream out = getOutputStream(bnum);
        try {
            ctx.update(data);
            out.write(ctx.digest());
            out.write(data);
        }
        finally {
            out.close();
        }
    }

    public final DataOutputStream writeBlock(int bnum) throws IOException {
        return new DataOutputStream(new BlockOutputStream(bnum));
    }

    private final class BlockOutputStream extends ByteArrayOutputStream {
        final int bnum;
        BlockOutputStream(int bnum) {
            super(dataWidth);
            this.bnum = bnum;
        }
        public final void close() throws IOException {
            super.close();
            if (buf.length != dataWidth) {
                throw new IOException("data buffer overrun");
            }
            putBlock(bnum, buf);
        }
    }

    /**
     * Delete the block on disk by zeroing the checksum field.
     */
    public void destroyBlock(int bnum) throws IOException {
        OutputStream out = getOutputStream(bnum);
        try {
            out.write(zeroes);
        }
        finally {
            out.close();
        }
    }

    public final void destroyTable() throws IOException {
        for (int i=0; i<blockCount; ++i)
            destroyBlock(i);
    }

    /**
     * Get an input stream from disk for this block.
     */
    private InputStream getInputStream(int bnum) throws IOException {
        FragmentIndex fi = search(bnum);
        if (fi == null)
            throw new AccountingException("block number out of bounds");
        long lo = fi.range.getLowerBound() + (bnum - fi.blo) * blockWidth;
        long hi = lo + blockWidth - 1;
        return ReadLock.getInputStream(lg, lo, hi);
    }
    
    /**
     * Get an output stream to disk for this block.
     */
    private OutputStream getOutputStream(int bnum) throws IOException {
        FragmentIndex fi = search(bnum);
        if (fi == null)
            throw new AccountingException("block number out of bounds");
        long lo = fi.range.getLowerBound() + (bnum - fi.blo) * blockWidth;
        long hi = lo + blockWidth - 1;
        return WriteLock.getOutputStream(lg, lo, hi);
    }

    private FragmentIndex search(int bnum) {
        int lo = 0, hi = fragmentIndices.length - 1;
        while (lo <= hi) {
            int m = (lo + hi) >> 1;
            FragmentIndex fi = fragmentIndices[m];
            if (bnum < fi.blo)
                hi = m-1;
            else if (bnum > fi.bhi)
                lo = m+1;
            else
                return fi;
        }
        return null;
    }
}


