package freenet.message;
import freenet.*;
import freenet.support.*;
import freenet.node.NodeReference;
import freenet.node.BadReferenceException;
import freenet.node.State;
import freenet.node.states.announcement.NewAnnouncement;

/**
 * The message that initiates an announcement transaction.
 *
 * @author oskar
 */

public class NodeAnnouncement extends NodeMessage {

    public static final String messageName = "NodeAnnouncement";

    private int hopsToLive;
    private int depth;
    private NodeReference announcee;
    private NodeReference anSource;
    private byte[] commitVal;

    public NodeAnnouncement(long id, int htl, int depth, 
                            NodeReference source, NodeReference announcee, 
                            byte[] commitVal) {
        super(id);
        this.hopsToLive = htl;
        this.anSource = source;
        this.depth = depth;
        this.announcee = announcee;
        this.commitVal = commitVal;
    }

    public NodeAnnouncement(long id, int htl, int depth, 
                            NodeReference source, NodeReference announcee, 
                            byte[] commitVal, FieldSet otherFields) {
        super(id, otherFields);
        this.hopsToLive = htl;
        this.anSource = source;
        this.depth = depth;
        this.announcee = announcee;
        this.commitVal = commitVal;
    }

    public NodeAnnouncement(ConnectionHandler conn, RawMessage raw) 
        throws InvalidMessageException {

        super(conn, raw);
        String hopS = otherFields.get("HopsToLive");
        String depthS = otherFields.get("Depth");
        FieldSet afs = otherFields.getSet("Announcee");
        FieldSet sfs = otherFields.getSet("Source");
        String commitS = otherFields.get("CommitValue");
        if (hopS == null || hopS.equals("")) {
            throw new InvalidMessageException("No HopsToLive field.");
        } else if (depthS == null || depthS.equals("")) {
            throw new InvalidMessageException("No Depth field.");
        } else if (afs == null || afs.isEmpty()) {
            throw new InvalidMessageException("No Announcee!");
        } else if (sfs == null || sfs.isEmpty()) {
            throw new InvalidMessageException("No Source!");
        } else if (commitS == null || commitS.equals("")) {
            throw new InvalidMessageException("No Commit value");
        } else {
            try {
                hopsToLive = (int) Fields.stringToLong(hopS);
                depth = (int) Fields.stringToLong(depthS);
                announcee = new NodeReference(afs, true);
                anSource = new NodeReference(sfs, false);
                commitVal = Fields.hexToBytes(commitS);
            } catch (NumberFormatException e) {
                throw 
                  new InvalidMessageException("Malformed depth/htl/commitval");
            } catch (BadReferenceException e) {
                throw new InvalidMessageException("Corrupt Announcee");
            }
        }
        otherFields.remove("HopsToLive");
        otherFields.remove("Depth");
        otherFields.remove("Announcee");
        otherFields.remove("Source");
        otherFields.remove("CommitValue");
    }

    public RawMessage toRawMessage(Presentation p) {
        RawMessage raw = super.toRawMessage(p);
        //raw.messageType = messageName;
        raw.fs.add("HopsToLive",Fields.longToString(hopsToLive));
        raw.fs.add("Depth", Fields.longToString(depth));
        raw.fs.add("Announcee", announcee.getFieldSet());
        raw.fs.add("CommitValue", Fields.bytesToHex(commitVal));
        raw.fs.add("Source", anSource.getFieldSet());
        return raw;

    }

    public String getMessageName() {
        return messageName;
    }

    public int hopsToLive() {
        return hopsToLive;
    }

    public void decHopsToLive() {
        this.hopsToLive--;
    }

    public void incDepth() {
        this.depth++;
    }

    public NodeReference getSource() {
        return anSource;
    }

    public void setSource(NodeReference nr) {
        this.anSource = nr;
    }
    
    public void setCommitVal(byte[] commitVal) {
        this.commitVal = commitVal;
    }

    public State getInitialState() {
        return new NewAnnouncement(id, announcee, 
                                   depth + 1, hopsToLive - 1, commitVal);
    }
}


