package freenet.node.states.request;

import freenet.*;
import freenet.fs.dir.BufferException;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.message.*;
import freenet.node.states.data.*;
import freenet.support.Logger;
import freenet.support.io.*;
import java.io.*;
import java.util.Enumeration;

/**
 * This is the abstract superclass for states pertaining to Insert and
 * Data Requests that are being processed.
 */
public abstract class Pending extends RequestState {

    /** the auxiliary state that runs the data receive */
    ReceiveData receivingData;
    
    /** the auxiliary state that runs the data send */
    SendData sendingData;

    /** may need to cache it here */
    DataReceived dataReceived;

    /** may need to cache it here */
    DataSent dataSent;
    
    /** may need to cache it here */
    StoreData storeData;

    /** got Accepted yet? */
    boolean accepted = false;

    /** Time marked when routing so that we can register time taken for diag */
    long routedTime = -1;
    
    /** Pending states may be created from scratch.
      */
    Pending(long id, int htl, Key key, Peer orig, FeedbackToken ft, RequestInitiator ri) {
        super(id, htl, key, orig, ft, ri);
    }

    /** We don't retain the state variables above
      * in a transition back to Pending (a restart).
      */
    Pending(RequestState ancestor) {
        super(ancestor);
    }
    
    /** State implementation.
      */
    public void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostRequestState", 1);
        fail(n, "Request state lost due to overflow");
    }
    
    /** @return  A request Message that could be routed upstream
      *          (note that unrecognized fields are lost)
      */
    abstract Request createRequest(FieldSet otherFields, NodeReference ref);

    
    //=== message handling =====================================================

    void receivedStoreData(Node n, StoreData sd) throws BadStateException {
        if (!fromLastPeer(sd)) {
            throw new BadStateException("StoreData from the wrong peer!");
        }
        // just hold on to it, when the Pending state is ready to finish, it
        // can check for it and go to AwaitingStoreData if it's still null
        storeData = sd;
    }
    
    void receivedQueryRestarted(Node n, QueryRestarted qr) throws BadStateException,
                                                                  RequestAbortException {

        routedTime = -1; // this time is no longer meaningful

        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRestarted from the wrong peer!");
        }
        cancelRestart();
        long timeout = Core.hopTime(hopsToLive);
        relayRestarted(n, timeout);
        scheduleRestart(n, timeout);
    }

    /**
     * Note the EndOfRouteException and RequestAbortException.
     * Must be called by subclass as part of the implementation
     * for receivedMessage(Node n, QueryRejected qr)
     */
    void receivedQueryRejected(Node n, QueryRejected qr) throws BadStateException,
                                                                RequestAbortException,
                                                                EndOfRouteException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRejected from the wrong peer!");
        }

        ++rejected;

        // reduce our HTL like the man sez, but by at least 1
        hopsToLive = (int) Math.min(hopsToLive-1, qr.hopsToLive);
        
        routes.queryRejected(qr.attenuation);

        // FIXME - do we want to do a QueryRestarted if the sanity check fails?
        
        cancelRestart();
        searchData(n);  // check for data in case of
                        // a parallel request succeeding
        sendOn(n, createRequest(qr.otherFields, n.myRef));
    }
                                    
    /**
     * Note the EndOfRouteException and RequestAbortException.
     * Must be called by subclass as part of the implementation
     * for receivedMessage(Node n, *RequestInitiator ri)
     */
    void receivedRequestInitiator(Node n, RequestInitiator ri) throws BadStateException,
                                                                      RequestAbortException,
                                                                      EndOfRouteException {        
        // it must be my own RequestInitiator
        if (this.ri == null || this.ri != ri) {
            throw new BadStateException("Not my request initiator: "+ri);
        }

        // is this the first time?
        if (routes == null) {
            n.logger.log(this, "Starting Pending chain " + 
                         Long.toHexString(id), Logger.DEBUG);
            // check for data in cache
            searchData(n);
            routes = n.rt.route(searchKey);
        }
        else {

            ++restarted;
            
            n.logger.log(this, "Restarting Pending chain " + 
                            Long.toHexString(id), Logger.DEBUG);
            
            // decrease contact probability for node that didn't reply
            if (lastPeer != null)
                routes.timedOut();

            // Don't allow indefinite restarts.
            //
            // Note:
            // sendOn will throw an EndOfRouteException below
            // if hopsToLive hits 0.

            if (--hopsToLive > 0) {
                // send QueryRestarted to initiating chain
                relayRestarted(n, Core.hopTime(hopsToLive + 1));
                // +1 for Accepted
            }

            // check for data in cache in case of a parallel request
            searchData(n);
        }
        
        // note the null; unknown fields are not restored after 
        // restarts since they are not saved. This means that 
        // Requests should actually not carry unknown fields... 
        sendOn(n, createRequest(null, n.myRef));
    }

    /**
     * Because method selection on superclasses doesn't work,
     * implementation must call this from a method with the actual
     * message (ie, receivedMessage(Node n, DataRequest dr)
     */
    void receivedRequest(Node n, Request r) {
        // This is a loop, no real problem here.
        n.logger.log(this, "Backtracking", Logger.DEBUG);
        try {
            n.sendMessage(
                new QueryRejected(id, hopsToLive, "Looped request", r.otherFields),
                r.getSource(), 0
            );
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to send backtrack reply: "+e, Logger.MINOR);
        }
        r.drop(n);
    }

    void receivedAccepted(Node n, Accepted a) throws BadStateException {
        if (!fromLastPeer(a)) {
            throw new BadStateException("Accepted from the wrong peer!");
        }
        if (!accepted) {
            scheduleRestart(n, Core.hopTime(hopsToLive));
            routedTime = System.currentTimeMillis();
            accepted = true;
            routes.routeAccepted();
        }
    }

    State receivedDataReply(Node n, DataReply dr) throws BadStateException {
        if (!fromLastPeer(dr)) {
            throw new BadStateException("DataReply from wrong peer!");
        }

        accepted = true;  // more or less..
        routes.routeAccepted();
        cancelRestart();
        
        try {
            receivingData = dr.cacheData(n, searchKey);
            n.ft.remove(searchKey); // remove if in FT.
        }
        catch (DataNotValidIOException e) {
            dr.drop(n);
            n.logger.log(this,
                "Got DNV: "+Presentation.getCBdescription(e.getCode()),
                Logger.MINOR);
            scheduleRestart(n, 0); // schedule restart now
            return this;           // back to pending
        }
        catch (KeyCollisionException e) {
            // oh well, try to go to SendingReply
            dr.drop(n);
            try {
                searchData(n);
            }
            catch (RequestAbortException rae) {
                return rae.state;
            }
            // damn, already gone
            fail(n, "I/O error replying with data");
            n.logger.log(this,
                "Failed to find data after key collision while caching DataReply",
                Logger.NORMAL);
            return new RequestDone(this);
        }
        catch (IOException e) {
            dr.drop(n);
            fail(n, "I/O error replying with data");
            n.logger.log(this, "I/O error caching DataReply", e, Logger.ERROR);
            return new RequestDone(this);
        }
        
        try {
            KeyInputStream kin = receivingData.getKeyInputStream();
            boolean worked = false;
            try {
                sendingData = sendData(n, kin);
                sendingData.schedule(n);
                worked = true;
            }
            finally {
                if (!worked)
                    kin.close();
            }
        }
        catch (IOException e) {
            // couldn't get the KeyInputStream
            fail(n, "I/O error replying with data");
            if (e instanceof BufferException) {
                n.logger.log(this, "Failed to get KeyInputStream: "+e,
                             Logger.MINOR);
            }
            else {
                n.logger.log(this, "I/O error getting KeyInputStream",
                             e, Logger.ERROR);
            }
            return new ReceivingReply(this);
        }
        catch (CommunicationException e) {
            n.logger.log(this, "Error replying to peer: "+e, Logger.MINOR);
            return new ReceivingReply(this);
        }
        finally {
            receivingData.schedule(n);
        }

        return new TransferReply(this);
    }

    
    //=== support methods ======================================================

    private void sendOn(Node n, Request r) throws EndOfRouteException,
                                                  RequestAbortException {

        if (hopsToLive <= 0)
            throw new EndOfRouteException("Reached last node in route");
        
        Peer addr = null;
        NodeReference nr;

        int attemptCount = 0;
        while (true) { // until the send doesn't fail
            nr = routes.getNextRoute();
            if (nr == null) {
                fail(n, "No route found", r.otherFields);
                throw new RequestAbortException(new RequestDone(this));
            }
            addr = n.getPeer(nr);
            if (addr == null) {  // bad node ref
                // or, one we just don't have a transport for?
                //n.logger.log(this,
                //    "Found bad node ref while routing, removing: "+nr,
                //    Logger.MINOR);
                //n.rt.dereference(nr.getIdentity());
                continue;
            }
            if (origPeer != null && origPeer.equalsIdent(addr)) {
                // funnily enough, this happens quite often
                // and results in some pretty dumb "loops"
                continue;
            }
            n.logger.log(this, "Forwarding query to "+addr, Logger.DEBUG);
            try {
                attemptCount++;
                n.sendMessage(r, addr, n.routeConnectTimeout);
                break;
            }
            catch (CommunicationException e) {
                ++unreachable;
                // don't care if it's terminal or nonterminal
                // because routing is too time-critical
                n.logger.log(this, "Routing failure to: "+e.peerAddress()
                                   + " -- " + e, Logger.DEBUG);
                             //e, Logger.DEBUG);
                if (e instanceof AuthenticationFailedException) {
                    routes.routeConnected();
                    routes.authFailed();
                }
                else {
                    routes.connectFailed();
                }
            }
        }

        // Count outbound Requests.
        n.diagnostics.occurrenceBinomial("outboundAggregateRequests", attemptCount, 1);
        if (n.outboundRequests != null) {
            n.outboundRequests.incTotal(addr.getAddress().toString());
        }

        // Keep track of outbound requests for rate limiting.
        n.outboundRequestLimit.inc();

        lastPeer = addr;
        routes.routeConnected();

        scheduleRestart(n, Core.hopTime(1));  // timeout to get the Accepted
    }

    private final void relayRestarted(Node n, long timeout) throws RequestAbortException {
        try {
            ft.restarted(n, timeout);
        }
        catch (CommunicationException e) {
            n.logger.log(this,
                         "Couldn't restart because relaying QueryRestarted failed: "+e,
                         Logger.MINOR);
            throw new RequestAbortException(new RequestDone(this));
        }
    }

    /** Given an existing KeyInputStream, sets up a SendData state to
      * transfer the data back to the requester. 
      */
    SendData sendData(Node n, KeyInputStream doc) throws CommunicationException {
        Storables storables = doc.getStorables();
        OutputStream out = ft.dataFound(n, storables, doc.length());
        // this means the initiator is not interested in seeing the 
        // data (e.g., KeyCollision response in FCP)
        if (out == null) out = new NullOutputStream();

        // FIXME: don't waste resources piping data to the NullOutputStream
        
        return new SendData(n.randSource.nextLong(), this.id, out,
                            doc, doc.length(), storables.getPartSize());
    }
                                                  
    /** Attempts to retrieve the key from the cache and transition to
      * SendingReply.  Will transition to null in the event of an
      * unrecoverable error.  Does nothing if the key is not found.
      */
    void searchData(Node n) throws RequestAbortException {
        KeyInputStream doc = null;
        try {
            doc = n.ds.getData(searchKey);
            if (doc != null) {
                sendingData = sendData(n, doc);
                sendingData.schedule(n);
                doc = null;
                throw new RequestAbortException(new SendingReply(this));
            }
        }
        catch (IOException e) {
            fail(n, "I/O error replying with data");
            n.logger.log(this, "I/O error replying with data",
                         e, Logger.MINOR);
            throw new RequestAbortException(new RequestDone(this));
        }
        catch (CommunicationException e) {
            n.logger.log(this, "Error replying to peer: "+e, Logger.MINOR);
            throw new RequestAbortException(new RequestDone(this));
        }
        finally {
            if (doc != null) {
                try {
                    doc.close();
                }
                catch (IOException e) {
                    n.logger.log(this,
                        "Failed to close KeyInputStream after failing",
                        e, Logger.MINOR);
                }
            }
        }
    }
}









