package freenet.support;

import java.util.*;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Implements a thread-safe Blocking Stack. A BlockingStack
 * is a LIFO (Last In First Out) buffer
 * with the property that calls to pop() on an empty stack will block
 * until an element becomes available to pop.
 *
 * @author Scott G. Miller <scgmille@indiana.edu>
 */
public class BlockingStack {
    private Stack stack;

    /**
     * Construct an empty BlockingStack.
     *
     *
     */
    
    public BlockingStack() {
	stack=new Stack();
    }

    /**
     * Pushes an object onto this stack.
     *
     * @param o the object to push
     */
    public void push(Object o) {
	synchronized(stack) {
	    stack.push(o);
	    stack.notify();
	}
    }

    /**
     * Pops an object from this BlockingStack.  This method will return the
     * top element in this BlockingStack, or block until one is available.
     *
     * @return the object on the top of this BlockingStack
     * @throws InterruptedException if this thread is blocked in this method 
     * and something interrupts this thread.
     */
    public Object pop() throws InterruptedException {
        synchronized(stack) {
            while (stack.isEmpty()) stack.wait();
            return stack.pop();
        }
    }

    /**
     * Returns the empty status of this BlockingStack.
     *
     * @return true if this BlockingStack is empty, false otherwise.
     */
    
    public boolean isEmpty() {
	return stack.isEmpty();
    }
    
    /**
     * Returns the number of objects in this BlockingStack.
     *
     * @return the number of objects in this BlockingStack.
     */

    public int size() {
	return stack.size();
    }
}
    
