package freenet.support.io;
import freenet.support.Logger;
import freenet.Core;
import java.io.*;

public class Bandwidth {

     private static final int ticksPerSecond = 10;
     private static final int millisPerTick = 1000 / ticksPerSecond;

     protected int bandwidthPerTick; /* quanity to increment every tick */
     private int available;          /* how much is availble right now */
     private long moreBandwidthTime; /* when more bandwidth is available */

     /**
      * Sets the upper bandwidth limit, in multiples of 10Bps (Bytes, not bits),
      * for all ThrottledOutputStreams. This class will treat any bandwidth
      * under 100Bps as equals to 0Bps! A setting of 0 or less will turn
      * bandwidth limiting off for all ThrottledOutputStreams, and prevent
      * new ones from being created.
      *
      * @param bandwidth the upper bandwidth limit in multiples of 10Bps.
      */
    
     public Bandwidth(int bandwidth) {
         bandwidthPerTick = bandwidth / ticksPerSecond;
     }

    synchronized protected void chargeBandwidth(int used) {
        do {
            waitForBandwidth();
            int result = Math.min(used, available);
            available -= result;
            used -= result;
        } while(used > 0);
    }
    
    synchronized protected int getBandwidth(int desired) {
        // Core.logger.log(this,
        //        desired + " bytes desired", Logger.DEBUG);
        waitForBandwidth();
        int result = Math.min(desired, available);
        available -= result;
        return result;
    }
    
    // Too bad "void" is such a short word.  (-:
    synchronized protected void waitForBandwidth() {
         for (;;) {
             long now = System.currentTimeMillis();
             refillAvailableBandwidth(now);
             if (available != 0)
                 break;
             try {
                 Thread.currentThread().sleep(moreBandwidthTime - now);
             } catch (InterruptedException e) {}
         }
     }
     
    synchronized protected void refillAvailableBandwidth(long now) {
         if (now >= moreBandwidthTime) {
             available = bandwidthPerTick;
             moreBandwidthTime = now + millisPerTick;
         }
     }
}
