package freenet.transport;

import freenet.*;
import java.util.StringTokenizer;
import java.io.IOException;
import java.net.*;

/** A tcpTransport is any Transport based on Sockets.  Subclasses implementing
  * this provide their own socket factories so that there can be a choice
  * between plain TCP, SSL, etc.
  */
abstract class tcpTransport implements Transport {

    private final InetAddress bindAddr;
    private final int preference;
    private final int designator;

    /** Starts a TCP transport object.
      */
    tcpTransport(int preference) {
        this(null, -1, preference);
    }

    /** Starts a TCP transport object that only listens on one local 
      * interface.
      */
    tcpTransport(InetAddress bindAddr, int preference) {
        this(bindAddr, -1, preference);
    }

    /** Starts a TCP transport object that only listens on one local 
      * interface.
      * @param  designator A session protocol designator to put first 
      *                    on incoming streams,
      *                    so that protocols without designator numbers
      *                    work within Freenet's scheme.
      *                    If < 0 , it will not be used.
      */
    tcpTransport(InetAddress bindAddr, int designator, int preference) {
        this.bindAddr   = bindAddr;
        this.designator = designator;
        this.preference = preference;
    }

    public final int preference() {
        return preference;
    }

    public Address getAddress(String s) throws BadAddressException {
        StringTokenizer st = new StringTokenizer(s, ":");
        if (st.countTokens() != 2) {
            throw new BadAddressException("Bad tcp address: "+s);
        }
        try {
            return new tcpAddress(this, st.nextToken(),
                                  Integer.parseInt(st.nextToken()));
            
        } catch (NumberFormatException e) {
            throw new BadAddressException("Illegal port value: "+s);
        }
    }

    public ListeningAddress getListeningAddress(String s) throws BadAddressException {
        try {
            return new tcpListeningAddress(this, bindAddr, designator,
                                           Integer.parseInt(s));
        } catch (NumberFormatException e) {
            throw new BadAddressException("Illegal port value: "+s);
        }
    }

    public final boolean equals(Object o) {
        return o instanceof tcpTransport && equals((tcpTransport) o);
    }

    public final boolean equals(tcpTransport o) {
        return getClass().equals(o.getClass())
            && preference == o.preference
            && (bindAddr == null && o.bindAddr == null
                || bindAddr != null && o.bindAddr != null && bindAddr.equals(o.bindAddr));
    }


    // implementation methods
    
    public abstract String getName();

    abstract tcpSocketFactory getSocketFactory();

    abstract tcpServerSocketFactory getServerSocketFactory();

    /* Keep out of compile...
    public static void main(String[] args) {
        SSL ssl = new SSL(10);
        TCP tcp = new TCP(10);
        System.out.println("TCP equals TCP: "+tcp.equals(tcp));
        System.out.println("TCP equals SSL: "+tcp.equals(ssl));
        System.out.println("SSL equals SSL: "+ssl.equals(ssl));
        System.out.println("SSL equals TCP: "+ssl.equals(tcp));
    }
    */
}


