/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  vscan.c -- Called from localcache.c if virus scanning is enabled. We
  have two phases of operation: INCOMING and OUTGOING.

  When RETR is received it is forwarded to the server and we enter
  INCOMING mode. All incoming data is written to a temporary file, and
  the buffer length zeroed so it doesn't get written to either cache
  or client. The server's 150 reply is intercepted, and a multiline
  150 reply started instead - a line at a time every few seconds to
  prevent timeouts.

  On data connection close during INCOMING we scan the temporary file. 
  If infected we send an error and return -1. If clean we switch to
  OUTGOING mode, and reopen the file for reading. This fd is returned
  and will become the new server_data fd.

  During the OUTGOING phase we do nothing. The data read from our
  temporary file will be sent to both client and cache file. On close
  we delete the temporary file.

problems:
  o Uploads not scanned
  o Sensitive to order of calls in l_retr_end and l_inc_data.
  o The file is written to disk on two occasions.

TODO Modify localcache.c to delete cache file header on failed scan.
  ***************************************/
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>

#include "common.h"
#include "control.h"
#include "cache.h"
#include "vscan.h"

static char *argv[20];

static enum { NONE, INCOMING, OUTGOING } status = NONE;
static enum { STARTING, FINISHED, NOTHING} expected_reply = NOTHING;
static int size;
static int tsize;
static int fd = -1;
static char scanfile[BUF_LEN];
static time_t lastprog;

int vscan_scan(void);

/*This function run as root to allow making tmp dir*/
int vscan_init(void)
{
	int i;
	char *p=config.vscanner;
	char name[512];
	struct stat tmp;
	
	if (!config.vscanner)
		return 0;

	for(i=0;i<19;i++) {
		while(*p!=0 && *p++!='"');
		if(*p==0) break;
		argv[i]=p;
		while(*p!=0 && *p!='"') p++;
		if(*p==0) break;
		*p++=0;
		if(!strcmp(argv[i], "%s")) argv[i]=scanfile;
	}
	argv[i]=NULL;

	sprintf(name, "%s/tmp", config.chroot);
	if(stat(name, &tmp)==-1) {
		if (mkdir(name, S_IRWXU) == -1) {
			write_log(ERROR, "Unable to make vscan dir %s",
				  name);
			return (-1);
		}
		chown(name, config.uid, config.gid);
	}	
	snprintf(scanfile, BUF_LEN, "%s/tmp/VS_%d", config.chroot, getpid());
	debug2("VS: Virus scanner temp file is %s\n", scanfile);
	return 0;
}

void vscan_new(int sz)
{
	if (!config.vscanner)
		return;
	fd = creat(scanfile, S_IRUSR | S_IWUSR);
	status = INCOMING;
	expected_reply = STARTING;
	time(&lastprog);
	size = sz;
	tsize = 0;
	debug("VS: Downloading to temporary file\n");
}

void vscan_inc(sstr * inc)
{
	time_t tmp;

	if (!config.vscanner)
		return;
	if (status == INCOMING) {
		tsize += sstr_len(inc);
		sstr_write(fd, inc, 0);
		sstr_empty(inc);
		time(&tmp);
		if (config.vscanpm && tmp - lastprog > config.vscanpm && 
		    expected_reply!=STARTING) {
			sstr *msg;
			msg = sstr_init(500);
			sstr_apprintf(msg,
				      "150-Downloaded %d/%d bytes to proxy",
				      tsize, size);
			send_message(0, msg);
			sstr_free(msg);
			lastprog = tmp;
		}
	}
}

int vscan_switchover(void)
{
	int tmp;
	
	if(status != INCOMING) return FALSE;
	
	close(fd);
	fd = -1;

	status = OUTGOING;
	if (!vscan_scan()) {
		debug("VS: Scan failed");
		if(config.vscanpm) send_cmessage(150, "Not starting Transfer");
		send_cmessage(451, "File contains virus. Aborting");
		unlink(scanfile);
		status = NONE;
		return FALSE;
	}
	debug("VS: Scan complete. Changing fd\n");
	send_cmessage(150, "Starting Transfer");
	tmp = open(scanfile, O_RDONLY);
	unlink(scanfile);
	if(dup2(tmp, info->server_data.fd)==-1) {
		debug_perr("dup2");
		quit(NO_MESSAGE);
	}
	close(tmp);
	return TRUE;
}

int vscan_end(void)
{
	if(status == INCOMING) {
		debug_err("In vscan_end() and shouldn't be");
		quit(NO_MESSAGE);
	}
	if (status == OUTGOING) {
		status = NONE;
		debug("VS: Finished forwarding scanned file\n");
		send_cmessage(226, "Transfer Complete");
		return (VSCAN_OK);
	}
	return (VSCAN_OK);
}

int vscan_parsed_reply(int code, sstr * msg)
{
	switch (expected_reply) {
	case NOTHING:
		return (FALSE);
	case STARTING:
		if (code <= 0)
			return (TRUE);
		if(code >299) { /*Failure*/
			expected_reply = NOTHING;
			status = NONE;
			close(fd);
			return(FALSE);
		}
		if(config.vscanpm) {
			send_cmessage(-150, "Starting Transfer");
			send_cmessage(0, "150-There'll be a delay while we "
				      "scan for viruses");
		}
		expected_reply = FINISHED;
		return (TRUE);
	case FINISHED:
		if (code <= 0)
			return (TRUE);
		expected_reply=NOTHING;
		if (code > 299) { /*Failure*/
			status = NONE;
			close(fd);
			if(config.vscanpm)
				send_cmessage(150, "Error. Aborting.");
			return(FALSE);
		}
		return (TRUE);
	}

	if (status == INCOMING)
		return (TRUE);
	return (FALSE);
}

int vscan_scan(void)
{
	int i;

	debug("VS: Now scanning file");
	if(config.vscanpm)
		send_cmessage(0, "150-Scanning file for viruses");

	switch(fork()){
	case 0: /*Child*/
		execvp(argv[0], argv);
		write_log(ERROR, "Failed to exec virus scanner");
		quit(NO_MESSAGE);
	case -1:
		debug_err("Error");
		break; /*FIXME*/
	default:
	}
	wait(&i);
	if(!WIFEXITED(i)) return(FALSE);

	return( WEXITSTATUS(i) == config.vscanok );
}
