
/*
    FUNIONFS: UNIONFS over FUSE Usermode filesystem
    Copyright (C) 2005-2006  Stephane APIOU <stephane.apiou@free.fr>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fuse.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <dirent.h>

#include "funionfs.h"

extern f_opt_t f_opt;
extern FILE *log_fd;

struct unionfs_desc *unionfs_list;

extern char initial_working_dir[PATH_MAX + 1];

f_cbuf_t f_cbuf = { "", 0, 0 };

int
funionfs_relist(void)
{
	int ret;

	if (*f_opt.firstdir != '\0')
	{
		if (strcmp(f_opt.firstdir, "NONE")
		    && strcmp(f_opt.firstdir, "none")
		    && strcmp(f_opt.firstdir, "None"))
		{
			int fd;
			char *s, *firstdir;
			struct unionfs_desc *walk;

			if (*f_opt.firstdir == '/')
				firstdir = strdup(f_opt.firstdir);
			else
				firstdir =
					rel2abspath(f_opt.firstdir,
						    initial_working_dir);
			DEBUG_PRINTF("firstdir=%s\n", firstdir);
			s = (char *) malloc(strlen(firstdir) +
					    strlen(CONTROL_FILE) + 2);
			strcpy(s, firstdir);
			strcat(s, "/");
			strcat(s, CONTROL_FILE);
			DEBUG_PRINTF("s=%s\n", s);
			free(firstdir);
			if (!s)
				return -ENOMEM;
			ret = unlink(s);
			if ((ret < 0) && (errno != ENOENT))
			{
				free(s);
				return -errno;
			}
			fd = open(s, O_CREAT | O_EXCL | O_WRONLY, 0600);
			free(s);
			if (fd < 0)
				return -errno;
			DEBUG_PRINTF("Starting to walk list...\n");
			walk = unionfs_list;
			while (walk)
			{
				ret = write(fd, walk->path, strlen(walk->path));
				if (ret < 0)
					return -errno;
				if (walk->ro)
					ret = write(fd, "=ro", 3);
				else
					ret = write(fd, "=rw", 3);
				if (ret < 0)
					return -errno;
				ret = write(fd, "\n", 1);
				if (ret < 0)
					return -errno;
				walk = walk->pnext;
			}
			close(fd);
		}
	}
	return 0;

}

static int
command_is(const char *text, const char *word)
{
	int l;

	l = strlen(word);
	return !strncmp(text, word, l) && (isspace(text[l]) || !text[l]);
}

static int
funionfs_execute(const char *command)
{
	int ret;

	if (!*command)
		return 0;

	LOG("execute %s\n", command);

	if (command_is(command, "relist"))
		return funionfs_relist();

	if (command_is(command, "umount"))
	{
		if (!fuse_get_context()->uid
		    || (fuse_get_context()->uid == getuid())
		    || (fuse_get_context()->uid == geteuid()))
		{
			if (*f_opt.mountpoint == '/')
				fuse_unmount(f_opt.mountpoint);
			else
				fuse_unmount(rel2abspath
					     (f_opt.mountpoint,
					      initial_working_dir));
		}
		return -1;
	}
	if (command_is(command, "add_ro") && command[6])
	{
		ret = funionfs_addpath(command + 7, 1);
		if (ret != 1)
			return ret;
		return funionfs_relist();
	}
	if (command_is(command, "add_rw") && command[6])
	{
		ret = funionfs_addpath(command + 7, 0);
		if (ret != 1)
			return ret;
		return funionfs_relist();
	}
	if (command_is(command, "make_ro") && command[7])
	{
		struct unionfs_desc *walk;

		if (command[8] != '/')
			return -ENOENT;

		walk = unionfs_list;
		while (walk)
		{
			if (!strcmp(command + 8, walk->path))
				walk->ro = 1;
			walk = walk->pnext;
		}
		return funionfs_relist();
	}
	return 0;
}

int
add_to_control_buffer(f_cbuf_t * to, const char *buf, size_t size)
{
	LOG("add_to_control_buf:\n");
	if (to->length + size >= CONTROL_BUFFER_SIZE)
		return -ENOSPC;
	{
		unsigned i, j;

		for (i = 0; i < size; i++)
		{
			j = (to->start + to->length + i) % CONTROL_BUFFER_SIZE;
			to->data[j] = buf[i];
			LOG("%c", buf[i]);
		}
		j = (to->start + to->length + size) % CONTROL_BUFFER_SIZE;
		to->data[j] = 0;
		LOG("\nend add_to_control_buf\n");
	}
	to->length += size;
	return 0;
}

void
flush_control_buffer(f_cbuf_t * what)
{
	what->start = 0;
	what->length = 0;
	what->data[0] = 0;
}

char *
take_control_buffer_line(f_cbuf_t * from)
{
	char *head, *tail;
	long n;

	n = 0;
	head = from->data + from->start;
	tail = head;
	while (*tail != 0 && *tail != 10)
	{
		tail++;
		n++;
		if (tail - from->data >= CONTROL_BUFFER_SIZE)
			tail = from->data;
	}
	*tail = 0;
	from->length -= n + 1;
	if (from->length < 0)
	{
		from->length = 0;
		from->data[from->start] = 0;
	}
	from->start = (from->start + n + 1) % CONTROL_BUFFER_SIZE;
	return strdup(head);
}

static int
is_control_buffer_line(f_cbuf_t * from)
{
	int n;

	n = 0;
	while ((from->data[(from->start + n) % CONTROL_BUFFER_SIZE] != 10)
	       && (from->data[(from->start + n) % CONTROL_BUFFER_SIZE] != 10)
	       && (n < CONTROL_BUFFER_SIZE))
		n++;
	return (from->data[(from->start + n) % CONTROL_BUFFER_SIZE] == 10);
}


int
funionfs_control_write(const char *buf, size_t size)
{
	int ret;
	char *s;

	ret = add_to_control_buffer(&f_cbuf, buf, size);
	if (ret)
		return ret;
	while (is_control_buffer_line(&f_cbuf))
	{
		s = take_control_buffer_line(&f_cbuf);
		funionfs_execute(s);
		free(s);
	}
	return size;
}
