/* 

                          Firewall Builder

                 Copyright (C) 2004 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: instDialog.cpp,v 1.65.4.4 2005/09/13 04:55:41 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "../../config.h"
#include "global.h"
#include "utils.h"

#include "instDialog.h"
#include "instConf.h"
#include "FWBSettings.h"
#include "SSHUnx.h"
#include "SSHPIX.h"

#include <qcheckbox.h>
#include <qlineedit.h>
#include <qtextbrowser.h>
#include <qtextedit.h>
#include <qtimer.h>
#include <qfiledialog.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qprogressbar.h>
#include <qprocess.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qspinbox.h>

#include "fwbuilder/Resources.h"
#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/XMLTools.h"
#include "fwbuilder/Policy.h"
#include "fwbuilder/InterfacePolicy.h"
#include "fwbuilder/NAT.h"
#include "fwbuilder/Interface.h"

#ifndef _WIN32
#  include <unistd.h>     // for access(2) and getdomainname
#endif

#include <errno.h>
#include <iostream>

using namespace std;
using namespace libfwbuilder;




instDialog::instDialog(instConf *_c)
{
    ready=false;
    cnf=_c;
    activationCommandDone=false;

    progressDisplay->setTextFormat(Qt::RichText);

    fwb_prompt="--**--**--";
    session = NULL;

    pwd->setEchoMode( QLineEdit::Password );
    epwd->setEchoMode( QLineEdit::Password );

    setHelpEnabled( page(0), false );
    setHelpEnabled( page(1), false );

    if (cnf->fwobj!=NULL)
    {
        QString fwobjname=cnf->fwobj->getName().c_str();
        setTitle( page(0), title( page(0) ).arg(fwobjname) );
        setTitle( page(1), title( page(1) ).arg(fwobjname) );
    }

    cnf->incremental=st->readBoolEntry("/FirewallBuilder2/Installer/incr"    );
    cnf->save_diff  =st->readBoolEntry("/FirewallBuilder2/Installer/savediff");
    cnf->saveStandby=st->readBoolEntry("/FirewallBuilder2/Installer/saveStandby");
    cnf->dry_run    =st->readBoolEntry("/FirewallBuilder2/Installer/dryrun"  );
    cnf->quiet      =st->readBoolEntry("/FirewallBuilder2/Installer/quiet"   );
    cnf->verbose    =st->readBoolEntry("/FirewallBuilder2/Installer/verbose" );
    cnf->stripComments=st->readBoolEntry("/FirewallBuilder2/Installer/stripComments" );
    cnf->compressScript=st->readBoolEntry("/FirewallBuilder2/Installer/compressScript" );
    cnf->testRun     =st->readBoolEntry("/FirewallBuilder2/Installer/testRun" );
    cnf->rollback    =st->readBoolEntry("/FirewallBuilder2/Installer/rollback" );
    cnf->rollbackTime=st->readNumEntry("/FirewallBuilder2/Installer/rollbackTime");

/* TODO: set cnf->pgm to ssh path here */

    QString platform=cnf->fwobj->getStr("platform").c_str();

    cnf->rollbackTimeUnit=
        Resources::getTargetOptionStr(cnf->fwobj->getStr("host_OS"),
                                      "activation/timeout_units").c_str();

    if (platform!="pix" && platform!="fwsm")
    {
        incr->hide();
        test->hide();
        saveDiff->hide();
        saveStandby->hide();
        backupConfigFile->hide();
        backupConfigFileLbl->hide();
        epwd->hide();
        epwdLbl->hide();
    } else
    {
        incr->show();
        test->show();
        saveDiff->show();
        saveStandby->show();
        backupConfigFile->show();
        backupConfigFileLbl->show();
        epwd->show();
        epwdLbl->show();
    }

/* hide anyway, diff does not work for pix 6.3(3) */
    saveDiff->hide();

    progressBar->hide();
    stripComments->hide();
    compressScript->hide();

    if (platform=="pix" || platform=="fwsm")
    {
        progressBar->show();
        stripComments->show();
    }

    if (cnf->fwobj->getStr("host_OS")=="linksys")
    {
        compressScript->show();
    }


    uname->setFocus();
    uname->setText( cnf->user );
    incr->setChecked( cnf->incremental );
    test->setChecked( cnf->dry_run );
    backupConfigFile->setText( cnf->backup_file );
    saveDiff->setChecked( cnf->save_diff );
    saveStandby->setChecked( cnf->saveStandby );
    altAddress->setText( cnf->maddr );
    quiet->setChecked( cnf->quiet );
    verbose->setChecked( cnf->verbose );
    stripComments->setChecked( cnf->stripComments );
    compressScript->setChecked( cnf->compressScript );
    testRun->setChecked( cnf->testRun );
    rollback->setChecked( cnf->rollback );
    rollbackTime->setValue( cnf->rollbackTime );

    testRunRequested();
    

/* we initialize these in FWBSettings constructor on Unix, but do not
 * do it on Windows since there is no standard ssh package there. User
 * is supposed to fill these in in the Preferences dialog, otherwise
 * they can't use installer
 */

    ssh=st->getSSHPath();
    scp=st->getSCPPath();

/* as of v2.0.3, build 437, incremental install actually installs only
 * ACL and nat commands on PIX. It does not use fwb_pix_diff so there
 *  is no need to disable it anymore
 *
    if ( access(cnf->diff_pgm.latin1(), F_OK|X_OK)!=0 )
    {
        cerr << "could not access " << cnf->diff_pgm << endl;

        incr->setChecked(false);
        incr->setEnabled(false);
        saveDiff->setChecked(false);
        saveDiff->setEnabled(false);
    }
*/

    try
    {
        if (cnf->fwobj!=NULL && ! cnf->fwbfile.isEmpty())
        {
            cnf->maddr = cnf->fwobj->getManagementAddress().toString().c_str();
//            altAddress->setText( cnf->maddr );
        }

        

        setReady(true);

    } catch(FWException &ex)
    {
        setReady(false);
        showPage( page(1) );
        progressDisplay->append( ex.toString().c_str() );
    } catch (std::string s) {
        setReady(false);
        showPage( page(1) );
        progressDisplay->append( s.c_str() );
    } catch (std::exception ex) {
        setReady(false);
        showPage( page(1) );
        progressDisplay->append( ex.what() );
    } catch (...) {
        setReady(false);
        showPage( page(1) );
        progressDisplay->append( QObject::tr("Unsupported exception") );
    }
}

instDialog::~instDialog()
{

    st->writeEntry("/FirewallBuilder2/Installer/incr",    cnf->incremental);
    st->writeEntry("/FirewallBuilder2/Installer/savediff",cnf->save_diff  );
    st->writeEntry("/FirewallBuilder2/Installer/saveStandby",cnf->saveStandby);
    st->writeEntry("/FirewallBuilder2/Installer/dryrun"  ,cnf->dry_run    );
    st->writeEntry("/FirewallBuilder2/Installer/quiet",   cnf->quiet      );
    st->writeEntry("/FirewallBuilder2/Installer/verbose", cnf->verbose    );
    st->writeEntry("/FirewallBuilder2/Installer/stripComments", cnf->stripComments);
    st->writeEntry("/FirewallBuilder2/Installer/compressScript", cnf->compressScript);
    st->writeEntry("/FirewallBuilder2/Installer/testRun",  cnf->testRun);
    st->writeEntry("/FirewallBuilder2/Installer/rollback", cnf->rollback);
    st->writeEntry("/FirewallBuilder2/Installer/rollbackTime", cnf->rollbackTime);
}

void instDialog::append(const QString &s)
{
    progressDisplay->moveCursor( QTextEdit::MoveEnd , false );
    progressDisplay->insert( s );
//    progressDisplay->append(s);
}

void instDialog::appendRich(const QString &s)
{
    progressDisplay->append(s);
//    progressDisplay->insertParagraph(s,-1);
}

QString instDialog::getUName() { return uname->text(); }
QString instDialog::getPWD()   { return pwd->text();   }
QString instDialog::getEPWD()  { return epwd->text();  }

void instDialog::summary()
{
    appendRich( "<hr>" + QObject::tr("<b>Summary:</b>") );
    appendRich( QObject::tr("* firewall name : %1")
            .arg(cnf->fwobj->getName().c_str()) );
    appendRich( QObject::tr("* user name : %1")
            .arg(cnf->user) );
    appendRich( QObject::tr("* management address : %1")
            .arg(cnf->maddr) );
    appendRich( QObject::tr("* platform : %1")
            .arg(cnf->fwobj->getStr("platform").c_str())  );
    appendRich( QObject::tr("* host OS : %1")
            .arg(cnf->fwobj->getStr("host_OS").c_str()) );
    appendRich( QObject::tr("* Loading configuration from file %1")
            .arg(cnf->fwbfile));

    if (cnf->incremental)
    {
        appendRich( QObject::tr("* Incremental install"));
    }
    if (cnf->save_diff && cnf->incremental)
    {
        appendRich(
QObject::tr("* Configuration diff will be saved in file %1").arg(cnf->diff_file));
    }
    if (cnf->dry_run)
    {
        appendRich(
QObject::tr("* Commands will not be executed on the firewall"));
    }
    appendRich("<hr>\n");
}



void instDialog::selected(const QString &title)
{
    int p = indexOf( currentPage() );

    if (p==0)
    {
        if (session!=NULL)
        {
            disconnect(session,SIGNAL(printStdout_sign(const QString&)),
                       this,SLOT(append(const QString&)));

            disconnect(session,SIGNAL(sessionFinished_sign()),
                       this,SLOT(processExited()));

            disconnect(session,SIGNAL(updateProgressBar_sign(int,bool)),
                       this,SLOT(updateProgressBar(int,bool)));

            session->terminate();

            delete session;
            session=NULL;
        }

        activationCommandDone=false;

        setFinishEnabled( page(1), false );
        return;
    }

    if (p==1 && ready)
    {
/* change of the page when flag ready is 'true' means we should start
 * operation */

        cnf->incremental = incr->isChecked();
        cnf->dry_run     = test->isChecked();
        cnf->backup_file = backupConfigFile->text();
        cnf->backup      = !cnf->backup_file.isEmpty();
        cnf->save_diff   = saveDiff->isChecked();
        cnf->saveStandby = saveStandby->isChecked();
	QString aaddr    = altAddress->text();
	if (!aaddr.isEmpty())
	{
/* alternative address can also be putty session name. In any case,
 * leave it up to ssh to resolve it and signal an error if it can't be
 * resolved ( Putty session name does not have to be in DNS at all ).
 */
            cnf->maddr = aaddr;
	} else
            cnf->maddr = cnf->fwobj->getManagementAddress().toString().c_str();

//        cnf->maddr         = altAddress->text();
        cnf->user          = uname->text();
        cnf->pwd           = pwd->text();
        cnf->epwd          = epwd->text();
        cnf->quiet         = quiet->isChecked();
        cnf->verbose       = verbose->isChecked();
        cnf->stripComments = stripComments->isChecked();
        cnf->compressScript= compressScript->isChecked();
        cnf->testRun       = testRun->isChecked();
        cnf->rollback      = rollback->isChecked();
        cnf->rollbackTime  = rollbackTime->value();

/* check for a common error when multiple interfaces are marked as
 * 'management'
 */
        int nmi = 0;
        list<FWObject*> ll = cnf->fwobj->getByType(Interface::TYPENAME);
        for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
        {
            Interface *intf = Interface::cast( *i );
            if (intf->isManagement()) nmi++;
        }
        if (nmi>1)
        {
	   appendRich(
		QObject::tr("Only one interface of the firewall '%1' must be marked as management interface.")
		.arg(cnf->fwobj->getName().c_str()).latin1() );
	    return;
        }
        if (nmi==0)
        {
	   appendRich(
		QObject::tr("One of the interfaces of the firewall '%1' must be marked as management interface.")
		.arg(cnf->fwobj->getName().c_str()).latin1() );
	    return;
        }
        if ((cnf->maddr == "" || cnf->maddr == "0.0.0.0") &&
            altAddress->text().isEmpty())
	{
	   appendRich(
               QObject::tr("Management interface does not have IP address, can not communicate with the firewall.") );
	    return;
	}

        confScript="";

/* read manifest from the conf file */

        QFile cf(  cnf->wdir+"/"+cnf->conffile );
        if (cf.open( IO_ReadOnly ) )
        {
            QTextStream stream(&cf);
            QString line;
            while (!stream.eof())
            {
                line    = stream.readLine();
                int pos = -1;
                if ( (pos=line.find(MANIFEST_MARKER))!=-1 )
                {
                    int n = pos + QString(MANIFEST_MARKER).length();
//                if (line[n]=='*')  confScript=line.mid(n+2);
//                else               confFiles.push_back( line.mid(n+2) );

                    confFiles.push_back( line.mid(n+2) );

                    if (fwbdebug)
                        qDebug("instDialog: adding %c %s",
                               line[n].latin1(),
                               line.mid(n+2).ascii());

                }
                line = "";
            }
            cf.close();
        } else
        {
            QMessageBox::critical(this, "Firewall Builder",
                 tr("File %1 not found.").arg(cnf->wdir+"/"+cnf->conffile),
                                  tr("&Continue") );
            return;
        }

//    if (confScript.isEmpty()) confScript=cnf->conffile;
        if (confFiles.size()==0) confFiles.push_back(cnf->conffile);

	progressDisplay->setText("");
        progressBar->setProgress(0);

        summary();

        QStringList  args;

        if (cnf->fwobj->getStr("platform")=="pix" || cnf->fwobj->getStr("platform")=="fwsm")
        {
#ifdef _WIN32
            args.push_back(ssh);

/*
 * putty ignores protocol and port specified in the session file if
 * command line option -ssh is given.
 *
 * On the other hand,the sign of session usage is an empty user name,
 * so we can check for that. If user name is empty, then putty will
 * use current Windows account name to log in to the firewall and this
 * is unlikely to work anyway. This seems to be a decent workaround.
 */
            if (!cnf->user.isEmpty() &&
                ssh.find("plink.exe")!=-1) args.push_back("-ssh");

            args.push_back("-pw");
            args.push_back(cnf->pwd);
#else
            args.push_back(argv0.c_str());
            args.push_back("-X");   // fwbuilder works as ssh wrapper
//            args.push_back("-d");
            args.push_back("-t");
            args.push_back("-t");
#endif
            if (!cnf->sshArgs.isEmpty())
                args += QStringList::split(" ",cnf->sshArgs);

            if (cnf->verbose) args.push_back("-v");
            if (!cnf->user.isEmpty())
            {
                args.push_back("-l");
                args.push_back(cnf->user);
                args.push_back(cnf->maddr);
            } else
                args.push_back(cnf->maddr);

            if (cnf->verbose) displayCommand(args);

            phase=1;

            runSSH( new SSHPIX(cnf->fwobj->getName().c_str(),
                               args,
                               cnf->pwd,
                               cnf->epwd,
                               "") );
        } else
        {
/*
 * if user requested test run, store firewall script in a temp file.
 * Always store it in a temp file on linksys
 */
            QString s;

/* user_can_change_install_dir */
            bool uccid=Resources::getTargetOptionBool(
                cnf->fwobj->getStr("host_OS"),"user_can_change_install_dir");

            if (uccid)
                s=cnf->fwobj->getOptionsObject()->getStr("firewall_dir").c_str();

            if (s.isEmpty()) s=Resources::getTargetOptionStr(
                cnf->fwobj->getStr("host_OS"),
                "activation/fwdir").c_str();

            cnf->fwdir     = s;

            continueRun();
        }
    }
}

/*
 * This method builds and returns activation command 
 * This method is used for all firewall platforms but PIX
 */
QString instDialog::getActivationCmd()
{
    if (!cnf->activationCmd.isEmpty())
    {
        return cnf->activationCmd;
    }

    QString cmd="";

    string optpath="activation/";

    if (cnf->user=="root")   optpath += "root/";
    else                     optpath += "reg_user/";

    if (cnf->testRun)
    {       
        optpath += "test/";
        if (cnf->rollback) optpath += "rollback/";
        else               optpath += "no_rollback/";
    } else
    {
        optpath += "run/";
        if (cnf->compressScript) optpath += "compression/";
        else                     optpath += "no_compression/";
    }

    cmd=Resources::getTargetOptionStr(cnf->fwobj->getStr("host_OS"),
                                      optpath).c_str();

/* replace macros in activation command:
 *
 * %FWSCRIPT%, %FWDIR%, %FWBPROMPT%, %RBTIMEOUT%
 */
    cmd.replace("%FWSCRIPT%",cnf->conffile);
    cmd.replace("%FWDIR%",cnf->fwdir);
    cmd.replace("%FWBPROMPT%",fwb_prompt);

    int r = cnf->rollbackTime;
    if (cnf->rollbackTimeUnit=="sec")  r = r*60;

    QString rbt;
    rbt.setNum(r);
    cmd.replace("%RBTIMEOUT%",rbt);

    return cmd;
}

void instDialog::initiateCopy(const QString &file)
{
    QStringList args;
    QStringList allConfig;

    if (fwbdebug)
        qDebug("instDialog::initiateCopy for the file %s",file.ascii());

    QString platform=cnf->fwobj->getStr("platform").c_str();
    if (platform!="pix" && platform!="fwsm") progressBar->show();

    QString ff = cnf->wdir+"/"+file;

    std::ifstream   *config_file;

    config_file = new ifstream(ff.latin1());
    if ( ! *config_file)
    {
        appendRich(QObject::tr("Can not open file %1").arg(ff));
        return;
    }
    string s0;
    while ( !config_file->eof() )
    {
        getline( *config_file, s0);
        
        QString s(s0.c_str());
        s.stripWhiteSpace();
        allConfig.push_back(s);
    }
    config_file->close();
    delete config_file;
    config_file=NULL;

//    allConfig.push_back("\004");

#ifdef _WIN32
    args.push_back(ssh);

    if (!cnf->user.isEmpty() &&
        ssh.find("plink.exe")!=-1) args.push_back("-ssh");

    args.push_back("-pw");
    args.push_back(cnf->pwd);
#else
    args.push_back(argv0.c_str());
    args.push_back("-X");   // fwbuilder works as ssh wrapper
//    if (fwbdebug>1) args.push_back("-d");
//    args.push_back("-t");
//    args.push_back("-t");
#endif
/* do not change destination, we do chmod on it later */
//        args.push_back( cnf->wdir+"/"+file);

    if (!cnf->sshArgs.isEmpty())
        args += QStringList::split(" ",cnf->sshArgs);

    if (cnf->verbose) args.push_back("-v");
    if (!cnf->user.isEmpty())
    {
        args.push_back("-l");
        args.push_back(cnf->user);
        args.push_back(cnf->maddr);
    } else
        args.push_back(cnf->maddr);

    string optpath="activation/";

    if (cnf->user=="root")   optpath += "root/";
    else                     optpath += "reg_user/";

    if (cnf->testRun)        optpath += "test/";
    else                     optpath += "run/";

    optpath+="copy";

    QString cmd=Resources::getTargetOptionStr(cnf->fwobj->getStr("host_OS"),
                                              optpath).c_str();

/* replace macros in activation command:
 *
 * %FWSCRIPT%, %FWDIR%, %FWBPROMPT%, %RBTIMEOUT%
 */
    cmd.replace("\n","");
    cmd.replace("%FWSCRIPT%",file);
    cmd.replace("%FWDIR%",cnf->fwdir);
    cmd.replace("%FWBPROMPT%",fwb_prompt);

    args.push_back(cmd);
        
    appendRich( tr("\nCopying %1 -> %2:%3\n")
                .arg(cnf->wdir+"/"+file).arg(cnf->maddr).arg(cnf->fwdir) );
    appendRich( "\n");

    if (cnf->verbose) displayCommand(args);

    phase=1;

    SSHSession *s= new SSHUnx(cnf->fwobj->getName().c_str(),
                              args,
                              cnf->pwd,
                              "",
                              allConfig);
    
    s->setCloseStdin(true);
    runSSH(s);
}

void instDialog::displayCommand(const QStringList &args)
{
    QStringList a1 = args;

    for (QStringList::iterator i=a1.begin(); i!=a1.end(); i++)
    {
        if ( (*i)=="-pw" )
        {
            i++;
            *i = "XXXXXX";
            break;
        }
    }
    QString s=a1.join(" ");
    appendRich( tr("Running command '%1'\n\n").arg(s) );
    appendRich("\n");
}

void instDialog::continueRun()
{
    if (session)
    {
        if (session->getErrorStatus())
        {
            appendRich( tr("ERROR: Terminating install sequence\n") );
            setFinishEnabled( page(1), true );
            return;
        }

        delete session;
        session=NULL;
    }

    if (activationCommandDone)
    {
        appendRich( tr("Done\n") );
        setFinishEnabled( page(1), true );
        return;
    }

    if (cnf->fwobj->getStr("platform")=="pix" || cnf->fwobj->getStr("platform")=="fwsm")
    {
        setFinishEnabled( page(1), true );
        return;
    } else
    {

        if (!confFiles.empty())
        {
            QString cnffile = confFiles.front();
            confFiles.pop_front();
            initiateCopy( cnffile );
        } else
        {
            QStringList args;

            progressBar->hide();

#ifdef _WIN32
            args.push_back(ssh);

            if (!cnf->user.isEmpty() &&
                ssh.find("plink.exe")!=-1) args.push_back("-ssh");

            args.push_back("-pw");
            args.push_back(cnf->pwd);
#else
            args.push_back(argv0.c_str());
            args.push_back("-X");   // fwbuilder works as ssh wrapper
            args.push_back("-t");
            args.push_back("-t");
#endif
            if (!cnf->sshArgs.isEmpty())
                args += QStringList::split(" ",cnf->sshArgs);

            if (cnf->verbose) args.push_back("-v");

            if (!cnf->user.isEmpty())
            {
                args.push_back("-l");
                args.push_back(cnf->user);
                args.push_back(cnf->maddr);
            } else
                args.push_back(cnf->maddr);
#if 0
            if (!cnf->user.isEmpty())
                args.push_back(cnf->user + "@" + cnf->maddr);
            else
                args.push_back(cnf->maddr);
#endif

            QString cmd = getActivationCmd();
            args.push_back( cmd );

            appendRich( tr("Activating new policy\n") );
            appendRich( "\n");

            if (cnf->verbose) displayCommand(args);

            activationCommandDone=true;

            runSSH( new SSHUnx(cnf->fwobj->getName().c_str(),
                               args,
                               cnf->pwd,
                               "",
                               "") );
        }

    }
}

void instDialog::runSSH(SSHSession *s)
{
    if (fwbdebug) qDebug("instDialog::runSSH()");

    session = s;

    session->setOptions(cnf);
    session->setFWBPrompt(fwb_prompt);

    connect(session,SIGNAL(printStdout_sign(const QString&)),
            this,SLOT(append(const QString&)));

    connect(session,SIGNAL(sessionFinished_sign()),
            this,SLOT(processExited()));

    connect(session,SIGNAL(updateProgressBar_sign(int,bool)),
            this,SLOT(updateProgressBar(int,bool)));

    session->startSession();
}

void instDialog::updateProgressBar(int n,bool setsize)
{
    if (fwbdebug)
        qDebug("instDialog::updateProgressBar  n=%d setsize=%d",n,setsize);

    if (setsize) progressBar->setTotalSteps(n);
    else
        progressBar->setProgress(progressBar->totalSteps()-n);
}


void instDialog::processExited()
{
    QTimer::singleShot( 0, this, SLOT(continueRun()) );
}

/* user clicked 'Cancel' */
void instDialog::reject()
{
    if (fwbdebug) qDebug("instDialog::reject()");
    if (session!=NULL)
    {
        if (fwbdebug)
            qDebug("instDialog::reject()  killing ssh session");

        disconnect(session,SIGNAL(printStdout_sign(const QString&)),
                   this,SLOT(append(const QString&)));

        disconnect(session,SIGNAL(sessionFinished_sign()),
                   this,SLOT(processExited()));

        disconnect(session,SIGNAL(updateProgressBar_sign(int,bool)),
                   this,SLOT(updateProgressBar(int,bool)));

        session->terminate();

        delete session;
        session=NULL;
    }
    QWizard::reject();
}


void instDialog::showEvent( QShowEvent *ev)
{
    st->restoreGeometry(this, QRect(200,100,480,500) );
    QDialog::showEvent(ev);
}

void instDialog::hideEvent( QHideEvent *ev)
{
    st->saveGeometry(this);
    QDialog::hideEvent(ev);
}
    
void instDialog::testRunRequested()
{
    bool f = testRun->isChecked();
    if (cnf->fwobj->getStr("host_OS")=="pix_os" || cnf->fwobj->getStr("host_OS")=="fwsm_os")
    {
        rollback->setChecked( false );
        f=false;
    }

    rollback->setEnabled(f);
    rollbackTime->setEnabled(f);
    rollbackTimeUnit->setEnabled(f);
}

