/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: ObjectTreeView.cpp,v 1.43 2007/07/07 05:39:34 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "FWBTree.h"
#include "ObjectTreeView.h"
#include "ObjectTreeViewItem.h"
#include "ObjectManipulator.h"
#include "ObjectEditor.h"
#include "FWObjectDrag.h"
#include "FWWindow.h"

#include "fwbuilder/FWObject.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/Group.h"

#include <qdragobject.h>
#include <qlistview.h>
#include <qheader.h>
#include <qpainter.h>
#include <qpixmapcache.h>

#include <iostream>
#include <algorithm>

using namespace std;
using namespace libfwbuilder;

ObjectTreeView* ObjectTreeViewItem::getTree()
{
    return dynamic_cast<ObjectTreeView*>(listView());
}

/****************************************************************************
 *
 *    class ObjectTreeView
 *
 ****************************************************************************/

ObjectTreeView::ObjectTreeView(QWidget* parent, const char * name, WFlags f) :
    QListView(parent,name,f),
    singleClickTimer(this)
{
//    setAcceptDrops( TRUE );
    item_before_drag_started=NULL;
    lastSelected = NULL;
    second_click = false;
    selectionFrozen = false;
    expandOrCollapse = false;
    Lockable = false;
    Unlockable = false;
    visible = false;
    /*
     * note about process_mouse_release_event
     *
     * we use mouseReleaseEvent event to switch object opened in the
     * editor panel (i.e. we open new object when mouse button is
     * released rather than when it is pressed). This allows us to
     * start drag without switching object in the editor. The problem
     * is that mouseReleaseEvent is received in this widget after the
     * d&d ends with a drop somewhere else, which triggers call to
     * contentsMouseReleaseEvent which switches object in the
     * editor. This is undesired when the editor shows a group and we
     * try to drag and drop an object into that group. Flag
     * process_mouse_release_event is used to suppress object
     * switching when contentsMouseReleaseEvent is called after
     * successfull drop.
     */
    process_mouse_release_event = true;

    connect( this, SIGNAL(currentChanged(QListViewItem*)),
             this, SLOT(currentChanged(QListViewItem*)) );

    connect( this, SIGNAL( selectionChanged() ),
             this, SLOT( selectionChanged() ) );

    connect(this, SIGNAL( collapsed(QListViewItem*)),
            this, SLOT( collapsed(QListViewItem*)) );

    connect(this, SIGNAL( expanded(QListViewItem*)),
            this, SLOT( expanded(QListViewItem*)) );

    connect( &singleClickTimer, SIGNAL( timeout() ),
             this, SLOT( resetSelection() ) );



    addColumn( tr( "Object" ) );

    header()->hide();

    setMinimumSize( QSize( 200, 0 ) );

//    QFont objTreeView_font(  font() );
//    setFont( objTreeView_font ); 
//    setCursor( QCursor( 0 ) );
    setResizePolicy( QListView::AutoOneFit );

    setColumnWidthMode(0, QListView::Maximum);
    setItemMargin( 2 );

    setDragAutoScroll( TRUE );
    setAllColumnsShowFocus( TRUE );
    setSelectionMode( QListView::Extended );
    setShowSortIndicator( FALSE );
    setRootIsDecorated( TRUE );
    setResizeMode( QListView::AllColumns );
    setTreeStepSize( 15 );

    setSorting(0,true);
}

void ObjectTreeView::currentChanged(QListViewItem *itm)
{
    if (fwbdebug)
        qDebug("ObjectTreeView::currentChanged  itm=%s",itm->text(0).ascii());
    expandOrCollapse = false;

//    lastSelected = ovi;
//    lastSelected = currentItem();
}

void ObjectTreeView::collapsed(QListViewItem* itm)
{
    if (fwbdebug)
        qDebug("ObjectTreeView::collapsed  itm=%s",itm->text(0).ascii());
    expandOrCollapse = true;
}

void ObjectTreeView::expanded(QListViewItem* itm)
{
    if (fwbdebug)
        qDebug("ObjectTreeView::expanded  itm=%s",itm->text(0).ascii());
    expandOrCollapse = true;
}

/*
 * This method makes list selectedObjects flat. If user selects
 * several objects in the tree, and some of them have children, QT
 * puts all the children in the selected objects list even if
 * corresponding subtrees are collapsed. This method eliminates these
 * selected children objects.
 *
 */
std::vector<libfwbuilder::FWObject*> ObjectTreeView::getSimplifiedSelection()
{
    vector<FWObject*> so  = selectedObjects;
    vector<FWObject*> so2 = selectedObjects;
    for (vector<FWObject*>::iterator i=so2.begin();  i!=so2.end(); ++i)
    {
        for (vector<FWObject*>::iterator j=i;  j!=so2.end(); ++j)
        {
            vector<FWObject*>::iterator k=std::find(so.begin(),so.end(),*j);
            if ( (*j)->isChildOf( *i ) && k!=so.end())
                so.erase( k );
        }
    }
    return so;
}

FWObject* ObjectTreeView::getCurrentObject()
{
    QListViewItem *ovi = currentItem();
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    assert(otvi!=NULL);
    return otvi->getFWObject();
}

bool ObjectTreeView::isLockable()
{
    return Lockable;
}

bool ObjectTreeView::isUnlockable()
{
    return Unlockable;
}

void ObjectTreeView::focusInEvent(QFocusEvent* ev)
{
    if (fwbdebug) qDebug("ObjectTreeView::focusInEvent 1");
    QListView::focusInEvent(ev);
    QListViewItem *ci = currentItem();
    if (ci) repaintItem(ci);
    if (fwbdebug) qDebug("ObjectTreeView::focusInEvent 2");
}

void ObjectTreeView::focusOutEvent(QFocusEvent* ev)
{
    if (fwbdebug) qDebug("ObjectTreeView::focusOutEvent 1");
    QListView::focusOutEvent(ev);
    QListViewItem *ci = currentItem();
    if (ci) repaintItem(ci);
    if (fwbdebug) qDebug("ObjectTreeView::focusOutEvent 2");
}

void ObjectTreeView::updateTreeItems()
{
    if (fwbdebug) qDebug("ObjectTreeView::updateTreeItems 1");
    QListViewItemIterator it(this);
    QListViewItem *itm;
    ObjectTreeViewItem *otvi;
    FWObject *obj;
    QString icn;

    QPixmap pm_lock;
    if ( ! QPixmapCache::find( "lock.png", pm_lock) )
    {
        pm_lock = QPixmap::fromMimeSource( "lock.png" );
        QPixmapCache::insert( "lock.png", pm_lock);
    }
    
    while ( it.current() ) 
    {
        itm= it.current();
        otvi=dynamic_cast<ObjectTreeViewItem*>(itm);
        obj=otvi->getFWObject();
        
        if (FWBTree::isSystem(obj))
            icn="folder1.png";
        else
            icn=Resources::global_res->getObjResourceStr(obj, "icon-tree").c_str();
        
        QPixmap pm_obj;
        if ( ! QPixmapCache::find( icn, pm_obj) )
        {
            pm_obj = QPixmap::fromMimeSource( icn );
            QPixmapCache::insert( icn, pm_obj);
        }

        if (obj->getBool("ro"))  itm->setPixmap(0, pm_lock );
        else                     itm->setPixmap(0, pm_obj );
        
        Firewall *fw=Firewall::cast(obj);
        if (fw!=NULL)
        {
            itm->setText(0,(fw->needsInstall())?
                    QString::fromUtf8(fw->getName().c_str())+" *":
                    QString::fromUtf8(fw->getName().c_str()));
        }
        
        ++it;
    }
    triggerUpdate ();  
    if (fwbdebug) qDebug("ObjectTreeView::updateTreeItems 2");
}

/*
 * This is where drag starts
 */
QDragObject* ObjectTreeView::dragObject()
{
    if (fwbdebug) qDebug("ObjectTreeView::dragObject");

    QListViewItem *ovi = currentItem();
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);

    FWObject *current_obj = getCurrentObject();

/* can't drag system folders 

    in fact, I have to allow to drag system folders because otherwise
    QListView triggers highlighting of objects in the tree when user
    drags mouse cursor across them. This is weird behavior and there
    does not seem to be any way to turn it off. It happens close to
    the end of void QListView::contentsMouseMoveEvent( QMouseEvent * e)
    (See code after they decided that they do not need to call startDrag())

    if (FWBTree::isSystem(obj)) return NULL;
*/
    QString icn =
        Resources::global_res->getObjResourceStr(current_obj, "icon-ref").c_str();

    vector<FWObject*> so = getSimplifiedSelection();

    list<FWObject*> dragobj;
    for (vector<FWObject*>::iterator v=so.begin(); v!=so.end(); v++)
        dragobj.push_back( *v );

    FWObjectDrag    *drag = new FWObjectDrag(dragobj, this);
    //QPixmap          pm   = QPixmap::fromMimeSource( icn_filename );

    QPixmap pm;
    if ( ! QPixmapCache::find( icn, pm) )
    {
        pm = QPixmap::fromMimeSource( icn );
        QPixmapCache::insert( icn, pm);
    }

    if (dragobj.size()>1)
    {
        QPixmap npm(32,32);
        QPainter p( &npm );
        p.fillRect( 0,0,32,32, QBrush( QColor("white"),Qt::SolidPattern ) );
        p.setBackgroundMode( TransparentMode );
        p.drawPixmap( 0, 32-pm.rect().height(), pm);
        p.setPen( QColor("red") );
        p.setBrush( QBrush( QColor("red"),Qt::SolidPattern ) );
        p.drawPie( 16, 0, 16,16, 0, 5760 );
        QString txt;
        txt.setNum(dragobj.size());
        QRect br=p.boundingRect(0, 0, 1000, 1000,
                                Qt::AlignLeft|Qt::AlignVCenter,
                                txt );
        p.setPen( QColor("white") );
        p.drawText( 24-br.width()/2 , 4+br.height()/2, txt );
        npm.setMask( npm.createHeuristicMask() );
        drag->setPixmap( npm,
                         QPoint( npm.rect().width() / 2,
                                 npm.rect().height() / 2 ) );
    } else
        drag->setPixmap( pm,
                         QPoint( pm.rect().width() / 2,
                                 pm.rect().height() / 2 ) );

/*
 * This fragment returns selection in the tree back to the object that
 * was selected before drag operation has started. This help in the
 * following case:
 *
 *  - open a group for editing (group is selected in the tree)
 *  - left-click on another object in the tree, start dragging it
 *
 * at this point selection in the tree returns to the group, so when
 * user finishes d&d operation, the selection in the tree is consisten
 * with object currently opened in the editor panel.
 *
 * There is a problem with this however. If user wants to put an
 * object from a different library into the group, they have to switch
 * to that library before doing d&d. When they switch, ObjectTree
 * shown in the left panel becomes different from the tree in which
 * the group is located. When d&d finishes, the ObjectTree object
 * receives contentsMouseReleaseEvent event. Since it is not the right
 * tree object, it can not properly restore selection and choses an
 * object that was previously opened in that tree, which in turn
 * changes the object opened in the editor panel. To make things
 * worse, this event is only delivered to the tree object on Mac OS X.
 *
 * 
 */
    if (fwbdebug) qDebug("ObjectTreeView::dragObject()  this=%p visible=%d",
                         this,visible);

    FWObject *edit_obj = oe->getOpened();

    if (oe->isVisible() && 
        dragobj.size()==1 &&
        edit_obj!=NULL &&
        current_obj->getLibrary()==edit_obj->getLibrary() )
    {
        if (fwbdebug) qDebug("ObjectTreeView::dragObject() reset selection");
        setSelected(otvi,false);
        resetSelection();
    }

#if 0
    /*
     * need to reset selection if:
     *
     * object editor is opened, and
     * we are dragging one object, and
     * object opened in editor is not the same as the one we are dragging
     */
    if (oe->isVisible() && dragobj.size()==1 && oe->getOpened()!=obj)
    {
        setSelected(otvi,false);
        resetSelection();
    }
#endif
    if (fwbdebug) qDebug("ObjectTreeView::dragObject()  returns !NULL");

    return drag;
}

void ObjectTreeView::dragEnterEvent( QDragEnterEvent *ev)
{
    ev->accept( FWObjectDrag::canDecode(ev) );
    QListViewItem *ovi = currentItem();
    item_before_drag_started = ovi;
}

void ObjectTreeView::dragMoveEvent( QDragMoveEvent *ev)
{
    bool    acceptE = false;
    
    if (FWObjectDrag::canDecode(ev))
    {
        int hy;

//        hy=header()->height();    // if header is shown
        hy=0;

        QListViewItem *ovi = itemAt( ev->pos() );

        ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
        if (otvi==NULL)
        {   
            ev->accept(acceptE);
            return;
        }

        FWObject *trobj = otvi->getFWObject();

/* the tree can accept drop only if it goes into a group and if that group
 * validates the object and tree is not read-only
 */
        if (Group::cast(trobj)!=NULL  && 
            !FWBTree::isSystem(trobj) &&
            !trobj->isReadOnly()
        )
        {
            acceptE = true;

            Group    *g     = Group::cast(trobj);
            list<FWObject*> dragol;
            if (FWObjectDrag::decode(ev, dragol))
            {
                for (list<FWObject*>::iterator i=dragol.begin();
                     i!=dragol.end(); ++i)
                {
                    FWObject *dragobj = *i;
                    assert(dragobj!=NULL);

                    if (FWBTree::isSystem(dragobj))
                    {
/* can not drop system folder anywhere */
                        acceptE = false;
                        break;
                    }

                    bool t= g->validateChild(dragobj);
                    if (!t)
                    {
                        acceptE = false;
                        break;
                    }

                    if (g->getPath(true) == "Services/Groups" && t)
                        setOpen(ovi,true);

                    if (g->getPath(true) == "Objects/Groups" && t)
                        setOpen(ovi,true);
                }
            }
        }
    }
    ev->accept(acceptE);
}

void ObjectTreeView::dropEvent(QDropEvent *ev)
{
    QListViewItem *ovi = itemAt( ev->pos() );
    ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(ovi);
    if (otvi==NULL) return;
    FWObject *trobj = otvi->getFWObject();

/* the tree can accept drop only if it goes into a group and if that group
 * validates the object and the tree is not read-only
 */
    if (Group::cast(trobj)!=NULL  && 
        !FWBTree::isSystem(trobj) &&
        !trobj->isReadOnly()
    )
    {
        Group *g=Group::cast(trobj);

        item_before_drag_started=NULL;

        list<FWObject*> dragol;
        if (FWObjectDrag::decode(ev, dragol))
        {
            for (list<FWObject*>::iterator i=dragol.begin();
                 i!=dragol.end(); ++i)
            {
                FWObject *dragobj = *i;
                assert(dragobj!=NULL);

/* check for duplicates */
                string cp_id=dragobj->getId();
                list<FWObject*>::iterator j;
                for(j=g->begin(); j!=g->end(); ++j)     
                {
                    FWObject *o1=*j;
                    if(cp_id==o1->getId()) continue;

                    FWReference *ref;
                    if( (ref=FWReference::cast(o1))!=NULL &&
                        cp_id==ref->getPointerId()) return;
                }

                g->addRef(dragobj);
            }
            setCurrentItem(ovi);
            setSelected(ovi, true);

//            emit objectDropped_sign(g);
        }
    }
}

void ObjectTreeView::dragLeaveEvent( QDragLeaveEvent *ev)
{
    QListView::dragLeaveEvent(ev);

    if (item_before_drag_started!=NULL)
    {
        setCurrentItem(item_before_drag_started);
        setSelected(item_before_drag_started, true);
    }
}

void ObjectTreeView::contentsMouseMoveEvent( QMouseEvent * e )
{
    if (e==NULL)  return;

    QListView::contentsMouseMoveEvent(e);
}


void ObjectTreeView::contentsMousePressEvent( QMouseEvent *e )
{
    if (fwbdebug)
        qDebug("ObjectTreeView::contentsMousePressEvent");

    second_click = false;
    process_mouse_release_event = true;

    if (fwbdebug)
    {
        qDebug(QString("ObjectTreeView::contentsMousePressEvent :: currentItem=%1")
               .arg((currentItem())?currentItem()->text(0):"nil")
        );
        qDebug(QString("ObjectTreeView::contentsMousePressEvent :: lastSelected=%2")
               .arg((lastSelected)?lastSelected->text(0):"nil")
        );
    }

    lastSelected = currentItem();

    QListView::contentsMousePressEvent(e);
}

/*
 * Two modes of operation of this widget:
 *
 * 1.  this widget can intercept single mouse click and return
 * selection back to the object that was current before it. If user
 * double ckicks mouse button, then this reset is not done and new
 * object is selected. This is done using timer.
 *
 * 2. this widget can act as usual QListView does, that is, select an object
 * on a single click.
 *
 * uncomment the line that starts timer for mode #1.
 *
 *
 * we use mouseReleaseEvent event to switch object opened in the
 * editor panel (i.e. we open new object when mouse button is released
 * rather than when it is pressed). This allows us to start drag
 * without switching object in the editor. The problem is that
 * mouseReleaseEvent is received in this widget after the d&d ends
 * with a drop somewhere else, which triggers call to
 * contentsMouseReleaseEvent which switches object in the editor. This
 * is undesired when the editor shows a group and we try to drag and
 * drop an object into that group. Flag process_mouse_release_event is
 * used to suppress object switching when contentsMouseReleaseEvent is
 * called after successfull drop.
 *
 */
void ObjectTreeView::contentsMouseReleaseEvent( QMouseEvent *e )
{
    if (fwbdebug)
        qDebug("ObjectTreeView::contentsMouseReleaseEvent 1 this=%p  process_mouse_release_event=%d",
               this,process_mouse_release_event);

    QListView::contentsMouseReleaseEvent(e);


    if (!process_mouse_release_event)
    {
        // just do not switch object in the editor, otherwise
        // process this event as usual
        process_mouse_release_event = true;
        return;
    }

    if (fwbdebug)
        qDebug("ObjectTreeView::contentsMouseReleaseEvent 2 selectedObjects.size()=%d getCurrentObject()=%p current object %s",
               selectedObjects.size(),
               getCurrentObject(),
               (getCurrentObject()!=NULL)?getCurrentObject()->getName().c_str():"nil");
    
    if (expandOrCollapse) return;  // user expanded or collapsed subtree,
                                   // no need to change object in the editor

    if (selectedObjects.size()==1)
        emit switchObjectInEditor_sign( getCurrentObject() );
    else
    {
        // user selected multiple objects
        // do not let them if editor has unsaved changes
        // 
        if (oe->isVisible() && oe->isModified())
            emit switchObjectInEditor_sign( getCurrentObject() );
        else
            oe->blank();
    }
}

/*
 * sends signal that should be connected to a slot in
 * ObjectManipulator which opens editor panel if it is closed and then
 * opens current object in it
 */
void ObjectTreeView::editCurrentObject()
{
    if (fwbdebug)
        qDebug("ObjectTreeView::editCurrentObject");

    emit editCurrentObject_sign();

    if (fwbdebug)
        qDebug("ObjectTreeView::editCurrentObject done");
}

void ObjectTreeView::contentsMouseDoubleClickEvent( QMouseEvent *e )
{
    if (fwbdebug)
        qDebug("ObjectTreeView::contentsMouseDoubleClickEvent");

    second_click=true;
    singleClickTimer.stop();

    FWObject *obj = getCurrentObject();

/* system folders open on doubleclick, while for regular objects it
 * opens an editor 
 */
    if (FWBTree::isSystem(obj))
        QListView::contentsMouseDoubleClickEvent(e);
    else
        editCurrentObject();
}

void ObjectTreeView::keyPressEvent( QKeyEvent* ev )
{
    FWObject *obj = getCurrentObject();

    if (ev->key()==Qt::Key_Enter || ev->key()==Qt::Key_Return)
    {
        editCurrentObject();
        ev->accept();
        return;
    }
    if (ev->key()==Qt::Key_Delete)
    {
        emit deleteObject_sign(obj);
        ev->accept();
        return;
    }    
    QListView::keyPressEvent(ev);
}

void ObjectTreeView::keyReleaseEvent( QKeyEvent* ev )
{
    if (fwbdebug)
        qDebug("ObjectTreeView::keyReleaseEvent");

    QListView::keyReleaseEvent(ev);

    if (selectedObjects.size()==1)
        emit switchObjectInEditor_sign( getCurrentObject() );
    else
    {
        // user selected multiple objects
        // do not let them if editor has unsaved changes
        // 
        if (oe->isVisible() && oe->isModified())
            emit switchObjectInEditor_sign( getCurrentObject() );
        else
            oe->blank();
    }
}

void ObjectTreeView::returnPressed(QListViewItem *itm)
{
    if (fwbdebug)
        qDebug("ObjectTreeView::returnPressed");

    editCurrentObject();

    QListView::returnPressed(itm);
}

void ObjectTreeView::clearLastSelected()
{
    lastSelected = NULL;
}


void ObjectTreeView::resetSelection()
{
    if (fwbdebug)
        qDebug(QString("ObjectTreeView::resetSelection :: lastSelected=%1").arg(lastSelected->text(0)));

    setCurrentItem(lastSelected);
    setSelected( lastSelected , true);
}

void ObjectTreeView::selectionChanged()
{
    if (fwbdebug)
        qDebug("ObjectTreeView::selectionChanged selectionFrozen=%d",
               selectionFrozen);

    if (selectionFrozen) return;

/* in extended selection mode there may be several selected items */

    selectedObjects.clear();

    QListViewItemIterator it(this);
    while ( it.current() )
    {
        if (it.current()->isSelected())
        {
            QListViewItem *itm= it.current();
            ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(itm);
            selectedObjects.push_back(otvi->getFWObject());

            if (fwbdebug)
                qDebug("ObjectTreeView::selectionChanged: selected otvi=%p object %s", otvi, otvi->getFWObject()->getName().c_str());
        }
        ++it;
    }
    setLockFlags();
/* now list  selectedObjects   holds all selected items */
}

void ObjectTreeView::setLockFlags()
{
    QListViewItemIterator it(this);
    Lockable=false;
    Unlockable=false;
    while ( it.current() )
    {
        if (it.current()->isSelected())
        {
            QListViewItem *itm= it.current();
            ObjectTreeViewItem *otvi=dynamic_cast<ObjectTreeViewItem*>(itm);

            FWObject *lib = otvi->getFWObject()->getLibrary();
            // these lbraries are locked anyway, do not let the user
            // lock objects inside because they won't be able to unlock them.
            if (lib->getId()!=STANDARD_LIB && lib->getId()!=TEMPLATE_LIB)
            {
                if (otvi->getFWObject()->getBool("ro"))  Unlockable=true;
                else                                     Lockable=true;
            }
        }
        ++it;
    }
}

bool ObjectTreeView::isSelected(FWObject* obj)
{
    for (vector<FWObject*>::iterator i=selectedObjects.begin();
         i!=selectedObjects.end(); ++i)
    {
        if ( (*i)==obj)  return true;
    }
    return false;
}

int  ObjectTreeView::getNumSelected()
{
    return selectedObjects.size();
}

