/* readico.c
 * 
 * Copyright (C) 1998-2001  Oskar Liljeblad
 *
 * This file is part of icoutils.
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file "LICENSE" for
 * details.
 */

#include "icotool.h"

#include <stdlib.h>
#include <stdio.h>

#include "win32-endian.h"

/*
 *
 * Macros, local
 *
 */

/* CHECK_OFFSET:
 * Check if mem+size is outside the range of the file.
 */
#define CHECK_OFFSET(itf, mem, len) \
    if (bad_offset((itf)->contents, (itf)->size, (mem), (len))) { \
    	put_error("%s: format error (not an icon file)\n", (itf)->name); \
		return FALSE; \
    }

/*
 *
 * Functions
 *
 */

/* initialize_icotool_file:
 * Initialize the various fields in the specified file.
 */
bool
initialize_icotool_file(IconFile *itf)
{
    itf->ci_dir = (Win32CursorIconFileDir *) itf->contents;
    CHECK_OFFSET(itf, itf->ci_dir, sizeof(Win32CursorIconFileDir));
    fix_win32_cursor_icon_file_dir_endian(itf->ci_dir);

    if (itf->ci_dir->reserved != 0
    	    || !(itf->ci_dir->type == 1 || itf->ci_dir->type == 2)) {
    	put_error("%s: format error (not an icon file)\n", (itf)->name);
    	return FALSE;
    }

    itf->resource_type = (itf->ci_dir->type == 1 ? RESOURCE_ICON : RESOURCE_CURSOR);
    return TRUE;
}

/* initialize_icotool_icon:
 * Initialize the IconImage structure to represent the
 * icon with the specified index in the specified file.
 */
bool
initialize_icotool_icon(IconFile *itf, int index, IconImage *icon)
{
    uint32_t offset;

    icon->entry = &itf->ci_dir->entries[index];
    CHECK_OFFSET(itf, icon->entry, sizeof(Win32CursorIconFileDirEntry));
    fix_win32_cursor_icon_file_dir_entry_endian(icon->entry);

    offset = icon->entry->dib_offset;
    icon->bitmap = (Win32BitmapInfoHeader *) &itf->contents[offset];
    CHECK_OFFSET(itf, icon->bitmap, sizeof(Win32BitmapInfoHeader));
    fix_win32_bitmap_info_header_endian(icon->bitmap);

    if (icon->bitmap->compression != 0) {
    	put_error("%s: compressed image data not supported\n", itf->name);
		return FALSE;
    }
    icon->flipped = (icon->bitmap->height < 0);
    if (icon->flipped)
    	icon->bitmap->height = -icon->bitmap->height;
    if (icon->bitmap->width < 0)
    	icon->bitmap->width = -icon->bitmap->width;

    offset += icon->bitmap->size;
	if (icon->bitmap->clr_used == 0)
		icon->color_count = 1 << icon->bitmap->bit_count;
	else
		icon->color_count = icon->bitmap->clr_used;

    if (icon->color_count != 1<<24) {
        icon->rgb_data = (Win32RGBQuad *) &itf->contents[offset];
		CHECK_OFFSET(itf, icon->rgb_data, sizeof(Win32RGBQuad) * icon->color_count);
		offset += icon->color_count * sizeof(Win32RGBQuad);
    }

    icon->image_data = &itf->contents[offset];
    icon->image_line_width = LINE_SIZE(icon->bitmap->width * icon->bitmap->bit_count);
    CHECK_OFFSET(itf, icon->image_data, icon->image_line_width * icon->entry->height);

    offset += icon->image_line_width * icon->entry->height;
    icon->mask_data = &itf->contents[offset];
    icon->mask_line_width = LINE_SIZE(icon->bitmap->width);
    CHECK_OFFSET(itf, icon->mask_data, icon->mask_line_width * icon->entry->height);

    return TRUE;
}

/* get_bitmap_image_pixel:
 * Get the value of a pixel in the icon bitmap image.
 */
int
get_bitmap_image_pixel(IconImage *icon, int x, int y)
{
    if (!icon->flipped)
    	y = icon->entry->height - y - 1;
    return vec(icon->image_data,
	    	x + (y * icon->image_line_width * 8)/icon->bitmap->bit_count,
	    	icon->bitmap->bit_count);
}

/* get_bitmap_mask_pixel:
 * Get the value of a pixel in the icon bitmap mask.
 */
int
get_bitmap_mask_pixel(IconImage *icon, int x, int y)
{
    if (!icon->flipped)
        y = icon->entry->height - y - 1;
    return vec(icon->mask_data, x + (y * icon->mask_line_width * 8), 1);
}

/* print_icotool_icon:
 * Print information on the specified icon in the specified file.
 */
void
print_icotool_icon(IconFile *itf, int c, IconImage *icon)
{
    switch (itf->resource_type) {
    case RESOURCE_CURSOR:
		put_message(0, "--cursor ");
		break;
    case RESOURCE_ICON:
		put_message(0, "--icon ");
		break;
    default:
	}

    put_message(0, "--index=%d --width=%d --height=%d --colors=%d",
    		c, icon->entry->width, icon->entry->height, icon->color_count);

    switch (itf->resource_type) {
    case RESOURCE_CURSOR:
    	put_message(0, " --hotspot-x=%d --hotspot-y=%d\n", 
	    		icon->entry->hotspot_x, icon->entry->hotspot_y);
		break;
    case RESOURCE_ICON:
    	put_message(0, "\n");
    	break;
    default:
    }
}
