#include "support.h"

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif

/* xstrtoi:
 * More simple version of strtol.
 */
bool
xstrtoi(char *str, int *result)
{
    char *endptr;
    long int lresult;

    lresult = strtol(str, &endptr, 10);
    if (str[0] == '\0' || endptr[0] != '\0')
    	return FALSE;

    *result = (int) lresult;
    return TRUE;
}

/* xstrdup:
 * Safe strdup.
 */
char *
xstrdup(char *str)
{
    str = strdup(str);
    assert_memory(str != NULL);
    return str;
}

/* xmalloc:
 * Safe malloc, that will never return NULL.
 */
void *
xmalloc(size_t size)
{
    void *mem = malloc(size);
    assert_memory(mem != NULL);
    return mem;
}

/* xrealloc:
 * Safe realloc, that will never return NULL.
 */
void *
xrealloc(void *memory, size_t size)
{
    memory = realloc(memory, size);
    assert_memory(memory != NULL);
    return memory;
}

/* xasprintf:
 * Safe and modified asprintf that will never return NULL.
 */
void
xasprintf(char **result, char *format, ...)
{
    va_list args;
    int iresult;

    va_start(args, format);
    iresult = vasprintf(result, format, args);
    assert_memory(iresult >= 0);
    va_end(args);
}

/* assert_memory:
 * Make sure that a memory allocation operation has succeeded.
 * Otherwise we print an message, and exit with non-zero status.
 */
void
assert_memory(bool expression)
{
    if (!expression) {
    	put_error("%s: %s\n", program_name, strerror(ENOMEM));
	exit(1);
    }
}

/* lastchar:
 * Return the last character in the string, or '\0' if it is
 * the empty string.
 */
char
lastchar(char *str)
{
    char last = '\0';
    while (str[0] != '\0')
    	last = *str++;
    return last;
}

/* llog2l:
 * Return the base-2 logarithm of the specified value.
 * If value is 0, the result is undefined.
 */
int
ilog2l(long value)
{
    uint32_t base = 1;
    value--;
    while ((value >>= 1) != 0)
    	base++;
    return base;
}

/* bad_offset:
 * Check if a chunk of data (determined by offset and size)
 * is within the bounds of the file. If the offset is bad,
 * return TRUE.
 */
bool
bad_offset(void *memory, size_t total_size, void *offset, size_t size)
{
/*FIXMEremove this    printf("check=%d %d %d : %d\n", (offset-memory), total_size, size, (((offset-memory) >= 0
    	    && (offset-memory) <= total_size
    	    && (offset-memory)+size >= 0
	    && (offset-memory)+size <= total_size))    );
*/
    return !((offset-memory) >= 0
    	    && (offset-memory) <= total_size
    	    && (offset-memory)+size >= 0
	    && (offset-memory)+size <= total_size);
}

/* is_directory:
 * Return TRUE if file exists, is readable and is a directory.
 */
bool
is_directory(char *filename)
{
    struct stat statbuf;
    if (stat(filename, &statbuf) == -1)
	return FALSE;
    return S_ISDIR(statbuf.st_mode);
}

/* put_error:
 * Print an error message to the standard error stream.
 */
void
put_error(char *format, ...)
{
    va_list args;

    va_start(args, format);
    fprintf(stderr, "%s: ", program_name);
    vfprintf(stderr, format, args);
    va_end(args);
}

/* vec:
 * An implementation of the perl vec() function.
 * Treats the string in vector as a vector of unsigned integers, and returns
 * the value of the bit field specified by offset. size specifies the number
 * of bits that are reserved for each entry in the bit vector.
 * Note: Due to the win32 image format, we start with bit 7 and work our way
 * down to 0. In the perl implementation, I believe bit 0 is really the first
 * bit we encounter.
 */
unsigned int
vec(char *vector, int offset, int size)
{
    int start_byte, start_bit;
    int end_byte, end_bit;
    unsigned int mod, value;

    start_byte = (offset * size) / 8;
    start_bit = 7 - (offset * size) % 8;

    end_byte = ((offset + 1) * size) / 8;
    end_bit = 7 - ((offset + 1) * size) % 8;

    value = 0;
    mod = 1<<(size-1);

    while (start_byte != end_byte || start_bit != end_bit) {
	if (vector[start_byte] & (1<<start_bit))
	    value |= mod;
	mod >>= 1;

	start_bit--;
	if (start_bit < 0) {
	    start_bit = 7;
	    start_byte++;
	}
    }

    return value;
}
