/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idataconsumer.h"


#include "idata.h"
#include "idatahandler.h"
#include "idatareader.h"
#include "ierror.h"
#include "iviewmodule.h"
#include "ivtk.h"


//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


int iDataConsumer::mModeGlobal = _DataConsumerOptimizeForSpeed;


//
//  helper class
//
iDataSyncRequest::iDataSyncRequest()
{
	mDataInfoPointer = new iDataInfo; IERROR_ASSERT(mDataInfoPointer);
	mVar = -1;
}


iDataSyncRequest::~iDataSyncRequest()
{
	delete mDataInfoPointer;
}


void iDataSyncRequest::Set(int var, const iDataInfo &info)
{
	mVar = var;
	*mDataInfoPointer = info;
}


//
//  Main class
//
iDataConsumer::iDataConsumer(iViewModule *vm, const iDataType &type) : iViewModuleComponent(vm), mPrimaryDataInfo(new iDataInfo(type))
{
	IERROR_ASSERT(mPrimaryDataInfo);
	mSecondaryDataInfo = new iDataInfo; IERROR_ASSERT(mSecondaryDataInfo);
	mModeLocal = mModeGlobal;
	mOverrideGlobal = false;
	
	this->GetViewModule()->RegisterDataConsumer(this);
}


iDataConsumer::~iDataConsumer()
{
	this->GetViewModule()->UnRegisterDataConsumer(this);
	delete mPrimaryDataInfo;
	delete mSecondaryDataInfo;

	//
	//  Check that all DataHandlers have been deleted too. 
	//  Fails for VTK 5.0.0 - Garbage collection is more convoluted.
	//
	if(mDataHandlers.Size() > 0)
	{
#ifdef I_DEBUG
#ifdef IVTK_4
		IERROR_REPORT("Not all DataHandlers have been deleted.");
#endif
#endif
	}
}


void iDataConsumer::AddSecondaryDataType(const iDataType &type)
{
	*mSecondaryDataInfo += type;
}


bool iDataConsumer::IsUsingData(const iDataInfo &info, bool onlyprimary) const
{
	int i;
	for(i=0; i<info.Count(); i++)
	{
		if(mPrimaryDataInfo->Includes(info.Type(i))) return true;
		if(!onlyprimary && mSecondaryDataInfo->Includes(info.Type(i))) return true;
	}
	return false;
}


const iDataType& iDataConsumer::GetDataType() const
{
	return mPrimaryDataInfo->Type(0);
}


bool iDataConsumer::IsDataPresent() const
{
	return this->GetViewModule()->GetReader()->IsThereData(this->GetDataType());
}


vtkDataSet* iDataConsumer::GetData() const
{
	return this->GetViewModule()->GetReader()->GetOutput(this->GetDataType());
}


void iDataConsumer::SetGlobalOptimizationMode(int mode)
{
	if(mode>=0 && mode<__NumDataConsumerOptimizationModes)
	{
		mModeGlobal = mode;
	}
}


void iDataConsumer::SetOptimizationMode(int mode)
{
	//
	//  Reset the global mode;
	//
	if(mode>=0 && mode<__NumDataConsumerOptimizationModes)
	{
		mModeLocal = mode;
		mOverrideGlobal = true;
	}
	else if(mode==_DataConsumerResetToGlobal)
	{
		mModeLocal = mModeGlobal;
		mOverrideGlobal = false;
	}
}


int iDataConsumer::GetOptimizationMode() const 
{
	if(mOverrideGlobal) return mModeLocal; else return mModeGlobal;
}


//
//  DataHandler registry functionality
//
void iDataConsumer::RegisterDataHandler(iDataHandler *c)
{
	mDataHandlers.AddUnique(c);
}


void iDataConsumer::UnRegisterDataHandler(iDataHandler *c)
{
	mDataHandlers.Remove(c);
}


void iDataConsumer::SyncWithData(const iDataSyncRequest &r)
{
	int i;
	this->SyncWithDataBody(r);
	for(i=0; i<mDataHandlers.Size(); i++) mDataHandlers[i]->SyncWithData(r);
}


const iDataSyncRequest& iDataConsumer::RequestAll()
{
	mRequest.Set(-1,(*mPrimaryDataInfo)+(*mSecondaryDataInfo));
	return mRequest;
}


const iDataSyncRequest& iDataConsumer::RequestPrimary()
{
	mRequest.Set(-1,*mPrimaryDataInfo);
	return mRequest;
}


const iDataSyncRequest& iDataConsumer::Request(int ind)
{
	mRequest.Set(ind,(*mPrimaryDataInfo)+(*mSecondaryDataInfo));
	return mRequest;
}


const iDataSyncRequest& iDataConsumer::Request(const iDataInfo &info, int ind) // for general use
{
	static iDataSyncRequest req;
	req.Set(ind,info);
	return req;
}
