/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iorthotextureplanefilter.h"


#include "ierror.h"
#include "iorthoslicer.h"
#include "imath.h"

#include <vtkCellArray.h>
#include <vtkPointData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


iOrthoTexturePlaneFilter::iOrthoTexturePlaneFilter(iViewSubject *vo) : iGenericFilter<vtkStructuredPointsToPolyDataFilter,vtkImageData,vtkPolyData>(vo,1,true,false) 
{
	mTexture = 0;
	mOffX = mOffY = 0;
}


void iOrthoTexturePlaneFilter::SetTexturePiece(vtkImageData *texture, int offx, int offy)
{
	mTexture = texture;
	mOffX = offx;
	mOffY = offy;

	this->Modified();
}


void iOrthoTexturePlaneFilter::ProduceOutput()
{
	int dims[3], dimsTexture[3];
	double org[3], spa[3];

	vtkImageData *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	output->Initialize();

	if(mTexture == 0) return;

	//
	//  Prepare texture support
	//
	input->GetOrigin(org);
	input->GetSpacing(spa);
	input->GetDimensions(dims);

	int i, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dims[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int Uidx, Vidx;
	iOrthoSlicer::GetUV(Axis,Uidx,Vidx);

	mTexture->GetDimensions(dimsTexture);

	float tc[2];
	static vtkIdType pts[4]={0,1,2,3};

	//
	//  AppenPolyDataFilter does not merge texture coordinates into one, so we need to create the correct texture support here.
	//  We create it for zero offset instance, and skip it altogether for other instances.
	//
	if(mOffX==0 && mOffY==0)
	{
		//
		// We'll create the building blocks of polydata including data attributes.
		//
		vtkPoints *points;
		points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(points);

		vtkCellArray *polys;
		polys = vtkCellArray::New(); IERROR_ASSERT(polys);

		vtkFloatArray *tcoords;
		tcoords = vtkFloatArray::New(); IERROR_ASSERT(tcoords);

		points->SetNumberOfPoints(4);
		tcoords->SetNumberOfComponents(2);
		tcoords->SetNumberOfTuples(4);

		//
		// Load the cell, and data attributes.
		//
		polys->InsertNextCell(4,pts);

		//
		//  Place the support plane
		//
		double x1[3];
		x1[Axis] = org[Axis];
		//
		//  LL point 
		//
		x1[Uidx] = org[Uidx];
		x1[Vidx] = org[Vidx];
		points->SetPoint(0,x1);
		tc[0] = 0.0;
		tc[1] = 0.0;
		tcoords->SetTuple(0,tc);
		//
		//  LR point 
		//
		x1[Uidx] = org[Uidx] + spa[Uidx]*dimsTexture[0];
		x1[Vidx] = org[Vidx];
		points->SetPoint(1,x1);
		tc[0] = 1.0;
		tc[1] = 0.0;
		tcoords->SetTuple(1,tc);
		//
		//  UR point 
		//
		x1[Uidx] = org[Uidx] + spa[Uidx]*dimsTexture[0];
		x1[Vidx] = org[Vidx] + spa[Vidx]*dimsTexture[1];
		points->SetPoint(2,x1);
		tc[0] = 1.0;
		tc[1] = 1.0;
		tcoords->SetTuple(2,tc);
		//
		//  UL point 
		//
		x1[Uidx] = org[Uidx];
		x1[Vidx] = org[Vidx] + spa[Vidx]*dimsTexture[1];
		points->SetPoint(3,x1);
		tc[0] = 0.0;
		tc[1] = 1.0;
		tcoords->SetTuple(3,tc);

		//
		// We now assign the pieces to the vtkPolyData.
		//
		output->SetPolys(polys);
		output->SetPoints(points);
		output->GetPointData()->SetTCoords(tcoords);
		polys->Delete();
		points->Delete();
		tcoords->Delete();
	}

	//
	//  Fill in out portion of the texture
	//
 	int ncomIn = input->GetPointData()->GetScalars()->GetNumberOfComponents();
    float *ptrIn = (float *)input->GetScalarPointer();
	float *ptrOut = (float *)mTexture->GetScalarPointer();

	if(ptrOut == 0)
	{
		vtkErrorMacro("Texture data has not been allocated properly.");
		return;
	}

	int ijk[3], ijkmin[3], ijknum[3];

	ijk[Axis] = 0;

	ijkmin[Uidx] = 0;
	ijkmin[Vidx] = 0;
	ijknum[Uidx] = dims[Uidx] - 1;
	ijknum[Vidx] = dims[Vidx] - 1;

	if(mOffX < 0) ijkmin[Uidx] -= mOffX;
	if(mOffY < 0) ijkmin[Vidx] -= mOffY;
	if(mOffX+ijknum[Uidx] > dimsTexture[0]) ijknum[Uidx] = dimsTexture[0] - mOffX;
	if(mOffY+ijknum[Vidx] > dimsTexture[1]) ijknum[Vidx] = dimsTexture[1] - mOffY;

	vtkIdType off1 = 0, off2 = 0, off3;
	switch(Axis)
	{
	case 0:
		{
			//
			//  Uidx = 1, Vidx = 2;
			//
			off1 = dims[0];
			off2 = dims[0]*dims[1];
			break;
		}
	case 1:
		{
			//
			//  Uidx = 0, Vidx = 1;
			//
			off1 = 1;
			off2 = dims[0]*dims[1];
			break;
		}
	case 2:
		{
			//
			//  Uidx = 0, Vidx = 1;
			//
			off1 = 1;
			off2 = dims[0];
			break;
		}
	}

	off1 *= ncomIn;
	off2 *= ncomIn;
	off3 = off1 + off2;

	vtkIdType lIn, lOut;
	for(ijk[Vidx]=ijkmin[Vidx]; ijk[Vidx]<ijknum[Vidx]; ijk[Vidx]++)
	{
		this->UpdateProgress((float)(ijk[Vidx]-ijkmin[Vidx])/(ijknum[Vidx]-ijkmin[Vidx]+1));
		if(this->GetAbortExecute()) break;

		for(ijk[Uidx]=ijkmin[Uidx]; ijk[Uidx]<ijknum[Uidx]; ijk[Uidx]++)
		{
			lIn = ncomIn*(ijk[0]+dims[0]*(ijk[1]+(vtkIdType)dims[1]*ijk[2]));
			lOut = mOffX + ijk[Uidx] + dimsTexture[0]*(mOffY+ijk[Vidx]);
			ptrOut[lOut] = 0.25f*(ptrIn[lIn]+ptrIn[lIn+off1]+ptrIn[lIn+off2]+ptrIn[lIn+off3]);
//			ptrOut[lOut] = pow10(-0.1*float(ijk[Uidx]));
		}
	}

	mTexture->Modified();
}

