/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_QT)


#include "iqtwidgettexteditorsubject.h"


#include "ierror.h"
#include "istring.h"

#include "iggwidgettext.h"

#include "ibgwindowhelper.h"

#include "iqthelper.h"
#include "iqtwidgethelper.h"

#if defined(IQT_3) || defined(IQT_3SUPPORT)
#else
#include <QtGui/QMouseEvent>
#endif

#include "iggparameter.h"
using namespace iggParameter;


#ifndef I_NO_SYNTAX_HIGHLIGHTING

//
//  Syntax highlighting class
//

#if defined(IQT_3) || defined(IQT_3SUPPORT)

#include <private/qrichtext_p.h>

#else

namespace Qt
{
	const int AlignAuto = 0;
};
#include <Qt3Support/private/q3richtext_p.h>
#include <Qt3Support/Q3IntDict>

#define QTextPreProcessor Q3TextPreProcessor
#define QTextDocument Q3TextDocument
#define QTextParagraph Q3TextParagraph
#define QTextFormat Q3TextFormat
#define QIntDict Q3IntDict

#endif

#ifndef QT_VERSION
#error "QT_VERSION is not defined."
!TERMINATE!
#endif

#if (QT_VERSION > 0x030004)
#define QTextParag QTextParagraph
#endif

#include "iarray.h"
#include "iarraytemplate.h"


class iqtSyntaxHighlighter : public QTextPreProcessor
{

public:

	iqtSyntaxHighlighter() : QTextPreProcessor(), mLastFormat(0), mLastFormatId(-1)
	{
		mActive = false;

		mCommentString = "";
		mHighlightComments = mHighlightStrings = false;

		mDefaultFont = QFont("courier",10,QFont::Normal);

		this->AddFormat(0,new QTextFormat(mDefaultFont,Qt::black));
		this->AddFormat(1,new QTextFormat(mDefaultFont,Qt::darkGreen));
		this->AddFormat(2,new QTextFormat(mDefaultFont,Qt::darkRed));

		mFirstFormat = 3;
	}

	virtual ~iqtSyntaxHighlighter()
	{
		while(mKeywords.Size() > 0) delete mKeywords.RemoveLast();
	}

	void SetPointSize(int s)
	{
		unsigned int i;
		for(i=0; i<mFormats.count(); i++)
		{
			this->format(i)->setPointSize(s);
			this->format(i)->setBold((s > 13));
		}
		mDefaultFont = QFont(mDefaultFont.family(),s,(s>13)?QFont::Bold:QFont::Normal);
	}

	void process(QTextDocument*, QTextParag *string, int, bool)
	{
		int i, j, len, index, index2, indexMax = -1;
		static QString line;
		static QString alphabeth = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-_";
		static QString mathChars = "xXeE";
		static QString numbers = "0123456789";

		line = string->string()->toString();
		line = line.left(line.length()-1);
		if(line.isEmpty() || !mActive) return;
		//
		//  Highlight comments first
		//
		if(mHighlightComments)
		{
#if defined(IQT_3) || defined(IQT_3SUPPORT)
			indexMax = line.find(mCommentString,0);
#else
			indexMax = line.indexOf(mCommentString,0);
#endif
			if(indexMax == -1) 
			{
				indexMax = line.length(); 
			}
			else
			{
				string->setFormat(indexMax,line.length()-indexMax,this->format(1),false);
			}

			if(indexMax == 0) return; // whole line is a comment
		}

		string->setFormat(0,indexMax,this->format(0),false);
		//
		//  Highlight reserved words
		//
		for(j=0; j<mKeywords.Size(); j++)
		{
			QStringList &list = *mKeywords[j];
			for(i=0; i<(int)list.count(); i++)
			{
				len = list[i].length();
				index = 0;
#if defined(IQT_3) || defined(IQT_3SUPPORT)
				while((index=line.find(list[i],index))!=-1 && index<indexMax && (index==0 || alphabeth.find(line.at(index-1),0)==-1) && alphabeth.find(line.at(index+len),0)==-1)
#else
				while((index=line.indexOf(list[i],index))!=-1 && index<indexMax && (index==0 || alphabeth.indexOf(line.at(index-1),0)==-1) && alphabeth.indexOf(line.at(index+len),0)==-1)
#endif
				{
					string->setFormat(index,len,this->format(mFirstFormat+j),false);
					index += len;
				}
			}
		}

		//
		//  Highlight strings
		//
		if(mHighlightStrings)
		{
			index2 = -1;
#if defined(IQT_3) || defined(IQT_3SUPPORT)
			while((index=line.find('"',index2+1)) != -1)
			{
				index2 = line.find('"',index+1);
#else
			while((index=line.indexOf('"',index2+1)) != -1)
			{
				index2 = line.indexOf('"',index+1);
#endif
				if(index2 == -1) index2 = indexMax - 1; 
				string->setFormat(index,index2-index+1,this->format(2),false);
			}
		}
	}

	QTextFormat* format(int id)
	{
		if(mLastFormatId==id && mLastFormat) return mLastFormat;

		QTextFormat *f = mFormats[id];
		mLastFormat = f ? f : mFormats[ 0 ];
		mLastFormatId = id;
		return mLastFormat;
	}

	void Activate(bool s)
	{
		mActive = s;
	}

	void AddKeyword(const QString &s, int category)
	{
		if(category>=0 && category<mKeywords.Size())
		{
			mKeywords[category]->append(s);
		}
	}

	void AddCategory(const QColor &c)
	{
		this->AddFormat(mFirstFormat+mKeywords.Size(),new QTextFormat(mDefaultFont,c));
		mKeywords.Add(new QStringList);
	}

	void SetHighlightComments(const QString &s, const QColor &c)
	{
		mHighlightComments = !s.isEmpty();
		if(mHighlightComments)
		{
			mCommentString = s;
			this->format(1)->setColor(c);
		}
	}

	void SetHighlightStrings(const QColor &c)
	{
		mHighlightStrings = true;
		if(mHighlightStrings)
		{
			this->format(2)->setColor(c);
		}
	}

private:

	void AddFormat(int id, QTextFormat *f)
	{
		mFormats.insert(id,f);
	}

	QTextFormat *mLastFormat;
	int mLastFormatId;
	QIntDict<QTextFormat> mFormats;

	iArray<QStringList*> mKeywords;
	QString mCommentString;
	bool mActive, mHighlightComments, mHighlightStrings;
	QFont mDefaultFont;
	int mFirstFormat;
};

#endif

//
//  I use Qt3 support for now, since QTextEdit changed so much.
//
#define IQT_3SUPPORT


//
// Constructs an editor and connects a syntax highlighter
//
iqtWidgetTextEditorSubject::iqtWidgetTextEditorSubject(iggWidgetTextEditor *owner, unsigned int mode) : QTextEdit(iqtHelper::Convert(owner->GetParent())), ibgWidgetTextEditorSubject(owner,mode), mMode(mode)
{
	mWidgetHelper = new iqtWidgetHelper(this,owner); IERROR_ASSERT(mWidgetHelper);

	this->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding));
	this->setFocusPolicy(QFocus::StrongFocus);

	//
	//  Configure editor
	//
	QFont f(this->font());
	f.setFamily("courier");
	f.setFixedPitch(true);
	iqtHelper::SetFont(this,f);
	if(mode & _TextEditorWrapping)
	{
#if defined(IQT_3) || defined(IQT_3SUPPORT)
		this->setWordWrap(QTextEdit::WidgetWidth);
#else
		this->setLineWrapMode(QTextEdit::WidgetWidth);
#endif
	}
	else
	{
#if defined(IQT_3) || defined(IQT_3SUPPORT)
		this->setWordWrap(QTextEdit::NoWrap);
#else
		this->setLineWrapMode(QTextEdit::NoWrap);
#endif
	}
	this->setReadOnly(mode & _TextEditorReadOnly);
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	if(mode & _TextEditorWithHTML) this->setTextFormat(Qt::RichText); else this->setTextFormat(Qt::PlainText);
#endif

#ifndef I_NO_SYNTAX_HIGHLIGHTING
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	mSyntaxHighlighter = new iqtSyntaxHighlighter; IERROR_ASSERT(mSyntaxHighlighter);
	this->document()->setPreProcessor(mSyntaxHighlighter);
#else
	mSyntaxHighlighter = 0;
#endif
#else
	mSyntaxHighlighter = 0;
#endif

	if(!connect(this,SIGNAL(textChanged()),this,SLOT(OnTextChanged()))) IERROR_LOW("Missed connection.");
	if(!connect(this,SIGNAL(selectionChanged()),this,SLOT(OnSelectionChanged()))) IERROR_LOW("Missed connection.");
	if(!connect(this,SIGNAL(cursorPositionChanged(int,int)),this,SLOT(OnCursorPositionChanged(int,int)))) IERROR_LOW("Missed connection.");
	if(!connect(this,SIGNAL(returnPressed()),this,SLOT(OnReturnPressed()))) IERROR_LOW("Missed connection.");
}


iqtWidgetTextEditorSubject::~iqtWidgetTextEditorSubject()
{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	this->document()->setPreProcessor(0);
	delete mSyntaxHighlighter;
#endif
#endif
}


bool iqtWidgetTextEditorSubject::SupportsHTML() const
{
	return (mMode & _TextEditorWithHTML);
}
	

bool iqtWidgetTextEditorSubject::TextModified() const
{
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	return this->isModified();
#else
	return this->document()->isModified();
#endif
}
	

bool iqtWidgetTextEditorSubject::AppendBreaksLine() const
{
	if(this->SupportsHTML()) return strcmp(qVersion(),"3.0.4") > 0; else return true;
}


//
//  Text manipulations
//
void iqtWidgetTextEditorSubject::Clear()
{
	this->blockSignals(true);
	this->clear();
	this->blockSignals(false);
}


void iqtWidgetTextEditorSubject::SetModified(bool s)
{
	this->blockSignals(true);
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	this->setModified(s);
#else
	this->document()->setModified(s);
#endif
	this->blockSignals(false);
}


void iqtWidgetTextEditorSubject::SetText(const iString &text)
{
	this->blockSignals(true);
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	this->setText(iqtHelper::Convert(text));
#else
	if(this->SupportsHTML()) this->setHtml(iqtHelper::Convert(text)); else this->setPlainText(iqtHelper::Convert(text));
#endif
	this->blockSignals(false);
}


iString iqtWidgetTextEditorSubject::GetText() const
{
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	return iqtHelper::Convert(this->text());
#else
	if(this->SupportsHTML()) return iqtHelper::Convert(this->toHtml()); else return iqtHelper::Convert(this->toPlainText());
#endif
}


void iqtWidgetTextEditorSubject::AppendText(const iString &text, bool scroll)
{
	this->blockSignals(true);
	//
	//  A work-around for a bug in some versions - needed on idunn, for example
	//
	// if(this->GetText().IsEmpty()) this->SetText(text); else
	this->append(iqtHelper::Convert(text));
#if defined(IQT_3) || defined(IQT_3SUPPORT)
	if(scroll) this->scrollToBottom();
#endif
	this->blockSignals(false);
}

//
//  Line-by-line manipulation
//
int iqtWidgetTextEditorSubject::GetNumberOfLines() const
{
	//
	//  Fix a bug that returns 1 for empty text
	//
	int n = this->paragraphs();
	if(n > 1) return n;
	if(this->text().isEmpty())
	{
		return 0;
	}
	else
	{
		return 1;
	}
}


void iqtWidgetTextEditorSubject::RemoveLine(int n)
{
	if(n>=0 && n<this->paragraphs())
	{
		this->blockSignals(true);
		this->removeParagraph(n);
		this->blockSignals(false);
	}
}


iString iqtWidgetTextEditorSubject::GetLine(int n) const
{
	static iString none;

	if(n>=0 && n<this->paragraphs()) return iqtHelper::Convert(this->text(n)); else return none;
}


//
//  Text attribute manipulations
//
void iqtWidgetTextEditorSubject::SetPointSize(int s)
{
	if(!this->SupportsHTML())
	{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
		mSyntaxHighlighter->SetPointSize(s);
#endif
		this->blockSignals(true);
		this->setPointSize(s);
		this->blockSignals(false);
	}
}


void iqtWidgetTextEditorSubject::SetTextColor(const iColor &color)
{
	if(!this->SupportsHTML())
	{
		this->blockSignals(true);
		this->setColor(iqtHelper::Convert(color));
		this->blockSignals(false);
	}
}


void iqtWidgetTextEditorSubject::SetBoldText(bool s)
{
	if(!this->SupportsHTML())
	{
		this->blockSignals(true);
		this->setBold(s);
		this->blockSignals(false);
	}
}


void iqtWidgetTextEditorSubject::SetLineColor(int line, const iColor &color)
{
	this->blockSignals(true);
	this->setParagraphBackgroundColor(line,iqtHelper::Convert(color));
	this->blockSignals(false);
}


void iqtWidgetTextEditorSubject::UnSetLineColor(int line)
{
	this->blockSignals(true);
	this->clearParagraphBackground(line);
	this->blockSignals(false);
}


iColor iqtWidgetTextEditorSubject::GetLineColor(int line) const
{
	return iqtHelper::Convert(this->paragraphBackgroundColor(line));
}


//
//  Selection
//
void iqtWidgetTextEditorSubject::Select(int paraFrom, int indexFrom, int paraTo, int indexTo, bool leftJustified, int selNum)
{
	this->blockSignals(true);
	if(paraFrom>paraTo || (paraFrom==paraTo && indexFrom>=indexTo))
	{
		this->removeSelection(selNum);
	}
	else
	{
		this->setSelection(paraFrom,indexFrom,paraTo,indexTo,selNum);
		if(leftJustified)
		{
			this->setCursorPosition(paraTo,0);
			this->ensureCursorVisible(); 
		}
	}
	this->blockSignals(false);
}


void iqtWidgetTextEditorSubject::OnTextChanged()
{
	this->OnTextChangedBody();
}


void iqtWidgetTextEditorSubject::OnSelectionChanged()
{
	int paraFrom, indexFrom, paraTo, indexTo;

	this->getSelection(&paraFrom,&indexFrom,&paraTo,&indexTo,0);
	if(paraFrom < 0) paraFrom = 0;
	this->OnSelectionChangedBody(paraFrom,indexFrom,paraTo,indexTo);
}


void iqtWidgetTextEditorSubject::OnCursorPositionChanged(int para, int index)
{
	this->OnCursorPositionChangedBody(para,index);
}


void iqtWidgetTextEditorSubject::OnReturnPressed()
{
	this->OnReturnPressedBody();
}


void iqtWidgetTextEditorSubject::GetCursorPosition(int &para, int &index)
{
	this->getCursorPosition(&para,&index);
}


bool iqtWidgetTextEditorSubject::HasEditingFunction(int type) const
{
	switch(type)
	{
	case _TextEditorFunctionUndo:
	case _TextEditorFunctionRedo:
	case _TextEditorFunctionCut:
	case _TextEditorFunctionCopy:
	case _TextEditorFunctionPaste:
		{
			return true;
		}
	default:
		{
			return false;
		}
	}
}


void iqtWidgetTextEditorSubject::UseEditingFunction(int type)
{
	switch(type)
	{
	case _TextEditorFunctionUndo:
		{
			this->undo();
			break;
		}
	case _TextEditorFunctionRedo:
		{
			this->redo();
			break;
		}
	case _TextEditorFunctionCut:
		{
			this->cut();
			break;
		}
	case _TextEditorFunctionCopy:
		{
			this->copy();
			break;
		}
	case _TextEditorFunctionPaste:
		{
			this->paste();
			break;
		}
	default:
		{
		}
	}
}


//
//  Search & find
//
bool iqtWidgetTextEditorSubject::Find(const iString &text, bool cs, bool wo, bool forward, int &para, int &index)
{
	return this->find(iqtHelper::Convert(text),cs,wo,forward,&para,&index);
}


//
//  Syntax highlighting
//
void iqtWidgetTextEditorSubject::HighlightSyntax(bool s)
{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
	mSyntaxHighlighter->Activate(s);
#endif
}


void iqtWidgetTextEditorSubject::AddKeyword(const iString &word, int category)
{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
	mSyntaxHighlighter->AddKeyword(iqtHelper::Convert(word),category);
#endif
}


void iqtWidgetTextEditorSubject::AddCategory(const iColor &color)
{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
	mSyntaxHighlighter->AddCategory(iqtHelper::Convert(color));
#endif
}


void iqtWidgetTextEditorSubject::HighlightComments(const iString &comment, const iColor &color)
{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
	mSyntaxHighlighter->SetHighlightComments(iqtHelper::Convert(comment),iqtHelper::Convert(color));
#endif
}


void iqtWidgetTextEditorSubject::HighlightStrings(const iColor &color)
{
#ifndef I_NO_SYNTAX_HIGHLIGHTING
	mSyntaxHighlighter->SetHighlightStrings(iqtHelper::Convert(color));
#endif
}


void iqtWidgetTextEditorSubject::enterEvent(QEvent * )
{
	//
	//  So that keyboard events are delivered without clicking first
	//
	if(!this->hasFocus())
	{
		this->setFocus();
	}
}


void iqtWidgetTextEditorSubject::contentsMouseDoubleClickEvent(QMouseEvent *e)
{
	if(this->isReadOnly() && e->button()==Qt::LeftButton)
	{
		e->accept();
		this->removeSelection();
		emit returnPressed();
	}
	else QTextEdit::contentsMouseDoubleClickEvent(e);
}


void iqtWidgetTextEditorSubject::keyPressEvent(QKeyEvent *e)
{
	if(this->isReadOnly())
	{
		switch(e->key())
		{
		case Qt::Key_Return:
		case Qt::Key_Enter:
		case Qt::Key_Space:
			{
				if(this->isReadOnly()) emit returnPressed();
				break;
			}
		}

#ifdef IQT_3
		if(e->state() & ControlButton)
#else
		if(e->modifiers() & Qt::CTRL)
#endif
		{
			int para, index;
			this->getCursorPosition(&para,&index);
			//
			//  Add cursor moving if Cntr button is pressed
			//
			switch(e->key())
			{
			case Qt::Key_Up:
				{
					para--;
					this->setCursorPosition(para,index);
					this->ensureCursorVisible();
					e->accept();
					return;
				}
			case Qt::Key_Down:
				{
					para++;
					this->setCursorPosition(para,index);
					this->ensureCursorVisible();
					e->accept();
					return;
				}
			case Qt::Key_PageUp:
				{
					para -= 10;
					this->setCursorPosition(para,index);
					this->ensureCursorVisible();
					e->accept();
					return;
				}
			case Qt::Key_PageDown:
				{
					para += 10;
					this->setCursorPosition(para,index);
					this->ensureCursorVisible();
					e->accept();
					return;
				}
			case Qt::Key_Home:
				{
					para = 0;
					this->setCursorPosition(para,index);
					this->ensureCursorVisible();
					e->accept();
					return;
				}
			case Qt::Key_End:
				{
					para = this->paragraphs() - 1;
					this->setCursorPosition(para,index);
					this->ensureCursorVisible();
					e->accept();
					return;
				}
			}
		}

#ifdef IQT_3
		if(e->state() & ShiftButton)
#else
		if(e->modifiers() & Qt::SHIFT)
#endif
		{
			int paraFrom, indexFrom, paraTo, indexTo;
			this->getSelection(&paraFrom,&indexFrom,&paraTo,&indexTo,0);

			//
			//  Add cursor moving if Cntr button is pressed
			//
			switch(e->key())
			{
			case Qt::Key_Up:
				{
					this->setSelection(--paraFrom,indexFrom,paraTo,indexTo);
					emit selectionChanged();
					e->accept();
					return;
				}
			case Qt::Key_Down:
				{
					this->setSelection(paraFrom,indexFrom,++paraTo,indexTo);
					emit selectionChanged();
					e->accept();
					return;
				}
			case Qt::Key_PageUp:
				{
					paraFrom -= 10;
					this->setSelection(paraFrom,indexFrom,paraTo,indexTo);
					emit selectionChanged();
					e->accept();
					return;
				}
			case Qt::Key_PageDown:
				{
					paraTo += 10;
					this->setSelection(paraFrom,indexFrom,paraTo,indexTo);
					emit selectionChanged();
					e->accept();
					return;
				}
			case Qt::Key_Home:
				{
					paraFrom = 0;
					this->setSelection(paraFrom,indexFrom,paraTo,indexTo);
					emit selectionChanged();
					e->accept();
					return;
				}
			case Qt::Key_End:
				{
					paraTo = this->paragraphs() - 1;
					this->setSelection(paraFrom,indexFrom,paraTo,indexTo);
					emit selectionChanged();
					e->accept();
					return;
				}
			}
		}
	}

	QTextEdit::keyPressEvent(e);
}


void iqtWidgetTextEditorSubject::contentsMousePressEvent(QMouseEvent *e)
{
//	int para;
//	int index = this->charAt(e->pos(),&para);
//	this->OnMousePressBody(para,index);

	QTextEdit::contentsMousePressEvent(e);
}


void iqtWidgetTextEditorSubject::contentsMouseReleaseEvent(QMouseEvent *e)
{
	QTextEdit::contentsMouseReleaseEvent(e);

//	int para;
//	int index = this->charAt(e->pos(),&para);
//	this->OnMouseReleaseBody(para,index);
}

#endif
