/*
 * Copyright (c) 1999 by Sun Microsystem, Inc.
 * All rights reserved.
 *
 * xaux_ext_common.c
 */
#pragma ident	"@(#)xaux_ext_common.c 1.21 99/12/08"

#include <stdio.h>
#include <limits.h>

#include <X11/X.h>
#include <X11/Xatom.h>

#include "logf.h"
#include "iiimpAux.h"
#include "xaux_common.h"
#include "xaux_ext_common.h"

static Bool xaux_xs_send_message();

static Bool
xaux_xs_get_sowin(
	xaux_class_t	*xc,
	Display		*display)
{
	size_t		i;


	xc->sowin = XGetSelectionOwner(display, xc->atom_sowin);

	if (xc->sowin == None)
		return False;
	else
		return True;
}

static Bool
xaux_xs_send_property(
	Display *		display,
	xaux_class_t *		xc,
	int			im_id,
	int			ic_id,
	const unsigned char *	p,
	int			len)
{
	if ((xc->sowin == None) &&
		(xaux_xs_get_sowin(xc, display) == False)) {
		return False;
	}

	XChangeProperty(display, xc->sowin,
		xc->atom_xs[xc->atom_xs_idx], XA_STRING,
		8, PropModeReplace, (unsigned char *)p, len);

	if (xaux_xs_send_message(display, xc, im_id, ic_id,
		AUX_EXT_DATA_SETVALUE, xc->atom_xs[xc->atom_xs_idx]) == False) {
			return False;
	}

	/* XFlush() has been called in xaux_xs_send_message() */

	if (++xc->atom_xs_idx == xc->atom_xs_num)
		xc->atom_xs_idx = 1;
	
	return True;
}

static Bool
xaux_xs_send_message(
	Display *	display,
	xaux_class_t *	xc,
	int		im_id,
	int		ic_id,
	aux_ext_data_type_t	type,
	Atom		atom)
{
	XClientMessageEvent	event;

	if ((xc->sowin == None) &&
		(xaux_xs_get_sowin(xc, display) == False)) {
		return False;
	}
		
	event.type = ClientMessage;
	event.serial = 0;
	event.send_event = True;
	event.display = display;
	event.window = xc->sowin;
	event.message_type = xc->atom_xs[0];
	event.format = 32;

	event.data.l[0] = xc->atom_classname;
	event.data.l[1] = ((CARD16)im_id << 8) | ((CARD16)ic_id & 0xffff);
	event.data.l[2] = xc->index;
	event.data.l[3] = type;
	if (type == AUX_EXT_DATA_SETVALUE) {
		event.data.l[4] = atom;
	} else {
		event.data.l[4] = 0; /* unused */
	}

	XSendEvent(display, xc->sowin, True, 0, (XEvent *)(&event));

	XFlush(display);

	return True;
}

static Bool
xaux_ext_process_property_update(
	Display	*		display,
	Window			window,
	Atom			atom)
{
	Atom		actual_type_return;
	int		actual_format_return;
	unsigned long	nitem_return;
	unsigned long	bytes_after_return;
	unsigned char *	prop_return;
	int		r;
	int		imid;
	int		size;
	unsigned char *	p;
	int		i;
	int		n=0;
	XPoint		point;
	Bool		rv;
	int		type;
	aux_ext_data_t	aux_ext_data_;
	aux_ext_data_t 	*aux_ext_data = &(aux_ext_data_);
	xaux_class_t	*xc = &xaux_class;
	char *		outbuf_ = NULL;

	if (window != xc->extwin && window != xc->sowin) {
		return False;
	}

	r = XGetWindowProperty(display, window,
			       atom, 0, INT_MAX, False,
			       AnyPropertyType, &actual_type_return,
			       &actual_format_return, &nitem_return,
			       &bytes_after_return, &prop_return);

	if (r != Success || actual_type_return == 0) {
		return False;
	}

	type = SX_PROP_TYPE(prop_return);

	/* ClientMessage detoured to property, due to extwin not
	   available yet */
	if (type == AUX_DATA_START || type == AUX_DATA_DONE) {

		if (SX_PROP_ATOM_AUX_NAME(prop_return) != xc->atom_classname) {
			return False;
		}

		aux_ext_data->im = SX_PROP_IMID(prop_return);
		aux_ext_data->ic = SX_PROP_ICID(prop_return);
		aux_ext_data->aux_index = SX_PROP_INDEX(prop_return);

		switch (type) {
		case AUX_EXT_DATA_START:
			logf("[%s] received START via property\n",
				xc->classname);
			return xaux_ext_Start(xc, aux_ext_data);
			break;
		case AUX_EXT_DATA_DONE:
			logf("[%s] received DONE via property\n",
				xc->classname);
			return xaux_ext_Done(xc, aux_ext_data);
			break;
		default:
			return False;
		}
	}

	/* header */

	aux_ext_data->type = AUX_EXT_DATA_DRAW;
	aux_ext_data->im = SX_PROP_IMID(prop_return);
	aux_ext_data->ic = SX_PROP_ICID(prop_return);
	aux_ext_data->aux_index = xc->index;
	aux_ext_data->aux_name = (unsigned char *)xc->classname;
	aux_ext_data->aux_name_length =
		strlen((const char *)aux_ext_data->aux_name);

	/* int values */

	aux_ext_data->integer_count = SX_PROP_INT_COUNT(prop_return);

	if (aux_ext_data->integer_count > 0) {
		aux_ext_data->integer_list =
			(int *)SX_PROP_INT_LIST(prop_return);
	} else {
		aux_ext_data->integer_list = NULL;
	}

	/* string values */

	aux_ext_data->string_count = SX_PROP_STR_COUNT(prop_return);

	if (aux_ext_data->string_count > 0) {
		unsigned char * prop_str = SX_PROP_STR_LIST(prop_return);
		char *		outbuf;
		size_t		outbufsiz, c;

		if ((aux_ext_data->string_list =
			(aux_ext_string_t *)malloc(sizeof (aux_ext_string_t) *
				aux_ext_data->string_count)) == NULL) {
				XFree(prop_return);
				return False;
		}

		for(i = 0; i < aux_ext_data->string_count; i++) {
			char *		ib;
			size_t		ibl;

			/* assign length of a string to ibl */
			ibl = (size_t)*((CARD16 *)(prop_str));
			/* move prop_str to point head of the string */
			prop_str += sizeof(CARD16);
			/* assign head of the string to ib */
			ib = (char *)prop_str;
			/* move prop_str to point lenght of next string */
			prop_str += (ibl + padding[(sizeof(CARD16) + ibl) % 4]);

			aux_ext_data->string_list[i].ptr = (unsigned char *)ib;
			aux_ext_data->string_list[i].length = ibl;
		}
	} else {
		aux_ext_data->string_list = NULL;
	}

	aux_ext_data->string_ptr = NULL;

	aux_ext_data->clientwin = SX_PROP_CLIENTWIN(prop_return);
	aux_ext_data->point.x = SX_PROP_POSX(prop_return);
	aux_ext_data->point.y = SX_PROP_POSY(prop_return);
	aux_ext_data->focuswin = SX_PROP_FOCUSWIN(prop_return);

	logf("[%s] received DRAW via property\n",
		xc->classname);
	rv = xaux_ext_Draw(xc, aux_ext_data);

	free(aux_ext_data->string_list);
	XFree(prop_return);

	return rv;
}

Bool
xaux_ext_process_client_message(
	Display	*		display,
	XClientMessageEvent *	event)
{
	aux_ext_data_t	aux_ext_data_;
	aux_ext_data_t	*aux_ext_data = &(aux_ext_data_);
	aux_ext_data_type_t	type;
	xaux_class_t	*xc = &xaux_class;

	if (event->data.l[0] != xc->atom_classname) {
		return False;
	}

	if (event->message_type != xc->atom_sx[0]) {
		return False;
	}

	if (event->window != xc->extwin) {
		return False;
	}

	aux_ext_data->im = ((CARD32)(event->data.l[1])) >> 16;
	aux_ext_data->ic = ((CARD32)(event->data.l[1])) & 0xffff;
	aux_ext_data->aux_index = (CARD32)(event->data.l[2]);

	type = (CARD32)(event->data.l[3]);

	switch (type) {
	case AUX_EXT_DATA_START:
		logf("[%s] received START via ClientMessage\n",
			xc->classname);
		return xaux_ext_Start(xc, aux_ext_data);
		break;
	case AUX_EXT_DATA_DRAW:
		logf("[%s] notified DRAW via ClientMessage\n",
			xc->classname);
		return xaux_ext_process_property_update(
			display, xc->extwin, (Atom)(CARD32)(event->data.l[4]));
	case AUX_EXT_DATA_DONE:
		logf("[%s] received DONE via ClientMessage\n",
			xc->classname);
		return xaux_ext_Done(xc, aux_ext_data);
		break;
	default:
		return False;
	}
}

#define _NET_WM_STATE_REMOVE        0    /* remove/unset property */
#define _NET_WM_STATE_ADD           1    /* add/set property */
#define _NET_WM_STATE_TOGGLE        2    /* toggle property  */

/* To fix bug 4765026: input method has a window in task list of Gnome desktop */
void xaux_ext_skip_taskbar_hint(Display *display, Window window, Bool add)
{
	static Atom net_wm_state_skip_taskbar = (Atom)0;

	if (!net_wm_state_skip_taskbar) {
		net_wm_state_skip_taskbar = XInternAtom(display, "_NET_WM_STATE_SKIP_TASKBAR", True);
	}

	if (net_wm_state_skip_taskbar != None) {
		XEvent xev;
		xev.xclient.type = ClientMessage;
		xev.xclient.serial = 0;
		xev.xclient.send_event = True;
		xev.xclient.window = window;
		xev.xclient.message_type = XInternAtom (display, "_NET_WM_STATE", True);
		xev.xclient.format = 32;
		xev.xclient.data.l[0] = add ? _NET_WM_STATE_ADD : _NET_WM_STATE_REMOVE;
		xev.xclient.data.l[1] = net_wm_state_skip_taskbar;
		xev.xclient.data.l[2] = net_wm_state_skip_taskbar;

		XSendEvent (display, RootWindow(display, DefaultScreen(display)),
				False,
				SubstructureRedirectMask | SubstructureNotifyMask,
				&xev);
	}
}

Bool
xaux_ext_init_classes(
	Display		*display,
	xaux_class_t	*p,
	Window		extwin)
{
	char		buf[1024];
	int		i;

	logf("xaux_ext_init_classes classname:%s, extwin:%d\n", p->classname, extwin);
	p->atom_classname = XInternAtom(display, p->classname, False);

	sprintf(buf, "%s%s", p->classname, XAUX_SOWIN_SUFFIX);
	p->atom_sowin = XInternAtom(display, buf, False);

	sprintf(buf, "%s%s", p->classname, XAUX_EXTWIN_SUFFIX);
	p->atom_extwin = XInternAtom(display, buf, False);

	for (i = 0; i < p->atom_sx_num; i++) {
		sprintf(buf, "%s%s_%d", p->classname, XAUX_SX_SUFFIX, i);
		p->atom_sx[i] = XInternAtom(display, buf, False);
	}

	p->atom_sx_idx = 1;

	for (i = 0; i < p->atom_xs_num; i++) {
		sprintf(buf, "%s%s_%d", p->classname, XAUX_XS_SUFFIX, i);
		p->atom_xs[i] = XInternAtom(display, buf, False);
	}

	p->atom_xs_idx = 1;

	p->sowin = (Window)0;

	p->extwin = extwin;

	if (XGetSelectionOwner(display, p->atom_extwin) != None) {
		logf("%s: %s already exists.[%s](1)\n",
			ME_EXT, ME_EXT, p->classname);
		return False;
	}

	XSetSelectionOwner(display, p->atom_extwin, p->extwin, CurrentTime);

	if (XGetSelectionOwner(display, p->atom_extwin) != p->extwin) {
		logf("%s: %s already exists.[%s](2)\n",
			ME_EXT, ME_EXT, p->classname);
		XDestroyWindow(display, p->extwin);
		p->extwin = (Window)0;
		return False;
	}

	/* process sx properties which has been put on sowin before
	   extwin is prepared */
	if (xaux_xs_get_sowin(p, display) == True) {
		for (i = p->atom_sx_idx; i < p->atom_sx_num; i++) {
			if (xaux_ext_process_property_update(
				display, p->sowin, p->atom_sx[i]) == False) {
				break;
			}
		}
	}
	return True;
}

Bool
xaux_ext_SetValue(
	Display		*display,
	xaux_class_t	*xc,
	aux_ext_data_t	*aux_ext_data)
{
	static char	*string_buf = NULL;
	static size_t	bufsize = 0;
	size_t		i;
	size_t		total = 0;
	int		*ip;
	char		*sp;
	Bool		rv = True;

	if (aux_ext_data == NULL) {
		/* reset; free string_buf */
		if (string_buf != NULL) {
			free(string_buf);
			string_buf = NULL;
		}
		bufsize = 0;
		return True;
	}

	total = XS_SIZE_PROP_HEADER_SETVALUE
		+ (sizeof (CARD32) * aux_ext_data->integer_count);

	if (aux_ext_data->string_count > 0) {
		for (i = 0; i < aux_ext_data->string_count; i++) {
			size_t	len;

			len = aux_ext_data->string_list[i].length
				* sizeof (CARD16);
			total += ((sizeof (CARD16) + len + 3) / 4) * 4;
		}

		/*
		 * "+1" is required by mb_utf16() method.
		 * The method uses the area for BOM.
		 */
		total += sizeof (CARD16);
	}

	/* tentatively use realloc(); may replace with malloc() later */
	/* othewise, cleanup (freeing) routine should be provided */
	if (total > bufsize) {
		string_buf = realloc(string_buf, total);
		if (string_buf == NULL) {
			bufsize = 0;
			return False;
		}
		bufsize = total;
	}

	XS_PROP_ATOM_AUX_NAME(string_buf) = xc->atom_classname;
	XS_PROP_IMID(string_buf) = aux_ext_data->im;
	XS_PROP_ICID(string_buf) = aux_ext_data->ic;

	XS_PROP_INT_COUNT(string_buf) = aux_ext_data->integer_count;
	XS_PROP_STR_COUNT(string_buf) = aux_ext_data->string_count;


	ip = (int *)XS_PROP_INT_LIST(string_buf);

	if (aux_ext_data->integer_count > 0) {

		for (i = 0; i < aux_ext_data->integer_count; i++) {
			*ip++ = aux_ext_data->integer_list[i];
		}
	}

	sp = (char *)XS_PROP_STR_LIST(string_buf);

	if (aux_ext_data->string_count > 0) {
		char *		ob;
		size_t		obl;

		ob = sp;

		for (i = 0; i < aux_ext_data->string_count; i++) {
			size_t		len;
			int		pn;
			unsigned char	*p;
			size_t		j;

			len = aux_ext_data->string_list[i].length;
			p = aux_ext_data->string_list[i].ptr;

			*(CARD16 *)ob = len;
			ob += sizeof (CARD16);

			for (j = 0; j < len; j++) {
				*ob++ = *p++;
			}
				
			pn = padding[(sizeof (CARD16) + len) % 4];

			/* padding */
			for (j = 0; j < pn; j++) {
				*ob++ = 0U;
			}
			sp = ob;
		}
	}

	logf("ext_SetValue[%s] im:0x%x ic:0x%x in=%d sn=%d\n",
		xc->classname, aux_ext_data->im, aux_ext_data->ic,
		aux_ext_data->integer_count, aux_ext_data->string_count);

	if (aux_ext_data->integer_count != 0 ||
		aux_ext_data->string_count != 0) {
		rv = xaux_xs_send_property(display, xc,
			aux_ext_data->im, aux_ext_data->ic,
			(unsigned char *)string_buf, (sp - &(string_buf[0])));
	}

	return (rv);
}

