#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <linux/ipmi_ioctls.h>
#include <sdr.h>
#include <sel.h>
#include <sensor_events.h>

static char *parse_sel_entry(SEL_RESP *entry);

extern SENSOR *sensor_head;

int dl_sel(int fd)
{
	SEL_INFO_RESP	sel_info_resp;
	SEL_REQ				sel_req;
	SEL_RESP			sel_resp;
	IPMI_XFER			xfer;
	IPMI_XFER			*xferp = &xfer;
	unsigned char	cc;
	int						num_entries, rc;
	char					*sel_string;
	
	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_GET_SEL_INFO);
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("dl_sel(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("dl_sel(): failed (0x%.2x)\n",cc);
		return(-1);
		}
	GET_RESPONSE_DATA(xferp, &sel_info_resp);
	num_entries = sel_info_resp.num_entries;

	printf("# of SEL entries -=> %d\n",num_entries);
	if (!num_entries)
		return(0);

	sel_req.resid = 0x0000;
	sel_req.recid = 0x0000;
	sel_req.offset = 0x00;
	sel_req.length = SEL_LENGTH_ALL;
	while(1)
		{
		INIT_XFER(xferp);
		SET_REQUEST_LUN(xferp, 0);
		SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
		SET_REQUEST_CMD(xferp, CMD_GET_SEL_ENTRY);
		SET_REQUEST_DATA(xferp, (unsigned char *) &sel_req, sizeof(sel_req));
		if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
			{
			printf("dl_sel(): failed (%m)\n");
			return(-1);
			}
		GET_RESPONSE_CC(xferp, cc);
		if (cc != 0x00)
			{
			printf("dl_sel(): failed (0x%.2x)\n",cc);
			return(-1);
			}
		GET_RESPONSE_DATA(xferp, &sel_resp);
		sel_string = parse_sel_entry(&sel_resp);
		printf("%s\n",sel_string);
		free(sel_string);

		num_entries--;
		if (sel_resp.next_id == (unsigned short) 0xffff)
			break;
		else
			{
			sel_req.resid = 0x0000;
			sel_req.recid = sel_resp.next_id;
			sel_req.offset = 0x00;
			sel_req.length = SEL_LENGTH_ALL;
			}
		}
}

int clear_sel(int fd)
{
	SEL_RESERVATION_RESP	sel_res_resp;
	SEL_CLEAR_REQ					sel_clear_req;
	IPMI_XFER							xfer;
	IPMI_XFER							*xferp = &xfer;
	unsigned short				res_id;
	unsigned char					cc;
	int										rc;

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_RESERVE_SEL);
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("clear_sel(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("clear_sel(): failed (0x%.2x)\n",cc);
		return(-1);
		}
	GET_RESPONSE_DATA(xferp, &sel_res_resp);

	printf("sel reservation resp = 0x%.4x\n",sel_res_resp.res_id);
	
	memset(&sel_clear_req, 0, sizeof(sel_clear_req));
	sel_clear_req.res_id = sel_res_resp.res_id;
	sel_clear_req.clr0=0x43;
	sel_clear_req.clr1=0x4c;
	sel_clear_req.clr2=0x52;
	sel_clear_req.op = 0xaa;

	INIT_XFER(xferp);
	SET_REQUEST_LUN(xferp, 0);
	SET_REQUEST_NETFN(xferp, STORAGE_REQUEST);
	SET_REQUEST_CMD(xferp, CMD_CLEAR_SEL);
	SET_REQUEST_DATA(xferp, (unsigned char *) &sel_clear_req, sizeof(sel_clear_req));
	if ((rc = ioctl(fd, IOCTL_IPMI_XFER, (void *) &xfer))<0)
		{
		printf("clear_sel(): failed (%m)\n");
		return(-1);
		}
	GET_RESPONSE_CC(xferp, cc);
	if (cc != 0x00)
		{
		printf("clear_sel(): failed (0x%.2x)\n",cc);
		return(-1);
		}
	printf("SEL has been erased\n");
	return(0);
}

/*
 * Structures and constants for converting SEL entry data into English
 */
struct posterror
  {
  unsigned char code0;
  unsigned char code1;
  char          *msg;
  };
static struct posterror posterrlist[] = {
  { 0x62, 0x01, "BIOS unable to apply BIOS update to processor 1"        },
  { 0x63, 0x01, "BIOS unable to apply BIOS update to processor 2"        },
  { 0x64, 0x01, "BIOS does not support current stepping for processor 1" },
  { 0x65, 0x01, "BIOS does not support current stepping for processor 2" },
  
  { 0x00, 0x02, "Failure Fixed Disk"                  },
  { 0x10, 0x02, "Stuck Key"                           },
  { 0x11, 0x02, "Keyboard error"                      },
  { 0x12, 0x02, "Keyboard Controller Failed"          },
  { 0x13, 0x02, "Keyboard locked"                     },
  { 0x20, 0x02, "Monitor type does not match CMOS"    },
  { 0x30, 0x02, "System RAM Failed at offset"         },
  { 0x31, 0x02, "Shadow RAM Failed at offset"         },
  { 0x32, 0x02, "Extended RAM Failed at offset"       },
  { 0x50, 0x02, "System battery is dead"              },
  { 0x51, 0x02, "System CMOS checksum bad"            },
  { 0x60, 0x02, "System timer error"                  },
  { 0x70, 0x02, "Real time clock error"               },
  { 0x97, 0x02, "ECC Memory error in bank"            },
  { 0xb2, 0x02, "Incorrect Drive A: type"             },
  { 0xb3, 0x02, "Incorrect Drive B: type"             },
  { 0xd0, 0x02, "SYstem cache error - Cache disabled" },
  { 0xf5, 0x02, "DMA Test Failed"                     },
  { 0xf6, 0x02, "Software NMI Failed"                 },
  
  { 0x01, 0x04, "Invalid System Configuration Data" },
  { 0x03, 0x04, "Resource Conflict"                 },
  { 0x04, 0x04, "Resource Conflict"                 },
  { 0x05, 0x04, "Expansion ROM not initialized"     },
  { 0x06, 0x04, "Warning: IRQ not configured"       },
  
  { 0x04, 0x05, "Resource Conflict"             },
  { 0x05, 0x05, "Expansion ROM not initialized" },
  { 0x06, 0x05, "Warning: IRQ not configured"   },
  
  { 0x01, 0x06, "Device configuration changed"          },
  { 0x02, 0x06, "Configuration error - device disabled" },
  
  { 0x00, 0x81, "processor 0 failed BIST"                            },
  { 0x01, 0x81, "processor 1 failed BIST"                            },
  { 0x04, 0x81, "processor 0 Internal Error (IERR) failure"          },
  { 0x05, 0x81, "processor 1 Internal Error (IERR) failure"          },
  { 0x06, 0x81, "processor 0 Thermal Trip failure"                   },
  { 0x07, 0x81, "processor 1 Thermal Trip failure"                   },
  { 0x08, 0x81, "Watchdog Timer failed on last boot, BSP switched"   },
  { 0x0a, 0x81, "processor 1 failed initialization on last boot"     },
  { 0x0b, 0x81, "processor 0 failed initialization on last boot"     },
  { 0x0c, 0x81, "processor 0 disabled, system in Uni-processor mode" },
  { 0x0d, 0x81, "processor 1 disabled, system in Uni-processor mode" },
  { 0x0e, 0x81, "processor 0 failed FRB Level 3 timer"               },
  { 0x0f, 0x81, "processor 1 failed FRB Level 3 timer"               },
  { 0x10, 0x81, "Server Management Interface failed to function"     },
  { 0x20, 0x81, "IOP sub-system is not functional"                   },
  { 0x50, 0x81, "NVRAM (CMOS) Cleared by Jumper"                     },
  { 0x51, 0x81, "NVRAM (CMOS) Checksum Error, NVRAM cleared"         },
  { 0x52, 0x81, "NVRAM (CMOS) Data Invalid, NVRAM cleared"           },
  { 0x00, 0x00, NULL}};
  
/*
 * Converts a binary SEL entry into the English equivalents
 * Note: Caller must free buffer 
 */
static char *parse_sel_entry(SEL_RESP *entry)
{
  SENSOR  *slist;
  char    *buffer;
  int     i;
  char    record_id[8];
  char    type[32];
  char    time[32];
  char    generator_id[32];
  char    emvrev[8];
  char    *sensortype;
  char    sensorname[64];
  char    *descp = NULL;
  char    description[64];
  char    log_msg[255];
  
  memset(description, 0, sizeof(description));
  sprintf(record_id, "0x%.4x", entry->recid);
  if (entry->type == 0x02)
    strcpy(type, "SEL");
  else
    sprintf(type, "Unknown (0x%.2x)", entry->type);
  sprintf(time, "0x%.8x",entry->timestamp);
  
  /*
   * Englishify the generator ID fields
   */
  if (entry->id_type == 0)
    sprintf(generator_id, "IPMB addr 0x%x (LUN %d)",
            entry->id_data,
            entry->ipmb_lun);
  else      
    sprintf(generator_id, "SSID 0x%x",entry->id_data);
    
  sprintf(emvrev, "0x%.2x", entry->evmrev);
  
  /*
   * Identify the type of sensor (temp, fan, volt, etc)
   */
  sensortype = sensor_get_type(entry->sensortype);
  
  /*
   * Try to lookup the actual sensor (proc 1 fan, +3.3 volt, etc)
   */
  slist = sensor_head;
  while(slist)
    {
    if (slist->rec_header.record_type == 0x01)
      {
      if (entry->sensornum==slist->sensor_type.type_01_sensor.sensor_number)
        {
        decode_string(slist->sensor_type.type_01_sensor.id_string_type_code,
                      0,
                      &slist->sensor_type.type_01_sensor.id_string[0],
                      &sensorname[0],
                      slist->sensor_type.type_01_sensor.id_string_num_bytes);
        break;        
        }
      } 
    else if (slist->rec_header.record_type == 0x02)
      {
      if (entry->sensornum==slist->sensor_type.type_02_sensor.sensor_number)
        {
        decode_string(slist->sensor_type.type_02_sensor.id_string_type_code,
                      0,
                      &slist->sensor_type.type_02_sensor.id_string[0],
                      &sensorname[0],
                      slist->sensor_type.type_02_sensor.id_string_num_bytes);
        break;
        }
      } 
    slist = slist->next;
    }
  if (!slist)
    strcpy(sensorname, "Unknown Sensor Name");
        
/*      
   * If it's a post err, display the error code
   */
  if (entry->sensortype == 0x0f) /* POST error */
    {
    for (i=0; posterrlist[i].code0 != 0x00; i++)
      {  
      if ((posterrlist[i].code0==entry->event_data[1])&&
          (posterrlist[i].code1==entry->event_data[2]))
        { 
        strcpy(description,posterrlist[i].msg);
        break;
        }
      } 
    if (posterrlist[i].code0 == 0x00)
      {
      sprintf(description,"0x%.2x 0x%.2x",entry->event_data[1], entry->event_data[0]);
      }
    } 
  if ((!descp) && (description[0] == 0x0))
    {
    int sensor_class;
    
    /* There was no 'canned' description for this entry */
    sensor_class = sensor_get_class(entry->event_type);
    if (sensor_class == SENSOR_CLASS_THRESHOLD)
      {
#if 1 
      THRESHOLD_EVENT_DATA  *event; 
      event = (THRESHOLD_EVENT_DATA *) entry->event_data;
      descp = sensor_get_trigger_description(entry->event_type,
                                            event->event_trigger_offset);
#else 
      descp = NULL;
#endif 
      }
    else if (sensor_class == SENSOR_CLASS_DIGITAL)
      {
#if 1  
      DIGITAL_EVENT_DATA  *event; 
      event = (DIGITAL_EVENT_DATA *) entry->event_data;
      descp = sensor_get_trigger_description(entry->event_type,
                                            event->digital_event_offset);
#else 
      descp = NULL;
#endif 
      }
    else if (sensor_class == SENSOR_CLASS_DISCRETE)
      {
#if 1 
      DISCRETE_EVENT_DATA *event;
      event = (DISCRETE_EVENT_DATA *) entry->event_data;
      descp = sensor_get_trigger_description(entry->event_type,
                                            event->event_trigger_offset);
#else                                       
      descp = NULL;                         
#endif                                      
      }                                     
    } 
    
  if (!(buffer = (char *) malloc(512)))
    {
    return(0);
    }
  sprintf(buffer,"%s:%s:%s:%s:%s:%s:%s:%s",
          record_id,
          type,
          time,
          generator_id,
          emvrev,
          sensortype,
          (sensorname != NULL ? sensorname : "Unknown Sensor Name"),
          ((descp != (char *) NULL ? descp : 
           (description[0] != (char) NULL ? description : "No Description Available"))));
           
  return(buffer);
}

