/*
 printtext.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint mirc_colors[] =
{
    BWHITE,
    BLACK,
    BLUE,
    GREEN,
    RED,
    YELLOW,
    MAGENTA,
    ORANGE,
    BYELLOW,
    BGREEN,
    CYAN,
    BCYAN,
    BBLUE,
    BMAGENTA,
    BBLACK,
    WHITE,
    BWHITE
};

static gint max_textwidget_lines;
static gint block_remove_lines;

static gboolean is_sb_down(WINDOW_REC *window)
{
    GtkAdjustment *adj;
    WINDOW_VIEW_REC *view;

    g_return_val_if_fail(window != NULL, TRUE);

    view = WINDOW_VIEW(window);
    adj = view->itext ?
	GTK_ITEXT(view->text)->adjustment :
	GTK_TEXT(view->text)->vadj;

    if ((adj->value + (view->itext ? 0 : 10)) < (adj->upper - adj->lower - adj->page_size))
	    return FALSE; /* scrollbar is not down, leave it alone */
    return TRUE; /* scrollbar is down, move it down again after printing */
}

static void set_sb_down(WINDOW_REC *window)
{
    GtkAdjustment *adj;
    WINDOW_VIEW_REC *view;

    g_return_if_fail(window != NULL);

    view = WINDOW_VIEW(window);
    adj = view->itext ?
	GTK_ITEXT(view->text)->adjustment :
	GTK_TEXT(view->text)->vadj;
    gtk_adjustment_set_value(adj, adj->upper - adj->lower - adj->page_size);
}

#ifdef GTK_HEBREW
/* defined in gui-mainwindows.c: */
extern void hebrew_log2vis(guchar *log, guchar *vis);
#endif

static gboolean gui_printtext_itext(CHANNEL_REC *channel, gint fgcolor, gint bgcolor, gint flags, gchar *str)
{
    GUI_WINDOW_REC *gui;
    WINDOW_VIEW_REC *view;
    WINDOW_REC *window;
    GdkColor *fg, *bg;
    gint iflags;

    g_return_val_if_fail(str != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    gui = WINDOW_GUI(window);
    view = WINDOW_VIEW(window);

    if ((flags & PRINTFLAG_BEEP))
        gdk_beep();

    if (flags & PRINTFLAG_MIRC_COLOR)
    {
	/* use mirc coloring */
	fg = fgcolor < 0 || fgcolor > 16 ? fg = &THEME_GUI(current_theme)->colors[current_theme->default_color] :
	    &THEME_GUI(current_theme)->colors[mirc_colors[fgcolor]];
	bg = bgcolor < 0 || bgcolor > 16 ? NULL : &THEME_GUI(current_theme)->colors[mirc_colors[bgcolor]];
    }
    else
    {
	/* default colors */
	fg = fgcolor < 0 || fgcolor > 8 ? fg = &THEME_GUI(current_theme)->colors[current_theme->default_color] :
	    &THEME_GUI(current_theme)->colors[(fgcolor == 0 || fgcolor == 8) ? fgcolor : fgcolor+8];
	bg = bgcolor == -1 ? NULL : &THEME_GUI(current_theme)->colors[bgcolor];
    }

    if (flags & PRINTFLAG_REVERSE)
    {
        GdkColor *tmp;

	tmp = fg;
        fg = bg != NULL ? bg : &THEME_GUI(current_theme)->colors[COLOR_BACKGROUND];
	bg = tmp;
    }

    gtk_itext_freeze(GTK_ITEXT(view->text));
    if (*str == '\n' && max_textwidget_lines > 0 &&
        max_textwidget_lines+block_remove_lines <= window->lines &&
        channel->type != CHANNEL_TYPE_SETUP)
    {
        /* need to remove lines from start.. */
	while (window->lines > max_textwidget_lines)
	{
	    gtk_itext_remove_first_line(GTK_ITEXT(view->text));
	    window->lines--;
	}
    }

    iflags = (flags & PRINTFLAG_BOLD) == 0 && fgcolor < 8 ? 0 : ITEXT_LINECOL_BOLD;
    if (flags & PRINTFLAG_UNDERLINE) iflags |= ITEXT_LINECOL_UNDERLINE;
    gtk_itext_append(GTK_ITEXT(view->text), str, fg, bg, iflags, 10);

    gtk_itext_thaw(GTK_ITEXT(view->text));
    return TRUE;
}

/* Write text to window */
static gboolean gui_printtext(CHANNEL_REC *channel, gpointer fgcolor, gpointer bgcolor, gpointer flags, gchar *str)
{
    GUI_WINDOW_REC *gui;
    WINDOW_VIEW_REC *view;
    WINDOW_REC *window;
    GdkColor *fg, *bg;
    GdkFont *font;
    gint len, fgc, bgc;
#ifdef GTK_HEBREW
    static gchar *vis = NULL;
#endif

    g_return_val_if_fail(str != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    gui = WINDOW_GUI(window);
    view = WINDOW_VIEW(window);

    if (!view->freeze)
    {
	/* freeze the text widget */
	view->freeze = TRUE;
	view->sb_down = is_sb_down(window);

	if (view->itext)
	    gtk_itext_freeze(GTK_ITEXT(view->text));
	else
	    gtk_text_freeze(GTK_TEXT(view->text));
    }

    fgc = GPOINTER_TO_INT(fgcolor);
    bgc = GPOINTER_TO_INT(bgcolor);

    if (view->itext)
    {
        /* Use Irssi text widget */
        return gui_printtext_itext(channel, fgc, bgc, GPOINTER_TO_INT(flags), str);
    }

    /* \n should always be the only character in the string or something weird
       will happen.. (and this code even looks awful ...) */
    if (*str == '\n')
    {
        view->linepos = g_list_append(view->linepos, GINT_TO_POINTER(view->lastlinelen)+1);
        view->lastlinelen = 0;
    }
    else
        view->lastlinelen += strlen(str);

    gtk_text_freeze(GTK_TEXT(view->text));
    if (*str == '\n' && max_textwidget_lines > 0 &&
        max_textwidget_lines+block_remove_lines <= window->lines &&
        channel->type != CHANNEL_TYPE_SETUP)
    {
        /* need to remove lines from start.. */
        len = 0;
        while (window->lines > max_textwidget_lines)
        {
            GList *link;

            link = g_list_first(view->linepos);
            len += GPOINTER_TO_INT(link->data);
            view->linepos = g_list_remove_link(view->linepos, link);
            g_list_free_1(link);
            window->lines--;
        }
        gtk_text_set_point(GTK_TEXT(view->text), 0);
        gtk_text_forward_delete(GTK_TEXT(view->text), len);
        gtk_text_set_point(GTK_TEXT(view->text), gtk_text_get_length(GTK_TEXT(view->text)));
    }

    if (GPOINTER_TO_INT(flags) & PRINTFLAG_BOLD || fgc >= 8)
        font = /*GPOINTER_TO_INT(flags) & PRINTFLAG_UNDERLINE ? font_bolditalic : */CHANNEL_GUI(channel)->font_bold;
    /*else if (GPOINTER_TO_INT(flags) & PRINTFLAG_UNDERLINE)
        font = font_italic;*/
    else
        font = CHANNEL_GUI(channel)->font_normal;

    if ((GPOINTER_TO_INT(flags) & PRINTFLAG_BEEP))
        gdk_beep();

    if (GPOINTER_TO_INT(flags) & PRINTFLAG_MIRC_COLOR)
    {
	/* use mirc coloring */
	fg = fgc < 0 || fgc > 16 ? fg = &THEME_GUI(current_theme)->colors[current_theme->default_color] :
	    &THEME_GUI(current_theme)->colors[mirc_colors[fgc]];
	bg = bgc < 0 || bgc > 16 ? NULL : &THEME_GUI(current_theme)->colors[mirc_colors[bgc]];
    }
    else
    {
	/* default colors */
	fg = fgc < 0 ? fg = &THEME_GUI(current_theme)->colors[current_theme->default_color] :
	    &THEME_GUI(current_theme)->colors[(fgc == 0 || fgc == 8) ? fgc : fgc+8];
	bg = bgc == -1 ? NULL : &THEME_GUI(current_theme)->colors[bgc];
    }

    if (GPOINTER_TO_INT(flags) & PRINTFLAG_REVERSE)
    {
        GdkColor *tmp;

        tmp = fg;
        fg = bg != NULL ? bg : &THEME_GUI(current_theme)->colors[COLOR_BACKGROUND];
        bg = tmp;
    }

#ifdef GTK_HEBREW
    if (vis == NULL)
       vis = malloc(2048);
    hebrew_log2vis(str, vis);
    gtk_text_insert(GTK_TEXT(view->text), font, fg, bg, vis, -1);
#else
    gtk_text_insert(GTK_TEXT(view->text), font, fg, bg, str, -1);
#endif
    gtk_text_thaw(GTK_TEXT(view->text));

    return TRUE;
}

static gboolean gui_printtext_finished(CHANNEL_REC *channel)
{
    WINDOW_VIEW_REC *view;

    g_return_val_if_fail(channel != NULL, FALSE);

    view = WINDOW_VIEW(CHANNEL_PARENT(channel));
    if (view->freeze)
    {
	view->freeze = FALSE;
	if (view->itext)
	    gtk_itext_thaw(GTK_ITEXT(view->text));
	else
	    gtk_text_thaw(GTK_TEXT(view->text));

	if (view->sb_down) set_sb_down(CHANNEL_PARENT(channel));
    }

    return TRUE;
}

void gui_printtext_get_fonts(gchar *font, gchar **normal, gchar **bold)
{
    gchar *fonthead, *fonttail;
    gint num;
    gchar **fonttokens;

    if (font == NULL)
	font = "-misc-fixed-medium-r-semicondensed-*-*-120-*-*-c-*-iso8859-1";

    fonthead = fonttail = g_strdup(font);    
    for (num = 0; num < 5 && *fonttail != '\0'; fonttail++)
    {
        if (*fonttail == '-')
        {
            num++;
            if (num == 3)
            {
                /* here's the end of the head. */
                *fonttail = '\0';
            }
        }
    }

    if (setup_get_bool("toggle_use_fontset"))
    {
	/* tokenize the font string */
	fonttokens = g_strsplit(font, "-", 0);

	*normal = g_strdup_printf("%s-medium-r-%s,-*-*-*-*-*-*-*-%s-*-*-*-*-*-*", fonthead, fonttail, fonttokens[8]);
	*bold = g_strdup_printf("%s-bold-r-%s,-*-*-*-*-*-*-*-%s-*-*-*-*-*-*", fonthead, fonttail, fonttokens[8]);

	g_strfreev(fonttokens);
    }
    else
    {
	*normal = g_strdup_printf("%s-medium-r-%s", fonthead, fonttail);
	*bold = g_strdup_printf("%s-bold-r-%s", fonthead, fonttail);
    }
    g_free(fonthead);

}

gboolean gui_printtext_configure(void)
{
    gchar *normal, *bold;

    gui_printtext_get_fonts(current_theme->font, &normal, &bold);

    if (setup_get_bool("toggle_use_fontset"))
    {
	font_normal = gdk_fontset_load(normal);
	font_bold = gdk_fontset_load(bold);
    }
    else
    {
	font_normal = gdk_font_load(normal);
	font_bold = gdk_font_load(bold);
    }
    if (font_bold == NULL && font_normal != NULL)
	font_bold = font_normal;

    g_free(normal); g_free(bold);

    return TRUE;
}

static gboolean sig_setup(void)
{
    max_textwidget_lines = setup_get_int("max_textwidget_lines");
    block_remove_lines = setup_get_int("block_remove_lines");
    return TRUE;
}

void gui_printtext_init(void)
{
    gui_printtext_configure();

    signal_add("gui print text", (SIGNAL_FUNC) gui_printtext);
    signal_add("print text finished", (SIGNAL_FUNC) gui_printtext_finished);
    signal_add_first("setup changed", (SIGNAL_FUNC) gui_printtext_configure);
    signal_add("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_add("setup changed", (SIGNAL_FUNC) sig_setup);
}

void gui_printtext_deinit(void)
{
    signal_remove("gui print text", (SIGNAL_FUNC) gui_printtext);
    signal_remove("print text finished", (SIGNAL_FUNC) gui_printtext_finished);
    signal_remove("setup changed", (SIGNAL_FUNC) gui_printtext_configure);
    signal_remove("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_remove("setup changed", (SIGNAL_FUNC) sig_setup);
}
