/*
 gui-windows.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "gui-dcc.h"
#include "gui-menu-commands.h"
#include "wordclick.h"

#define iswordcut(a) (isspace((gint) a) || \
    (a) == '\n' || (a) == '\0' || (a) == '"' || \
    (a) == '(' || (a) == ')' || (a) == '[' || (a) == ']' || \
    (a) == '<' || (a) == '>')

#define URL_MAX_LENGTH 200 /*FIXME: no max. limits*/

static gchar mode_flags[] = "spmint";
static gchar *mode_tooltips[] =
{
    N_("Secret"),
    N_("Private"),
    N_("Moderated"),
    N_("Joining needs an invitation"),
    N_("No external messages"),
    N_("Only operators can change the topic"),
    N_("Maximum number of people in channel"),
    N_("Channel password")
};

static gchar *querybuttons[] =
{
    N_("Whois"),
    N_("Send"),
    N_("Chat"),
    N_("Ping"),
    NULL
};

static gint window_create_override;

static void init_colors(GtkWidget *window)
{
    gint n;

    g_return_if_fail(window != NULL);

    for (n = 0; n < sizeof(THEME_GUI(current_theme)->colors)/sizeof(THEME_GUI(current_theme)->colors[0]); n++)
    {
        THEME_GUI(current_theme)->colors[n].pixel =
            (gulong)((THEME_GUI(current_theme)->colors[n].red & 0xff00)*256 +
                     (THEME_GUI(current_theme)->colors[n].green & 0xff00) +
                     (THEME_GUI(current_theme)->colors[n].blue & 0xff00)/256);
        gdk_color_alloc(gtk_widget_get_colormap(window), &THEME_GUI(current_theme)->colors[n]);
    }
}

void gui_window_set_background_pixmap(WINDOW_REC *window, gchar *path)
{
    WINDOW_VIEW_REC *view;
    GtkStyle *style;
    GdkPixmap *pixmap;
    GdkBitmap *mask;
#if defined (HAVE_IMLIB) || defined (HAVE_GNOME)
    GdkImlibImage *img;
#endif
    gint flags;

    g_return_if_fail(window != NULL);
    g_return_if_fail(path != NULL);

    if (*path == '\0')
	return;

    view = WINDOW_VIEW(window);

    flags = (current_theme->flags & THEME_FLAG_BG_SCALED) ? ITEXT_BG_SCALED : 0;
    flags |= (current_theme->flags & THEME_FLAG_BG_SHADED) ? ITEXT_BG_SHADED : 0;
    flags |= (current_theme->flags & THEME_FLAG_BG_SCROLLABLE) ? ITEXT_BG_SCROLLABLE : 0;

    if (view->pixmap != NULL && strcmp(view->pixmap, path) == 0 &&
	(!view->itext || (GTK_ITEXT(view->text)->bg_flags & (ITEXT_BG_SCALED|ITEXT_BG_SHADED|ITEXT_BG_SCROLLABLE)) == flags))
    {
	/* same pixmap, same properties */
	return;
    }

    if (view->pixmap != NULL) g_free(view->pixmap);
    view->pixmap = g_strdup(path);

    if (view->itext)
    {
	gtk_itext_set_background(GTK_ITEXT(view->text), NULL, path, flags);
        return;
    }

#if defined (HAVE_IMLIB) || defined (HAVE_GNOME)
    mask = NULL;
    img = gdk_imlib_load_image(path);
    if (img == NULL)
	pixmap = NULL;
    else
    {
	gdk_imlib_render(img, img->rgb_width, img->rgb_height);
	pixmap = gdk_imlib_move_image(img);
	gdk_imlib_destroy_image(img);
    }
#else
    pixmap = gdk_pixmap_create_from_xpm(view->text->window, &mask, NULL, path);
#endif

    style = gtk_style_copy(view->text->style);
    gtk_style_ref(style);
    style->bg_pixmap[GTK_STATE_NORMAL] = pixmap;
    gtk_widget_set_style(view->text, style);
    gtk_style_unref(style);
}

void gui_window_set_background(WINDOW_REC *window)
{
    WINDOW_VIEW_REC *view;
    GtkStyle *style;
    gchar *path;
    gint flags;

    g_return_if_fail(window != NULL);

    view = WINDOW_VIEW(window);
    if (view->itext && (setup_get_bool("toggle_background_transparent") || current_theme->bg_pixmap == NULL))
    {
	flags = (current_theme->flags & THEME_FLAG_BG_SCALED) ? ITEXT_BG_SCALED : 0;
	flags |= (current_theme->flags & THEME_FLAG_BG_SHADED) ? ITEXT_BG_SHADED : 0;
	flags |= (current_theme->flags & THEME_FLAG_BG_SCROLLABLE) ? ITEXT_BG_SCROLLABLE : 0;
	flags |= setup_get_bool("toggle_background_transparent") ? ITEXT_BG_TRANSPARENT : 0;

	gtk_itext_set_background(GTK_ITEXT(view->text), &THEME_GUI(current_theme)->colors[COLOR_BACKGROUND],
				 current_theme->bg_pixmap, flags);
        if (setup_get_bool("toggle_background_transparent"))
        {
            if (view->pixmap != NULL)
            {
                g_free(view->pixmap);
                view->pixmap = NULL;
            }
	}
	return;
    }
    else if (!view->itext)
    {
        style = gtk_style_copy(view->text->style);
        gtk_style_ref(style);
        style->base[GTK_STATE_NORMAL] = THEME_GUI(current_theme)->colors[COLOR_BACKGROUND];
	style->bg_pixmap[GTK_STATE_NORMAL] = NULL;
        gtk_widget_set_style(view->text, style);
        gtk_style_unref(style);
    }

    if (current_theme->bg_pixmap == NULL)
    {
	if (view->pixmap != NULL)
	{
	    g_free(view->pixmap);
	    view->pixmap = NULL;
	}
    }
    else
    {
	path = convert_home(current_theme->bg_pixmap);
	gui_window_set_background_pixmap(window, path);
	g_free(path);
    }
}

/* signal: button pressed in GtkIText widget */
static void sig_itext_word_clicked(GtkIText *itext, gchar *word, GdkEventButton *event, WINDOW_REC *window)
{
    g_return_if_fail(event != NULL);
    g_return_if_fail(window != NULL);

    if (event->button == 3 && setup_get_bool("toggle_show_menubar"))
    {
        /* create popup menu */
        gui_popupmenu_create(window->active, word, event);
    }
    else if (word != NULL)
    {
        /* left mouse button clicked */
        word_clicked(window, word, NULL, event);
    }
}

/* signal: button pressed in text window */
static gint sig_text_butpress(GtkWidget *widget, GdkEventButton *event, WINDOW_REC *window)
{
    static gboolean cludging = FALSE;
    GdkEventButton kludge_event;

    WINDOW_VIEW_REC *view;
    guint pos, spos, len;
    gchar *text, *word;

    g_return_val_if_fail(widget != NULL, 0);
    g_return_val_if_fail(event != NULL, 0);
    g_return_val_if_fail(window != NULL, 0);

    /* so, here's the problem: we need know what word was just clicked, but
       text widget haven't get button_press_event yet so cursor position is
       still in the last clicked position.. We could use
       gtk_signal_connect_after(), but if we create a popup menu with it,
       text widget's selection gets broken, it never receives the
       button_release_event .. We could also create the popup menu in
       button_release_event but it seems to work in some weird way too ..
       So this kludge seems to work best. */
    if (cludging) return 0;

    /* send faked left button click to text widget */
    cludging = TRUE;
    memcpy(&kludge_event, event, sizeof(kludge_event));
    kludge_event.button = 1;
    kludge_event.type = GDK_BUTTON_PRESS;
    gtk_widget_event(widget, (GdkEvent *) &kludge_event);
    kludge_event.type = GDK_BUTTON_RELEASE;
    gtk_widget_event(widget, (GdkEvent *) &kludge_event);
    cludging = FALSE;

    /* finished, cursor is now in right position! */

    view = WINDOW_VIEW(window);
    len = gtk_text_get_length(GTK_TEXT(view->text));
    pos = gtk_editable_get_position(GTK_EDITABLE(view->text));

    word = NULL;
    if (pos < len)
    {
        text = gtk_editable_get_chars(GTK_EDITABLE(view->text), pos < URL_MAX_LENGTH ? 0 : pos-URL_MAX_LENGTH,
                                      pos+URL_MAX_LENGTH > len ? len : pos+URL_MAX_LENGTH);
        len = spos = pos < URL_MAX_LENGTH ? pos : URL_MAX_LENGTH;
        if (!iswordcut(text[spos]))
            while (spos > 0 && !iswordcut(text[spos-1])) spos--;
        while (!iswordcut(text[len])) len++; len -= spos;

        word = g_new(gchar, len+1);
        memcpy(word, text+spos, len); word[len] = '\0';
        g_free(text);
    }

    if (event->button == 3 && setup_get_bool("toggle_show_menubar"))
    {
        /* create popup menu */
        gui_popupmenu_create(window->active, word, event);
        gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "button_press_event");
    }
    else if (event->button != 3 && word != NULL)
    {
        /* left mouse button clicked */
        word_clicked(window, word, NULL, event);
    }

    if (word != NULL) g_free(word);
    return 0;
}

static gint sig_topicentry_press(GtkWidget *entry, GdkEventButton *event)
{
    gboolean editable;

    g_return_val_if_fail(entry != NULL, 0);
    g_return_val_if_fail(event != NULL, 0);

    if (event->type == GDK_2BUTTON_PRESS)
    {
        editable = !(gboolean) GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(entry), "editable"));
        gtk_object_set_data(GTK_OBJECT(entry), "editable", GINT_TO_POINTER((gint) editable));
        gtk_entry_set_editable(GTK_ENTRY(entry), editable);
    }

    return 0;
}

static void sig_topicentry_activated(GtkWidget *entry, GUI_WINDOW_REC *gui)
{
    CHANNEL_REC *channel;
    gchar *str;

    g_return_if_fail(entry != NULL);
    g_return_if_fail(gui != NULL);

    channel = gui->active->window->active;
    if (channel == NULL || channel->type != CHANNEL_TYPE_CHANNEL || (!channel->chanop && channel->mode_optopic))
    {
        /* can't change topic. */
        gtk_entry_set_text(GTK_ENTRY(entry),
                           (channel == NULL || channel->topic == NULL) ?
                           "" : channel->topic);
        return;
    }

    str = g_strdup_printf("TOPIC %s :%s", channel->name,
			  gtk_entry_get_text(GTK_ENTRY(entry)));
    irc_send_cmd(cur_channel->server, str);
    g_free(str);
}

static void sig_modebutton_toggled(GtkToggleButton *button, gint mode)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;
    NICK_REC *nick;
    GString *str;

    g_return_if_fail(button != NULL);
    if (gtk_object_get_data(GTK_OBJECT(button), "toggling")) return;

    gui = gtk_object_get_data(GTK_OBJECT(button), "window");
    window = gui->active->window;

    nick = nicklist_find(window->active, window->active->server->nick);
    if (!window->active->chanop)
    {
        /* can't change mode */
        gtk_object_set_data(GTK_OBJECT(button), "toggling", GINT_TO_POINTER(TRUE));
        gtk_toggle_button_set_active(button, !button->active);
        gtk_object_set_data(GTK_OBJECT(button), "toggling", GINT_TO_POINTER(FALSE));
        return;
    }

    str = g_string_new(NULL);
    g_string_sprintf(str, "* %c%c", button->active ? '+' : '-', mode_flags[mode]);

    signal_emit("command mode", 3, str->str, window->active->server, window->active);
    g_string_free(str, TRUE);
}

static void sig_limit_set(GtkWidget *widget, GUI_WINDOW_REC *gui)
{
    WINDOW_REC *window;
    gchar *text, *cmd;
    gboolean set;

    if (gtk_object_get_data(GTK_OBJECT(gui->modebuttons[6]), "toggling")) return;

    window = gui->active->window;
    if (!window->active->chanop)
    {
        /* not chanop, can't change limit */
        gtk_object_set_data(GTK_OBJECT(gui->modebuttons[6]), "toggling", GINT_TO_POINTER(TRUE));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[6]), window->active->limit != 0);
        text = g_strdup_printf("%d", window->active->limit);
	gtk_entry_set_text(GTK_ENTRY(gui->limit_entry), text);
	g_free(text);
        gtk_object_set_data(GTK_OBJECT(gui->modebuttons[6]), "toggling", GINT_TO_POINTER(FALSE));
        return;
    }

    text = gtk_entry_get_text(GTK_ENTRY(gui->limit_entry));
    set = widget == gui->limit_entry ? text != '\0' :
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui->modebuttons[6]));
    if (!set)
    {
	/* remove limit */
	cmd = g_strdup("* -l");
    }
    else
    {
	/* set limit */
	if (*text == '\0')
	{
	    /* clicked the L button with limit entry being empty */
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[6]), FALSE);
	    return;
	}
	cmd = g_strdup_printf("* +l %s", text);
    }
    signal_emit("command mode", 3, cmd, window->active->server, window->active);
    g_free(cmd);
}

static void sig_key_set(GtkWidget *widget, GUI_WINDOW_REC *gui)
{
    WINDOW_REC *window;
    gchar *text, *cmd;
    gboolean set;

    if (gtk_object_get_data(GTK_OBJECT(gui->modebuttons[7]), "toggling")) return;

    window = gui->active->window;
    if (!window->active->chanop)
    {
        /* not chanop, can't change key */
        gtk_object_set_data(GTK_OBJECT(gui->modebuttons[7]), "toggling", GINT_TO_POINTER(TRUE));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[7]), window->active->key != NULL);
	gtk_entry_set_text(GTK_ENTRY(gui->key_entry), window->active->key == NULL ? "" : window->active->key);
	gtk_object_set_data(GTK_OBJECT(gui->modebuttons[7]), "toggling", GINT_TO_POINTER(FALSE));
        return;
    }

    text = gtk_entry_get_text(GTK_ENTRY(gui->key_entry));
    set = widget == gui->key_entry ? text != '\0' :
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui->modebuttons[7]));

    if (window->active->key != NULL)
    {
	/* key already set, we need to remove it first.. */
	cmd = g_strdup_printf("* -k %s", window->active->key);
	signal_emit("command mode", 3, cmd, window->active->server, window->active);
	g_free(cmd);
    }

    if (set)
    {
	/* set new key */
	if (*text == '\0')
	{
	    /* clicked the K button with key entry being empty */
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui->modebuttons[7]), FALSE);
	    return;
	}
	cmd = g_strdup_printf("* +k %s", text);
	signal_emit("command mode", 3, cmd, window->active->server, window->active);
	g_free(cmd);
    }
}

static void sig_querybutton_clicked(GtkWidget *button, gpointer num)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;

    gui = gtk_object_get_data(GTK_OBJECT(button), "window");

    window = gui->active->window;
    switch (GPOINTER_TO_INT(num))
    {
	case 0:
	    signal_emit("command whois", 3, window->active->name, window->active->server, window->active);
	    break;
	case 1:
	    gui_dcc_send(window->active->server, window->active->name, NULL);
	    break;
	case 2:
	    signal_emit("command dcc chat", 3, window->active->name, window->active->server, window->active);
	    break;
	case 3:
	    signal_emit("command ping", 3, window->active->name, window->active->server, window->active);
	    break;
    }
}

#ifdef HAVE_GNOME
static GtkWidget *create_button(GtkWidget *widget, gchar *name)
{
    GtkWidget *button, *pixmap;

    button = gtk_button_new();
    pixmap = gnome_stock_pixmap_widget_at_size(widget, name, 12, 14);
    gtk_container_add(GTK_CONTAINER(button), pixmap);
    return button;
}
#endif

static void sig_window_close(GtkWidget *widget, GUI_WINDOW_REC *gui)
{
    if (channels->next != NULL)
	ui_window_destroy(gui->active->window);
}

static void sig_tab_left(GtkWidget *widget, GUI_WINDOW_REC *gui)
{
    GUI_WINDOW_REC *prevwin;
    GtkNotebook *notebook;
    GList *w1, *w2;
    gint num;

    notebook = GTK_NOTEBOOK(gui->parent->notebook);

    /* get the current window's position */
    num = gtk_notebook_page_num(notebook, gui->window);
    if (num <= 0) return;

    /* get the window in left */
    widget = gtk_notebook_get_nth_page(notebook, num-1);
    if (widget == NULL) return;
    prevwin = gtk_object_get_data(GTK_OBJECT(widget), "window");

    /* swap the windows */
    w1 = g_list_find(windows, gui->active->window);
    w2 = g_list_find(windows, prevwin->active->window);

    w1->data = prevwin->active->window;
    w2->data = gui->active->window;

    /* move window in the notebook */
    gtk_notebook_reorder_child(notebook, gui->window, num-1);

    /* redraw the window tabs so that window numbers gets redrawed */
    signal_emit("channel name changed", 2, gui->active->window->active, gui->active->window->active->name);
    signal_emit("channel name changed", 2, prevwin->active->window->active, prevwin->active->window->active->name);
}

static void sig_tab_right(GtkWidget *widget, GUI_WINDOW_REC *gui)
{
    GUI_WINDOW_REC *nextwin;
    GtkNotebook *notebook;
    GList *w1, *w2;
    gint num;

    notebook = GTK_NOTEBOOK(gui->parent->notebook);

    /* get the current window's position */
    num = gtk_notebook_page_num(notebook, gui->window);

    /* get the window in right */
    widget = gtk_notebook_get_nth_page(notebook, num+1);
    if (widget == NULL) return;
    nextwin = gtk_object_get_data(GTK_OBJECT(widget), "window");

    /* swap the windows */
    w1 = g_list_find(windows, gui->active->window);
    w2 = g_list_find(windows, nextwin->active->window);

    w1->data = nextwin->active->window;
    w2->data = gui->active->window;

    /* move window in the notebook */
    gtk_notebook_reorder_child(notebook, gui->window, num+1);

    /* redraw the window tabs so that window numbers gets redrawed */
    signal_emit("channel name changed", 2, gui->active->window->active, gui->active->window->active->name);
    signal_emit("channel name changed", 2, nextwin->active->window->active, nextwin->active->window->active->name);
}

static void gui_window_new(WINDOW_REC *window)
{
    MAIN_WINDOW_REC *parent;
    GUI_WINDOW_REC *gui;
    GtkWidget *hbox, *hbox2, *scrollwin, *label, *paned, *button;
    GtkTooltips *tooltips;
    GtkStyle *style;
    gint num;

    gui = WINDOW_GUI(window);
    parent = gui->parent;

    gui->window = gtk_vbox_new(FALSE, 2);
    gtk_object_set_data(GTK_OBJECT(gui->window), "window", gui);
    gtk_container_border_width(GTK_CONTAINER(gui->window), 2);

    tooltips = gtk_tooltips_new();

    /* add the generic window move buttons */
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(gui->window), hbox, FALSE, FALSE, 0);

    hbox2 = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(hbox), hbox2, FALSE, FALSE, 0);

#ifdef HAVE_GNOME
    button = create_button(hbox, GNOME_STOCK_BUTTON_CANCEL);
    gtk_tooltips_set_tip(tooltips, button, _("Close window"), NULL);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_window_close), gui);
    gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);

    button = create_button(hbox, GNOME_STOCK_PIXMAP_BACK);
    gtk_tooltips_set_tip(tooltips, button, _("Move window tab to left"), NULL);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_tab_left), gui);
    gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);

    button = create_button(hbox, GNOME_STOCK_PIXMAP_FORWARD);
    gtk_tooltips_set_tip(tooltips, button, _("Move window tab to right"), NULL);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_tab_right), gui);
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
#else
    button = gtk_button_new_with_label("X");
    gtk_tooltips_set_tip(tooltips, button, _("Close window"), NULL);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_window_close), gui);
    gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);

    button = gtk_button_new();
    gtk_tooltips_set_tip(tooltips, button, _("Move window tab to left"), NULL);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_tab_left), gui);
    gtk_container_add(GTK_CONTAINER(button), gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
    gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);

    button = gtk_button_new();
    gtk_tooltips_set_tip(tooltips, button, _("Move window tab to right"), NULL);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(sig_tab_right), gui);
    gtk_container_add(GTK_CONTAINER(button), gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
#endif
    /* Add channel topic */
    gui->topicentry = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(gui->topicentry), FALSE);
    gtk_signal_connect(GTK_OBJECT(gui->topicentry), "button_press_event",
                       GTK_SIGNAL_FUNC(sig_topicentry_press), gui);
    gtk_signal_connect(GTK_OBJECT(gui->topicentry), "activate",
                       GTK_SIGNAL_FUNC(sig_topicentry_activated), gui);
    gtk_box_pack_start(GTK_BOX(hbox), gui->topicentry, TRUE, TRUE, 3);

    /* add channel mode buttons */
    gui->channelwidget = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), gui->channelwidget, FALSE, FALSE, 0);

    for (num = 0; num < 6; num++)
    {
        gchar str[2];

        str[0] = toupper(mode_flags[num]); str[1] = '\0';
        gui->modebuttons[num] = gtk_toggle_button_new_with_label(str);
        gtk_tooltips_set_tip(tooltips, gui->modebuttons[num], dgettext(PACKAGE, mode_tooltips[num]), NULL);
        gtk_widget_set_usize(gui->modebuttons[num], 20, -1);
        gtk_object_set_data(GTK_OBJECT(gui->modebuttons[num]), "window", gui);
        gtk_signal_connect(GTK_OBJECT(gui->modebuttons[num]), "toggled",
                           GTK_SIGNAL_FUNC(sig_modebutton_toggled), GINT_TO_POINTER(num));
        gtk_box_pack_start(GTK_BOX(gui->channelwidget), gui->modebuttons[num], FALSE, FALSE, 0);
    }

    /* add query buttons */
    gui->querywidget = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), gui->querywidget, FALSE, FALSE, 0);

    for (num = 0; querybuttons[num] != NULL; num++)
    {
	button = gtk_button_new_with_label(querybuttons[num]);
	gtk_object_set_data(GTK_OBJECT(button), "window", gui);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(sig_querybutton_clicked), GINT_TO_POINTER(num));
	gtk_box_pack_start(GTK_BOX(gui->querywidget), button, FALSE, FALSE, 0);
    }

    paned = gtk_hpaned_new();
    gtk_box_pack_start(GTK_BOX(gui->window), paned, TRUE, TRUE, 0);

    /* View widget */
    gui->views_widget = gtk_vpaned_new();
    gtk_paned_pack1(GTK_PANED(paned), gui->views_widget, TRUE, TRUE);

    /* add channel mode widgets */
    gui->nlist_widget = gtk_vbox_new(FALSE, 0);
    gtk_paned_pack2(GTK_PANED(paned), gui->nlist_widget, FALSE, TRUE);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_widget_set_usize(hbox, 120, -1);
    gtk_box_pack_start(GTK_BOX(gui->nlist_widget), hbox, FALSE, FALSE, 0);

    /* limit */
    gui->modebuttons[6] = gtk_toggle_button_new_with_label("L");
    gtk_tooltips_set_tip(tooltips, gui->modebuttons[6], dgettext(PACKAGE, mode_tooltips[6]), NULL);
    gtk_signal_connect(GTK_OBJECT(gui->modebuttons[6]), "toggled",
		       GTK_SIGNAL_FUNC(sig_limit_set), gui);
    gtk_widget_set_usize(gui->modebuttons[6], 20, -1);
    gtk_box_pack_start(GTK_BOX(hbox), gui->modebuttons[6], FALSE, FALSE, 0);

    gui->limit_entry = gtk_entry_new();
    gtk_signal_connect(GTK_OBJECT(gui->limit_entry), "activate",
		       GTK_SIGNAL_FUNC(sig_limit_set), gui);
    gtk_widget_set_usize(gui->limit_entry, 40, -1);
    gtk_box_pack_start(GTK_BOX(hbox), gui->limit_entry, FALSE, FALSE, 0);

    /* key */
    gui->modebuttons[7] = gtk_toggle_button_new_with_label("K");
    gtk_tooltips_set_tip(tooltips, gui->modebuttons[7], dgettext(PACKAGE, mode_tooltips[7]), NULL);
    gtk_signal_connect(GTK_OBJECT(gui->modebuttons[7]), "toggled",
		       GTK_SIGNAL_FUNC(sig_key_set), gui);
    gtk_widget_set_usize(gui->modebuttons[7], 20, -1);
    gtk_box_pack_start(GTK_BOX(hbox), gui->modebuttons[7], FALSE, FALSE, 0);

    gui->key_entry = gtk_entry_new();
    gtk_signal_connect(GTK_OBJECT(gui->key_entry), "activate",
		       GTK_SIGNAL_FUNC(sig_key_set), gui);
    gtk_widget_set_usize(gui->key_entry, -1, -1);
    gtk_box_pack_start(GTK_BOX(hbox), gui->key_entry, FALSE, FALSE, 0);

    /* number of nicks */
    gui->nlist_label = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(gui->nlist_widget), gui->nlist_label, FALSE, FALSE, 0);

    /* Add nick list */
    scrollwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(gui->nlist_widget), scrollwin, TRUE, TRUE, 0);

    gui->nicklist = gtk_clist_new(1);
    gtk_container_add(GTK_CONTAINER(scrollwin), gui->nicklist);

    /* nick list background */
    style = gtk_style_copy(GTK_WIDGET(gui->nicklist)->style);
    gtk_style_ref(style);
    style->base[0] = THEME_GUI(current_theme)->colors[COLOR_NICKLIST_BACKGROUND];
    gtk_widget_set_style(gui->nicklist, style);
    gtk_style_unref(style);

    gtk_widget_show_all(gui->window);
    gtk_widget_hide(gui->nlist_widget);

    label = gtk_label_new(_("(empty)"));
    gtk_widget_show(label);

    if (GTK_NOTEBOOK(parent->notebook)->children == NULL)
    {
        /* only one child, don't show the notebook tabs.. */
        gtk_notebook_set_show_tabs(GTK_NOTEBOOK(parent->notebook), FALSE);
        gtk_notebook_set_show_border(GTK_NOTEBOOK(parent->notebook), FALSE);
    }
    else if (GTK_NOTEBOOK(parent->notebook)->children->next == NULL)
    {
        /* two children, show the notebook tabs.. */
        gtk_notebook_set_show_tabs(GTK_NOTEBOOK(parent->notebook), TRUE);
        gtk_notebook_set_show_border(GTK_NOTEBOOK(parent->notebook), TRUE);
    }

    gtk_notebook_append_page(GTK_NOTEBOOK(parent->notebook), gui->window, label);

    gtk_widget_hide(gui->channelwidget);
    gtk_widget_hide(gui->querywidget);

    signal_emit("gui window created", 1, window);
}

static gboolean gui_window_created(WINDOW_REC *window, gpointer automatic)
{
    MAIN_WINDOW_REC *parent;
    WINDOW_VIEW_REC *view;
    GUI_WINDOW_REC *gui;
    GtkWidget *hbox, *scrollwin, *paned;

    g_return_val_if_fail(window != NULL, FALSE);

    parent = (!setup_get_bool("toggle_use_tabbed_windows") || cur_channel == NULL || window_create_override == 0) &&
	window_create_override < 1 ? gui_mainwindow_create(window) :
	(automatic ? mainwindows->data : WINDOW_GUI(CHANNEL_PARENT(cur_channel))->parent);

    view = g_new0(WINDOW_VIEW_REC, 1);
    window->gui_data = view;

    if (window_create_override == 2)
    {
	/* create split window */
	gui = WINDOW_GUI(CHANNEL_PARENT(cur_channel));
    }
    else
    {
	/* new tab */
	gui = g_new0(GUI_WINDOW_REC, 1);
	gui->active = view;
	gui->parent = parent;
    }

    view->gui = gui;
    view->window = window;
    gui->views = g_list_append(gui->views, view);

    if (parent->children == NULL) parent->active = window;
    parent->children = g_list_append(parent->children, window);

    if (window_create_override != 2)
	gui_window_new(window);

    /* Text window */
    hbox = gtk_hbox_new(FALSE, 3);
    if (window_create_override == 2)
    {
	paned = gtk_vpaned_new();
	gtk_paned_pack1(GTK_PANED(gui->views_widget), paned, TRUE, TRUE);
	gtk_paned_pack2(GTK_PANED(paned), hbox, TRUE, TRUE);
	gui->views_widget = paned;
	gtk_widget_show_all(paned);
    }
    else
    {
	gtk_paned_pack2(GTK_PANED(gui->views_widget), hbox, TRUE, TRUE);
    }

    view->itext = setup_get_bool("toggle_use_itext");
    if (view->itext)
    {
        GtkWidget *scrollbar;

        view->text = gtk_itext_new();
        gtk_signal_connect(GTK_OBJECT(view->text), "word_clicked",
                           GTK_SIGNAL_FUNC(sig_itext_word_clicked), window);
        gtk_itext_set_fonts(GTK_ITEXT(view->text), font_normal, font_bold);
        gtk_box_pack_start(GTK_BOX(hbox), view->text, TRUE, TRUE, 0);

        scrollbar = gtk_vscrollbar_new(GTK_ITEXT(view->text)->adjustment);
        gtk_box_pack_start(GTK_BOX(hbox), scrollbar, FALSE, FALSE, 0);
    }
    else
    {
        view->text = gtk_text_new(NULL, NULL);
        gtk_widget_set_usize(view->text, -1, -1);
        gtk_signal_connect(GTK_OBJECT(view->text), "button_press_event",
                           GTK_SIGNAL_FUNC(sig_text_butpress), window);
        gtk_text_set_word_wrap(GTK_TEXT(view->text), TRUE);

        scrollwin = gtk_scrolled_window_new(NULL, NULL);
        gtk_widget_set_usize(scrollwin, -1, -1);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
                                       GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
        gtk_box_pack_start(GTK_BOX(hbox), scrollwin, TRUE, TRUE, 0);
        gtk_container_add(GTK_CONTAINER(scrollwin), view->text);

    }
    gtk_widget_show_all(hbox);

    init_colors(view->text);
    gui_window_set_background(window);

    if (!setup_get_bool("toggle_show_menubar"))
    {
        GtkWidget *popup;

        /* add popup menu created by mainwindow */
        popup = gtk_object_get_data(GTK_OBJECT(parent->window), "popup");
        if (popup != NULL) gnome_popup_menu_attach(popup, view->text, parent);
    }

    signal_emit("gui view created", 1, window);
    window_create_override = -1;
    return TRUE;
}

static gboolean sig_window_create_override(gpointer tab)
{
    window_create_override = GPOINTER_TO_INT(tab);
    return TRUE;
}

static gboolean gui_window_destroyed(WINDOW_REC *window)
{
    MAIN_WINDOW_REC *parent;
    WINDOW_VIEW_REC *view;
    GUI_WINDOW_REC *gui;
    GtkWidget *widget;
    gint page;

    g_return_val_if_fail(window != NULL, FALSE);

    gui = WINDOW_GUI(window);
    parent = gui->parent;
    parent->children = g_list_remove(parent->children, window);

    view = WINDOW_VIEW(window);
    if (view->pixmap != NULL) g_free(view->pixmap);
    g_free(view);

    gui->views = g_list_remove(gui->views, view);
    if (gui->views != NULL)
    {
	/* destroy the next view (and recursively all..) */
	view = gui->views->data;
	ui_window_destroy(view->window);
	return TRUE;
    }

    signal_emit("gui window destroyed", 1, window);

    page = gtk_notebook_page_num(GTK_NOTEBOOK(parent->notebook), gui->window);
    widget = gtk_notebook_get_nth_page(GTK_NOTEBOOK(parent->notebook), page);
    if (widget != NULL) gtk_object_set_data(GTK_OBJECT(widget), "window", NULL);
    gtk_notebook_remove_page(GTK_NOTEBOOK(parent->notebook), page);

    g_free(gui);

    if (parent->children == NULL)
        gui_mainwindow_destroy(parent);
    else if (g_list_length(parent->children) == 1)
    {
        /* only one child left, hide the notebook tabs.. */
        gtk_notebook_set_show_tabs(GTK_NOTEBOOK(parent->notebook), FALSE);
        gtk_notebook_set_show_border(GTK_NOTEBOOK(parent->notebook), FALSE);
    }
    
    return TRUE;
}

static gboolean gui_window_server_changed(CHANNEL_REC *channel)
{
    MAIN_WINDOW_REC *parent;
    SERVER_REC *server;
    gint num;

    g_return_val_if_fail(channel != NULL, FALSE);

    parent = WINDOW_GUI(CHANNEL_PARENT(channel))->parent;
    server = CHANNEL_PARENT(channel)->active->server;

    if (parent->active->active != channel)
	return TRUE;

    if (parent->usermodes[0] == NULL) return TRUE;

    for (num = 0; num < 4; num++)
        gtk_object_set_data(GTK_OBJECT(parent->usermodes[num]), "toggling", GINT_TO_POINTER(TRUE));

    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(parent->usermodes[0]),
                                  server == NULL ? FALSE : server->usermode_away);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(parent->usermodes[1]),
                                  server == NULL ? FALSE : server->usermode_invisible);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(parent->usermodes[2]),
                                  server == NULL ? FALSE : server->usermode_wallops);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(parent->usermodes[3]),
                                  server == NULL ? FALSE : server->usermode_servernotes);

    for (num = 0; num < 4; num++)
        gtk_object_set_data(GTK_OBJECT(parent->usermodes[num]), "toggling", GINT_TO_POINTER(FALSE));

    return TRUE;
}

static gboolean gui_windows_usermode_changed(SERVER_REC *server)
{
    GList *winlist;

    for (winlist = g_list_first(mainwindows); winlist != NULL; winlist = winlist->next)
    {
        MAIN_WINDOW_REC *rec = winlist->data;

        if (server == rec->active->active->server)
            signal_emit("channel server changed", 1, rec->active->active);
    }

    return TRUE;
}

static gboolean cmd_clear(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    WINDOW_VIEW_REC *view;
    WINDOW_REC *window;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    view = WINDOW_VIEW(window);

    if (view->itext)
        gtk_itext_clear(GTK_ITEXT(view->text));
    else
    {
	g_list_free(view->linepos);
	view->linepos = NULL;
	gtk_text_set_point(GTK_TEXT(view->text), 0);
	gtk_text_forward_delete(GTK_TEXT(view->text), gtk_text_get_length(GTK_TEXT(view->text)));
    }
    window->lines = 0;

    return TRUE;
}

static gboolean signal_channel_changed(CHANNEL_REC *channel)
{
    MAIN_WINDOW_REC *mainwindow;
    WINDOW_REC *window;
    GUI_WINDOW_REC *gui;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    mainwindow = WINDOW_GUI(window)->parent;

    if (WINDOW_VIEW(window) != WINDOW_GUI(window)->active)
    {
	WINDOW_GUI(window)->active = WINDOW_VIEW(window);
	mainwindow->active = window;
    }
 
    gui = WINDOW_GUI(window);
    gui->active = WINDOW_VIEW(window);

    /* remove old extra menus */
    if (mainwindow->extramenu != NULL)
    {
        gui_menus_remove(mainwindow, mainwindow->extramenu);
        g_free(mainwindow->extramenu);
        mainwindow->extramenu = NULL;
    }
    if (channel == NULL) return TRUE;

    if (channel->type == CHANNEL_TYPE_CHANNEL)
    {
        /* Add channel menu */
        mainwindow->extramenu = g_strdup(channel_menu[0].label);
        gui_menus_insert(mainwindow, NULL, channel_menu, channel);
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(channel_submenu[2].widget),
				       CHANNEL_GUI(channel)->nicklist);
	gtk_widget_hide(gui->querywidget);
        gtk_widget_show(gui->channelwidget);
    }
    else
    {
        gtk_widget_hide(gui->nlist_widget);
	gtk_widget_hide(gui->channelwidget);

        if (channel->type != CHANNEL_TYPE_QUERY)
	    gtk_widget_hide(gui->querywidget);
	else
        {
            /* Add query menu */
            mainwindow->extramenu = g_strdup(query_menu[0].label);
            gui_menus_insert(mainwindow, NULL, query_menu, channel);
	    gtk_widget_show(gui->querywidget);
        }
    }

    if (CHANNEL_GUI(channel)->servermenu)
    {
        /* change server selection */
        gtk_option_menu_set_history(GTK_OPTION_MENU(mainwindow->servermenu),
                                    g_list_index(servers, channel->server));
        gtk_widget_show(mainwindow->servermenu);
    }
    else
        gtk_widget_hide(mainwindow->servermenu);

    /* draw title bar */
    signal_emit("channel topic changed", 1, channel);

    return TRUE;
}

static gboolean gui_windows_setup_changed(void)
{
    WINDOW_VIEW_REC *view;
    GList *tmp;

    for (tmp = g_list_first(windows); tmp != NULL; tmp = tmp->next)
    {
        WINDOW_REC *rec = tmp->data;

	gui_window_set_background(rec);
	view = WINDOW_VIEW(rec);
        if (view->itext)
	    gtk_itext_set_fonts(GTK_ITEXT(view->text), font_normal, font_bold);
    }

    return TRUE;
}

void gui_windows_init(void)
{
    window_create_override = -1;

    signal_add("window created", (SIGNAL_FUNC) gui_window_created);
    signal_add("window destroyed", (SIGNAL_FUNC) gui_window_destroyed);
    signal_add("setup changed", (SIGNAL_FUNC) gui_windows_setup_changed);
    signal_add("user mode changed", (SIGNAL_FUNC) gui_windows_usermode_changed);
    signal_add("channel server changed", (SIGNAL_FUNC) gui_window_server_changed);
    signal_add("channel changed", (SIGNAL_FUNC) signal_channel_changed);
    signal_add("gui window create override", (SIGNAL_FUNC) sig_window_create_override);
    command_bind("clear", "GUI commands", (SIGNAL_FUNC) cmd_clear);
}

void gui_windows_deinit(void)
{
    signal_remove("window created", (SIGNAL_FUNC) gui_window_created);
    signal_remove("window destroyed", (SIGNAL_FUNC) gui_window_destroyed);
    signal_remove("setup changed", (SIGNAL_FUNC) gui_windows_setup_changed);
    signal_remove("user mode changed", (SIGNAL_FUNC) gui_windows_usermode_changed);
    signal_remove("channel server changed", (SIGNAL_FUNC) gui_window_server_changed);
    signal_remove("channel changed", (SIGNAL_FUNC) signal_channel_changed);
    signal_remove("gui window create override", (SIGNAL_FUNC) sig_window_create_override);
    command_unbind("clear", (SIGNAL_FUNC) cmd_clear);
}
