/*

 ui-completion.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    time_t time;
    gchar *nick;
}
COMPLETION_REC;

static gint comptag;
static GList *complist;

static COMPLETION_REC *completion_find(GList *list, gchar *nick)
{
    GList *tmp;

    for (tmp = g_list_first(list); tmp != NULL; tmp = tmp->next)
    {
        COMPLETION_REC *rec = tmp->data;

        if (g_strcasecmp(rec->nick, nick) == 0) return rec;
    }

    return NULL;
}

static void completion_destroy(GList **list, COMPLETION_REC *rec)
{
    *list = g_list_remove(*list, rec);

    g_free(rec->nick);
    g_free(rec);
}

static COMPLETION_REC *completion_create(GList **list, time_t time, gchar *nick)
{
    COMPLETION_REC *rec;

    rec = completion_find(*list, nick);
    if (rec != NULL)
    {
        /* remove the old one */
        completion_destroy(list, rec);
    }

    rec = g_new(COMPLETION_REC, 1);
    *list = g_list_prepend(*list, rec);

    rec->time = time;
    rec->nick = g_strdup(nick);
    return rec;
}

static void completion_checklist(GList **list, gint timeout, time_t t)
{
    GList *tmp, *next;

    for (tmp = g_list_first(*list); tmp != NULL; tmp = next)
    {
        COMPLETION_REC *rec = tmp->data;

        next = tmp->next;
        if (t-rec->time > timeout)
            completion_destroy(list, rec);
    }
}

static gint completion_timeout(void)
{
    GList *tmp, *link;
    time_t t;
    gint len;

    t = time(NULL);
    for (tmp = g_list_first(servers); tmp != NULL; tmp = tmp->next)
    {
        SERVER_REC *rec = tmp->data;

        len = g_list_length(rec->lastmsgs);
        if (len > 0 && len >= setup_get_int("completion_keep_privates"))
        {
            link = g_list_last(rec->lastmsgs);
            g_free(link->data);
            rec->lastmsgs = g_list_remove_link(rec->lastmsgs, link);
            g_list_free_1(link);
        }
    }

    for (tmp = g_list_first(channels); tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

        completion_checklist(&rec->lastownmsgs, setup_get_int("completion_keep_ownpublics"), t);
        completion_checklist(&rec->lastmsgs, setup_get_int("completion_keep_publics"), t);
    }

    return 1;
}

static void add_private_msg(SERVER_REC *server, gchar *nick)
{
    GList *link;

    link = glist_find_icase_string(server->lastmsgs, nick);
    if (link != NULL)
    {
	g_free(link->data);
	server->lastmsgs = g_list_remove_link(server->lastmsgs, link);
	g_list_free_1(link);
    }
    server->lastmsgs = g_list_prepend(server->lastmsgs, g_strdup(nick));
}

static gboolean event_privmsg(gchar *data, SERVER_REC *server, gchar *nick)
{
    gchar *params, *target, *msg;
    GList **list;

    g_return_val_if_fail(server != NULL, FALSE);
    if (nick == NULL) return TRUE; /* from server */

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);

    if (*msg == 1)
    {
        /* ignore ctcp messages */
        g_free(params);
        return TRUE;
    }

    if (ischannel(*target))
    {
        /* channel message */
        CHANNEL_REC *channel;

        channel = channel_find(server, target);
        if (channel == NULL)
        {
            g_free(params);
            return TRUE;
        }

        list = ui_completion_msgtoyou(server, msg) ?
            &channel->lastownmsgs :
            &channel->lastmsgs;
        completion_create(list, time(NULL), nick);
    }
    else
    {
	/* private message */
        add_private_msg(server, nick);
    }

    g_free(params);
    return TRUE;
}

static gboolean cmd_msg(gchar *data, SERVER_REC *server)
{
    gchar *params, *target, *msg;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &msg);
    if (*target != '\0' && *msg != '\0')
    {
	if (!ischannel(*target) && *target != '=' && server != NULL)
	    add_private_msg(server, target);
    }

    g_free(params);
    return TRUE;
}

gboolean ui_completion_msgtoyou(SERVER_REC *server, gchar *msg)
{
    gchar *stripped;
    gboolean ret;
    gint len;

    g_return_val_if_fail(msg != NULL, FALSE);

    if (g_strncasecmp(msg, server->nick, strlen(server->nick)) == 0 &&
        !isalnum((gint) msg[strlen(server->nick)])) return TRUE;

    stripped = nick_strip(server->nick);
    msg = nick_strip(msg);

    len = strlen(stripped);
    ret = *stripped != '\0' &&
        g_strncasecmp(msg, stripped, len) == 0 &&
	!isalnum((gint) msg[len]) &&
	(guchar) msg[len] < 128;

    g_free(msg);
    g_free(stripped);
    return ret;
}

static void complete_list(GList **outlist, GList *list, gchar *nick)
{
    GList *tmp;
    gint len;

    len = strlen(nick);
    for (tmp = g_list_first(list); tmp != NULL; tmp = tmp->next)
    {
        COMPLETION_REC *rec = tmp->data;

        if (g_strncasecmp(rec->nick, nick, len) == 0 &&
            glist_find_icase_string(*outlist, rec->nick) == NULL)
            *outlist = g_list_append(*outlist, g_strdup(rec->nick));
    }
}

static GList *completion_getlist(CHANNEL_REC *channel, gchar *nick)
{
    GList *nicks, *list, *tmp;
    gint len;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(nick != NULL, NULL);
    if (*nick == '\0') return NULL;

    list = NULL;
    complete_list(&list, channel->lastownmsgs, nick);
    complete_list(&list, channel->lastmsgs, nick);

    len = strlen(nick);
    nicks = nicklist_getnicks(channel);
    for (tmp = nicks; tmp != NULL; tmp = tmp->next)
    {
        NICK_REC *rec = tmp->data;

        if (g_strncasecmp(rec->nick, nick, len) == 0 &&
            glist_find_icase_string(list, rec->nick) == NULL &&
            g_strcasecmp(rec->nick, channel->server->nick) != 0)
            list = g_list_append(list, g_strdup(rec->nick));
    }
    g_list_free(nicks);

    return list;
}

static GList *completion_getmsglist(SERVER_REC *server, gchar *nick)
{
    GList *list, *tmp;
    gint len;

    list = NULL; len = strlen(nick);
    for (tmp = g_list_first(server->lastmsgs); tmp != NULL; tmp = tmp->next)
    {
        if (len == 0 || g_strncasecmp(tmp->data, nick, len) == 0)
            list = g_list_append(list, g_strdup(tmp->data));
    }

    return list;
}

static gboolean event_command(gchar *line, SERVER_REC *server, CHANNEL_REC *channel)
{
    GList *comp;
    gchar *str, *ptr;

    g_return_val_if_fail(channel != NULL, FALSE);
    g_return_val_if_fail(line != NULL, FALSE);

    if (*line == CMD_CHAR || channel->type == CHANNEL_TYPE_EMPTY) return TRUE;

    line = g_strdup(line);

    /* check for nick completion */
    if (setup_get_bool("completion_disable_auto") || *setup_get_str("completion_char") == '\0')
    {
	ptr = NULL;
	comp = NULL;
    }
    else
    {
	ptr = strchr(line, *setup_get_str("completion_char"));
	if (ptr != NULL) *ptr++ = '\0';

	comp = ptr == NULL || channel->type != CHANNEL_TYPE_CHANNEL ||
	    nicklist_find(channel, line) != NULL ? NULL :
	    completion_getlist(channel, line);
    }

    /* message to channel */
    if (ptr == NULL)
        str = g_strdup_printf("%s %s", channel->name, line);
    else
    {
        str = g_strdup_printf("%s %s%s%s", channel->name,
                              comp != NULL ? (gchar *) comp->data : line,
                              setup_get_str("completion_char"), ptr);
    }
    signal_emit("command msg", 3, str, channel->server, channel);

    g_free(str);
    g_free(line);

    if (comp != NULL)
    {
        g_list_foreach(comp, (GFunc) g_free, NULL);
        g_list_free(comp);
    }
    return TRUE;
}

static GList *completion_joinlist(GList *list1, GList *list2)
{
    while (list2 != NULL)
    {
	if (!glist_find_icase_string(list1, list2->data))
	    list1 = g_list_append(list1, list2->data);
	else
	    g_free(list2->data);

	list2 = list2->next;
    }
    g_list_free(list2);
    return list1;
}

gchar *ui_auto_completion(gchar *line, gint *pos)
{
    LIST_REC *rec;
    gchar *word, *ret;
    gint spos, epos, n, wordpos;
    GString *result;

    g_return_val_if_fail(line != NULL, NULL);
    g_return_val_if_fail(pos != NULL, NULL);

    spos = *pos;

    /* get the word we are completing.. */
    while (spos > 0 && isspace((gint) line[spos-1])) spos--;
    epos = spos;
    while (spos > 0 && !isspace((gint) line[spos-1])) spos--;
    while (line[epos] != '\0' && !isspace((gint) line[epos])) epos++;

    word = g_strdup(line+spos);
    word[epos-spos] = '\0';

    /* word position in line */
    wordpos = 0;
    for (n = 0; n < spos; )
    {
        while (n < spos && isspace((gint) line[n])) n++;
        while (n < spos && !isspace((gint) line[n])) n++;
        if (n < spos) wordpos++;
    }

    result = g_string_new(line);
    g_string_erase(result, spos, epos-spos);

    /* check for words in autocompletion list */
    rec = list_find(replaces, word); g_free(word);
    if (rec != NULL)
    {
        *pos = spos+strlen(rec->value);

        g_string_insert(result, spos, rec->value);
        ret = result->str;
        g_string_free(result, FALSE);
        return ret;
    }

    g_string_free(result, TRUE);
    return NULL;
}

#define issplit(a) ((a) == ',' || (a) == ' ')

gchar *ui_completion_line(CHANNEL_REC *channel, gchar *line, gint *pos)
{
    static gboolean msgcomp = FALSE;
    LIST_REC *rec;
    gchar *word, *ret;
    gint spos, epos, len, n, wordpos;
    gboolean msgcompletion;
    GString *result;

    g_return_val_if_fail(channel != NULL, NULL);
    g_return_val_if_fail(line != NULL, NULL);
    g_return_val_if_fail(pos != NULL, NULL);

    spos = *pos;

    /* get the word we are completing.. */
    while (spos > 0 && issplit((gint) line[spos-1])) spos--;
    epos = spos;
    if (line[epos] == ',') epos++;
    while (spos > 0 && !issplit((gint) line[spos-1])) spos--;
    while (line[epos] != '\0' && !issplit((gint) line[epos])) epos++;

    word = g_strdup(line+spos);
    word[epos-spos] = '\0';

    /* word position in line */
    wordpos = 0;
    for (n = 0; n < spos; )
    {
        while (n < spos && issplit((gint) line[n])) n++;
        while (n < spos && !issplit((gint) line[n])) n++;
        if (n < spos) wordpos++;
    }

    msgcompletion = channel->server != NULL &&
        (*line == '\0' || ((wordpos == 0 || wordpos == 1) && g_strncasecmp(line, "/msg ", 5) == 0));

    if (msgcompletion && wordpos == 0 && issplit((gint) line[epos]))
    {
        /* /msg <tab> */
        *word = '\0'; epos++; spos = epos; wordpos = 1;
    }

    /* are we completing the same nick as last time?
       if not, forget the old completion.. */
    len = strlen(word)-(msgcomp == FALSE && word[strlen(word)-1] == *setup_get_str("completion_char"));
    if (complist != NULL && (strlen(complist->data) != len || g_strncasecmp(complist->data, word, len) != 0))
    {
        g_list_foreach(complist, (GFunc) g_free, NULL);
        g_list_free(complist);

        complist = NULL;
    }

    result = g_string_new(line);
    g_string_erase(result, spos, epos-spos);

    /* check for words in completion list */
    rec = list_find(completions, word);
    if (rec != NULL)
    {
        g_free(word);
        *pos = spos+strlen(rec->value);

        g_string_insert(result, spos, rec->value);
        ret = result->str;
        g_string_free(result, FALSE);
        return ret;
    }

    if (complist == NULL && !msgcompletion && channel->type != CHANNEL_TYPE_CHANNEL)
    {
        /* don't try nick completion */
        g_free(word);
        g_string_free(result, TRUE);
        return NULL;
    }

    if (complist == NULL)
    {
        /* start new nick completion */
	complist = completion_getlist(channel, word);

        if (!msgcompletion)
        {
            /* nick completion in channel */
            msgcomp = FALSE;
	}
	else
	{
	    GList *tmpcomp;

            /* /msg completion */
            msgcomp = TRUE;

	    /* first get the list of msg nicks and then nicks from current
	       channel. */
	    tmpcomp = complist;
	    complist = completion_getmsglist(channel->server, word);
	    complist = completion_joinlist(complist, tmpcomp);
            if (*line == '\0')
            {
                /* completion in empty line -> /msg <nick> */
                g_free(word);
                g_string_free(result, TRUE);

                if (complist == NULL)
                    ret = g_strdup("/msg ");
                else
                    ret = g_strdup_printf("/msg %s ", (gchar *) complist->data);
                *pos = strlen(ret);
                return ret;
            }
        }

        if (complist == NULL)
        {
            g_free(word);
            g_string_free(result, TRUE);
            return NULL;
        }
    }
    else
    {
        /* continue the last completion */
        complist = complist->next == NULL ? g_list_first(complist) : complist->next;
    }
    g_free(word);

    /* insert the nick.. */
    g_string_insert(result, spos, complist->data);
    *pos = spos+strlen(complist->data);

    if (!msgcomp && wordpos == 0)
    {
        /* insert completion character */
        g_string_insert(result, *pos, setup_get_str("completion_char"));
        *pos += strlen(setup_get_str("completion_char"));
    }
    if (msgcomp || wordpos == 0)
    {
        if (!issplit((gint) result->str[*pos]))
        {
            /* insert space */
            g_string_insert(result, *pos, " ");
        }
        (*pos)++;
    }

    ret = result->str;
    g_string_free(result, FALSE);
    return ret;
}

static gboolean completion_deinit_server(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    g_list_foreach(server->lastmsgs, (GFunc) g_free, NULL);
    g_list_free(server->lastmsgs);

    return TRUE;
}

static gboolean completion_deinit_channel(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    while (channel->lastmsgs != NULL)
        completion_destroy(&channel->lastmsgs, channel->lastmsgs->data);
    while (channel->lastownmsgs != NULL)
        completion_destroy(&channel->lastownmsgs, channel->lastownmsgs->data);
    g_list_free(channel->lastmsgs);
    g_list_free(channel->lastownmsgs);

    return TRUE;
}

void ui_completion_init(void)
{
    signal_add("event privmsg", (SIGNAL_FUNC) event_privmsg);
    signal_add("send command", (SIGNAL_FUNC) event_command);
    signal_add("server disconnected", (SIGNAL_FUNC) completion_deinit_server);
    signal_add("channel destroyed", (SIGNAL_FUNC) completion_deinit_channel);
    command_bind("msg", NULL, (SIGNAL_FUNC) cmd_msg);

    comptag = gui_timeout_add(1000, (GUITimeoutFunction) completion_timeout, NULL);
    complist = NULL;
}

void ui_completion_deinit(void)
{
    if (complist != NULL)
    {
	g_list_foreach(complist, (GFunc) g_free, NULL);
	g_list_free(complist);
    }

    gui_timeout_remove(comptag);

    signal_remove("event privmsg", (SIGNAL_FUNC) event_privmsg);
    signal_remove("send command", (SIGNAL_FUNC) event_command);
    signal_remove("server disconnected", (SIGNAL_FUNC) completion_deinit_server);
    signal_remove("channel destroyed", (SIGNAL_FUNC) completion_deinit_channel);
    command_unbind("msg", (SIGNAL_FUNC) cmd_msg);
}
