/*****************************************************************************
 *                               BitVector.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements a bit-vector and arithmetic
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#include "BitVector.h"
#include <string.h>
#include <ctype.h>
#include <stdio.h>

#define FP stderr

Uint32 *BitVector::tmp_arr     = 0;
Uint32  BitVector::tmp_arr_len = 0;

s_vpi_vecval  *BitVector::d_vecval      = 0;
Uint32         BitVector::d_vecval_len  = 0;

/****************************************************************
 * BitVector()
 ****************************************************************/
BitVector::BitVector(const String  &bitstr) :
    storage(bitstr), display(bitstr)
{
    /**** Convert the input string to something in binary radix
     ****/
    d_width = 0;
    radix   = 0; /* auto-detect the number type... */
    toBin(bitstr, storage, &radix);

    displayValid = 0;
    d_width = storage.length();
}

/****************************************************************
 * BitVector()
 ****************************************************************/
BitVector::BitVector(const String  &bitstr, Uint32 flags) :
    storage(bitstr), display(bitstr)
{
    /**** Convert the input string to something in binary radix
     ****/
    d_width = 0;
    radix = flags; /* auto-detect the number type... */
    toBin(bitstr, storage, &radix);

    displayValid = 0;
    d_width = storage.length();
}

/****************************************************************
 * BitVector()
 ****************************************************************/
BitVector::BitVector(void) :
    storage(String("")), display(String(""))
{
    /**** Convert the input string to something in binary radix
     ****/

    displayValid = 0;
    d_width = 0;

    radix = Radix_Bin;
}

/****************************************************************
 * BitVector(BitVector)
 ****************************************************************/
BitVector::BitVector(const BitVector &bv) :
    storage(bv.storage), display(bv.storage)
{
    radix = bv.radix;
    storage = bv.storage;
    d_width = bv.d_width;
}

/****************************************************************
 * binToBinV
 ****************************************************************/
Int32 BitVector::binToBinV(
        const String    &in,
        Uint32           start_idx,
        String          &out,
        Uint32           size,
        Uint32           flags)
{
    Uint32       i, idx = 0;
    Char         fill;
    Uint32       in_str_len = in.length();
    Uint32       in_bit_len = in_str_len-start_idx;

    /**** Deal with padding a vector... 
     **** TODO: Deal with truncation...
     ****/
    if (flags & Radix_Sized) {
        out.chkSize(size+1);
        if (size > in_bit_len) {
            /**** extend output vector ***/
            switch (in[0]) {
                case 'z': case 'Z': fill = 'z'; break;
                case 'x': case 'X': fill = 'x'; break;
                case '0': case '1': 
                    if (flags & Radix_Signed) {
                        fill = in[0];
                    } else {
                        fill = '0';
                    }
                    break;
                default: 
                    fprintf(stderr, "ERROR :: Unknown char %c\n", in[0]);
                    break;
            }

            for (i=0; i<(size-in_bit_len); i++) {
                out[idx++] = fill;
            }
        }
    } else {
        out.chkSize(in_bit_len+1);
    }

    for (i=start_idx; i<in_str_len; i++) {
        switch (in[i]) {
            case 'z': case 'Z': out[idx++] = 'z'; break;
            case 'x': case 'X': out[idx++] = 'x'; break;
            case '0':           out[idx++] = '0'; break;
            case '1':           out[idx++] = '1'; break;
            case '_':                             break;
            default:            out[idx++] = 'x'; break;
        }
    }

    out[idx] = 0;

    out.setLen(idx);

    return 0;
}

/****************************************************************
 * decToBinV()
 ****************************************************************/
Int32 BitVector::decToBinV(
        const String    &in,
        Uint32           start_idx,
        String          &out,
        Uint32           size,
        Uint32           flags)
{
    Int32     i, top, idx;
    Int32     in_str_len = in.length();
    Char      ch;
    String    t_str(in);

    /**** First, convert each ASCII digit to binary...
     ****
     **** We just flipped the vector... Now, the msd is
     **** at the end of the vector rather than the beginning...
     ****/
    for (i=start_idx; i<(in_str_len); i++) {
        t_str[in_str_len-i-1] = in[i] - '0';
    }

    top = in_str_len-start_idx-1;
    idx = 0;
    while (top >= 0) {
        /**** First, extract off the 0th bit... ****/
        ch = t_str[0];

        /**** Note, we're putting the bin digits in backward...
         ****/
        out.chkSize(idx+2);
        out[idx++] = (ch & 1)?'1':'0';
        out.setLen(idx);

        /**** Now, cycle through each vector digit... Divide the
         **** vector by two...
         ****/
        for (i=top; i>=0; i--) {
            ch = t_str[i];
            /**** If this digit has a one to propagate, then
             **** do so as long as there's a digit to propagate to...
             ****/
            if (ch & 1) {
                if (i > 0) {
                    t_str[i-1] += 10;
                } 
            }

            ch /= 2;
            t_str[i] = ch;

            if (!ch && (top==i)) {
                top--;
            }
        }
    }

    /**** Okay, now for the rest of the messiness: Swap the resulting
     **** vector...
     ****/
    for (i=0; i<(idx/2); i++) {
        ch = out[idx-i-1];
        out[idx-i-1] = out[i];
        out[i] = ch;
    }

    out[idx] = 0;
    out.setLen(idx);

    return 0;
}

/****************************************************************
 * hexToBinV()
 ****************************************************************/
Int32 BitVector::hexToBinV(
        const String    &in,
        Uint32           start_idx,
        String          &out,
        Uint32           size,
        Uint32           flags)
{
    Uint32       i, ii, idx = 0;
    Char         fill, ch;
    Uint32       in_str_len = in.length();
    Uint32       in_bit_len = (in_str_len-start_idx)*4;

    /**** First, pad out the vector if needed... *****/
    if (flags & Radix_Sized) {
        out.chkSize(size+1);
        ch = in[0];
        fill = '0';
        if (ch == 'z' || ch == 'Z' || ch == 'x' || ch == 'X') {
            fill = ch;
        } else if ((ch >= '8' && ch <= '9') && (flags & Radix_Signed)) { 
            fill = '1';
        } else if ((ch >= 'a' && ch <= 'f') && (flags & Radix_Signed)) {
            fill = '1';
        }

        /**** Vector is arranged with MSB in v[0]. Therefore, pad out
         **** 0-#fill with the fill byte...
         ****/
        if (size>in_bit_len) {
            for (i=0; i<(size-in_bit_len); i++) {
                out[idx++] = fill;
            }
        }
    } else {
        out.chkSize(in_bit_len+1);
    }

    /**** Now, get down to business... 
     **** NOTE: The output vector is MSB@0, LSB@size.. idx should start
     ****       initialized to size...
     ****/
    for (i=start_idx; i<in_str_len; i++) {
        switch (in[i]) {
            case 'x': case 'X':
            case 'z': case 'Z':
                ch = tolower(in[i]);
                for (ii=0; ii<4; ii++) {
                    out[idx++] = ch;
                }
                break;
                
            case '0': case '1': case '2': case '3': case '4':
            case '5': case '6': case '7': case '8': case '9':
                ch = in[i] - '0';
                for (ii=0; ii<4; ii++) {
                    out[idx++] = (ch & (1 << (3-ii)))?'1':'0';
                }
                break;

            case 'a': case 'A': case 'b': case 'B': case 'c': case 'C':
            case 'd': case 'D': case 'e': case 'E': case 'f': case 'F':
                ch = tolower(in[i]) - 'a' + 10;

                for (ii=0; ii<4; ii++) {
                    out[idx++] = (ch & (1 << (3-ii)))?'1':'0';
                }
                break;

            case '_': break;
        }
    }

    out[idx] = 0;
    out.setLen(idx);

    return 0;
}


/****************************************************************
 * toBin()
 *
 * NOTE :: *flags must be zero'd prior to calling this function
 ****************************************************************/
Int32 BitVector::toBin(
        const String    &in,
        String          &out,
        Uint32          *flags)
{
    Uint32    new_flags = 0;
    Uint32    width     = 0, i, idx = 0, ii;

    /**** See what type of number this is...
     ****/
    if (isdigit(in[idx])) {
        /**** Input is sized if there is a single-quote somewhere
         **** here... Otherwise, we're a decimal number
         ****/
        for (i=idx; i<in.length(); i++) {
            if (in[i] == '\'') {
                new_flags |= Radix_Sized;
                break;
            } else if (isdigit(in[i])) {
                width *= 10;
                width += in[i] - '0';
            } else {
                /**** Error methinks... ****/
                return -1;
            }
        }

        /**** We're sized... ****/
        if (new_flags & Radix_Sized) {
            idx += i;
            /*** Skip whitespace between size and single-quote ***/
            while (isspace(in[idx])) { idx++; }
        }
    } 
   
    /**** Now, try to find the radix... ****/
    if (in[idx] == '\'') {
        idx++;

        /*** Will be at most two radix-spec digits: [s|S]?[h|H|d|D|b|B]
         ***/
        for (ii=0; ii<2; ii++) {
            switch (in[idx++]) {
                case 'h': case 'H': new_flags |= Radix_Hex; ii=2; break;
                case 'd': case 'D': new_flags |= Radix_Dec; ii=2; break;
                case 'o': case 'O': new_flags |= Radix_Oct; ii=2; break;
                case 'b': case 'B': new_flags |= Radix_Bin; ii=2; break;
                case 's': case 'S': new_flags |= Radix_Signed; break;
                default: ii=2; break;
            }
        }
        /**** Okay, once again, skip whitespace...
         ****/
        while (isspace(in[idx])) { idx++; }
    } 

    if (!(new_flags & Radix_Mask)) {
        if ((*flags & Radix_Mask)) {
            new_flags |= (*flags & Radix_Mask);
        } else {
            new_flags |= Radix_Dec;
        }
    }
    new_flags |= Radix_Typed;

    /**** Okay, should know our config now... At this point, we branch
     **** off to convert the number...
     ****/
    switch (new_flags & Radix_Mask) {
        case Radix_Bin:
            BitVector::binToBinV(in, idx, out, width, new_flags);
            break;
        case Radix_Dec:
            BitVector::decToBinV(in, idx, out, width, new_flags);
            break;
        case Radix_Hex:
            BitVector::hexToBinV(in, idx, out, width, new_flags);
            break;
        case Radix_Oct:
            break;
        default:
            return -1;
    }

    *flags = new_flags;

    return 0;
}

/****************************************************************
 * operator =
 ****************************************************************/
void BitVector::operator = (const BitVector &rhs)
{
    Uint32  m = rhs.width();
    storage = rhs.storage;
    d_width = rhs.d_width;
    radix   = rhs.radix;
}

/****************************************************************
 * operator ~
 ****************************************************************/
const BitVector BitVector::operator ~ (void)
{
    BitVector     res(*this);
    Uint32        i, len = res.width();
    Char          b;

    for (i=0; i<len; i++) {
        b = storage[i];
        res.storage[i] = (b == '1')?'0':(b == '0')?'1':'x';
    }

    return res;
}


/****************************************************************
 * Operator +
 *
 * [0] is the MSB of each number... Want to process in rev order
 ****************************************************************/
const BitVector BitVector::operator + (const BitVector &rhs)
{
    const BitVector &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector &min_obj = (d_width > rhs.d_width)?rhs:*this;
    Uint32           max_len = max_obj.width();
    Uint32           min_len = min_obj.width();
    Uint32           i;
    BitVector        res(max_obj);
    Char             a, b, c = 0;

    for (i=0; i<min_len; i++) {
        a = min_obj.storage[min_len-i-1];
        b = max_obj.storage[max_len-i-1];
        a = (a == '0')?0:1;
        b = (b == '0')?0:1;

        res.storage[max_len-i-1] = (((a ^ b) ^ c))?'1':'0';
        c = (a | b) & (b | c);
    }


    for (i=(min_len+1); i<max_len; i++) {
        a = max_obj.storage[max_len-i-1];
        a = (a == '0')?0:1;
        b = 0;

        res.storage[max_len-i-1] = (((a ^ b) ^ c))?'1':'0';
        c = (a & c);
    }

    return res;
}

/****************************************************************
 * operator -
 ****************************************************************/
const BitVector BitVector::operator - (const BitVector &rhs)
{
    const BitVector &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector &min_obj = (d_width > rhs.d_width)?rhs:*this;
    Uint32           b_short = (d_width > rhs.d_width)?1:1;
    Uint32           max_len = max_obj.width();
    Uint32           min_len = min_obj.width();
    const BitVector &a_vect  = *this;
    const BitVector &b_vect  =  rhs;
    Uint32           a_len   = a_vect.width();
    Uint32           b_len   = b_vect.width();

    Uint32           i;
    BitVector        res(max_obj);
    Char             a, b, c = 1, short_top;
    
    for (i=0; i<min_len; i++) {
        a = a_vect.storage[a_len-i-1];
        b = b_vect.storage[b_len-i-1];
        a = (a == '0')?0:1;
        b = (b == '0')?1:0;

        res.storage[min_len-i-1] = (((a ^ b) ^ c))?'1':'0';
        c = (a | b) & (b | c);
    }

    short_top = min_obj.storage[min_obj.width()-1];
    for (i=min_len; i<max_len; i++) {
        a = ((b_short)?a_vect.storage[i]:short_top);
        b = ((b_short)?short_top:b_vect.storage[i]);

        a = (a == '0')?0:1;
        b = (b == '0')?1:0;

        res.storage[i] = (((a ^ b) ^ c))?'1':'0';
        c = (a | b) & (b | c);
    }

    return res;
}

/****************************************************************
 * operator &
 ****************************************************************/
const BitVector BitVector::operator & (const BitVector &rhs)
{
    const BitVector &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector &min_obj = (d_width > rhs.d_width)?rhs:*this;
    Uint32           max_len = max_obj.width();
    Uint32           min_len = min_obj.width();
    Int32            i;
    BitVector        res((d_width > rhs.d_width)?*this:rhs);
    Char             ch_a, ch_b;


    /**** NOTE: Vectors are arranged in reverse order => 
     ****     index 0 is really the msb
     ****/
    for (i=0; i<min_len; i++) {
        ch_a = min_obj.storage[min_len-i-1];
        ch_b = max_obj.storage[max_len-i-1];
        if ((ch_a != '0' && ch_a != '1') ||
                (ch_b != '0' && ch_b != '1')) {
            res.storage[max_len-i-1] = 'x';
        } else if (ch_a == '1' && ch_b == '1') {
            res.storage[max_len-i-1] = '1';
        } else {
            res.storage[max_len-i-1] = '0';
        }
    }

    /**** Zero out the top bits...
     ****/
    for (i=min_len; i<max_len; i++) {
        res.storage[max_len-i-1] = '0';
    }

    return res;
}

/****************************************************************
 * Operator |
 ****************************************************************/
const BitVector BitVector::operator | (const BitVector &rhs)
{
    Uint32       max_len = (d_width > rhs.d_width)?d_width:rhs.d_width;
    Uint32       min_len = (d_width > rhs.d_width)?rhs.d_width:d_width;
    const BitVector &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector &min_obj = (d_width > rhs.d_width)?rhs:*this;
    BitVector    res((d_width > rhs.d_width)?*this:rhs);
    Int32      i;
    Char       ch_a, ch_b;

    for (i=0; i<min_len; i++) {
        ch_a = min_obj.storage[min_len-i-1];
        ch_b = max_obj.storage[max_len-i-1];

        if ((ch_a != '0' && ch_a != '1') ||
                (ch_b != '0' && ch_b != '1')) {
            res.storage[max_len-i-1] = 'x';
        } else if (ch_a == '1' || ch_b == '1') {
            res.storage[max_len-i-1] = '1';
        } else {
            res.storage[max_len-i-1] = '0';
        }
    }

    for (i=min_len; i<max_len; i++) {
        res.storage[max_len-i-1] = max_obj.storage[max_len-i-1];
    }

    return res;
}

/****************************************************************
 * Operator ^
 ****************************************************************/
const BitVector BitVector::operator ^ (const BitVector &rhs)
{
    Uint32       max_len = (d_width > rhs.d_width)?d_width:rhs.d_width;
    Uint32       min_len = (d_width > rhs.d_width)?rhs.d_width:d_width;
    const BitVector &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector &min_obj = (d_width > rhs.d_width)?rhs:*this;
    BitVector    res((d_width > rhs.d_width)?*this:rhs);
    Int32        i;
    Char         ch_a, ch_b;

    for (i=0; i<min_len; i++) {
        ch_a = min_obj.storage[min_len-i-1];
        ch_b = max_obj.storage[max_len-i-1];

        if ((ch_a != '0' && ch_a != '1') ||
                (ch_b != '0' && ch_b != '1')) {
            res.storage[max_len-i-1] = 'x';
        } else if ((ch_a == '1' && ch_b == '0') ||
                (ch_a == '0' && ch_b == '1')) {
            res.storage[max_len-i-1] = '1';
        } else {
            res.storage[max_len-i-1] = '0';
        }
    }

    for (i=min_len; i<max_len; i++) {
        res.storage[max_len-i-1] = max_obj.storage[max_len-i-1];
    }

    return res;
}

/****************************************************************
 * operator *
 ****************************************************************/
const BitVector BitVector::operator * (const BitVector &rhs)
{
    Uint32       max_len = (d_width > rhs.d_width)?d_width:rhs.d_width;
    Uint32       min_len = (d_width > rhs.d_width)?rhs.d_width:d_width;
    const BitVector   &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector   &min_obj = (d_width > rhs.d_width)?rhs:*this;
    BitVector    res(String(" "));   /* Create a 1-bit-long string */
    Char         ch_a, ch_b, ch_carry;
    Int32        i;
  
    res.setLen(2*max_len);

    /***********************************************************
     * Okay... Algorithm is:
     *
     * mo_idx = 0;
     * for (i=0 ; i < max_len; i++) 
     *     ch_carry = min_obj[mo_idx++];  # Left-shift min_obj
     *     if (!is0(ch_carry)) then
     *         add max_obj to res[2*max_len-1:max_len]
     *         ch_carry = add_carry
     *     endif
     *
     *     ROR res, shift in ch_carry 
     *         - ROR should be implementable using shifting 
     *           indexes...
     * end
     *
     ***********************************************************/


    return res;
}


/****************************************************************
 * operator ==
 ****************************************************************/
Uint32 BitVector::operator == (const BitVector &rhs)
{
    Uint32       max_len = (d_width > rhs.d_width)?d_width:rhs.d_width;
    Uint32       min_len = (d_width > rhs.d_width)?rhs.d_width:d_width;
    const BitVector   &max_obj = (d_width > rhs.d_width)?*this:rhs;
    const BitVector   &min_obj = (d_width > rhs.d_width)?rhs:*this;
    BitVector    res((d_width > rhs.d_width)?*this:rhs);
    Char         ch_a, ch_b;
    Int32        i;

    for (i=0; i<min_len; i++) {
        ch_a = min_obj.storage[min_len-i-1];
        ch_b = max_obj.storage[max_len-i-1];

        if (ch_a != ch_b) {
            return 0;
        }
    }

    for (i=min_len; i<max_len; i++) {
        if (max_obj.storage[max_len-i-1] != '0') {
            return 0;
        }
    }

    return 1;
}

/****************************************************************
 * operator ()
 ****************************************************************/
void BitVector::operator () (const String &bitstr)
{
    storage = bitstr;
    d_width = bitstr.length();
}

/****************************************************************
 * write_digits()
 *
 * Helper routine for fmtDec()
 ****************************************************************/
Uint32 BitVector::write_digits(
        Uint32         v,
        String        &buf,
        Uint32        &idx,
        Uint32         zero_suppress)
{
    char     segment[BDIGITS];
    Int32    i;

    for (i=BDIGITS-1; i>=0; i--) {
        segment[i] = '0' + (v % 10);
        v = v / 10;
    }

    for (i=0; i<BDIGITS; i++) {
        if (!(zero_suppress &= (segment[i] == '0'))) {
            buf[idx++] = segment[i];
        }
    }

    return zero_suppress;
}

/****************************************************************
 * shift_in()
 *
 * Helper routine for fmtDec()
 ****************************************************************/
void BitVector::shift_in(
        Uint32       *valv, 
        Uint32        vlen,
        Uint32        val)
{
    Uint32 i;

    for (i=0; i<vlen; i++) {
        val = (valv[i] << BBITS) + val;
        valv[i] = (val % BASE);
        val = val / BASE;
    }
}

/****************************************************************
 * fmtDec()
 ****************************************************************/
void BitVector::fmtDec(
                const String   &in, 
                String         &out,
                Uint32          flags)
{
    Uint32 infoFlags;
    fmtDec(in, out, flags, infoFlags);
}

/****************************************************************
 * fmtDec()
 ****************************************************************/
void BitVector::fmtDec(
        const DFIOVal       *in,
        Uint32               in_len,
        String              &out,
        Uint32               flags,
        Uint32              &infoFlags)
{
    fprintf(stderr, "FIXME: BitVector::fmtDec()\n");
}

/****************************************************************
 * fmtDec()
 * Returns number represented as decimal. 
 * Expects input in msb:lsb form ([0] is msb)
 ****************************************************************/
void BitVector::fmtDec(
        const String   &in, 
        String         &out,
        Uint32          flags,
        Uint32         &infoFlags)
{
    Uint32    isSigned = (flags & Radix_Signed)?1:0;
    Uint32    isSized  = (flags & Radix_Sized)?1:0;
    Uint32    isTyped  = (flags & Radix_Typed)?1:0;

    Uint32    ii;
    Int32     i, x, idx = 0, mbits = in.length(), init, final;
    Uint32    comp  = 0;
    Uint32    num_z = 0, num_x = 0;
    Char      ch, tmpb[256];
    Uint32    vlen, val = 0;
    Uint32    i_off, s_off = 0;
    Uint32    nz=0, nx=0, n0=0, n1=0;

    /**** Find offset... ****/
    i_off = mbits % BBITS;

    /**** First, check the sign bit if output format is signed
     ****/
    if (isSigned) {
        ch = in[idx];
        if (isZ(ch)) { 
            num_z++;
            nz++;
        } else if (isX(ch)) { 
            num_x++;
            nx++;
        } else if (is1(ch)) { 
            comp = 1;
            n1++;
        } else {
            n0++;
        }
        s_off = 1;
        i_off++;
        idx++;
    }

    vlen = ((mbits*28+92)/93+3)/4;

    if (!tmp_arr || (vlen > tmp_arr_len)) {
        if (tmp_arr) {
            delete tmp_arr;
        }

        tmp_arr = new Uint32[vlen*2];
        tmp_arr_len = vlen*2;
    }
    memset(tmp_arr, 0, sizeof(Uint32)*(vlen*2));


    /**** Iterate from beginning to end. [0], again, is the sign bit
     ****/
    for (x=0; x<vlen; x++) {

        /**** For each 16-bit quantity (and the first ?non-16-bit? quantity),
         **** loop from msb to lsb
         ****/
        if (!x&&i_off) {
            init = i_off-1;
        } else {
            init = ((x+1)*BBITS-1+s_off)-i_off;
        }
        final = (x)?(x*BBITS)-i_off:0;

        if (init < mbits) {
            for (i=init; i>=final; i--) {
                ch = in[i];
                if (isZ(ch)) { 
                    num_z++;
                    nz++;
                } else if (isX(ch)) { 
                    num_x++;
                    nx++;
                } else if ( !comp && is1(ch)) { 
                    val |= (1 << ((mbits-i-1) % 16));
                    n1++;
                } else if (  comp && is0(ch)) {
                    val |= (1 << ((mbits-i-1) % 16));
                    n1++;
                } else {
                    n0++;
                }
            }
            /**** Something with sign-bit... 
            if (comp && (i == mbits-1)) 
             ****/

            shift_in(tmp_arr, vlen, val);
            val = 0;
        }
    }

    out.chkSize(in.length()/3+4);
    idx = 0;
    /**** Set size info if requested
     ****/
    if (isSized) {
        Uint32 tmp = in.length();
        ii = 0;
        do {
            tmpb[ii++] = "0123456789"[tmp%10];
            tmp /= 10;
        } while (tmp);
        for (i=0; i<ii; i++) {
            out[idx++] = tmpb[ii-i-1];
        }
    }

    if (isSized || isTyped) {
        out[idx++] = '\'';
    }

    if (isTyped) {
        if (isSigned) {
            out[idx++] = 's';
        }
        out[idx++] = 'd';
    }
    
    if (num_x == in.length()) {
        out[idx++] = 'x';
    } else if (num_x > 0) {
        out[idx++] = 'X';
    } else if (num_z == in.length()) {
        out[idx++] = 'z';
    } else if (num_z > 0) {
        out[idx++] = 'Z';
    } else {
        Uint32    zero_suppress = 1;

        if (comp) {
            out[idx++] = '-';
        }
        for (i=vlen-1; i>=0; i--) {
            zero_suppress = write_digits(tmp_arr[i],
                    out, (Uint32)idx, zero_suppress);
        }

        if (zero_suppress) {
            out[idx++] = ('0'+comp);
        }
    }

    out.setLen(idx);
    out[idx] = 0;

    infoFlags = 0;
    if (nz || nx) {
        infoFlags |= Fmt_HasXZ;
    }

    if (nz && !nx && !n0 && !n1) {
        infoFlags |= Fmt_ExclZ;
    }
}

/****************************************************************
 * fmtBin()
 ****************************************************************/
void BitVector::fmtBin(
        const String       &in,
              String       &out,
              Uint32        disp_radix)
{
    Uint32 infoFlags;
    fmtBin(in, out, disp_radix, infoFlags);
}

/****************************************************************
 * fmtBin()
 ****************************************************************/
void BitVector::fmtBin(
        const DFIOVal       *in,
        Uint32               in_len,
        String              &out,
        Uint32               flags,
        Uint32              &infoFlags)
{
    fprintf(stderr, "FIXME: BitVector::fmtBin()\n");
}

/****************************************************************
 * fmtBin()
 ****************************************************************/
void BitVector::fmtBin( 
        const String       &in,
              String       &out,
              Uint32        disp_radix,
              Uint32       &infoFlags)
{
    Uint32    isSigned = (disp_radix & Radix_Signed)?1:0;
    Uint32    isSized  = (disp_radix & Radix_Sized)?1:0;
    Uint32    isTyped  = (disp_radix & Radix_Typed)?1:0;
    Uint32    i, ii, len = in.length(), tmp, idx = 0;
    Uint32    num_x = 0, num_z = 0, offset;
    Uint32    nx=0, nz=0, n0=0, n1=0;
    Char      ch, tmpb[256];

    out.chkSize(in.length()+8);

    if (isSized) {
        tmp = in.length();
        ii = 0;
        do {
            tmpb[ii++] = "0123456789"[tmp%10];
            tmp /= 10;
        } while (tmp);
        for (i=0; i<ii; i++) {
            out[idx++] = tmpb[ii-i-1];
        }
    }

    if (isSized || isTyped) {
        out[idx++] = '\'';
    }

    if (isTyped) {
        out[idx++] = 'b';
    }

    for (i=0; i<len; i++) {
        out[idx++] = in[i];
        if (isZ(in[i])) {
            nz++;
        } else if (isX(in[i])) {
            nx++;
        } else if (is1(in[i])) {
            n1++;
        } else {
            n0++;
        }
    }

    out[idx] = 0;
    out.setLen(idx);

    infoFlags = 0;
    if (nz || nx) {
        infoFlags |= Fmt_HasXZ;
    }

    if (nz && !nx && !n0 && !n1) {
        infoFlags |= Fmt_ExclZ;
    }
}

/****************************************************************
 * fmtString()
 ****************************************************************/
void BitVector::fmtString(
        const String         &in,
              String         &out,
              Uint32          disp_radix)
{
    Uint32 infoFlags;
    fmtString(in, out, disp_radix, infoFlags);
}

/****************************************************************
 * fmtString()
 ****************************************************************/
void BitVector::fmtString(
        const DFIOVal   *in,
        Uint32           in_len,
        String          &out,
        Uint32           flags,
        Uint32          &infoFlags)
{
    fprintf(stderr, "FIXME: BitVector::fmtString()\n");
}

/****************************************************************
 * fmtString()
 *
 * Vector arranged with msb @ idx 0...
 ****************************************************************/
void BitVector::fmtString(
        const String         &in,
              String         &out,
              Uint32          disp_radix,
              Uint32         &infoFlags)
{
    Uint32    num_chars = (in.length()%8)?(in.length()/8+1):(in.length()/8);
    Int32     in_len = in.length();
    Uint32    val, idx, out_idx = 0, is_x;
    Int32     bit, msb, ch;
    Char      bit_val;
    Uint32    nz=0, nx=0, n0=0, n1=0;

    out.chkSize(num_chars);

    for (ch = 0; ch < num_chars; ch++) {
        /**** Loop through each character... Character extends 
         **** from ch*8+7 to 0
         ****/

        if ((ch*8+7) < in_len) {
            msb = ch*8+7;
        } else {
            msb = num_chars-1;
        }

        val = 0;
        idx = 0;
        is_x = 0;
        for (bit=msb; bit >= ch*8; bit--) {
            bit_val = in[bit];

            if (isZ(bit_val)) {
                nz++;
                out[out_idx++] = 'X';
                break;
            } else if (isX(bit_val)) {
                nx++;
                out[out_idx++] = 'X';
                break;
            } else if (is1(bit_val)) {
                n1++;
                val |= 1 << idx;
            } else {
                n0++;
            }

            idx++;
        }

        if (!is_x) {
            out[out_idx++] = val;
        }
    }

    out.setLen(out_idx);
    out[out_idx] = 0;

    infoFlags = 0;
    if (nz || nx) {
        infoFlags |= Fmt_HasXZ;
    }

    if (nz && !nx && !n0 && !n1) {
        infoFlags |= Fmt_ExclZ;
    }
}

/****************************************************************
 * fmtHex()
 *
 * Converts the input string to hexadecimal...
 *
 *
 ****************************************************************/
void BitVector::fmtHex( 
        const String       &in,
              String       &out,
              Uint32        disp_radix)
{
    Uint32 flags;
    BitVector::fmtHex(in, out, disp_radix, flags);
}

/****************************************************************
 * fmtHex()
 ****************************************************************/
void BitVector::fmtHex(
                const DFIOVal  *in,
                Uint32          in_len,
                String         &out,
                Uint32          flags,
                Uint32         &infoFlags)
{
    fprintf(stderr, "FIXME: BitVector::fmtHex()\n");
}

/****************************************************************
 * fmtHex()
 *
 * Converts the input string to hexadecimal...
 *
 *
 ****************************************************************/
void BitVector::fmtHex( 
        const String       &in,
              String       &out,
              Uint32        disp_radix,
              Uint32       &infoFlags)
{
    Uint32    isSigned = (disp_radix & Radix_Signed)?1:0;
    Uint32    isSized  = (disp_radix & Radix_Sized)?1:0;
    Uint32    isTyped  = (disp_radix & Radix_Typed)?1:0;
    Uint32    i, ii, len = in.length(), tmp, idx = 0;
    Uint32    num_x = 0, num_z = 0, offset;
    Char      ch, tmpb[256];
    Uint32    nx=0, n0=0, n1=0, nz=0;

    out.chkSize(in.length()/4+8);

    if (isSized) {
        tmp = in.length();
        ii = 0;
        do {
            tmpb[ii++] = "0123456789"[tmp%10];
            tmp /= 10;
        } while (tmp);
        for (i=0; i<ii; i++) {
            out[idx++] = tmpb[ii-i-1];
        }
    }

    if (isSized || isTyped) {
        out[idx++] = '\'';
    }

    if (isTyped) {
        if (isSigned) {
            out[idx++] = 's';
        }
        out[idx++] = 'h';
    }

    if ((offset = len%4)) {
        offset = (4-offset);
    }

    tmp = 0;
    for (i=offset; i<(len+offset); i++) {

        ch = in[i-offset];
        if (isX(ch)) { 
            num_x++;
            nx++;
        } else if (isZ(ch)) { 
            num_z++;
            nz++;
        } else if (is1(ch)) { 
            tmp |= (1 << 3-(i%4));
            n1++;
        } else {
            n0++;
        }

        if (i && (((i%4) == 3) || (i == (len+offset-1)))) {
            if (!num_x && !num_z) {
                out[idx++] = "0123456789ABCDEF"[tmp&0xF];
            } else if (num_z && (num_z == ((i-offset) % 4))) {
                out[idx++] = 'z';
            } else {
                out[idx++] = 'x';
            } 
            tmp = 0;
            num_z = num_x = 0;
        }
    }
    out.setLen(idx);
    out[idx] = 0;

    infoFlags = 0;
    if (nz || nx) {
        infoFlags |= Fmt_HasXZ;
    }

    if (nz && !nx && !n0 && !n1) {
        infoFlags |= Fmt_ExclZ;
    }
}

/****************************************************************
 * fmtOct()
 ****************************************************************/
void BitVector::fmtOct(
        const String     &in,
        String           &out,
        Uint32            disp_radix)
{
    Uint32 infoFlags;
    fmtOct(in, out, disp_radix, infoFlags);
}

/****************************************************************
 * fmtOct()
 ****************************************************************/
void BitVector::fmtOct(
        const DFIOVal       *in,
        Uint32               in_len,
        String              &out,
        Uint32               flags,
        Uint32              &infoFlags)
{
    fprintf(stderr, "FIXME: BitVector::fmtOct()\n");
}

/****************************************************************
 * fmtOct()
 ****************************************************************/
void BitVector::fmtOct(
        const String     &in,
        String           &out,
        Uint32            disp_radix,
        Uint32           &infoFlags)
{
    Uint32    isSigned = (disp_radix & Radix_Signed)?1:0;
    Uint32    isSized  = (disp_radix & Radix_Sized)?1:0;
    Uint32    isTyped  = (disp_radix & Radix_Typed)?1:0;
    Uint32    i, ii, len = in.length(), tmp, idx = 0;
    Uint32    num_x = 0, num_z = 0, offset;
    Char      ch, tmpb[256];
    Uint32    nx=0, nz=0, n0=0, n1=0;

    out.chkSize(in.length()/3+8);

    if (isSized) {
        tmp = in.length();
        ii = 0;
        do {
            tmpb[ii++] = "0123456789"[tmp%10];
            tmp /= 10;
        } while (tmp);
        for (i=0; i<ii; i++) {
            out[idx++] = tmpb[ii-i-1];
        }
    }

    if (isSized || isTyped) {
        out[idx++] = '\'';
    }

    if (isTyped) {
        if (isSigned) {
            out[idx++] = 's';
        }
        out[idx++] = 'o';
    }

    if ((offset = (len%3))) {
        offset = (3-offset);
    }

    tmp = 0;
    for (i=offset; i<(len+offset); i++) {
        ch = in[i-offset];
        if (isX(ch)) { 
            num_x++;
            nx++;
        } else if (isZ(ch)) { 
            num_z++;
            nz++;
        } else if (is1(ch)) {
            tmp |= (1 << 2-(i%3));
            n1++;
        } else {
            n0++;
        }

        if (i && (((i%3) == 2)|| (i == (len+offset-1)))) {
            if (!num_x && !num_z) {
                out[idx++] = "01234567"[tmp&0x7];
            } else if (num_z && (num_z == ((i-offset) % 3))) {
                out[idx++] = 'z';
            } else {
                out[idx++] = 'x';
            } 
            tmp = 0;
            num_z = num_x = 0;
        }
    }
    out.setLen(idx);
    out[idx] = 0;

    infoFlags = 0;
    if (nz || nx) {
        infoFlags |= Fmt_HasXZ;
    }

    if (nz && !nx && !n0 && !n1) {
        infoFlags |= Fmt_ExclZ;
    }
}

/****************************************************************
 * toString()
 *
 * Returns 'native' format of number.
 ****************************************************************/
String BitVector::toString(void) {

    if (!displayValid) {
        BitVector::toString(storage, display, radix);
    }

    return display;
}


/****************************************************************
 * toString(radix)
 ****************************************************************/
void BitVector::toString(
        String       &in,
        String       &out,
        Uint32  disp_radix)
{

    switch ((disp_radix & Radix_Mask)) {
        case Radix_Bin:
            BitVector::fmtBin(in, out, (disp_radix & Radix_FlagMask));
            break;

        case Radix_Hex:
            BitVector::fmtHex(in, out, (disp_radix & Radix_FlagMask));
            break;

        case Radix_Dec:
            BitVector::fmtDec(in, out, (disp_radix & Radix_FlagMask));
            break;

        case Radix_Oct:
            BitVector::fmtOct(in, out, (disp_radix & Radix_FlagMask));
            break;

        case Radix_Str:
            BitVector::fmtString(in, out, (disp_radix & Radix_FlagMask));
            break;
            
        default:
            break;
    }
}

/****************************************************************
 * setRadix()
 ****************************************************************/
void BitVector::setRadix(Uint32 new_radix) 
{
    if (radix != new_radix) {
        displayValid = 0;
    }

    radix = new_radix;
}

/****************************************************************
 * setWidth()
 *
 * Extends the width of a vector...
 ****************************************************************/
void BitVector::setWidth(Uint32 new_width)
{
    Int32   i;
    Char    sign_ext = '0';

    storage.setLen(new_width);
    storage.chkSize(new_width);
    storage[new_width] = 0;

    /**** First, compute the sign-bit...
     ****/
    if (radix & Radix_Signed) {
        if (storage[0] == '1') {
            sign_ext = '1';
        } 
    }

    if (isZ(storage[0]) || isX(storage[0])) {
        sign_ext = storage[0];
    }

    /**** Now, move the first 'd_width' bits to the low
     **** end of the vector...
     ****
     **** [d_width][new_width-d_width]
     ****
     **** [new_width-d_width][d_width]
     ****/
    if (new_width > d_width) {
        for (i=0; i<d_width; i++) {
            storage[i+(new_width-d_width)] = storage[i];
        }


        /**** Finally, sign-extend the whole thing...
         ****/
        for (i=(new_width-d_width-1); i >= 0; i--) {
            storage[i] = sign_ext;
        }
    }

    d_width = new_width;
}

/****************************************************************
 * get_signext()
 ****************************************************************/
Char BitVector::get_signext(Char top, Uint32 radix_f)
{
    Char    fill;

    if (isZ(top) || isX(top)) {
        fill = top;
    } else if (radix_f & Radix_Signed) {
        fill = top;
    } else {
        fill = '0';
    }

    return fill;
}

/****************************************************************
 * get_vecval()
 ****************************************************************/
void BitVector::get_vecval(
        Uint32           width,
        s_vpi_vecval   **vecval)
{
   BitVector::get_vecval(storage, width, radix, vecval);
}

/****************************************************************
 * get_vecval()
 ****************************************************************/
void BitVector::get_vecval(
        const DFIOVal *str,
        Uint32         width,
        Uint32         radix_f,
        s_vpi_vecval **vecval_p)
{
    fprintf(stderr, "FIXME: BitVector::get_vecval\n");
}

/****************************************************************
 * get_vecval()
 *
 *
 * - If the requested width is greater than the bit-vector width,
 *   then the vector is padded...
 ****************************************************************/
void BitVector::get_vecval(
        String        &str,
        Uint32         width,
        Uint32         radix_f,
        s_vpi_vecval **vecval_p)
{
    Uint32      b_idx = 0, w_idx = 0;
    Uint32      i, add = 0;
    Uint32      v_width = str.length();

    if (width > (d_vecval_len*32)) {
        if (d_vecval) {
            delete [] d_vecval;
        }

        d_vecval = new s_vpi_vecval[(width/32)+1];
        d_vecval_len = (width/32)+1;
    }

    /**** Clear the entire vector... ****/
    for (i=0; i<d_vecval_len; i++) {
        d_vecval[i].aval = 0;
        d_vecval[i].bval = 0;
    }

    /**** Now, pack the current value into the vecval ****/
    for (i=0; i<v_width; i++) {
        put_vecval(d_vecval, i, str[v_width-i-1]);
    }

    /**** Pad out the vector... ****/
    if (width > v_width) {
        Char fill = get_signext(str[0], radix_f);
        add = width-v_width;
        for (i=0; i<add; i++) {
            put_vecval(d_vecval, width-i-1, fill);
            b_idx++;
        }
    }

    *vecval_p = d_vecval;
}

