/****************************************************************************
 *                              StimulusMgr.h
 *
 * Author: Matthew Ballance
 * Desc:   Data structures to manage stimulus to the design
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "StimulusMgr.h"
#include "StimulusSeq.h"
#include "StimulusItem.h"
#include "StimSeqInst.h"
#include "StimTraceRdr.h"
#include "CmdSwitcher.h"
#include "WidgetManager.h"
#include "TclBitVector.h"
#include <stdlib.h>
#include <tk.h>
#include "tclConfig.h"
#include "tree_node.h"
#include "DesignDB.h"
#include "LogMgr.h"
#include "LogRegionType.h"
#include "LogInstance.h"
#include "StimTraceRdr.h"
#include "SigDB.h"
#include "CallbackMgr.h"

#define FP stderr
#undef  DEBUG_PUT
#undef  DEBUG_PARSE

#ifdef DEBUG_PUT
#define PUT_MSG(x) fprintf x
#else
#define PUT_MSG(x)
#endif

#ifdef DEBUG_PARSE
#define PARSE_MSG(x)   fprintf x
#else
#define PARSE_MSG(x)
#endif

/****************************************************************************
 * StimulusMgr Code
 ****************************************************************************/

/********************************************************************
 * prvStimMgrOptSpec
 ********************************************************************/
#define OPT_SIM    0x00000001
static Tk_OptionSpec    prvStimMgrOptSpec[] = {
    {TK_OPTION_STRING,    "-sim",    "sim", "Sim",
        (char *)NULL,  Tk_Offset(StimulusMgr, SimString),
        -1, 0, 0, OPT_SIM},

    {TK_OPTION_END, (char *)NULL, (char *)NULL,
        (char *)NULL, (char *)NULL, -1, 0, 0, 0, 0}
};
static Tk_OptionTable   prvStimMgrOptTab = 0;

/********************************************************************
 * StimulusMgr()
 ********************************************************************/
StimulusMgr::StimulusMgr(
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const objv[]) : instName(16)
{
    LogRegionType   *lrt;

    seq_insts   = new Vector<StimSeqInst>();
    trace_insts = new Vector<StimTraceRdr>();


    this->interp = interp;
    cmd = Tcl_CreateObjCommand(interp, Tcl_GetString(objv[1]),
            StimulusMgr::InstCmd, this, 0);
    SimString = 0;

    instName = Tcl_GetString(objv[1]);

    Tcl_InitHashTable(&hashTab, TCL_STRING_KEYS);

    Tcl_InitOptions(interp, (char *)this, prvStimMgrOptTab);

    if (LogMgr::FindRegionType("StimulusMgr", &lrt) < 0) {
        Tcl_AppendResult(interp, "cannot find LogRegionType StimulusMgr", 0);
        return;
    }

    log = new LogInstance(lrt, Tcl_GetString(objv[1]));
    LogMgr::AddInstance(log);
    log_low(log, LogMgr::DebugLevel_Low);
    log_med(log, LogMgr::DebugLevel_Med);
    log_high(log, LogMgr::DebugLevel_High);

    WidgetMgr_AddInst(interp, "StimulusMgr",
            Tcl_GetString(objv[1]), this);
    
    if (Configure(objc-2, &objv[2]) != TCL_OK) {
        fprintf(stderr, "ERROR :: \n");
    }
}

/********************************************************************
 * ~StimulusMgr()
 ********************************************************************/
StimulusMgr::~StimulusMgr()
{
    Tcl_DeleteCommandFromToken(interp, cmd);
    delete seq_insts;
}

/********************************************************************
 * addSeq()
 ********************************************************************/
void StimulusMgr::addSeq(StimulusSeq *seq)
{
    Tcl_HashEntry *entry;
    Int32          tmp;
    StimulusSeq   *oldSeq;

    if (!(entry = Tcl_FindHashEntry(&hashTab, seq->get_instName()))) {
        entry = Tcl_CreateHashEntry(&hashTab, seq->get_instName(), &tmp);
        Tcl_SetHashValue(entry, seq);
    } else {
        oldSeq = (StimulusSeq *)Tcl_GetHashValue(entry);

        if (oldSeq) {
            d_seqList.remove(oldSeq);
        }
        Tcl_SetHashValue(entry, seq);
    }
    d_seqList.append(seq);

    CallbackMgr_Invoke(CBTYPE_STIM_SEQ_LST_UPDATE, instName.value(), 0, 0);
}

/********************************************************************
 * lookupSeq()
 ********************************************************************/
StimulusSeq *StimulusMgr::lookupSeq(const char *inst)
{
    StimulusSeq       *seq;
    Tcl_HashEntry    *entry;

    if ((entry = Tcl_FindHashEntry(&hashTab, inst))) {
        seq = (StimulusSeq *)Tcl_GetHashValue(entry);
        return seq;
    }
   
    return 0;
}

/********************************************************************
 * Configure()
 ********************************************************************/
int StimulusMgr::Configure(int objc, Tcl_Obj *const objv[])
{
    Tk_SavedOptions    savedOptions;
    Tcl_Obj           *errRes;
    int                mask, err;

    if (Tcl_SetOptions(interp, (char *)this, prvStimMgrOptTab,
                        objc, objv, &savedOptions, &mask) != TCL_OK) {
        Tcl_RestoreSavedOptions(&savedOptions);
        return TCL_ERROR;
    }

    if ((mask&OPT_SIM) && SimString) {
        sim = (IviSim *)WidgetMgr_GetObjHandle(interp, "IviSim", 
                Tcl_GetString(SimString));

        if (!sim) {
            Tcl_AppendResult(interp, "unknown IviSim ", 
                    Tcl_GetString(SimString), 0);
            return TCL_ERROR;
        }
    }

    Tcl_FreeSavedOptions(&savedOptions);
    return TCL_OK;
}

/********************************************************************
 * GetDelay()
 ********************************************************************/
int StimulusMgr::GetDelay(Tcl_Obj *spec, Uint32 *delay)
{
    /**** For now, be simple... ****/
    *delay = strtoul(Tcl_GetString(spec), 0, 0);

    return TCL_OK;
}

static CmdSwitchStruct rad_list[] = {
    {"-bin",         BitVector::Radix_Bin   },
    {"-hex",         BitVector::Radix_Hex   },
    {"-dec",         BitVector::Radix_Dec   },
    {"-oct",         BitVector::Radix_Oct   },
    {"-str",         BitVector::Radix_Str   },
    {"-signed",      BitVector::Radix_Signed},
    {"-sized",       BitVector::Radix_Sized },
    {"",             0                      }
};

/********************************************************************
 * GetRadix
 ********************************************************************/
int StimulusMgr::GetRadix(
        Char                    *rstr,
        BitVector::RadixType    *radix)
{
    BitVector::RadixType         new_radix = 0;

    if (!(new_radix = CmdSwitch(rad_list, rstr))) {
        Tcl_AppendResult(interp, "unknown radix ", rstr, 0);
        return TCL_ERROR;
    }

    if (new_radix & BitVector::Radix_Mask) {
        *radix &= ~(BitVector::Radix_Mask);
    } 

    *radix |= new_radix;
    return TCL_OK;
}

/********************************************************************
 * LookupVpiHndl()
 ********************************************************************/
int StimulusMgr::LookupVpiHndl(
        DesignDB        *ddb,
        Char            *sigSpec,
        vpiHandle       *vpiHndl)
{
    Vector<TreeNode>   *sigVect;
    TreeNode           *sig;

    sigVect = ddb->globElems(0, sigSpec, "@signals@", 0);

    if (sigVect->length() != 1) {
        Tcl_AppendResult(interp, "may only specify 1 signal", 0);
        return TCL_ERROR;
    }

    sig = sigVect->idx(0);
    *vpiHndl = (vpiHandle)sig->findField("vpiHandle")->getVoidVal();

    if (!(*vpiHndl)) {
        Tcl_AppendResult(interp, "internal error - NULL vpiHandle", 0);
        return TCL_ERROR;
    }
    return TCL_OK;
}

/********************************************************************
 * Get()
 *
 * get signal [options]
 ********************************************************************/
int StimulusMgr::Get(
        Uint32        objc,
        Tcl_Obj      *const objv[])
{
    Tcl_Obj                *res_list;
    Vector<TreeNode>       *sigVect;
    TreeNode               *sig;
    BitVector::RadixType    radType = BitVector::Radix_Bin 
                                    | BitVector::Radix_Typed;
    Uint32                  i;
    DesignDB               *ddb = sim->get_ddb();
    BitVector              *bv;
    Tcl_Obj                *bv_obj;
    vpiHandle               vpiHndl;
    s_vpi_value             value_s;

    sigVect = ddb->globElems(0, Tcl_GetString(objv[0]), "@signals@", 0);
    if (!sigVect->length()) {
        Tcl_AppendResult(interp, "no signals match ", 
                Tcl_GetString(objv[0]), 0);
        return TCL_ERROR;
    }

    for (i=1; i<objc; i++) {
        if (GetRadix(Tcl_GetString(objv[i]), &radType) != TCL_OK) {
            return TCL_ERROR;
        }
    }

    value_s.format = vpiBinStrVal;
    res_list = Tcl_NewListObj(0, 0);
    /**** Okay, now process each signal... ****/
    for (i=0; i<sigVect->length(); i++) {
        sig = sigVect->idx(i);

        vpiHndl = (vpiHandle)sig->findField("vpiHandle")->getVoidVal();

        sim->vpi_get_value(vpiHndl, &value_s);

        bv_obj = Tcl_NewBitVector_BinStr(value_s.value.str, -1);
        bv = (BitVector *)bv_obj->internalRep.otherValuePtr;
        bv->setRadix(radType);

        Tcl_ListObjAppendElement(interp, res_list, bv_obj);
    }

    Tcl_SetObjResult(interp, res_list);
    return TCL_OK;
}

/********************************************************************
 * ParseStimSeq()
 ********************************************************************/
int StimulusMgr::ParseStimSeq(
        StimulusSeq       *seq,
        Uint32             objc,
        Tcl_Obj           *const objv[])
{
    Int32     delay, x, i;
    BitVector     *bv;
    StimulusItem  *item;
    StimulusSeq   *subseq;
    Char          *str;
    Uint32         msb, lsb;

    for (i=0; i<objc; i++) {
        delay = 0;

        str = Tcl_GetString(objv[i]);

        if (String::equal(str, "repeat")) {
            /**** If there is more stuff following the repeat statement,
             **** then this is invalid...
             ****/
            i++;

            seq->enableRepeat();

            /**** "repeat" may be followed by:
             **** [after delay] [for number]
             ****/
            for ( ; i<objc; i++) {
                str = Tcl_GetString(objv[i]);

                if (String::equal(str, "after")) {
                    i++;
                    /**** Get the delay... ****/ 
                    if (GetDelay(objv[i], (Uint32 *)&delay) != TCL_OK) {
                        return TCL_ERROR;
                    }
                    PARSE_MSG((FP, "\tset repeat delay of %d\n", delay));
                    seq->setRepeatDelay(delay);
                } else if (String::equal(str, "for")) {
                    i++;
                    if (Tcl_GetIntFromObj(interp, objv[i], &delay) != TCL_OK) {
                        return TCL_ERROR;
                    }
                    PARSE_MSG((FP, "\tSet repeat count to %d\n", delay));
                    seq->setNumRepeat(delay);
                } else {
                    Tcl_AppendResult(interp, "unknown postfix \"", 
                        str, "\" to \"repeat\" command", 0);
                    return TCL_ERROR;
                }
            }
        } else {
            if (str[0] == '%') {
                /**** Try to lookup a sequence-name ****/
                if (!(subseq = lookupSeq(&str[1]))) {
                    Tcl_AppendResult(interp, "no sequence named \"",
                    &str[1], "\"", 0);
                    return TCL_ERROR;
                }

                item = new StimulusItem(subseq, delay);
                seq->addItem(item);

            } else if (String::equal(str, "after")) {
                i++;

                if ((objc-i) == 0) {
                    Tcl_AppendResult(interp, "no argument to \"after\"", 0);
                    return TCL_ERROR;
                }

                if (GetDelay(objv[i], (Uint32 *)&delay) != TCL_OK) {
                    return TCL_ERROR;
                }

                /**** If this is the first item, treat the delay as an
                 **** initial offset, NOT a permanent delay...
                 ****/
                if (seq->length() <= 1) {
                    PARSE_MSG((FP, "\tSetting initial delay to %d\n", delay));
                    seq->setInitDelay(delay);
                } else {
                    PARSE_MSG((FP, "\tSetting delay on \"%d\" to %d\n",
                                seq->length()-1, delay));
                    item = seq->idx(seq->length()-1);
                    item->set_delay(delay);
                }
            } else {
                /**** Otherwise, just another value... ****/
                if (Tcl_GetBitVectorFromObj(interp, objv[i], &bv) != TCL_OK) {
                    Tcl_AppendResult(interp, "illegal bit-vector value ",
                            Tcl_GetString(objv[i]), 0);
                    return TCL_ERROR;
                }

                item = new StimulusItem(bv, delay);
                seq->addItem(item);
                seq->setMaxBitWidth(bv->width());
            }
        }

        if (!bv) {
            Tcl_AppendResult(interp, "Internal error: NULL bv", 0);
            return TCL_ERROR;
        }
    }

    PARSE_MSG((FP, "stim-len = %d\n", seq->get_values()->length()));

    return TCL_OK;
}

/********************************************************************
 * addSeqInst()
 *
 * Adds a stimulus sequence instance to the manager. This allows the
 * manager to track the instances currently executing
 ********************************************************************/
void StimulusMgr::addSeqInst(StimSeqInst *inst)
{
    seq_insts->append(inst);
}

/********************************************************************
 * delSeqInst()
 ********************************************************************/
void StimulusMgr::delSeqInst(StimSeqInst *inst)
{
    Uint32          len = seq_insts->length();
    Uint32          i, x;
    StimSeqInst    *seq;

    /**** This is a bit longer... ****/
    for (i=0; i<len; i++) {
        if (seq_insts->idx(i) == inst) {
            /**** Bubble all insts ahead of this one back... ****/
            for (x=i; x<len; x++) {
                seq_insts->setIdx(i, seq_insts->idx(x+1));
            }
            seq_insts->setLength(len-1);
            break;
        }
    }

    if (i == len) {
        fprintf(stderr, "ERROR :: Unknown StimSeqInst %x\n", inst);
    }
}

/********************************************************************
 * addTraceInst()
 ********************************************************************/
void StimulusMgr::addTraceInst(StimTraceRdr *rdr)
{
    trace_insts->append(rdr);
}

/********************************************************************
 * delTraceInst()
 ********************************************************************/
void StimulusMgr::delTraceInst(StimTraceRdr *rdr)
{
    Uint32          len = trace_insts->length();
    Uint32          i, x;

    /**** This is a bit longer... ****/
    for (i=0; i<len; i++) {
        if (trace_insts->idx(i) == rdr) {
            /**** Bubble all insts ahead of this one back... ****/
            for (x=i; x<len; x++) {
                trace_insts->setIdx(i, trace_insts->idx(x+1));
            }
            trace_insts->setLength(len-1);
            break;
        }
    }

    if (i == len) {
        fprintf(stderr, "ERROR :: Unknown StimSeqInst %x\n", rdr);
    }
}

/********************************************************************
 * StimulusCreate()
 *
 * create <id> <...>
 ********************************************************************/
int StimulusMgr::StimulusCreate(
        Uint32        objc,
        Tcl_Obj      *const objv[])
{
    StimulusSeq   *seq;

    /**** Check minimal case:
     **** create <sequence>
     ****/
    if (objc < 2) {
        Tcl_AppendResult(interp, "too few args: put <sig> <value>", 0);
        return TCL_ERROR;
    }
   
    seq = new StimulusSeq(Tcl_GetString(objv[0]), this);

    if (ParseStimSeq(seq, objc-1, &objv[1]) != TCL_OK) {
        return TCL_ERROR;
    }

    addSeq(seq);

    return TCL_OK;
}

/********************************************************************
 * putTrace()
 ********************************************************************/
int StimulusMgr::putTrace(
        Uint32        objc,
        Tcl_Obj      *const objv[])
{
    Vector<TreeNode>   *sigVect;
    TreeNode           *sig;
    SigDB              *sdb;
    DesignDB           *ddb, *s_ddb;
    DFIOTrace          *trace;
    vpiHandle           vpiHndl;
    char                tbuf[256];
    StimTraceRdr       *rdr;

    if (objc < 3) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    /**** Find the SDB ****/
    strcpy(tbuf, "::SDB.");
    strcat(tbuf, Tcl_GetString(objv[0]));
    if (!(sdb = (SigDB *)WidgetMgr_GetObjHandle(interp, "SDB", tbuf))) {
        Tcl_AppendResult(interp, "no SDB instance ", 
                Tcl_GetString(objv[0]), 0);
        return TCL_ERROR;
    } 
    ddb = sdb->get_ddb();

    /**** Look for the specified SDB trace in the SDB ****/
    sigVect = ddb->globElems(0, Tcl_GetString(objv[1]), "@signals@", 0);

    if (sigVect->length() > 1) {
        Tcl_AppendResult(interp, "sig-spec ", Tcl_GetString(objv[1]), 
                " specifies >1 signal", 0);
        return TCL_ERROR;
    } else if (sigVect->length() < 1) {
        Tcl_AppendResult(interp, "sig-spec ", Tcl_GetString(objv[1]), 
                " specifies no signals", 0);
        return TCL_ERROR;
    }

    sig = sigVect->idx(0);
    trace = (DFIOTrace *)sig->findField("vpiHandle")->getVoidVal();

    /**** Now, find the sim-signals... ****/
    s_ddb = sim->get_ddb();

    sigVect = s_ddb->globElems(0, Tcl_GetString(objv[2]), "@signals@", 0);
    if (sigVect->length() > 1) {
        Tcl_AppendResult(interp, "sig-spec ", Tcl_GetString(objv[1]), 
                " specifies >1 signal", 0);
        return TCL_ERROR;
    } else if (sigVect->length() < 1) {
        Tcl_AppendResult(interp, "sig-spec ", Tcl_GetString(objv[1]), 
                " specifies no signals", 0);
        return TCL_ERROR;
    }

    sig = sigVect->idx(0);
    if (!(vpiHndl = (vpiHandle)sig->findField("vpiHandle")->getVoidVal())) {
        Tcl_AppendResult(interp, "error locating vpiHndl for ", 
                Tcl_GetString(objv[1]), 0);
        return TCL_ERROR;
    }

    rdr = new StimTraceRdr(vpiHndl, trace, this);
    rdr->Start();

    return TCL_OK;
}

/********************************************************************
 * StimulusPut()
 ********************************************************************/
int StimulusMgr::StimulusPut(
        Uint32        objc,
        Tcl_Obj      *const objv[])
{
    Uint32         i, x, first_val = 0;
    Int32          delay;
    BitVector     *bv;
    StimulusItem  *item;
    StimulusSeq   *seq;
    Char          *str;
    vpiHandle      vpiHndl, tHndl;
    Uint32         width, msb, lsb;
    DesignDB      *ddb;
    s_vpi_value    v_val;

    /**** Check minimal case:
     **** put <sig> <value>
     ****/
    if (objc < 2) {
        Tcl_AppendResult(interp, "too few args: put <sig> <value>", 0);
        return TCL_ERROR;
    }
   
    ddb = sim->get_ddb();

    /**** Find the vpiHandle for this signal... ****/
    if (LookupVpiHndl(ddb, Tcl_GetString(objv[0]), &vpiHndl) != TCL_OK) {
        return TCL_ERROR;
    }

    v_val.format = vpiIntVal;
    tHndl = sim->vpi_handle(vpiRightRange, vpiHndl);
    sim->vpi_get_value(tHndl, &v_val);
    msb = v_val.value.integer;
    tHndl = sim->vpi_handle(vpiLeftRange, vpiHndl);
    sim->vpi_get_value(tHndl, &v_val);
    lsb = v_val.value.integer;
    width = (msb>lsb)?(msb-lsb+1):(lsb-msb+1);

    /**** Process the rest of the args... 
     ****
     **** <value> ?delay? 
     ****/
    seq = new StimulusSeq("", this);

    if (ParseStimSeq(seq, objc-1, &objv[1]) != TCL_OK) {
        return TCL_ERROR;
    }

    /**** Now, create an instance of this sequence... ****/
    new StimSeqInst(vpiHndl, seq);

    return TCL_OK;
}

typedef enum {
    StimCmd_Config,
    StimCmd_Cget,
    StimCmd_Put,
    StimCmd_Get,
    StimCmd_CreateSeq,
    StimCmd_PutTrace,
    StimCmd_SeqList,
    StimCmd_NumCmds
} StimCmds;

static CmdSwitchStruct   stim_cmds[] = {
    { "config",          StimCmd_Config    },
    { "configure",       StimCmd_Config    },
    { "cget",            StimCmd_Cget      },
    { "put",             StimCmd_Put       },
    { "put_value",       StimCmd_Put       },
    { "get",             StimCmd_Get       },
    { "get_value",       StimCmd_Get       },
    { "create_seq",      StimCmd_CreateSeq },
    { "put_trace",       StimCmd_PutTrace  },
    { "seq_list",        StimCmd_SeqList   },
    { "",                0                 }
};

/********************************************************************
 * InstCmd()
 *
 * Commands:
 * put <sig> <value> ?after time? ?<value> after <time>?
 ********************************************************************/
int StimulusMgr::InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[]) 
{
    StimulusMgr *mgr      = (StimulusMgr *)clientData;
    StimulusSeq *seq;
    Int32        cmd;

    if (objc < 2) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(stim_cmds, Tcl_GetString(objv[1]));

    switch (cmd) {

        /**** put <sig> <value> (?delay? <value>)* ?repeat ?delay??
         ****/
        case StimCmd_Put:
            return mgr->StimulusPut(objc-2, &objv[2]);
            break;

        case StimCmd_CreateSeq:
            return mgr->StimulusCreate(objc-2, &objv[2]);
            break;

        case StimCmd_Get:
            return mgr->Get(objc-2, &objv[2]);
            break;

        case StimCmd_Config:
            return mgr->Configure(objc-2, &objv[2]);
            break;

        case StimCmd_Cget:
            if (objc != 3) {
                Tcl_AppendResult(interp, "wrong number of args - expect 3", 0);
                return TCL_ERROR;
            }
            break;

        /************************************************************
         * StimCmd_PutTrace
         *
         * put_trace <sdb_name> <sdb_trace> <sim_path>
         ************************************************************/
        case StimCmd_PutTrace:
            return mgr->putTrace(objc-2, &objv[2]);
            break;

        /************************************************************
         * SeqList
         ************************************************************/
        case StimCmd_SeqList:
            {
                Tcl_Obj *lst = Tcl_NewListObj(0, 0);

                for (Uint32 i=0; i<mgr->d_seqList.length(); i++) {
                    seq = mgr->d_seqList.idx(i);

                    Tcl_ListObjAppendElement(interp, lst, 
                            Tcl_NewStringObj(seq->get_instName(), -1));
                }

                Tcl_SetObjResult(interp, lst);
            }
            break;

        default:
            Tcl_AppendResult(interp, "no stim-mgr sub-cmd ", 
                    Tcl_GetString(objv[1]), 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/********************************************************************
 * StimulusMgr_Cmd
 *
 * stimulus_mgr <inst> [options]
 ********************************************************************/
static int StimulusMgr_Cmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               objc,
        Tcl_Obj          *const objv[]) 
{
    StimulusMgr   *mgr;

    mgr = new StimulusMgr(interp, objc, objv);

    return TCL_OK;
}

/********************************************************************
 * StimulusMgr_Init()
 ********************************************************************/
extern "C" int StimulusMgr_Init(Tcl_Interp *interp)
{
    LogRegionType    *lr = new LogRegionType("StimulusMgr", 
            "Stimulus Generation Manager", 
            "",
            "",
            "");
    LogMgr::AddRegionType(lr);

    Tcl_CreateObjCommand(interp, "stimulus_mgr", StimulusMgr_Cmd, 0, 0);

    WidgetMgr_AddType(interp, "StimulusMgr");

    prvStimMgrOptTab = Tcl_CreateOptionTable(interp, prvStimMgrOptSpec);

    CallbackMgr_AddCbType(CBTYPE_STIM_SEQ_LST_UPDATE);
    return TCL_OK;
}


