/****************************************************************************
 *                          WaveWidgetCursors.cc
 *
 * Author: Matthew Ballance
 * Desc:   Code to manage cursors for the wave widget
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "WaveWidget.h"
#include "WidgetManager.h"
#include "WidgetCommon.h"
#include "SdbReader.h"
#include "SdbrCursor.h"
#include "CallbackMgr.h"
#include "PsPixmapObj.h"
#include "BitVector.h"
#include "BitVectorABVal.h"

#include "SigDB.h"
#include "SdbSignal.h"
#include "DFIO.h"
#include "DFIOTrace.h"

#include "LogInstance.h"
#include "LogDestination.h"
#include "LogMgr.h"

#include "TreeWidget.h"
#include "TreeWidgetNode.h"


/******************************************************************
 * DrawCursors()
 ******************************************************************/
void WaveWidget::DrawCursors(
        CursorMgr         *mgr,
        PixmapObj         &pixmap,
        Uint32             drawMain,
        Uint32             tagCenter,
        Uint32             drawTag)
{
    Vector<SdbrCursor>      *clist = mgr->getCursors();
    Vector<SdbrCursor>      *ctlist = mgr->getSortCursors();
    SdbrCursor              *cursor;
    Tk_TextLayout            layout;
    Char                     buf[32], buf2[32];
    Int32                    text_width, text_height;
    Int32                    last_text_width = 0;
    Uint32                   thisTagCenter, offset, i;
    Uint32                   traceSep = getTraceSep();
    Uint32                   endTime = displayStartTime + 
                                     PixelsToTime(widgetWidth);
    GCObj                   *gc = 0, *tgc = &(getGCs()[TC_ScaleColor]);
    GCObj                   *gcObjs = getGCs();

    /**** Ensure that the GCs are attached to this pixmap... ****/
    pixmap.addGCs(getGCs(), TC_NumColors);

    /**** Draw slot-separation lines ****/
    if (drawTag) {
        for (i=0; i<clist->length(); i++) {
            thisTagCenter = tagCenter + (getTraceSep() * i);
            tgc->line(0, (thisTagCenter-(traceSep/2))+1, pixmap.width(),
                    (thisTagCenter-(traceSep/2))+1);
        }
    }

    /**** Move through the list in time-sorted order ****/
    for (i=0; i<ctlist->length(); i++) {
        cursor = ctlist->idx(i);

        thisTagCenter = tagCenter + (getTraceSep() * cursor->getIdx());

        if (cursor->isLocked()) {
            if (cursor->isSelected()) {
                gc = &gcObjs[TC_CursorLockSelColor];
            } else {
                gc = &gcObjs[TC_CursorLockUnselColor];
            }
        } else if (cursor->isSelected()) {
            gc = &gcObjs[TC_CursorSelColor];
        } else {
            gc = &gcObjs[TC_CursorUnselColor];
        }

        if (cursor->getTime() >= displayStartTime &&
                cursor->getTime() <= endTime) {
            offset = TimeToPixels(cursor->getTime() - displayStartTime);

            /**** Setup tag for current cursor - this gives us tag-width,
             **** which is needed for drawing measurements
             ****/
            if (drawTag) {
                FormatTime(buf, cursor->getTime());
                layout = gc->ComputeTextLayout(
                        buf, -1, TK_JUSTIFY_CENTER, &text_width, &text_height);
            }

            /**** If there is a previous cursor, then see about drawing
             **** a measurement line...
             ****/
            if (i && drawTag) {
                GCObj         *mgc = &(getGCs()[TC_CursorMeasureColor]);
                SdbrCursor    *lastc;
                Uint32         delta, delta_pix, delta_center_pix;
                Uint32         measureTagCenter;
                Int32          delta_height, delta_width;
                Tk_TextLayout  delta_layout;
                bool           do_draw = false;

                Uint32         x1_1=0, x1_2=0, x2_1=0, x2_2=0;

                lastc = ctlist->idx(i-1);
                delta = cursor->getTime()-lastc->getTime();
                delta_pix = TimeToPixels(delta);
                delta_center_pix = TimeToPixels(lastc->getTime());
                FormatTime(buf2, delta);
                delta_layout = mgc->ComputeTextLayout(buf2, -1, 
                        TK_JUSTIFY_CENTER, &delta_width, &delta_height);

                /**** If the y-pos order of the last cursor is
                 **** greater than the current cursor, then the measurement
                 **** tag goes above us...
                 ****/
                if (lastc->getIdx() > cursor->getIdx()) {
                    measureTagCenter = tagCenter + 
                        (getTraceSep() * cursor->getIdx());
                    delta_center_pix = 
                        TimeToPixels(lastc->getTime()-displayStartTime) + 
                        (delta_pix/2) - (text_width/4) - (delta_width/2);
                    x1_1 = TimeToPixels(lastc->getTime()-displayStartTime)+2;
                    x1_2 = delta_center_pix-2;

                    x2_1 = delta_center_pix+delta_width+2;
                    x2_2 = TimeToPixels(cursor->getTime()-displayStartTime) - 
                            (text_width/2) - 2;

                    if (delta_pix >= (text_width/2)) {
                        delta_pix -= (text_width/2);
                    }
                } else {
                    measureTagCenter = tagCenter + 
                        (getTraceSep() * lastc->getIdx());
                    delta_center_pix = 
                        TimeToPixels(lastc->getTime()-displayStartTime) + 
                        (delta_pix/2) + (last_text_width/4) - (delta_width/2);

                    x1_1 = TimeToPixels(lastc->getTime()-displayStartTime) +
                            (last_text_width/2) + 2;
                    x1_2 = delta_center_pix-2;

                    x2_1 = delta_center_pix+delta_width+2;
                    x2_2 = TimeToPixels(cursor->getTime()-displayStartTime)-2;


                    if (delta_pix >= last_text_width/2) {
                        delta_pix -= (last_text_width/2);
                    }
                }

                if (lastc->getTime() >= displayStartTime) {
                    if (delta_pix > delta_width+4) {
                        mgc->DrawTextLayout(delta_layout,
                                delta_center_pix,
                                measureTagCenter-(d_config.traceHeight/2)+2);
                        do_draw = true;
                    } 
                } else {
                    /**** Last cursor falls before the display... See how 
                     **** much room is left between us and the display edge
                     ****/
                }
                mgc->FreeTextLayout(delta_layout);

                /**** Draw cursor to text lines ****/
                if (x1_1 != x1_2 && do_draw) {
                    mgc->line(x1_1, measureTagCenter, x1_2, measureTagCenter);
                }

                if (x2_1 != x2_2 && do_draw) {
                    mgc->line(x2_1, measureTagCenter, x2_2, measureTagCenter);
                }
            }

            /**** If we're drawing both tag and main, then we must adjust
             **** the lower bounds on the cursor slightly
             ****/
            if (drawMain && drawTag) {
                gc->line(offset, 0, offset, 
                        thisTagCenter-(getTraceSep()/2)+4);
            } else if (drawMain) {
                gc->line(offset, 0, offset, widgetHeight);
            }

            if (drawTag) {
                Uint32 hcp = offset+(text_width/2);
                Uint32 hcn = offset-(text_width/2);
                Uint32 vcp = thisTagCenter+(d_config.traceHeight/2)+2;
                Uint32 vcn = thisTagCenter-(d_config.traceHeight/2)+2;

                gc->line(offset, 0, offset, vcn-2);
           
                /*** Draw the time-surround box ***/
                gc->line(hcn-2, vcn, hcp+2, vcn);
                gc->line(hcn-2, vcp, hcp+2, vcp);
                gc->line(hcn-2, vcn, hcn-2, vcp);
                gc->line(hcp+2, vcn, hcp+2, vcp);

                gc->DrawTextLayout(layout, hcn, vcn);
                gc->FreeTextLayout(layout);

                last_text_width = text_width;
            }
        }
    }
}

/******************************************************************
 * CursorB1Action()
 ******************************************************************/
void WaveWidget::CursorB1Action(Int32 x, Int32 y, Int32 mod)
{
    Vector<SdbrCursor>    *cursors;
    CursorMgr             *cmgr;

    if (!sdbr) {
        return;
    }

    cmgr = sdbr->cursorMgr;
    cursors = cmgr->getCursors();

    /**** If no cursors are on the SDBR now, then add one ****/
    if (cursors->length() == 0) {
        grabbedCursor = cmgr->addCursor();
        grabbedCursor->setTime(PixelsToTime(x)+displayStartTime);
    }

    grabbedCursor = sdbr->cursorMgr->nearestCursor(
                PixelsToTime(x)+displayStartTime, displayStartTime,
                displayStartTime+PixelsToTime(widgetWidth),
                PixelsToTime(10));

    if (grabbedCursor) {
        grabbedCursor->select(1);

        /**** If we're not adding selection, then clear all other
         **** cursor's selected status
         ****/
        if (!mod) {
            for (Uint32 i=0; i<cursors->length(); i++) {
                if (cursors->idx(i) != grabbedCursor) {
                    cursors->idx(i)->select(0);
                }
            }
        }
    } else {
        for (Uint32 i=0; i<cursors->length(); i++) {
            cursors->idx(i)->select(0);
        }
    }
}

/******************************************************************
 * UpdateSigValTree()
 ******************************************************************/
void WaveWidget::UpdateSigValTree(Uint64 &time)
{
    Vector<SdbSignal>     *signals;
    Vector<DFIOValChg>    *chgs;
    static char            svals[] = {'0', '1', 'z', 'x'};

    if (!d_sigValTree || !sdbr) {
        fprintf(stderr, "ERROR: d_sigValTree = 0x%08x ; sdbr = 0x%08x\n",
                d_sigValTree, sdbr);
        return;
    }

    /**** Loop through each signal on the wave widget
     **** - find tree-node for value
     **** - set value for each
     ****/
    signals = &sdbr->d_signals;

    for (Uint32 i=0; i<signals->length(); i++) {
        SdbSignal        *signal = signals->idx(i);
        TreeWidgetNode   *node;
        DFIOTrace        *dfioTrace = signal->dfioTrace;
        char              buf[16];

        if (!(node = d_sigValTree->findNode(signal->id))) {
            fprintf(stderr, "Cannot find \"%s\"\n", signal->id);
            continue;
        }

        if (signal->isSeparator()) {
            continue;
        }

        chgs = dfioTrace->getValue(time.low, time.low, 0);

        if (chgs->length()) {
            Uint32 flags;
            DFIOValChg   *chg = chgs->idx(0);

            signal->FormatValue(chg, strVal, flags);
            node->SetText(strVal.value());

            /**** See if we must set values for sub-bit ****/
            if (node->IsOpen()) {
                Vector<TreeWidgetNode>    *children = node->getChildren();
                Uint32    numC = children->length();

                for (int y=0; y<numC; y++) {
                    TreeWidgetNode *child = children->idx(y);
                    char            bitv[2];
                    DFIOBitVal      bitval;

                    bitval = DFIOValChg_GetBit(chg->bitVal, (numC-y-1));
                    bitv[0] = svals[(Uint32)bitval];
                    bitv[1] = 0;

                    child->SetText(bitv);
                }
            }
        } else {
            node->SetText("-- No Data --");
        }
    }

    d_sigValTree->SetupRedraw();
}

#if 0
/******************************************************************
 * FillSigValTree()
 ******************************************************************/
void WaveWidget::FillSigValTree()
{
    Vector<SdbSignal>     *signals;
    
    signals = &sdbr->d_signals;

    for (Uint32 i=0; i<signals->length(); i++) {

    }
}
#endif


