/*
    Copyright 2008-2011
        Matthias Ehmann,
        Michael Gerhaeuser,
        Carsten Miller,
        Bianca Valentin,
        Alfred Wassermann,
        Peter Wilfahrt

    This file is part of JSXGraph.

    JSXGraph is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    JSXGraph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with JSXGraph.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @fileoverview The geometry object Line is defined in this file. Line stores all
 * style and functional properties that are required to draw and move a line on
 * a board.
 */

/**
 * The Line class is a basic class for all kind of line objects, e.g. line, arrow, and axis. It is usually defined by two points and can
 * be intersected with some other geometry elements.
 * @class Creates a new basic line object. Do not use this constructor to create a line. Use {@link JXG.Board#create} with
 * type {@link Line}, {@link Arrow}, or {@link Axis} instead.
 * @constructor
 * @augments JXG.GeometryElement
 * @param {String,JXG.Board} board The board the new line is drawn on.
 * @param {Point} p1 Startpoint of the line.
 * @param {Point} p2 Endpoint of the line.
 * @param {String} id Unique identifier for this object. If null or an empty string is given,
 * an unique id will be generated by Board
 * @param {String} name Not necessarily unique name. If null or an
 * empty string is given, an unique name will be generated.
 * @param {boolean} withLabel construct label, yes/no
 * @param {integer} layer display layer [0-9]
 * @see JXG.Board#generateName
 */
JXG.Line = function (board, p1, p2, id, name, withLabel, layer) {
    /* Call the constructor of GeometryElement */
    this.constructor();

   /**
     * Sets type of GeometryElement, value is OBJECT_TYPE_LINE.
     * @constant
     * @type int
     * @default JXG#OBJECT_TYPE_LINE
     * @private
     */
    this.type = JXG.OBJECT_TYPE_LINE;

    /**
     * Class of element, value is OBJECT_CLASS_LINE;
     * @type int
     * @constant
     * @default JXG#OBJECT_CLASS_LINE
     * @private
     */
    this.elementClass = JXG.OBJECT_CLASS_LINE;

    this.init(board, id, name);

    /**
     * Set the display layer.
     */
    if (layer == null) layer = board.options.layer['line'];
    this.layer = layer;
    
    /**
     * Startpoint of the line. You really should not set this field directly as it may break JSXGraph's
     * udpate system so your construction won't be updated properly.
     * @type JXG.Point
     */
    this.point1 = JXG.getReference(this.board, p1);

    /**
     * Endpoint of the line. Just like {@link #point1} you shouldn't write this field directly.
     * @type JXG.Point
     */
    this.point2 = JXG.getReference(this.board, p2);

    /**
     * This is just for the hasPoint() method.
     * @type int
     * @private
     */
    //this.r = this.board.options.precision.hasPoint;

    /* Wurde in GeometryElement schon dokumentiert. */
    this.visProp['fillColor'] = this.board.options.line.fillColor;
    this.visProp['highlightFillColor'] = this.board.options.line.highlightFillColor;
    this.visProp['strokeColor'] = this.board.options.line.strokeColor;
    this.visProp['highlightStrokeColor'] = this.board.options.line.highlightStrokeColor;

    /**
     * Determines if a line is drawn beyond {@link #point1}.
     * @name JXG.Line#straightFirst
     * @type boolean
     * @default JXG.Options.line#straightFirst
     * @field
     * @see JXG.Line#straightLast
     */
    this.visProp['straightFirst'] = this.board.options.line.straightFirst;

    /**
     * Determines if a line is drawn beyond {@link #point2}.
     * @name JXG.Line#straightLast
     * @type boolean
     * @default JXG.Options.line#straightLast
     * @field
     * @see JXG.Line#straightFirst
     */
    this.visProp['straightLast'] = this.board.options.line.straightLast;

    /* Already documented in JXG.GeometryElement */
    this.visProp['visible'] = true;

    /**
     * Determines if a line has an arrow at {@link #point1}.
     * @name JXG.Line#firstArrow
     * @type boolean
     * @default JXG.Options.line#firstArrow
     * @field
     * @see JXG.Line#lastArrow
     */
    this.visProp['firstArrow'] = this.board.options.line.firstArrow;

    /**
     * Determines if a line has an arrow at {@link #point1}.
     * @name JXG.Line#lastArrow
     * @type boolean
     * @default JXG.Options.line#lastArrow
     * @field
     * @see JXG.Line#firstArrow
     */
    this.visProp['lastArrow'] = this.board.options.line.lastArrow;

    /**
     * Array of ticks storing all the ticks on this line. Do not set this field directly and use
     * {@link #addTicks} and {@link #removeTicks} to add and remove ticks to and from the line.
     * @type array
     * @see JXG.Ticks
     */
    this.ticks = [];

    /**
     * Reference of the ticks created automatically when constructing an axis.
     * @type JXG.Ticks
     * @see JXG.Ticks
     */
    this.defaultTicks = null;

    /**
    * If the line is the border of a polygon, the polygon object is stored, otherwise null.
    * @type JXG.Polygon
    * @default null
    * @private
    */
    this.parentPolygon = null;

    /**
    * Label offsets from label anchor
    * @type array
     * @default JXG.Options.line#labelOffsets
    * @private
    */
    //this.labelOffsets = [].concat(this.board.options.line.labelOffsets);
    this.labelOffsets = [].concat(this.board.options.line.labelOffsets);
    //make sure we have absolute values
    this.labelOffsets[0] = Math.abs(this.labelOffsets[0]);
    this.labelOffsets[1] = Math.abs(this.labelOffsets[1]);

    // create Label
    this.createLabel(withLabel);

    /* Register line at board */
    this.id = this.board.setId(this, 'L');
    this.board.renderer.drawLine(this);
    this.board.finalizeAdding(this);

    /* Add arrow as child to defining points */
    this.point1.addChild(this);
    this.point2.addChild(this);
    this.needsUpdate = true; this.update();
};

JXG.Line.prototype = new JXG.GeometryElement;

/**
 * Checks whether (x,y) is near the line.
 * @param {int} x Coordinate in x direction, screen coordinates.
 * @param {int} y Coordinate in y direction, screen coordinates.
 * @return {boolean} True if (x,y) is near the line, False otherwise.
 */
 JXG.Line.prototype.hasPoint = function (x, y) {
    // Compute the stdform of the line in screen coordinates.
    var c = [], s,
        v = [1,x,y],
        vnew = [],
        mu, i, coords, p1Scr, p2Scr, distP1P, distP2P, distP1P2;

    c[0] = this.stdform[0] -
                this.stdform[1]*this.board.origin.scrCoords[1]/this.board.stretchX+
                this.stdform[2]*this.board.origin.scrCoords[2]/this.board.stretchY;
    c[1] = this.stdform[1]/this.board.stretchX;
    c[2] = this.stdform[2]/(-this.board.stretchY);

    // Project the point orthogonally onto the line 
    var vnew = [0,c[1],c[2]];
    vnew = JXG.Math.crossProduct(vnew,v); // Orthogonal line to c through v
    vnew = JXG.Math.crossProduct(vnew,c); // Intersect orthogonal line with line

    // Normalize the projected point
    vnew[1] /= vnew[0];
    vnew[2] /= vnew[0];
    vnew[0] = 1.0;
    
    // The point is too far away from the line
    // dist(v,vnew)^2 projective
    //if (JXG.Math.Geometry.distance(v,vnew)>this.board.options.precision.hasPoint) {
    s = (v[0]-vnew[0])*(v[0]-vnew[0])+(v[1]-vnew[1])*(v[1]-vnew[1])+(v[2]-vnew[2])*(v[2]-vnew[2]);
    if (isNaN(s) || s>this.board.options.precision.hasPoint*this.board.options.precision.hasPoint) {
        return false;
    }

    if(this.visProp['straightFirst'] && this.visProp['straightLast']) {
        return true;
    } else { // If the line is a ray or segment we have to check if the projected point is "inside" P1 and P2.
/*
        coords = new JXG.Coords(JXG.COORDS_BY_SCREEN, [vnew[1],vnew[2]], this.board);
        p1Scr = this.point1.coords.scrCoords;
        p2Scr = this.point2.coords.scrCoords;
        distP1P = coords.distance(JXG.COORDS_BY_SCREEN, this.point1.coords);
        distP2P = coords.distance(JXG.COORDS_BY_SCREEN, this.point2.coords);
        distP1P2 = this.point1.coords.distance(JXG.COORDS_BY_SCREEN, this.point2.coords);
*/
        p1Scr = this.point1.coords.scrCoords;
        p2Scr = this.point2.coords.scrCoords;
        distP1P2 = (p2Scr[1]-p1Scr[1])*(p2Scr[1]-p1Scr[1])+(p2Scr[2]-p1Scr[2])*(p2Scr[2]-p1Scr[2]);  // dist(p1,p2)^2 affine
        distP1P = (vnew[1]-p1Scr[1])*(vnew[1]-p1Scr[1])+(vnew[2]-p1Scr[2])*(vnew[2]-p1Scr[2]);       // dist(vnew,p1)^2 affine
        distP2P = (vnew[1]-p2Scr[1])*(vnew[1]-p2Scr[1])+(vnew[2]-p2Scr[2])*(vnew[2]-p2Scr[2]);       // dist(vnew,p2)^2 affine

        if((distP1P > distP1P2) || (distP2P > distP1P2)) { // Check if P(x|y) is not between  P1 and P2
            if(distP1P < distP2P) { // P liegt auf der Seite von P1
                if(!this.visProp['straightFirst']) {
                    return false;
                }
            } else { // P liegt auf der Seite von P2
                if(!this.visProp['straightLast']) {
                    return false;
                }
            }
        }
        return true;
    }
};

/**
 * TODO description. maybe. already documented in geometryelement?
 * @private
 */
JXG.Line.prototype.update = function() {
    var i, funps;

    if(this.constrained) {
    	if(typeof this.funps != 'undefined') {
    		funps = this.funps();
    		this.point1 = funps[0];
    		this.point2 = funps[1];
    	} else {
            this.point1 = this.funp1();
            this.point2 = this.funp2();
    	}
    }

    if (this.needsUpdate) {
        if (true || !this.board.geonextCompatibilityMode) {
            this.updateStdform();
        }
        /*
        // This is now done in Ticks.updateRenderer()
        for(i=0; i<this.ticks.length; i++) {
            // i don't know why we need this, but if we don't check it, an error will be reported
            // when the origin is moved. it seems like this.ticks.length is lying.
            if(typeof this.ticks[i] != 'undefined')
                this.ticks[i].calculateTicksCoordinates();
        }
        */
    }
    if(this.traced) {
        this.cloneToBackground(true);
    }
};

/**
 * TODO description. already documented in geometryelement?
 * @private
 */
JXG.Line.prototype.updateStdform = function() {
   /*
    var nx = -(this.point2.coords.usrCoords[2]-this.point1.coords.usrCoords[2]);
    var ny =  this.point2.coords.usrCoords[1]-this.point1.coords.usrCoords[1];
    var c = -(nx*this.point1.coords.usrCoords[1]+ny*this.point1.coords.usrCoords[2]);

    this.stdform[0] = c;
    this.stdform[1] = nx;
    this.stdform[2] = ny;
    */
    var v = JXG.Math.crossProduct(this.point1.coords.usrCoords,this.point2.coords.usrCoords);
    this.stdform[0] = v[0];
    this.stdform[1] = v[1];
    this.stdform[2] = v[2];
    this.stdform[3] = 0;
    this.normalize();
};

/**
 * Uses the boards renderer to update the line.
 * @private
 */
 JXG.Line.prototype.updateRenderer = function () {
    var wasReal, i;
    if (this.needsUpdate && this.visProp['visible']) {
        wasReal = this.isReal;
        this.isReal = (isNaN(this.point1.coords.usrCoords[1]+this.point1.coords.usrCoords[2]+this.point2.coords.usrCoords[1]+this.point2.coords.usrCoords[2]))?false:true;
        if (this.isReal) {
            if (wasReal!=this.isReal) {
                this.board.renderer.show(this);
                if(this.hasLabel && this.label.content.visProp['visible']) this.board.renderer.show(this.label.content);
            }
            this.board.renderer.updateLine(this);
/*            
            if (this.board.options.renderer == 'canvas') { 
                for (i=0;i<this.ticks.length;i++)                   // This is necessary for the CanvasRenderer
                    this.ticks[i].prepareUpdate().updateRenderer(); // No idea, why other Renderer work without it.
            }
*/            
        } else {
            if (wasReal!=this.isReal) {
                this.board.renderer.hide(this);
                if(this.hasLabel && this.label.content.visProp['visible']) this.board.renderer.hide(this.label.content);
            }
        }

        //this.board.renderer.updateLine(this); // Why should we need this?
        this.needsUpdate = false;
    }

    /* Update the label if visible. */
    if(this.hasLabel && this.label.content.visProp['visible'] && this.isReal) {
        //this.label.setCoordinates(this.coords);
        this.label.content.update();
        //this.board.renderer.updateLabel(this.label);
        this.board.renderer.updateText(this.label.content);
    }
};

/**
 * Used to generate a polynomial for a point p that lies on this line, i.e. p is collinear to {@link #point1}
 * and {@link #point2}.
 * @param p The point for that the polynomial is generated.
 * @return An array containing the generated polynomial.
 * @private
 */
JXG.Line.prototype.generatePolynomial = function (/** JXG.Point */ p) /** array */{
    var u1 = this.point1.symbolic.x,
        u2 = this.point1.symbolic.y,
        v1 = this.point2.symbolic.x,
        v2 = this.point2.symbolic.y,
        w1 = p.symbolic.x,
        w2 = p.symbolic.y;

    /*
     * The polynomial in this case is determined by three points being collinear:
     *
     *      U (u1,u2)      W (w1,w2)                V (v1,v2)
     *  ----x--------------x------------------------x----------------
     *
     *  The collinearity condition is
     *
     *      u2-w2       w2-v2
     *     -------  =  -------           (1)
     *      u1-w1       w1-v1
     *
     * Multiplying (1) with denominators and simplifying is
     *
     *    u2w1 - u2v1 + w2v1 - u1w2 + u1v2 - w1v2 = 0
     */

    return [['(',u2,')*(',w1,')-(',u2,')*(',v1,')+(',w2,')*(',v1,')-(',u1,')*(',w2,')+(',u1,')*(',v2,')-(',w1,')*(',v2,')'].join('')];
};

/**
 * Calculates the rise of the line.
 * @type float
 * @return The rise of the line.
 */
JXG.Line.prototype.getRise = function () {
    if (Math.abs(this.stdform[2])>=JXG.Math.eps) {
        return -this.stdform[0]/this.stdform[2];
    } else {
        return Infinity;
    }
};

/**
 * Calculates the slope of the line.
 * @type float
 * @return The slope of the line or Infinity if the line is parallel to the y-axis.
 */
JXG.Line.prototype.getSlope = function () {
    if (Math.abs(this.stdform[2])>=JXG.Math.eps) {
        return -this.stdform[1]/this.stdform[2];
    } else {
        return Infinity;
    }
};

/**
 * Determines whether the line is drawn beyond {@link #point1} and {@link #point2} and updates the line.
 * @param {boolean} straightFirst True if the Line shall be drawn beyond {@link #point1}, false otherwise.
 * @param {boolean} straightLast True if the Line shall be drawn beyond {@link #point2}, false otherwise.
 * @see #straightFirst
 * @see #straightLast
 * @private
 */
 JXG.Line.prototype.setStraight = function (straightFirst, straightLast) {
    this.visProp['straightFirst'] = straightFirst;
    this.visProp['straightLast'] = straightLast;

    this.board.renderer.updateLine(this);
};

/**
 * Determines whether the line has arrows at start or end of the line. Is stored in visProp['firstArrow'] and visProp['lastArrow']
 * @param {boolean} firstArrow True if there is an arrow at the start of the line, false otherwise.
 * @param {boolean} lastArrow True if there is an arrow at the end of the line, false otherwise.
 * @private
 */
/*
JXG.Line.prototype.setArrow = function (firstArrow, lastArrow) {
     this.visProp['firstArrow'] = firstArrow;
     this.visProp['lastArrow'] = lastArrow;

     this.board.renderer.updateLine(this);
};
*/

/**
 * Calculates TextAnchor. DESCRIPTION
 * @type JXG.Coords
 * @return Text anchor coordinates as JXG.Coords object.
 * @private
 */
JXG.Line.prototype.getTextAnchor = function() {
    return new JXG.Coords(JXG.COORDS_BY_USER, [this.point1.X()+0.5*(this.point2.X() - this.point1.X()),this.point1.Y() +0.5*(this.point2.Y() - this.point1.Y())],this.board);
};

/**
 * Adjusts Label coords relative to Anchor. DESCRIPTION
 * @private
 */
JXG.Line.prototype.setLabelRelativeCoords = function(relCoords) {
    if (typeof this.label.content!='undefined') { 
        this.label.content.relativeCoords = new JXG.Coords(JXG.COORDS_BY_SCREEN, [relCoords[0],-relCoords[1]],this.board);
    }
}

/**
 * Calculates LabelAnchor. DESCRIPTION
 * @type JXG.Coords
 * @return Text anchor coordinates as JXG.Coords object.
 * @private
 */
JXG.Line.prototype.getLabelAnchor = function() {
    var coords,screenCoords1,screenCoords2,
        relCoords, slope, xoffset = this.labelOffsets[0], yoffset = this.labelOffsets[1];

    if(!this.visProp['straightFirst'] && !this.visProp['straightLast']) {
        this.setLabelRelativeCoords(this.labelOffsets);
        return new JXG.Coords(JXG.COORDS_BY_USER, [this.point2.X()-0.5*(this.point2.X() - this.point1.X()),this.point2.Y()-0.5*(this.point2.Y() - this.point1.Y())],this.board);
    }
    else {
        screenCoords1 = new JXG.Coords(JXG.COORDS_BY_USER, this.point1.coords.usrCoords, this.board);
        screenCoords2 = new JXG.Coords(JXG.COORDS_BY_USER, this.point2.coords.usrCoords, this.board);
        this.board.renderer.calcStraight(this, screenCoords1, screenCoords2);

        if(this.visProp['straightFirst']) {
            coords = screenCoords1;
        }
        else {
            coords = screenCoords2;
        }
        // Hack
        if(this.label.content != null) {
            relCoords = [0,0];
            slope = this.getSlope();
            if(coords.scrCoords[2]==0) {
                if(slope == Infinity) {
                    relCoords = [xoffset,-yoffset];
                }
                else if(slope >= 0) {
                    relCoords = [xoffset,-yoffset];
                }
                else {
                    relCoords = [-xoffset,-yoffset];
                }
            }
            else if(coords.scrCoords[2]==this.board.canvasHeight) {
                if(slope == Infinity) {
                    relCoords = [xoffset,yoffset];
                }
                else if(slope >= 0) {
                    relCoords = [-xoffset,yoffset];
                }
                else {
                    relCoords = [xoffset,yoffset];
                }
            }
            if(coords.scrCoords[1]==0) {
                if(slope == Infinity) {
                    relCoords = [xoffset,yoffset]; // ??
                }
                else if(slope >= 0) {
                    relCoords = [xoffset,-yoffset];
                }
                else {
                    relCoords = [xoffset,yoffset];
                }
            }
            else if(coords.scrCoords[1]==this.board.canvasWidth) {
                if(slope == Infinity) {
                    relCoords = [-xoffset,yoffset]; // ??
                }
                else if(slope >= 0) {
                    relCoords = [-xoffset,yoffset];
                }
                else {
                    relCoords = [-xoffset,-yoffset];
                }
            }
            this.setLabelRelativeCoords(relCoords);
        }
        return coords;
    }
};

/**
 * Clone the element to the background to leave a trail of it on the board.
 * @param {boolean} addToTrace Not used.
 */
JXG.Line.prototype.cloneToBackground = function(addToTrace) {
    var copy = {}, r, s, er;

    copy.id = this.id + 'T' + this.numTraces;
    copy.elementClass = JXG.OBJECT_CLASS_LINE;
    this.numTraces++;
    copy.point1 = this.point1;
    copy.point2 = this.point2;

    copy.stdform = this.stdform;

    copy.board = {};
    copy.board.unitX = this.board.unitX;
    copy.board.unitY = this.board.unitY;
    copy.board.zoomX = this.board.zoomX;
    copy.board.zoomY = this.board.zoomY;
    copy.board.stretchX = this.board.stretchX;
    copy.board.stretchY = this.board.stretchY;
    copy.board.origin = this.board.origin;
    copy.board.canvasHeight = this.board.canvasHeight;
    copy.board.canvasWidth = this.board.canvasWidth;
    copy.board.dimension = this.board.dimension;
    //copy.board.algebra = this.board.algebra;

    copy.visProp = this.visProp;
    JXG.clearVisPropOld(copy);

    s = this.getSlope();
    r = this.getRise();
    copy.getSlope = function() { return s; };
    copy.getRise = function() { return r; };

    er = this.board.renderer.enhancedRendering;
    this.board.renderer.enhancedRendering = true;
    this.board.renderer.drawLine(copy);
    this.board.renderer.enhancedRendering = er;
    this.traces[copy.id] = copy.rendNode; //this.board.renderer.getElementById(copy.id);

    delete copy;

/*
    var id = this.id + 'T' + this.numTraces;
    this.traces[id] = this.board.renderer.cloneSubTree(this,id,'lines');
    this.numTraces++;
*/
};

/**
 * DESCRIPTION
 * @param transform A {@link #JXG.Transformation} object or an array of it.
 */
JXG.Line.prototype.addTransform = function (/** JXG.Transformation,array */ transform) {
    var list, i;
    if (JXG.isArray(transform)) {
        list = transform;
    } else {
        list = [transform];
    }
    for (i=0;i<list.length;i++) {
        this.point1.transformations.push(list[i]);
        this.point2.transformations.push(list[i]);
    }
};

/**
 * TODO DESCRIPTION. What is this method for? -- michael
 * @param method TYPE & DESCRIPTION. UNUSED.
 * @param x TYPE & DESCRIPTION
 * @param y TYPE & DESCRIPTION
 */
JXG.Line.prototype.setPosition = function (method, x, y) {
    //var oldCoords = this.coords;
    //if(this.group.length != 0) {
    // AW: Do we need this for lines?
        // this.coords = new JXG.Coords(method, [x,y], this.board);
        // this.group[this.group.length-1].dX = this.coords.scrCoords[1] - oldCoords.scrCoords[1];
        // this.group[this.group.length-1].dY = this.coords.scrCoords[2] - oldCoords.scrCoords[2];
        // this.group[this.group.length-1].update(this);
    //} else {
        var t = this.board.create('transform',[x,y],{type:'translate'});
        if (this.point1.transformations.length>0 && this.point1.transformations[this.point1.transformations.length-1].isNumericMatrix) {
            this.point1.transformations[this.point1.transformations.length-1].melt(t);
        } else {
            this.point1.addTransform(this.point1,t);
        }
        if (this.point2.transformations.length>0 && this.point2.transformations[this.point2.transformations.length-1].isNumericMatrix) {
            this.point2.transformations[this.point2.transformations.length-1].melt(t);
        } else {
            this.point2.addTransform(this.point2,t);
        }
        //this.addTransform(t);
        //this.update();
    //}
};

/**
 * Treat the line as parametric curve in homogeneuous coordinates.
 * <pre>x = 1 * sin(theta)*cos(phi)
 * y = 1 * sin(theta)*sin(phi)
 * z = 1 * sin(theta)</pre>
 * and the line is the set of solutions of <tt>a*x+b*y+c*z = 0</tt>.
 * It follows:
 * <pre>sin(theta)*(a*cos(phi)+b*sin(phi))+c*cos(theta) = 0</pre>
 * Define:
 * <pre>  A = (a*cos(phi)+b*sin(phi))
 *   B = c</pre>
 * Then
 * <pre>cos(theta) = A/sqrt(A*A+B*B)
 * sin(theta) = -B/sqrt(A*A+B*B)</pre>
 * and <tt>X(phi) = x</tt> from above.
 * phi runs from 0 to 1
 * @type float
 * @return X(phi) TODO description
 */
JXG.Line.prototype.X = function (phi) {
    var a = this.stdform[1],
        b = this.stdform[2],
        c = this.stdform[0],
        A, B, sq, sinTheta, cosTheta;
    phi *= Math.PI;
    A = a*Math.cos(phi)+b*Math.sin(phi);
    B = c;
    sq = Math.sqrt(A*A+B*B);
    sinTheta = -B/sq;
    cosTheta = A/sq;
    if (Math.abs(cosTheta)<JXG.Math.eps) { cosTheta = 1.0; }
    return sinTheta*Math.cos(phi)/cosTheta;
};

/**
 * Treat the line as parametric curve in homogeneous coordinates. See {@link #X} for a detailed description.
 * @type float
 * @return Y(phi) TODO description
 */
JXG.Line.prototype.Y = function (phi) {
    var a = this.stdform[1],
        b = this.stdform[2],
        c = this.stdform[0],
        A, B, sq, sinTheta, cosTheta;
    phi *= Math.PI;
    A = a*Math.cos(phi)+b*Math.sin(phi);
    B = c;
    sq = Math.sqrt(A*A+B*B);
    sinTheta = -B/sq;
    cosTheta = A/sq;
    if (Math.abs(cosTheta)<JXG.Math.eps) { cosTheta = 1.0; }
    return sinTheta*Math.sin(phi)/cosTheta;
};

/**
 * Treat the line as parametric curve in homogeneous coordinates. See {@link #X} for a detailed description.
 * @type float
 * @return Z(phi) TODO description
 */
JXG.Line.prototype.Z = function (phi) {
    var a = this.stdform[1],
        b = this.stdform[2],
        c = this.stdform[0],
        A, B, sq, cosTheta;
    phi *= Math.PI;
    A = a*Math.cos(phi)+b*Math.sin(phi);
    B = c;
    sq = Math.sqrt(A*A+B*B);
    cosTheta = A/sq;
    if (Math.abs(cosTheta)>=JXG.Math.eps) {
        return 1.0;
    } else {
        return 0.0;
    }
};

/**
 * TODO circle?!? --michael
 * private or public? --michael
 * Treat the circle as parametric curve:
 * t runs from 0 to 1
 * @private
 */
JXG.Line.prototype.minX = function () {
    return 0.0;
};

/**
 * TODO circle?!? --michael
 * private or public? --michael
 * Treat the circle as parametric curve:
 * t runs from 0 to 1
 * @private
 */
JXG.Line.prototype.maxX = function () {
    return 1.0;
};

/**
 * Adds ticks to this line. Ticks can be added to any kind of line: line, arrow, and axis.
 * @param {JXG.Ticks} ticks Reference to a ticks object which is describing the ticks (color, distance, how many, etc.).
 * @type String
 * @return Id of the ticks object.
 */
JXG.Line.prototype.addTicks = function(ticks) {
    if(ticks.id == '' || typeof ticks.id == 'undefined')
        ticks.id = this.id + '_ticks_' + (this.ticks.length+1);

    this.board.renderer.drawTicks(ticks);
    this.ticks.push(ticks);

    this.ticks[this.ticks.length-1].updateRenderer();

    return ticks.id;
};

/**
 * Removes all ticks from a line.
 */
JXG.Line.prototype.removeAllTicks = function() {
    var t;
    for(t=this.ticks.length; t>0; t--) {
        this.board.renderer.remove(this.ticks[t-1].rendNode);
    }
    this.ticks = new Array();
};

/**
 * Removes ticks identified by parameter named tick from this line.
 * @param {JXG.Ticks} tick Reference to tick object to remove.
 */
JXG.Line.prototype.removeTicks = function(tick) {
    var t, j;
    if(this.defaultTicks != null && this.defaultTicks == tick) {
        this.defaultTicks = null;
    }

    for(t=this.ticks.length; t>0; t--) {
        if(this.ticks[t-1] == tick) {
            this.board.renderer.remove(this.ticks[t-1].rendNode);

            for(j=0; j<this.ticks[t-1].ticks.length; j++) {
                if(this.ticks[t-1].labels[j] != null)
                    if (this.ticks[t-1].labels[j].show) this.board.renderer.remove(this.ticks[t-1].labels[j].rendNode);
            }
            delete(this.ticks[t-1]);
        }
    }
};

/**
 * @class This element is used to provide a constructor for a general line. A general line is given by two points. By setting additional properties
 * a line can be used as an arrow and/or axis.
 * @pseudo
 * @description
 * @name Line
 * @augments JXG.Line
 * @constructor
 * @type JXG.Line
 * @throws {Exception} If the element cannot be constructed with the given parent objects an exception is thrown.
 * @param {JXG.Point,array_JXG.Point,array} point1,point2 Parent elements can be two elements either of type {@link JXG.Point} or array of
 * numbers describing the coordinates of a point. In the latter case the point will be constructed automatically as a fixed invisible point.
 * @param {number_number_number} a,b,c A line can also be created providing three numbers. The line is then described by the set of solutions
 * of the equation <tt>a*x+b*y+c*z = 0</tt>.
 * @example
 * // Create a line using point and coordinates/
 * // The second point will be fixed and invisible.
 * var p1 = board.create('point', [4.5, 2.0]);
 * var l1 = board.create('line', [p1, [1.0, 1.0]]);
 * </pre><div id="c0ae3461-10c4-4d39-b9be-81d74759d122" style="width: 300px; height: 300px;"></div>
 * <script type="text/javascript">
 *   var glex1_board = JXG.JSXGraph.initBoard('c0ae3461-10c4-4d39-b9be-81d74759d122', {boundingbox: [-1, 7, 7, -1], axis: true, showcopyright: false, shownavigation: false});
 *   var glex1_p1 = glex1_board.create('point', [4.5, 2.0]);
 *   var glex1_l1 = glex1_board.create('line', [glex1_p1, [1.0, 1.0]]);
 * </script><pre>
 * @example
 * // Create a point using three coordinates
 * var l1 = board.create('line', [1.0, -2.0, 3.0]);
 * </pre><div id="cf45e462-f964-4ba4-be3a-c9db94e2593f" style="width: 300px; height: 300px;"></div>
 * <script type="text/javascript">
 *   var glex2_board = JXG.JSXGraph.initBoard('cf45e462-f964-4ba4-be3a-c9db94e2593f', {boundingbox: [-1, 7, 7, -1], axis: true, showcopyright: false, shownavigation: false});
 *   var glex2_l1 = glex2_board.create('line', [1.0, -2.0, 3.0]);
 * </script><pre>
 */
JXG.createLine = function(board, parents, atts) {
    var el, p1, p2, i,
        c = [];
        
    atts = JXG.checkAttributes(atts,{
        withLabel:JXG.readOption(board.options,'line','withLabel'),
        layer:null,
        labelOffsets:JXG.readOption(board.options,'line','labelOffsets')
    });

    var constrained = false;
    if (parents.length == 2) { // The line is defined by two points (or coordinates of two points)
        if (parents[0].length>1) { // point 1 given by coordinates
            p1 = board.create('point', parents[0], {visible:false,fixed:true});
        } else if (parents[0].elementClass == JXG.OBJECT_CLASS_POINT) {
            p1 =  JXG.getReference(board,parents[0]);
        } else if ((typeof parents[0] == 'function') && (parents[0]().elementClass == JXG.OBJECT_CLASS_POINT)) {
            p1 = parents[0]();
            constrained = true;
        } else
            throw new Error("JSXGraph: Can't create line with parent types '" + 
                            (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'." +
                            "\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");

        if (parents[1].length>1) { // point 2 given by coordinates
            p2 = board.create('point', parents[1], {visible:false,fixed:true});
        } else if (parents[1].elementClass == JXG.OBJECT_CLASS_POINT) {
            p2 =  JXG.getReference(board,parents[1]);
        } else if ((typeof parents[1] == 'function') && (parents[1]().elementClass == JXG.OBJECT_CLASS_POINT)) {
            p2 = parents[1]();
            constrained = true;
        } else
            throw new Error("JSXGraph: Can't create line with parent types '" + 
                            (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'." +
                            "\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");
        el = new JXG.Line(board, p1.id, p2.id, atts['id'], atts['name'],atts['withLabel'],atts['layer']);
        if(constrained) {
        	el.constrained = true;
        	el.funp1 = parents[0];
        	el.funp2 = parents[1];
        }
    }
    else if (parents.length==3) {  // Line is defined by three coordinates
        // free line
        for (i=0;i<3;i++) {
            if (typeof parents[i]=='number') {
                c[i] = function(z){ return function() { return z; }; }(parents[i]);
            } else if (typeof parents[i]=='function') {
                c[i] = parents[i];
            } else {
                throw new Error("JSXGraph: Can't create line with parent types '" + 
                                (typeof parents[0]) + "' and '" + (typeof parents[1]) + "' and '" + (typeof parents[2])+ "'." +
                                "\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");
            }
        }
        // point 1: (0,c,-b)
        /*
        p1 = board.create('point',[
                function() { return 0.0;},
                function() { return c[2]();},
                function() { return -c[1]();}],{visible:false,name:' '});
        */
        // New version: point1 is the midpoint between (0,c,-b) and point 2. => point1 is finite.
        p1 = board.create('point',[
                function() { return (0.0 + c[2]()*c[2]()+c[1]()*c[1]())*0.5;},
                function() { return (c[2]() - c[1]()*c[0]()+c[2]())*0.5;},
                function() { return (-c[1]() - c[2]()*c[0]()-c[1]())*0.5;}],{visible:false,name:' '});
        // point 2: (b^2+c^2,-ba+c,-ca-b)
        p2 = board.create('point',[
                function() { return c[2]()*c[2]()+c[1]()*c[1]();},
                function() { return -c[1]()*c[0]()+c[2]();},
                function() { return -c[2]()*c[0]()-c[1]();}],{visible:false,name:' '});
        el = new JXG.Line(board, p1.id, p2.id, atts['id'], atts['name'],atts['withLabel']);
    }
    else if ((parents.length==1) && (typeof parents[0] == 'function') && (parents[0]().length == 2) &&
    		 (parents[0]()[0].elementClass == JXG.OBJECT_CLASS_POINT) && (parents[0]()[1].elementClass == JXG.OBJECT_CLASS_POINT)) {
    	var ps = parents[0]();
        el = new JXG.Line(board, ps[0].id, ps[1].id, atts['id'], atts['name'],atts['withLabel'],atts['layer']);
        el.constrained = true;
        el.funps = parents[0];
    } else
        throw new Error("JSXGraph: Can't create line with parent types '" + 
                        (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'." +
                        "\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");

    el.labelOffsets = atts['labelOffsets'];
    return el;
};

JXG.JSXGraph.registerElement('line', JXG.createLine);

/**
 * @class This element is used to provide a constructor for a segment. It's strictly spoken just a wrapper for element {@link Line} with {@link JXG.Line#straightFirst}
 * and {@link JXG.Line#straightLast} properties set to false.
 * @pseudo
 * @description
 * @name Segment
 * @augments JXG.Line
 * @constructor
 * @type JXG.Line
 * @throws {Exception} If the element cannot be constructed with the given parent objects an exception is thrown.
 * @param {JXG.Point,array_JXG.Point,array} point1,point2 Parent elements can be two elements either of type {@link JXG.Point} or array of numbers describing the
 * coordinates of a point. In the latter case the point will be constructed automatically as a fixed invisible point.
 * @param {number_number_number} a,b,c A line can also be created providing three numbers. The line is then described by the set of solutions
 * of the equation <tt>a*x+b*y+c*z = 0</tt>.
 * @see Line
 * @example
 * // Create a segment providing two points.
 *   var p1 = board.create('point', [4.5, 2.0]);
 *   var p2 = board.create('point', [1.0, 1.0]);
 *   var l1 = board.create('segment', [p1, p2]);
 * </pre><div id="d70e6aac-7c93-4525-a94c-a1820fa38e2f" style="width: 300px; height: 300px;"></div>
 * <script type="text/javascript">
 *   var slex1_board = JXG.JSXGraph.initBoard('d70e6aac-7c93-4525-a94c-a1820fa38e2f', {boundingbox: [-1, 7, 7, -1], axis: true, showcopyright: false, shownavigation: false});
 *   var slex1_p1 = slex1_board.create('point', [4.5, 2.0]);
 *   var slex1_p2 = slex1_board.create('point', [1.0, 1.0]);
 *   var slex1_l1 = slex1_board.create('segment', [slex1_p1, slex1_p2]);
 * </script><pre>
 */
 JXG.createSegment = function(board, parents, atts) {
    var el;

    atts = JXG.checkAttributes(atts,{withLabel:JXG.readOption(board.options,'line','withLabel'), layer:null});
    atts['straightFirst'] = false;
    atts['straightLast'] = false;
    el = board.create('line', parents, atts);

    return el;
};

JXG.JSXGraph.registerElement('segment', JXG.createSegment);

/**
 * @class This element is used to provide a constructor for arrow, which is just a wrapper for element {@link Line} with {@link JXG.Line#straightFirst}
 * and {@link JXG.Line#straightLast} properties set to false and {@link JXG.Line#lastArrow} set to true.
 * @pseudo
 * @description
 * @name Arrow
 * @augments JXG.Line
 * @constructor
 * @type JXG.Line
 * @throws {Exception} If the element cannot be constructed with the given parent objects an exception is thrown.
 * @param {JXG.Point,array_JXG.Point,array} point1,point2 Parent elements can be two elements either of type {@link JXG.Point} or array of numbers describing the
 * coordinates of a point. In the latter case the point will be constructed automatically as a fixed invisible point.
 * @param {number_number_number} a,b,c A line can also be created providing three numbers. The line is then described by the set of solutions
 * of the equation <tt>a*x+b*y+c*z = 0</tt>.
 * @see Line
 * @example
 * // Create an arrow providing two points.
 *   var p1 = board.create('point', [4.5, 2.0]);
 *   var p2 = board.create('point', [1.0, 1.0]);
 *   var l1 = board.create('arrow', [p1, p2]);
 * </pre><div id="1d26bd22-7d6d-4018-b164-4c8bc8d22ccf" style="width: 300px; height: 300px;"></div>
 * <script type="text/javascript">
 *   var alex1_board = JXG.JSXGraph.initBoard('1d26bd22-7d6d-4018-b164-4c8bc8d22ccf', {boundingbox: [-1, 7, 7, -1], axis: true, showcopyright: false, shownavigation: false});
 *   var alex1_p1 = alex1_board.create('point', [4.5, 2.0]);
 *   var alex1_p2 = alex1_board.create('point', [1.0, 1.0]);
 *   var alex1_l1 = alex1_board.create('arrow', [alex1_p1, alex1_p2]);
 * </script><pre>
 */
JXG.createArrow = function(board, parents, attributes) {
    var el;

    attributes = JXG.checkAttributes(attributes,{withLabel:JXG.readOption(board.options,'line','withLabel'), layer:null});
    //if ( (JXG.isPoint(parents[0])) && (JXG.isPoint(parents[1])) ) { // The constructability decision is delkegated to the line object
        el = board.create('line',parents,attributes);
        //el = new JXG.Line(board, parents[0], parents[1], attributes['id'], attributes['name'],attributes['withLabel']);
        el.setStraight(false,false);
        el.setArrow(false,true);
        el.type = JXG.OBJECT_TYPE_VECTOR;
    //} // Ansonsten eine fette Exception um die Ohren hauen
    //else
    //    throw new Error("JSXGraph: Can't create arrow with parent types '" + (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'.");

    return el;
};

JXG.JSXGraph.registerElement('arrow', JXG.createArrow);

/**
 * @class This element is used to provide a constructor for an axis. It's strictly spoken just a wrapper for element {@link Line} with {@link JXG.Line#straightFirst}
 * and {@link JXG.Line#straightLast} properties set to true. Additionally {@link JXG.Line#lastArrow} is set to true and default {@link Ticks} will be created.
 * @pseudo
 * @description
 * @name Axis
 * @augments JXG.Line
 * @constructor
 * @type JXG.Line
 * @throws {Exception} If the element cannot be constructed with the given parent objects an exception is thrown.
 * @param {JXG.Point,array_JXG.Point,array} point1,point2 Parent elements can be two elements either of type {@link JXG.Point} or array of numbers describing the
 * coordinates of a point. In the latter case the point will be constructed automatically as a fixed invisible point.
 * @param {number_number_number} a,b,c A line can also be created providing three numbers. The line is then described by the set of solutions
 * of the equation <tt>a*x+b*y+c*z = 0</tt>.
 * @example
 * // Create an axis providing two coord pairs.
 *   var l1 = board.create('axis', [[0.0, 1.0], [1.0, 1.3]]);
 * </pre><div id="4f414733-624c-42e4-855c-11f5530383ae" style="width: 300px; height: 300px;"></div>
 * <script type="text/javascript">
 *   var axex1_board = JXG.JSXGraph.initBoard('4f414733-624c-42e4-855c-11f5530383ae', {boundingbox: [-1, 7, 7, -1], axis: true, showcopyright: false, shownavigation: false});
 *   var axex1_l1 = axex1_board.create('axis', [[0.0, 1.0], [1.0, 1.3]]);
 * </script><pre>
 */
JXG.createAxis = function(board, parents, attributes) {
    var point1,
        point2,
        line, dist, c1, c2, len, defTicks;

    // Arrays oder Punkte, mehr brauchen wir nicht.
    if ( (JXG.isArray(parents[0]) || JXG.isPoint(parents[0]) ) && (JXG.isArray(parents[1]) || JXG.isPoint(parents[1])) ) {
        if( JXG.isPoint(parents[0]) )
            point1 = parents[0];
        else
            point1 = new JXG.Point(board, parents[0],'','',false);

        if( JXG.isPoint(parents[1]) )
            point2 = parents[1];
        else
            point2 = new JXG.Point(board,parents[1],'','',false);

        /* Make the points fixed */
        point1.fixed = true;
        point2.fixed = true;

        attributes = JXG.checkAttributes(attributes,
            {'lastArrow':true, 'straightFirst':true, 'straightLast':true,
             'strokeWidth':1, 'withLabel':false,
             'strokeColor':board.options.axis.strokeColor});

        attributes.highlightStrokeColor = attributes.highlightStrokeColor || attributes.strokeColor || board.options.axis.highlightStrokeColor;

        line = board.create('line', [point1, point2], attributes);
        line.setProperty({needsRegularUpdate : false});  // Axes only updated after zooming and moving of  the origin.

        attributes = JXG.checkAttributes(attributes,{'minorTicks':4, 'insertTicks': true});

        if(attributes.ticksDistance != 'undefined' && attributes.ticksDistance != null) {
            dist = attributes.ticksDistance;
        } else if(JXG.isArray(attributes.ticks)) {
            dist = attributes.ticks;
        } else {
            c1 = new JXG.Coords(JXG.COORDS_BY_USER, [line.point1.coords.usrCoords.slice(1)],board);
            c2 = new JXG.Coords(JXG.COORDS_BY_USER, [line.point2.coords.usrCoords.slice(1)],board);
            board.renderer.calcStraight(line, c1, c2);
            len = c1.distance(JXG.COORDS_BY_USER,c2);
            //len *= 0.33;
            dist = 1.0; //len;
        }
        
        line.defaultTicks = board.create('ticks', [line, dist], attributes);
        line.defaultTicks.setProperty({needsRegularUpdate : false});
    }
    else
        throw new Error("JSXGraph: Can't create point with parent types '" + 
                        (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'." +
                        "\nPossible parent types: [point,point], [[x1,y1],[x2,y2]]");

    return line;
};

JXG.JSXGraph.registerElement('axis', JXG.createAxis);

/**
 * @class With the element tangent the slope of a line, circle, or curve in a certain point can be visualized. A tangent is always constructed
 * by a glider on a line, circle, or curve and describes the tangent in the glider point on that line, circle, or curve.
 * @pseudo
 * @description
 * @name Tangent
 * @augments JXG.Line
 * @constructor
 * @type JXG.Line
 * @throws {Exception} If the element cannot be constructed with the given parent objects an exception is thrown.
 * @param {Glider} g A glider on a line, circle, or curve.
 * @example
 * // Create a tangent providing a glider on a function graph
 *   var c1 = board.create('curve', [function(t){return t},function(t){return t*t*t;}]);
 *   var g1 = board.create('glider', [0.6, 1.2, c1]);
 *   var t1 = board.create('tangent', [g1]);
 * </pre><div id="7b7233a0-f363-47dd-9df5-4018d0d17a98" style="width: 400px; height: 400px;"></div>
 * <script type="text/javascript">
 *   var tlex1_board = JXG.JSXGraph.initBoard('7b7233a0-f363-47dd-9df5-4018d0d17a98', {boundingbox: [-6, 6, 6, -6], axis: true, showcopyright: false, shownavigation: false});
 *   var tlex1_c1 = tlex1_board.create('curve', [function(t){return t},function(t){return t*t*t;}]);
 *   var tlex1_g1 = tlex1_board.create('glider', [0.6, 1.2, tlex1_c1]);
 *   var tlex1_t1 = tlex1_board.create('tangent', [tlex1_g1]);
 * </script><pre>
 */
JXG.createTangent = function(board, parents, attributes) {
    var p,
        c,
        g, f, i, j, el, Dg, Df, tangent;

    if (parents.length==1) { // One arguments: glider on line, circle or curve
        p = parents[0];
        c = p.slideObject;
    } else if (parents.length==2) {     // Two arguments: (point,line|curve|circle|conic) or (line|curve|circle|conic,point). // Not yet: curve!
        if (JXG.isPoint(parents[0])) {  // In fact, for circles and conics it is the polar.
            p = parents[0];
            c = parents[1];
        } else if (JXG.isPoint(parents[1])) {
            c = parents[0];
            p = parents[1];
        } else {
            throw new Error("JSXGraph: Can't create tangent with parent types '" + 
                            (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'." +
                            "\nPossible parent types: [glider], [point,line|curve|circle|conic]");
        }
    } else {
        throw new Error("JSXGraph: Can't create tangent with parent types '" + 
                        (typeof parents[0]) + "' and '" + (typeof parents[1]) + "'." +
                        "\nPossible parent types: [glider], [point,line|curve|circle|conic]");
    }

    attributes = JXG.checkAttributes(attributes,{withLabel:JXG.readOption(board.options,'line','withLabel'), layer:null});
    
    if (c.elementClass == JXG.OBJECT_CLASS_LINE) {
        tangent = board.create('line', [c.point1,c.point2], attributes);
    } else if (c.elementClass == JXG.OBJECT_CLASS_CURVE && !(c.type == JXG.OBJECT_TYPE_CONIC)) {
        if (c.curveType!='plot') {
            g = c.X;
            f = c.Y;
            tangent = board.create('line', [
                    function(){ return -p.X()*board.D(f)(p.position)+p.Y()*board.D(g)(p.position);},
                    function(){ return board.D(f)(p.position);},
                    function(){ return -board.D(g)(p.position);}
                    ], attributes );
            p.addChild(tangent);
            // this is required for the geogebra reader to display a slope
            tangent.glider = p;
        } else {  // curveType 'plot'
            // equation of the line segment: 0 = y*(x1-x2) + x*(y2-y1) + y1*x2-x1*y2
            tangent = board.create('line', [
                    function(){ i=Math.floor(p.position);
                                if (i==c.numberPoints-1) i--;
                                if (i<0) return 1.0;
                                return c.Y(i)*c.X(i+1)-c.X(i)*c.Y(i+1);},
                    function(){ i=Math.floor(p.position);
                                if (i==c.numberPoints-1) i--;
                                if (i<0) return 0.0;
                                return c.Y(i+1)-c.Y(i);},
                    function(){ i=Math.floor(p.position);
                                if (i==c.numberPoints-1) i--;
                                if (i<0) return 0.0;
                                return c.X(i)-c.X(i+1);}
                    ], attributes );
            p.addChild(tangent);
            // this is required for the geogebra reader to display a slope
            tangent.glider = p;
        }
    } else if (c.type == JXG.OBJECT_TYPE_TURTLE) {
            tangent = board.create('line', [
                    function(){ i=Math.floor(p.position);
                                for(j=0;j<c.objects.length;j++) {  // run through all curves of this turtle
                                    el = c.objects[j];
                                    if (el.type==JXG.OBJECT_TYPE_CURVE) {
                                        if (i<el.numberPoints) break;
                                        i-=el.numberPoints;
                                    }
                                }
                                if (i==el.numberPoints-1) i--;
                                if (i<0) return 1.0;
                                return el.Y(i)*el.X(i+1)-el.X(i)*el.Y(i+1);},
                    function(){ i=Math.floor(p.position);
                                for(j=0;j<c.objects.length;j++) {  // run through all curves of this turtle
                                    el = c.objects[j];
                                    if (el.type==JXG.OBJECT_TYPE_CURVE) {
                                        if (i<el.numberPoints) break;
                                        i-=el.numberPoints;
                                    }
                                }
                                if (i==el.numberPoints-1) i--;
                                if (i<0) return 0.0;
                                return el.Y(i+1)-el.Y(i);},
                    function(){ i=Math.floor(p.position);
                                for(j=0;j<c.objects.length;j++) {  // run through all curves of this turtle
                                    el = c.objects[j];
                                    if (el.type==JXG.OBJECT_TYPE_CURVE) {
                                        if (i<el.numberPoints) break;
                                        i-=el.numberPoints;
                                    }
                                }
                                if (i==el.numberPoints-1) i--;
                                if (i<0) return 0.0;
                                return el.X(i)-el.X(i+1);}
                    ], attributes );
            p.addChild(tangent);
            // this is required for the geogebra reader to display a slope
            tangent.glider = p;
    } else if (c.elementClass == JXG.OBJECT_CLASS_CIRCLE || c.type == JXG.OBJECT_TYPE_CONIC) {
        /*
        Dg = function(t){ return -c.Radius()*Math.sin(t); };
        Df = function(t){ return c.Radius()*Math.cos(t); };
        return board.create('line', [
                    function(){ return -p.X()*Df(p.position)+p.Y()*Dg(p.position);},
                    function(){ return Df(p.position);},
                    function(){ return -Dg(p.position);}
                    ], attributes );
        */

        // If p is not on c, the tangent is the polar.
        // This construction should work on conics, too. p has to lie on c.
        tangent = board.create('line', [
                    function(){ return JXG.Math.matVecMult(c.quadraticform,p.coords.usrCoords)[0]; },
                    function(){ return JXG.Math.matVecMult(c.quadraticform,p.coords.usrCoords)[1]; },
                    function(){ return JXG.Math.matVecMult(c.quadraticform,p.coords.usrCoords)[2]; }
                ] , attributes);

        p.addChild(tangent);
        // this is required for the geogebra reader to display a slope
        tangent.glider = p;
    }
    
    return tangent;
};

/**
 * Register the element type tangent at JSXGraph
 * @private
 */
JXG.JSXGraph.registerElement('tangent', JXG.createTangent);
JXG.JSXGraph.registerElement('polar', JXG.createTangent);
// vim: et ts=4
