/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-------------------------------------------------------------------------------
FILENAME:                utils.h
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:      gpiv_scan_parfiles,
                         gpiv_scan_resourcefiles
			 gpiv_alloc_imgdata
			 gpiv_free_imgdata
                         gpiv_cp_img

			 gpiv_null_pivdata
                         gpiv_alloc_pivdata
			 gpiv_free_pivdata
                         gpiv_0_pivdata
                         gpiv_cp_pivdata
                         gpiv_add_dxdy_pivdata
                         gpiv_sum_dxdy_pivdata

			 gpiv_alloc_scdata
			 gpiv_free_scdata

			 gpiv_alloc_bindata
			 gpiv_free_bindata

                         gpiv_matrix, 
                         gpiv_free_matrix
			 gpiv_ucmatrix, 
                         gpiv_free_ucmatrix
                         gpiv_matrix_guint16
                         gpiv_free_matrix_guint16
			 gpiv_imatrix, 
                         gpiv_free_imatrix
                         gpiv_double_matrix
                         gpiv_free_double_matrix
                         gpiv_fftw_real_matrix
                         gpiv_free_fftw_real_matrix

                         gpiv_vector,
                         gpiv_free_vector, 
                         gpiv_vector_index,
                         gpiv_free_vector_index, 

                         gpiv_dvector,
                         gpiv_free_dvector, 
                         gpiv_dvector_index,
                         gpiv_free_dvector_index, 

                         gpiv_nulvector,
                         gpiv_free_nulvector,
                         gpiv_nulvector_index,
                         gpiv_free_nulvector_index, 

                         gpiv_ulvector,
                         gpiv_free_ulvector, 
                         gpiv_ulvector_index,
                         gpiv_free_ulvector_index, 

                         gpiv_ivector,
                         gpiv_free_ivector, 
                         gpiv_ivector_index,
                         gpiv_free_ivector_index,

                         gpiv_max, 
                         gpiv_min, 
                         gpiv_scan_iph,
                         gpiv_scan_iph_nl,
                         gpiv_fscan_iph, 
                         gpiv_scan_cph, 
                         gpiv_fscan_cph, 
                         gpiv_scan_fph, 
                         gpiv_fscan_fph, 
                         gpiv_scan_sph, 
                         gpiv_fscan_sph, 
                         gpiv_sort_3,
                         gpiv_histo
                         gpiv_cumhhisto
                         gpiv_histo_gnuplot
                         gpiv_gnuplot_pivdata
                         gpiv_scalar_gnuplot
                         gpiv_error 

LAST MODIFICATION DATE:  $Id: utils.h,v 1.16 2006/01/31 13:27:22 gerber Exp $
 --------------------------------------------------------------------------- */

#ifndef __LIBGPIV_UTILS_H__
#define __LIBGPIV_UTILS_H__


void
gpiv_scan_parameter(const char * PAR_KEY, 
                    const char * PARFILE,
                    void * pstruct,
                    int print_par
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Read parameters from local parameter file PARFILE
 *
 * INPUTS:
 *      PAR_KEY:        Parameter key, specific for each process
 *      PARFILE:        Parameter file, specific for each process
 *      print_par:      flag for printing parameters
 *
 * OUTPUTS:
 *      pstruct:        pointer to parameter structure
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_scan_resourcefiles(const char * PAR_KEY, 
                        void * pstruct,
                        int print_par
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Opens resource files ~/GPIV_HOME_RSC_FILE (hidden) and 
 *      SYSTEM_RSC_DIR/GPIV_SYSTEM_RSC_FILE
 *      Reads parameters from it
 *
 * INPUTS:
 *      par_key:        Parameter key, specific for each process
 *      print_par:      parameter to print to stdout
 *
 * OUTPUTS:
 *      pstruct:        pointer to parameter structure
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/


guint16 **
gpiv_alloc_img(GpivImagePar image_par
               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for image
 *
 * INPUTS:
 *      image_par:      image parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      img:            2-dim image data arry
 *---------------------------------------------------------------------------*/



void 
gpiv_free_img(guint16 ** img, 
              GpivImagePar image_par
              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for image
 *
 * INPUTS:
 *      image_par:      image parameters
 *      img:            2-dim image data arry
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_cp_img(guint16 **img_src, 
            guint16 **img_dest,
            GpivImagePar image_par
            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies contents of img_src to img_dest
 *
 * INPUTS:
 *      image_par:      image parameters
 *      img_src:            2-dim image data arry
 *
 * OUTPUTS:
 *      img_dest:            2-dim image data arry
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_null_pivdata(GpivPivData * piv_data
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets all elements of piv_data structure to NULL
 *
 * INPUTS:
 *      piv_data:      Input PIV data structure
 *
 * OUTPUTS:
 *      piv_data:      Output PIV data structure
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_alloc_pivdata(GpivPivData * piv_data
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for GpivPivData
 *
 * INPUTS:
 *      piv_data:       .nx and .ny members of GpivPivData structure
 *
 * OUTPUTS:
 *      piv_data:        point_x, point_y, dx, dy, snr, peak_no of GpivPivData
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_free_pivdata(GpivPivData * piv_data
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for GpivPivData
 *
 * INPUTS:
 *      piv_data:       PIV data structure
 *
 * OUTPUTS:
 *      piv_data:      NULL pointer to point_x, point_y, dx, dy, snr, peak_no
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_0_pivdata(GpivPivData * piv_data
               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets estimators, snr and peak_nr of piv_data to 0 or 0.0. 
 *      The structure will have to be allocated before (with 
 *      gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data:   PIV data structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_cp_pivdata(GpivPivData * piv_data_in,
                GpivPivData * piv_data_out
                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies data from piv_data_in to piv_data_out. Both structures will have
 *      to be allocated before (with gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data_in:   Input PIV data structure
 *
 * OUTPUTS:
 *      piv_data_out:  Output PIV data structure
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_add_dxdy_pivdata(GpivPivData * piv_data_in,
                      GpivPivData * piv_data_out
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Adds displacements (dx, dy), snr and peak_nr from piv_data_in to 
 *      piv_data_out. Both structures will have to be allocated before 
 *      (with gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data_in:   Input PIV data structure
 *
 * OUTPUTS:
 *      piv_data_out:  Output PIV data structure
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char * 
gpiv_sum_dxdy_pivdata(GpivPivData gpd,
                      gfloat * sum
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Adds all displacements in order to calculate residuals
 *      The structure will have to be allocated before (with 
 *      gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data:   PIV data structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_alloc_scdata(GpivScalarData * scal_data
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for GpivScalarData
 *
 * INPUTS:
 *      scal_data:      .nx and .ny members of GpivScalarData structure
 *
 * OUTPUTS:
 *      scal_data:       point_x, point_y, scalar, flag of GpivScalarData
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_free_scdata(GpivScalarData * scal_data
                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for GpivScalarData
 *
 * INPUTS:
 *      scal_data:      scalar data structure
 *
 * OUTPUTS:
 *      scal_data:       NULL pointer to point_x, point_y, scalar, flag
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_alloc_bindata(GpivBinData * bin_data
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for GpivBinData
 *
 * INPUTS:
 *      bin_data:       nbins of GpivBinData
 *
 * OUTPUTS:
 *      bin_data:       count, bound, centre of GpivBinData
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void
gpiv_free_bindata(GpivBinData * bin_data
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for GpivBinData
 *
 * INPUTS:
 *      bin_data:       data of bins
 *
 * OUTPUTS:
 *      bin_data:       NULL pointer to count, bound, centre
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



float **
gpiv_matrix_index(long nrl,
                  long nrh,
                  long ncl,
                  long nch
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of float data 
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      long nrl:	lowest row index number
 *      nrh:		highest row index number
 *      ncl:		lowest column index number
 *      nch	        highest column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      2-dimensional array
 *---------------------------------------------------------------------------*/



void 
gpiv_free_matrix_index(float ** m,
		 long nrl,
		 long nrh,
		 long ncl,
		 long nch
		 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of float data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      m:	        matrix
 *      nrl:	        lowest row index number
 *      long nrh:	highest row index number
 *	long ncl:	lowest column index number
 *	long nch:	highest column index number
 *
 * OUTPUTS:
 *      m:             NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



float **
gpiv_matrix(long nr,
            long nc
            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of float data
 *
 * INPUTS:
 *      nr:	number of rows
 *      nc:	number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_matrix(float ** m
		 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of float data
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



unsigned char **
gpiv_ucmatrix(long nr,
              long nc
              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of guint16 data
 *
 * INPUTS:
 *      nrl:	lowest row index number
 *      nrh:	highest row index number 
 *      ncl:	lowest column index number
 *      nch:	highest column index number 
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_ucmatrix(unsigned char ** m
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of guint16 data
 *
 * INPUTS:
 *      m:	        matrix 
 *	nrl:	        lowest row index number
 *	nrh:	        highest row index number
 *	ncl:	        lowest column index number
 *	nch:	        highest column index number
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



guint8 **
gpiv_matrix_guint8(long nr, 
                    long nc
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a guint8 matrix with subscript range m[0..nr][0..nc]
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_matrix_guint8(guint8 ** m
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a guint8 matrix allocated by gpiv_matrix_guint8
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/


guint16 **
gpiv_matrix_guint16(long nr, 
                    long nc
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a guint16 matrix with subscript range m[0..nr][0..nc]
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_matrix_guint16(guint16 ** m
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a guint16 matrix allocated by gpiv_matrix_guint16
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int **
gpiv_imatrix_index(long nrl,
                   long nrh,
                   long ncl,
                   long nch
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of integer data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *	nrl:	        lowest row index number
 *	nrh:	        highest row index number
 *	ncl:	        lowest column index number
 *	nch:	        highest column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_imatrix_index(int ** m,
                        long nrl,
                        long nrh,
                        long ncl,
                        long nch
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of integer data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      m:	        matrix 
 *	nrl:	        lowest row index number
 *	nrh:	        highest row index number
 *	ncl:	        lowest column index number
 *	nch:	        highest column index number
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int **
gpiv_imatrix(long nr,
             long nc
             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of integer data
 *      depreciated, use gpiv_imatrix_index instead
 *
 * INPUTS:
 *	nrl:	        number of rows
 *	ncl:	        number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_imatrix(int ** m
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of integer data
 *      unappreciated, use gpiv_free_imatrix_index instead
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



double **
gpiv_double_matrix(glong nr, 
                    glong nc 
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a contiguous 2-dimensional double matrix 
 *      of nr x nc
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_double_matrix(double **m
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a double matrix allocated by gpiv_double_matrix()
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



fftw_real **
gpiv_fftw_real_matrix(glong nr, 
                      glong nc 
                      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a contiguous 2-dimensional fftw_real_matrix matrix 
 *      of nr x nc
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_fftw_real_matrix(fftw_real **m
                           );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a gdouble matrix allocated by gpiv_fftw_real_matrix_matrix()
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



fftw_complex **
gpiv_fftw_complex_matrix(long nr, 
                         long nc 
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a contiguous 2-dimensional fftw_complex matrix 
 *      of nr x nc
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/



void 
gpiv_free_fftw_complex_matrix(fftw_complex ** m
                              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a fftw_real matrix allocated by gpiv_fftw_complex_matrix()
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



float *
gpiv_vector(long nl
            );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of float data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_vector(float * v
		 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of float data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



float *
gpiv_vector_index(long nl,
                  long nh
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of float data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_vector_index(float * v,
                       long nl,
                       long nh
                       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of float data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



gdouble *
gpiv_dvector(glong nl
             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of gdouble data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void
gpiv_free_dvector(double * v
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of double data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



double *
gpiv_dvector_index(long nl, 
                   long nh
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of double data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void
gpiv_free_dvector_index(double * v, 
                        long nl, 
                        long nh
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of double data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



long *
gpiv_nulvector(long nl
               );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of long data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_nulvector(long * v
		    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of long data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



long *
gpiv_nulvector_index(long nl,
                     long nh
                     );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of long data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_nulvector_index(long * v,
                          long nl,
                          long nh
                          );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of long data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



unsigned long *
gpiv_ulvector(long nl
              );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of unsigned long data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_ulvector(unsigned long * v
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of unsigned long data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



unsigned long *
gpiv_ulvector_index(long nl,
                    long nh
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of unsigned long data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_ulvector_index(unsigned long * v,
                         long nl,
                         long nh
                         );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of unsigned long data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int *
gpiv_ivector(long nl
             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of integer data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_ivector(int * v
		  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of integer data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



int *
gpiv_ivector_index(long nl,
                   long nh
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of integer data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_ivector_index(int * v,
                        long nl,
                        long nh
                        );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of integer data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



gboolean *
gpiv_gbolvector(glong nl
                );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of gboolean data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/



void 
gpiv_free_gbolvector(gboolean *v
                     );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of gboolean data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional boolean data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



long 
gpiv_lmax(long a,
	  long b
	  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates maximum of longs a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of maximum
 *---------------------------------------------------------------------------*/



long 
gpiv_lmin(long a,
	  long b
	  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates minimum of longs a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of minimum
 *---------------------------------------------------------------------------*/



int 
gpiv_max(int a,
	 int b
	 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates maximum of integers a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of maximum
 *---------------------------------------------------------------------------*/



int 
gpiv_min(int a,
	 int b
	 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates minimum of integers a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of minimum
 *---------------------------------------------------------------------------*/



gboolean
gpiv_scan_iph(const char * PAR_KEY,
	      char * line,
	      char * par_name,
	      char * key,
	      int * parameter,
	      gboolean print_par,
	      gboolean flag_par
	      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **integer** parameter key
 *      and value or image header value (without program key).
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      flag_par:       flag to use program key
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



gboolean
gpiv_scan_iph_nl (FILE *fp_h, 
                  const char *PROGNAME, 
                  char *line, 
                  char *par_name, 
                  char *key, 
                  int *parameter, 
                  gboolean print_par, 
                  gboolean flag_par
                  );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **integer** parameter key
 *      and value or image header value (without program key). The
 *      value to be read is on the next line of the key.
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *      fp_h:           file pointer of header to read the data from
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      flag_par:       flag to use program key
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



gboolean
gpiv_fscan_iph(const char * PAR_KEY,
	       char * line,
	       char * par_name,
	       char * key,
	       int * parameter,
	       int print_par,
	       FILE * fp,
	       gboolean flag_par
	       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **integer** parameter key
 *      and value or image header value (without program key). Prints result
 *      to file.
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      fp:             file pointer to print parameter to file
 *      flag_par:       flag to use program key
 *
 * OUTPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



gboolean
gpiv_scan_cph(const char * PAR_KEY,
	      char * line,
	      char * par_name,
	      char * key,
	      char * parameter,
	      char print_par,
	      int flag_par
	      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **char** parameter key
 *      and value or image header value (without program key).
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



gboolean
gpiv_fscan_cph(const char * PAR_KEY,
	       char * line,
	       char * par_name,
	       char * key,
	       char * parameter,
	       char print_par,
	       FILE * fp,
	       int flag_par
	       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **char** parameter key
 *      and value or image header value (without program key). Prints result
 *      to file.
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      fp:             file pointer to print parameter to file
 *      flag_par:       flag to use program key
 *
 * OUTPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



gboolean
gpiv_scan_fph(const char * PAR_KEY,
	      char * line,
	      char * par_name,
	      char * key,
	      float * parameter,
	      char print_par,
	      int flag_par
	      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **float** parameter key
 *      and value or image header value (without program key).
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */




gboolean
gpiv_scan_sph (const char * PROGNAME, 
	       char * line, 
	       char * par_name, 
	       char * key, 
	       char parameter[], 
	       char print_par, 
	       int flag_par
	       );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **string** parameter key
 *      and value or image header value (without program key)
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



gboolean
gpiv_fscan_sph (const char * PROGNAME, 
		char * line, 
		char * par_name, 
		char * key, 
		char parameter[], 
		char print_par, FILE *fp, 
		int flag_par
		);
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **string** parameter key
 *      and value or image header value (without program key). Prints result
 *      to file.
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      fp:             file pointer to print parameter to file
 *      flag_par:       flag to use program key
 *
 * OUTPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */



char *
gpiv_sort_3(unsigned long n,
	    float arr[],
	    float arr_2[],
	    float arr_3[]
	    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Orders array arr AND its belonging arrays arr_2 and arr_3
 *      Also swaps accompanying arrays
 *
 * INPUTS:
 *      n:	        length of array
 *      arr[]:	        array to be sorted
 *      arr_3[]:	second belonging array to arr
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_histo(GpivPivData * data,
	   GpivBinData * klass 
	   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates histogram from GpivPivData (NOT from GpivScalarData!!)
 *
 * INPUTS:
 *      data:           Input data
 *
 * OUTPUTS:
 *      klass:          Output data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_cumhisto(GpivPivData * data, 
	      GpivBinData * klass
	      );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates cumulative histogram from GpivPivData (NOT from GpivScalarData!!)
 *
 * INPUTS:
 *      data:           Input data
 *
 * OUTPUTS:
 *      klass:          Output data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



char *
gpiv_histo_gnuplot(char * fname_out,
		   char * title,
		   const char *GNUPLOT_DISPLAY_COLOR,
		   const int GNUPLOT_DISPLAY_SIZE
		   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Plots histogram on screen with gnuplot
 *
 * INPUTS:
 *      fname_out:	output filename
 *      title:	        plot title
 *      GNUPLOT_DISPLAY_COLOR:  display color of window containing graph
 *      GNUPLOT_DISPLAY_SIZE:   display size of window containing graph
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_piv_gnuplot(char * fname, 
                 char * title, 
                 float gnuplot_scale,
                 const char * GNUPLOT_DISPLAY_COLOR, 
                 const int GNUPLOT_DISPLAY_SIZE,
                 GpivImagePar image_par, 
                 GpivEvalPar piv_eval_par,
                 GpivPivData piv_data,
                 const char *RCSID
                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Plots piv data as vectors on screen with gnuplot
 *
 * INPUTS:
 *      fname:         file name containing plot
 *      title:         title of plot
 *      gnuplot_scale: vector scale
 *      GNUPLOT_DISPLAY_COLOR:  display color of window containing graph
 *      GNUPLOT_DISPLAY_SIZE:   display size of window containing graph
 *      image_par:      image parameters
 *      piv_eval_par:   piv evaluation parameters
 *      piv_data:       piv data
 *      RCSID:          program name and version that interrogated the image
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



char *
gpiv_scalar_gnuplot(char * fname_out, 
		    char * title, 
		    const char * GNUPLOT_DISPLAY_COLOR, 
		    const int GNUPLOT_DISPLAY_SIZE
		    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Graphical output with gnuplot for scalar data
 *
 * INPUTS:
 *      fname_out:     file name containing plot
 *      title:         title of plot
 *      GNUPLOT_DISPLAY_COLOR:  display color of window containing graph
 *      GNUPLOT_DISPLAY_SIZE:   display size of window containing graph
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_warning(char * message, ...
             );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints warning message with variable argumanent list
 *
 * INPUTS:
 *      message:       warning message
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



void 
gpiv_error(char * message, ...
	   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints error handling with variable argumanent list to stdout 
 *      and exits program
 *
 * INPUTS:
 *      message:       warning message
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



#endif /* __LIBGPIV_UTILS_H__ */
