/*******************************************************************************************************************************************
 cboxlayout.h
 
 CClass	
 |--CSerialized
    |--CMetaModule
       |--CObject
          |--CComponent
   	     |--CControl
	        |--CWidget
	   	   |--CContainer
		      |--CLayout
		         |--CBoxLayout

 Box layout abstraction, base class for box containers. CBoxLayout implements the GtkBox widget and is an abstract class which encapsulates 
 functionallity for a particular kind of containers, one that organizes a variable number of widgets into a rectangular area.
 CBoxLayout uses a notion of packing. Packing refers to adding widgets with reference to a particular position in a container. For a 
 CLayoutBox, there are two reference positions : the start and the end of the box.
*******************************************************************************************************************************************/

#ifndef __CBOXLAYOUT_H__
#define __CBOXLAYOUT_H__

#include "clayout.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// box pack definition
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef enum
{
	BOXPACK_START=0,
	BOXPACK_END
} TBoxPack;

//-----------------------------------------------------------------------------------------------------------------------------------------
// cboxlayout xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <cboxlayout spacing="UInt32" homogeneous="bool">
//    <cboxlayout-child box="TBoxPack" expand="bool" fill="bool" padding="UInt32">
//  [ <cboxlayout-child box="TBoxPack" expand="bool" fill="bool" padding="UInt32"> ]
//    ...
// </cboxlayout>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_BOXLAYOUT_ELEMENT			("cboxlayout");
static CString XML_BOXLAYOUT_ATTR_SPACING		("spacing");
static CString XML_BOXLAYOUT_ATTR_HOMOGENEOUS		("homogeneous");
static CString XML_BOXLAYOUT_CHILD_ELEMENT		("cboxlayout-child");
static CString XML_BOXLAYOUT_ATTR_CHILD_BOX_PACK	("box");
static CString XML_BOXLAYOUT_ATTR_CHILD_EXPAND		("expand");
static CString XML_BOXLAYOUT_ATTR_CHILD_FILL		("fill");
static CString XML_BOXLAYOUT_ATTR_CHILD_PADDING		("padding");

//-----------------------------------------------------------------------------------------------------------------------------------------
// CBoxLayout class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CBoxLayout : public CLayout
{
	// instanciation section
	public :

		CBoxLayout			(CContainer *inOwner=NULL, const CWidgetListener *inListener=NULL);
		virtual ~CBoxLayout		() =0;

	// box layout widget specifics
	protected :

		// widget initialization
		virtual void			PerformWidgetInitialize		();

	// box layout specific protected functions
        protected :

                // those two functions handle the box specific container behaviour when adding or removing a child widget
                virtual void                    PerformContainerAdd     	(CWidget *inChild);
                virtual void                    PerformContainerRemove  	(CWidget *inChild);

	// box layout specific functions
	public :

		// set the box packing style for the next children to be added: inExpand parameter must be true if the next child is to be
		// given extra space allocated to box. The extra space will be divided evenly between all children of the box that use this
		// option. The inFill parameter must be true if space given to a child by the expand option is actually allocated to child,
		// rather than just padding it. This parameter has no effect if expand is set to false. A child is always allocated the
		// full dimension a box. This option affects the other dimensions. The inPadding parameters specifies extra space in pixels
		// to put between this child and its neighbors, over and above the global amount specified by spacing. If the next child is
		// a widget at one of the reference ends of box, padding pixels are also put between the child and the reference box edge
		virtual void			SetBoxPack			(const TBoxPack inBoxPack, const bool inExpand=true, 
									 	 const bool inFill=true, const UInt16 inPadding=0L);

		// controlling whether or not all children are given equal space in the box
		virtual void			SetHomogeneous			(const bool inHomogeneous);
		bool				GetHomogeneous			() const;

		// basic spacing between children
		virtual void			SetSpacing			(const UInt16 inSpacing);
		UInt16				GetSpacing			() const;

		// get the box pack attributes for a given widget
		TBoxPack			GetBoxPack			(const CWidget *inChild) const; 
		bool				GetExpand			(const CWidget *inChild) const;
		bool				GetFill				(const CWidget *inChild) const;
		UInt16				GetPadding			(const CWidget *inChild) const;

	// CSerialized redefinition
	public :

		// box layout specific xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

	// protected attributes
	protected :

		// child box pack attributes
		struct TChildBoxPack
		{
			TBoxPack			m_BoxPack;
			bool				m_Expand;
			bool				m_Fill;
			UInt16				m_Padding;

			TChildBoxPack			(const TBoxPack =BOXPACK_START, const bool inExpand=true, const bool inFill=true,
							 const UInt16 inPadding=0L);

			TChildBoxPack &	operator =	(const TChildBoxPack &);
		};

		// a list of children box properties
		TBuffer <TChildBoxPack>		m_ChildrenBoxPack;

		// the box specific attributes
		bool				m_Homogeneous;
		UInt16				m_Spacing;

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// class tag and metaclass declaration
DECLARE_GENERIC_METACLASS ('blyt', CBoxLayout, CLayout);

#endif
