/* Item Factory
 *
 * The Gtk::ItemFactory object allows the easy creation of menus
 * from an item factory map of menu items. The only limitiations
 * are (1) each class can only have one map and (2) each item in
 * the map should have its own slot function that gets called
 * when the item is actviated.
 *
 * This example is similar to the item factory example used in the
 * application main window except its item factory displays the
 * menu items with images.
 */

#include "demowindow.h"
#include <inti/gtk/accelgroup.h>
#include <inti/gtk/itemfactory.h>
#include <inti/gtk/menubar.h>
#include <inti/gtk/checkmenuitem.h>
#include <inti/gtk/box.h>
#include <inti/gtk/label.h>
#include <inti/gtk/separator.h>
#include <inti/gtk/button.h>

using namespace Inti;

class ItemFactoryWindow : public Gtk::Window
{
	// Smart pointers don't need to unreferenced
	Pointer<Gtk::ItemFactory> item_factory;

protected:
	virtual bool on_delete_event(const Gdk::EventAny& event) { return true; }

public:
	ItemFactoryWindow();
	~ItemFactoryWindow();

	void menu_item_selected(const char *parent, const char *item);

	void on_tear_off1();

	void on_file_new();
	void on_file_open();
	void on_file_save();
	void on_file_save_as();
	void on_file_quit();

	void on_color_red();
	void on_color_green();
	void on_color_blue();

	void on_shape_square();
	void on_shape_rectangle();
	void on_shape_oval();

	void on_help_about();

	DECLARE_ITEM_FACTORY_MAP(ItemFactoryWindow)
};
//!

ItemFactoryWindow::ItemFactoryWindow()
{
	// Create the menubar
	Pointer<Gtk::AccelGroup> accel_group = new Gtk::AccelGroup;
 	item_factory = new Gtk::ItemFactory(GTK_TYPE_MENU_BAR, "<main>", accel_group);
	add_accel_group(accel_group);
	set_title("Item Factory");
	set_border_width(0);

	// Create menu items
	item_factory->create_items(*this);
	Gtk::MenuBar *menu = item_factory->menu_bar();

	// Preselect /Preferences/Shape/Oval over the other radios
	Gtk::CheckMenuItem *item = static_cast<Gtk::CheckMenuItem*>(item_factory->get_item("/Preferences/Shape/Oval"));
	item->set_active(true);

	// Add the menubar
	Gtk::VBox *vbox1 = new Gtk::VBox;
	add(*vbox1);
	vbox1->pack_start(*menu, false, false);

	// Add the remaining widgets
	Gtk::Label *label = new Gtk::Label("Type\n<alt>\nto start");
	label->set_size_request(200, 200);
	label->set_alignment(0.5, 0.5);
	vbox1->pack_start(*label);

	Gtk::HSeparator *separator = new Gtk::HSeparator;
	vbox1->pack_start(*separator, false);

 	Gtk::VBox *vbox2 = new Gtk::VBox;
	vbox2->set_border_width(10);
	vbox1->pack_start(*vbox2, false);

	Gtk::Button *button = new Gtk::Button("close");
	button->sig_clicked().connect(slot(this, &Gtk::Object::dispose));
	vbox2->pack_start(*button);
	button->set_flags(Gtk::CAN_DEFAULT);
	button->grab_default();

	show_all();
}

ItemFactoryWindow::~ItemFactoryWindow()
{
}

void
ItemFactoryWindow::menu_item_selected(const char *parent, const char *item)
{
	g_message("ItemFactory: activated the \"%s\" menu item: \"%s\"", parent, item);
}

void
ItemFactoryWindow::on_tear_off1()
{
	menu_item_selected("File", "New");
}

void
ItemFactoryWindow::on_file_new()
{
	menu_item_selected("File", "New");
}

void
ItemFactoryWindow::on_file_open()
{
	menu_item_selected("File", "Open");
}

void
ItemFactoryWindow::on_file_save()
{
	menu_item_selected("File", "Save");
}

void
ItemFactoryWindow::on_file_save_as()
{
	menu_item_selected("File", "Save As");
}

void
ItemFactoryWindow::on_file_quit()
{
	menu_item_selected("File", "Quit");
}

void
ItemFactoryWindow::on_color_red()
{
	menu_item_selected("Color", "Red");
}

void
ItemFactoryWindow::on_color_green()
{
	menu_item_selected("Color", "Green");
}

void
ItemFactoryWindow::on_color_blue()
{
	menu_item_selected("Color", "Blue");
}

void
ItemFactoryWindow::on_shape_square()
{
	menu_item_selected("Shape", "Square");
}

void
ItemFactoryWindow::on_shape_rectangle()
{
	menu_item_selected("Shape", "Rectangle");
}

void
ItemFactoryWindow::on_shape_oval()
{
	menu_item_selected("Shape", "Oval");
}

void
ItemFactoryWindow::on_help_about()
{
	menu_item_selected("Help", "About");
}

BEGIN_ITEM_FACTORY_MAP(ItemFactoryWindow)
	IFM_BRANCH("/_File"),
	IFM_TEAROFF_ITEM("/File/tearoff1", on_tear_off1),
	IFM_ITEM("/File/_New", "<control>N", on_file_new),
	IFM_ITEM("/File/_Open", "<control>0", on_file_open),
	IFM_ITEM("/File/_Save", "<control>S", on_file_save),
	IFM_ITEM("/File/Save As", 0, on_file_save_as),
	IFM_SEPARATOR("/File/sep1"),
	IFM_ITEM("/File/Quit", "<control>Q", on_file_quit),

	IFM_BRANCH("/_Preferences"),
	IFM_BRANCH("/_Preferences/_Color"),
	IFM_RADIO_ITEM("/_Preferences/Color/_Red", 0, on_color_red),
	IFM_RADIO_ITEM_LINK("/_Preferences/Color/_Green", on_color_green, "/Preferences/Color/Red"),
	IFM_RADIO_ITEM_LINK("/_Preferences/Color/_Blue", on_color_blue, "/Preferences/Color/Red"),

	IFM_BRANCH("/_Preferences/_Shape"),
	IFM_RADIO_ITEM("/_Preferences/Shape/_Square", 0, on_shape_square),
	IFM_RADIO_ITEM_LINK("/_Preferences/Shape/_Rectangle", on_shape_rectangle, "/Preferences/Shape/Square"),
	IFM_RADIO_ITEM_LINK("/_Preferences/Shape/_Oval", on_shape_oval, "/Preferences/Shape/Rectangle"),

	IFM_LAST_BRANCH("/_Help"),
	IFM_ITEM("/Help/_About", 0, on_help_about),
END_ITEM_FACTORY_MAP

Gtk::Window*
do_item_factory()
{
	static Gtk::Window *window = 0;

	if (!window)
		window = new ItemFactoryWindow;
	else
	{
		window->dispose();
		window = 0;
	}
	return window;
}

