/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/base.h
//! @brief The primaray base class interface.
//!
//! Provides ReferencedBase, the primary base class for the ReferencedObject,
//! G::Boxed, and G::TypeInstance base classes.

#ifndef INTI_BASE_H
#define INTI_BASE_H

#ifndef INTI_POINTER_H
#include <inti/pointer.h>
#endif

#ifndef INTI_UTF_STRING_H
#include <inti/utf-string.h>
#endif

namespace Inti {

//! @class ReferencedBase base.h inti/base.h
//! @brief The primary base class.
//!
//! ReferencedBase is a reference counting base class. It implements
//! the reference counting features required by the Inti smart pointer,
//! Inti::Pointer<>.

class ReferencedBase
{
	template<typename T> friend class Pointer;

	ReferencedBase(const ReferencedBase&);
	ReferencedBase& operator=(const ReferencedBase&);

	bool referenced;

protected:
	ReferencedBase();
	//!< Constructor.

	virtual ~ReferencedBase() = 0;
	//!< Destructor.

	void set_referenced(bool reference);
	//!< Set the internal referenced flag.
	//!< @param reference Set <EM>true</EM> if the initial reference count must be removed by owner.
	//!<
	//!< <BR>Called by derived classes to set the referenced flag. A G::Object sets this flag
	//!< to true , indicating that it owns the initial reference count and unref() must be called.
	//!< As a general rule a Gtk::Object sets this flag to false, indicating that the initial
	//!< reference count is floating and unref() only needs to be called if you take ownership. 
	//!< If you pass a Gtk::Object to an owner or add a Gtk::Widget to a container you only
	//!< need to call unref() if you called ref().

public:
	bool is_referenced() const;
	//!< The referenced flag setting.
	//!< @return <EM>true</EM> if unref() must be explicitly called on this object.

	virtual void ref();
	//!< Increase an object's reference count by one.

	virtual void unref();
	//!< Decrease an object's reference count by one.
	//!< After this call, if the object reference is zero and the object was allocated on the 
	//!< heap, delete is automatically called. <B>You don't need to call delete on objects
	//!< derived from ReferenceBase</B>.
};

} // namespace Inti

#endif // INTI_BASE_H

