/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  colorselection.cc - GtkColorSelection and GtkColorSelectionDialog C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "colorselection.h"
#include "private/colorselection_p.h"
#include "button.h"
#include "../gdk/color.h"

using namespace Inti;

/*  Gtk::ColorSelection
 */
 
Gtk::ColorSelection::ColorSelection(GtkColorSelection *color_selection, bool reference)
: VBox((GtkVBox*)color_selection, reference)
{
}

Gtk::ColorSelection::ColorSelection()
: VBox((GtkVBox*)ColorSelectionClass::create())
{
}

Gtk::ColorSelection::~ColorSelection()
{
}

GtkColorSelectionClass*
Gtk::ColorSelection::gtk_color_selection_class() const
{ 
	return get_class<GtkColorSelectionClass>();
}
	
Gtk::ColorSelection::operator GtkColorSelection* () const 
{ 
	return this ? gtk_color_selection() : 0; 
}
	
bool
Gtk::ColorSelection::get_has_opacity_control() const
{
	return gtk_color_selection_get_has_opacity_control(gtk_color_selection());
}

bool
Gtk::ColorSelection::get_has_palette() const
{
	return gtk_color_selection_get_has_palette(gtk_color_selection());
}

Gdk::Color
Gtk::ColorSelection::get_current_color() const
{
	GdkColor color;
	gtk_color_selection_get_current_color(gtk_color_selection(), &color);
	return Gdk::Color(&color, true);
}

unsigned short
Gtk::ColorSelection::get_current_alpha() const
{
	return gtk_color_selection_get_current_alpha(gtk_color_selection());
}

Gdk::Color
Gtk::ColorSelection::get_previous_color() const
{
	GdkColor color;
	gtk_color_selection_get_previous_color(gtk_color_selection(), &color);
	return Gdk::Color(&color, true);
}

unsigned short
Gtk::ColorSelection::get_previous_alpha() const
{
	return gtk_color_selection_get_previous_alpha(gtk_color_selection());
}

bool 
Gtk::ColorSelection::is_adjusting() const
{
	return gtk_color_selection_is_adjusting(gtk_color_selection());
}

bool
Gtk::ColorSelection::palette_from_string(const String& str, std::vector<Pointer<Gdk::Color> >& colors)
{
	g_return_val_if_fail(colors.empty(), false);
	GdkColor *tmp_colors = 0;
	int n_colors = 0;
	
	bool result = gtk_color_selection_palette_from_string(str.c_str(), &tmp_colors, &n_colors);
	
	if (result)
	{
		int i = 0;
		while (i < n_colors)
		{
			Pointer<Gdk::Color> tmp_color(G::Boxed::wrap<Gdk::Color>(GDK_TYPE_COLOR, &tmp_colors[i], true));
			colors.push_back(tmp_color);
			++i;
		}
		result = !colors.empty();
		g_free(tmp_colors);
	}
	return result;
}
	
void 
Gtk::ColorSelection::set_has_opacity_control(bool has_opacity)
{
	gtk_color_selection_set_has_opacity_control(gtk_color_selection(), has_opacity);
}

void 
Gtk::ColorSelection::set_has_palette(bool has_palette)
{
	return gtk_color_selection_set_has_palette(gtk_color_selection(), has_palette);
}

void 
Gtk::ColorSelection::set_current_color(const Gdk::Color& color)
{
	gtk_color_selection_set_current_color(gtk_color_selection(), color.gdk_color());
}

void 
Gtk::ColorSelection::set_current_alpha(unsigned short alpha)
{
	gtk_color_selection_set_current_alpha(gtk_color_selection(), alpha);
}

void 
Gtk::ColorSelection::set_previous_color(const Gdk::Color& color)
{
	gtk_color_selection_set_previous_color(gtk_color_selection(), color.gdk_color());
}

void 
Gtk::ColorSelection::set_previous_alpha(unsigned short alpha)
{
	gtk_color_selection_set_previous_alpha(gtk_color_selection(), alpha);
}

String
Gtk::ColorSelection::palette_to_string(const std::vector<Pointer<Gdk::Color> >& colors)
{
	g_return_val_if_fail(!colors.empty(), "");
	int count = colors.size();
	GdkColor *tmp_colors = new GdkColor[count];
	
	int i = 0;
	while (i < count)
	{
		tmp_colors[i] = *colors[i]->gdk_color();
		++i;
	}
	
	char *s = gtk_color_selection_palette_to_string(tmp_colors, count);
	String str(s);
	g_free(s);
	delete [] tmp_colors;
	return str;
}

/*  Gtk::ColorSelectionClass
 */

void
Gtk::ColorSelectionClass::init(GtkColorSelectionClass *g_class)
{
	VBoxClass::init((GtkVBoxClass*)g_class);
	g_class->color_changed = &color_changed_proxy;
}

GType
Gtk::ColorSelectionClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_COLOR_SELECTION, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::ColorSelectionClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::ColorSelectionClass::color_changed_proxy(GtkColorSelection *color_selection)
{
	ColorSelection *tmp_color_selection = G::Object::pointer<ColorSelection>(color_selection);
	if (tmp_color_selection)
		tmp_color_selection->on_color_changed();
	else
	{
		GtkColorSelectionClass *tmp_class = GTK_COLOR_SELECTION_GET_CLASS(color_selection);
		GtkColorSelectionClass *g_class = G::TypeInstance::class_peek_parent<GtkColorSelectionClass>(tmp_class);
		if (g_class->color_changed)
			g_class->color_changed(color_selection);
	}
}

/*  Gtk::ColorSelection Signal handlers
 */

void
Gtk::ColorSelection::on_color_changed()
{
	GtkColorSelectionClass *g_class = class_peek_parent<GtkColorSelectionClass>(gtk_color_selection_class());
	if (g_class->color_changed)
		g_class->color_changed(gtk_color_selection());
}

/*  Gtk::ColorSelection properties
 */

const Gtk::ColorSelection::HasOpacityControlPropertyType Gtk::ColorSelection::has_opacity_control_property("has_opacity_control");

const Gtk::ColorSelection::HasPalettePropertyType Gtk::ColorSelection::has_palette_property("has_palette");

const Gtk::ColorSelection::CurrentColorPropertyType Gtk::ColorSelection::current_color_property("current_color");

const Gtk::ColorSelection::CurrentAlphaPropertyType Gtk::ColorSelection::current_alpha_property("current_alpha");

/*  Gtk::ColorSelection Signals
 */

const Gtk::ColorSelection::ColorChangedSignalType Gtk::ColorSelection::color_changed_signal("color_changed");

/*  Gtk::ColorSelectionDialog
 */

Gtk::ColorSelectionDialog::ColorSelectionDialog(GtkColorSelectionDialog *color_selection_dialog, bool reference)
: Dialog((GtkDialog*)color_selection_dialog, reference)
{
}

Gtk::ColorSelectionDialog::ColorSelectionDialog() 
: Dialog((GtkDialog*)ColorSelectionDialogClass::create())
{
}

Gtk::ColorSelectionDialog::ColorSelectionDialog(const String& title) 
: Dialog((GtkDialog*)ColorSelectionDialogClass::create())
{
	set_title(title);
}

Gtk::ColorSelectionDialog::~ColorSelectionDialog()
{
}

GtkColorSelectionDialogClass* 
Gtk::ColorSelectionDialog::gtk_color_selection_dialog_class() const 
{ 
	return get_class<GtkColorSelectionDialogClass>(); 
}

Gtk::ColorSelectionDialog::operator GtkColorSelectionDialog* () const 
{ 
	return this ? gtk_color_selection_dialog() : 0; 
}
	
Gtk::ColorSelection*
Gtk::ColorSelectionDialog::colorsel() const
{
	return G::Object::wrap<ColorSelection>((GtkColorSelection*)gtk_color_selection_dialog()->colorsel);
}

Gtk::Button*
Gtk::ColorSelectionDialog::ok_button() const
{
	return G::Object::wrap<Button>((GtkButton*)gtk_color_selection_dialog()->ok_button);
}

Gtk::Button*
Gtk::ColorSelectionDialog::cancel_button() const
{
	return G::Object::wrap<Button>((GtkButton*)gtk_color_selection_dialog()->cancel_button);
}

Gtk::Button*
Gtk::ColorSelectionDialog::help_button() const
{
	return G::Object::wrap<Button>((GtkButton*)gtk_color_selection_dialog()->help_button);
}

/*  Gtk::ColorSelectionDialogClass
 */

void
Gtk::ColorSelectionDialogClass::init(GtkColorSelectionDialogClass *g_class)
{
	DialogClass::init((GtkDialogClass*)g_class);
}

GType
Gtk::ColorSelectionDialogClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_COLOR_SELECTION_DIALOG, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::ColorSelectionDialogClass::create()
{
	return g_object_new(get_type(), 0);
}

