/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/pango/types.h
//! @brief Pango types C++ wrapper interface.
//!
//! Provides common typedefs, enumerations and utility classes.

#ifndef INTI_PANGO_TYPES_H
#define INTI_PANGO_TYPES_H

#ifndef __PANGO_TYPES_H__
#include <pango/pango-types.h>
#endif

namespace Inti {

namespace Pango {
	
//! Glyph is a convenient typedef for PangoGlyph. Glyph structure represents a single glyph
//! in the output form of a glyph string.

typedef PangoGlyph Glyph;

//! @enum Pango::Direction
//! The Direction type represents the direction of writing for text.

enum Direction
{
	DIRECTION_LTR = PANGO_DIRECTION_LTR,
	//!< The text is written left-to-right.

	DIRECTION_RTL = PANGO_DIRECTION_RTL,
	//!< The text is written right-to-left.

	DIRECTION_TTB_LTR = PANGO_DIRECTION_TTB_LTR,
	//!< The text is written vertically top-to-bottom, with the rows ordered from left to right.
	
	DIRECTION_TTB_RTL = PANGO_DIRECTION_TTB_RTL
	//!< The text is written vertically top-to-bottom, with the rows ordered from right to left.
};

//! @class Rectangle types.h inti/pango/types.h
//! @brief A PangoRectangle C++ wrapper class.
//!
//! Rectangle represents a rectangle. It is frequently used to represent the logical or ink extents
//! of a single glyph or section of text (see, for instance, Pango::Font::get_glyph_extents()).

class Rectangle
{
	PangoRectangle rectangle_;

public:
//! @name Constructors
//! @{

	Rectangle();
	//!< Construct a with its x, y, width and height set to 0;

	Rectangle(int x, int y, int width, int height);
	//!< Construct a new rectangle.
	//!< @param x The X coordinate of the rectangle.
	//!< @param y The Y coordinate of the rectangle.
	//!< @param width The width of the rectangle.
	//!< @param height The height of the rectangle.
	
	Rectangle(const PangoRectangle& rectangle);
	//!< Construct a new rectangle from an existing PangoRectangle.
	//!< @param rectangle A PangoRectangle.

//! @}
//! @name Accessors
//! @{

	PangoRectangle* pango_rectangle() const { return const_cast<PangoRectangle*>(&rectangle_); }
	//!< Get a pointer to the PangoRectangle structure.

	operator PangoRectangle* () const;
	//!< Conversion operator; safely converts a Rectangle to a PangoRectangle pointer.

	int x() const;
	//!< Returns the X coordinate of the top-left corner of the rectangle.

	int y() const;
	//!< Returns the Y coordinate of the top-left corner of the rectangle.

	int width() const;
	//!< Returns the width of the rectangle.
	
	int height() const;
	//!< Returns the height of the rectangle.
	
	int ascent() const;
	//!< Extracts the ascent from a Pango::Rectangle representing glyph extents.
	//!< The ascent is the distance from the baseline to the highest point of 
	//!< the character. This is positive if the glyph ascends above the baseline.

	int descent() const;
	//!< Extracts the descent from a Pango::Rectangle representing glyph extents.
	//!< The descent is the distance from the baseline to the lowest point of 
	//!< the character. This is positive if the glyph descends below the baseline. 

	int lbearing() const;
	//!< Extracts the left bearing from a Pango::Rectangle representing glyph extents.
	//!< The left bearing is the distance from the horizontal origin to the farthest
	//!< left point of the character. This is positive for characters drawn completely
	//!< to the right of the glyph origin.

	int rbearing() const;
	//!< Extracts the right bearing from a Pango::Rectangle representing glyph extents.
	//!< The right bearing is the distance from the horizontal origin to the farthest
	//!< right point of the character. This is positive except for characters drawn 
	//!< completely to the left of the horizontal origin.

//! @}
//! @name Methods
//! @{

	void set(int x, int y, int width, int height);
	//!< Sets the position and size of the rectangle.
	//!< @param x The new X coordinate of the rectangle.
	//!< @param y The new Y coordinate of the rectangle.
	//!< @param width The new width of the rectangle.
	//!< @param height The new height of the rectangle.
	
//! @}
};

} // namespace Pango

} // namespace Inti

#endif // INTI_PANGO_TYPES_H

