#include <string.h>
#include <limits.h>
#include "mb.h"

int
mb_char_dec(const char *s, size_t *p_b, size_t e)
{
  size_t b = *p_b;
  int c1;

  if (b < e) {
    if ((c1 = (unsigned char)s[b]) >= 0xC0 && c1 <= 0xFD) {
      size_t i, j;
      int c, mask;

      for (i = b + 2, mask = 0x20 ; c1 & mask ; mask >>= 1, ++i)
	;

      for (c = (c1 & (mask - 1)), j = b + 1 ; j < e && j < i ; ++j)
	if (((unsigned char)s[j] & ~0x3F) == 0x80)
	  c = c * 0x40 + ((unsigned char)s[j] & 0x3F);
	else
	  break;

      if (j == i) {
	*p_b = j;
	return c;
      }
      else if (j == e) {
	return -c1;
      }
    }

    return c1;
  }
  else
    return EOF;
}

static struct {
  char *seq;
  size_t len;
} mb_lsl[] = {
  {"\x0F", 1},
  {"\x0E", 1},
  {"\x1B\x6E", 2},
  {"\x1B\x6F", 2},
}, mb_lsr[] = {
  {"", 0},
  {"\x1B\x7E", 2},
  {"\x1B\x7D", 2},
  {"\x1B\x7C", 2},
};

void
mb_store_esc(mb_char_t *ch, mb_info_t *info)
{
  if (info->GR != mb_FAKEUTF8) {
    int set = ch->set & MB_ESC_SET_MASK;
    int fc = ch->fc & MB_ESC_FC_MASK;
    int gn = ch->gn & MB_ESC_Gn_MASK;
    int sn = ch->sn & MB_ESC_Sn_MASK;

    if (set == mb_128) {
      if (fc == (0x47 & MB_ESC_FC_MASK)) {
	if (info->GR != mb_UTF8 && info->io_func.out("\x1B\x25\x47", 3, info->io_arg) == 3) {
	  info->GRB4UTF8 = info->GR;
	  info->GR = mb_UTF8;
	}
      }
    }
    else if (info->GR != mb_UTF8 || set != mb_94_0 || fc != (0x42 & MB_ESC_FC_MASK) || gn != mb_G0 || sn != mb_SL) {
      int esc = MB_ESC_ENC(set, fc);
      size_t esclen;

      if (info->GR == mb_UTF8) {
	if (info->io_func.out("\x1B\x25\x40", 3, info->io_arg) == 3)
	  info->GR = info->GRB4UTF8;
      }

      if (info->G[gn] != esc) {
	char buf[ISO2MB_ESC_LEN_MAX] = "\x1B";
	int fc6 = 0;
	size_t i = 1;

	switch (ch->set) {
	case mb_94x94:
	  buf[i++] = 0x24;

	  switch (fc) {
	  case 0x40 & MB_ESC_FC_MASK:
	  case 0x41 & MB_ESC_FC_MASK:
	  case 0x42 & MB_ESC_FC_MASK:
	    if (!(info->flag & MB_FLAG_28FOR94X94G0) && gn == mb_G0)
	      break;
	  default:
	    buf[i++] = 0x28 + gn;
	    break;
	  }

	  break;
	case mb_96_1:
	  fc6 = 1 << 5;
	case mb_96_0:
	  buf[i++] = 0x2C + gn;
	  break;
	case mb_94_3:
	  fc6 = 1 << 5;
	case mb_94_2:
	  buf[i++] = 0x28 + gn;
	  buf[i++] = 0x21;
	  break;
	case mb_94_1:
	  fc6 = 1 << 5;
	case mb_94_0:
	  buf[i++] = 0x28 + gn;
	  break;
	default:
	  return;
	}

	buf[i++] = fc | fc6 | 0x40;
	info->io_func.out(buf, i, info->io_arg);
	info->G[gn] = esc;
      }

      switch (sn) {
      case mb_SL:
	if (info->GL != gn && (esclen = mb_lsl[gn].len) && info->io_func.out(mb_lsl[gn].seq, esclen, info->io_arg) == esclen)
	  info->GL = gn;

	break;
      case mb_SR:
	if (info->GR != gn && (esclen = mb_lsr[gn].len) && info->io_func.out(mb_lsr[gn].seq, esclen, info->io_arg) == esclen)
	  info->GR = gn;
      default:
	break;
      }
    }
  }
}

void
mb_parse_badutf8(int c, mb_char_t *ch)
{
  int gnsn;

  if (c & (1 << 0)) {
    c >>= (1 + 0);
    ch->fc = MB_BADUTF8_FC(c >> 13);
    gnsn = MB_BADUTF8_GnSn(c >> 13);

    if ((c = (c & ((1 << 13) - 1)) + (1 << 10)) < 0x5F * 0x60)
      ch->set = mb_94x94;
    else {
      c -= 0x5F * 0x60;
      ch->set = mb_96_1;
    }
  }
  else if (c & (1 << 1)) {
    c >>= (1 + 1);
    ch->fc = MB_BADUTF8_FC(c >> 10);
    gnsn = MB_BADUTF8_GnSn(c >> 10);

    if ((c = (c & ((1 << 10) - 1))) < 0x60)
      ch->set = mb_96_0;
    else
      ch->set = mb_94x94;
  }
  else if (c & (1 << 2)) {
    c >>= (1 + 2);
    ch->fc = MB_BADUTF8_FC(c >> 13);
    gnsn = MB_BADUTF8_GnSn(c >> 13);
    c = (c & ((1 << 13) - 1)) + (1 << 10);
    ch->set = mb_128;
  }
  else if (c & (1 << 3)) {
    c >>= (1 + 3);
    ch->fc = MB_BADUTF8_FC(c >> 10);
    gnsn = MB_BADUTF8_GnSn(c >> 10);
    c &= (1 << 10) - 1;
    ch->set = mb_128;
  }
  else {
    c >>= (1 + 3);
    ch->set = mb_94_0 + ((c >> 7) & 0x03);
    ch->fc = MB_BADUTF8_FC(c >> (7 + 2));
    gnsn = MB_BADUTF8_GnSn(c >> (7 + 2));
    c &= (1 << 7) - 1;
  }

  ch->c = c;

  switch (gnsn) {
  default:
  case mb_G0SL:
    ch->gn = mb_G0;
    ch->sn = mb_SL;
    break;
  case mb_G1SL:
    ch->gn = mb_G1;
    ch->sn = mb_SL;
    break;
  case mb_G2SL:
    ch->gn = mb_G2;
    ch->sn = mb_SL;
    break;
  case mb_G3SL:
    ch->gn = mb_G3;
    ch->sn = mb_SL;
    break;
  case mb_G0SR:
    ch->gn = mb_G0;
    ch->sn = mb_SR;
    break;
  case mb_G1SR:
    ch->gn = mb_G1;
    ch->sn = mb_SR;
    break;
  case mb_G2SR:
    ch->gn = mb_G2;
    ch->sn = mb_SR;
    break;
  case mb_G3SR:
    ch->gn = mb_G3;
    ch->sn = mb_SR;
    break;
  case mb_G2SSL:
    ch->gn = mb_G2;
    ch->sn = mb_SSL;
    break;
  case mb_G3SSL:
    ch->gn = mb_G3;
    ch->sn = mb_SSL;
    break;
  case mb_G2SSR:
    ch->gn = mb_G2;
    ch->sn = mb_SSR;
    break;
  case mb_G3SSR:
    ch->gn = mb_G3;
    ch->sn = mb_SSR;
    break;
  }
}

static struct {
  char beg, end;
  mb_128decoder_t func;
} mb_128decoderv[] = {
#undef def_mb128
#define def_mb128(mac, fcb, fce, convg0sl, encoder, decoder, cwidth) {(fcb) & MB_ESC_FC_MASK, (fce) & MB_ESC_FC_MASK, decoder},
#include "mb128.h"
};

size_t
mb_store_parsed_badutf8(mb_char_t *ch, char *buf)
{
  int fc = ch->fc & MB_ESC_FC_MASK;
  int off;
  size_t e, tb, te;

  switch (ch->sn & MB_ESC_Sn_MASK) {
  case mb_SSL:
    off = 0x20;
    buf[0] = 0x1B;
    buf[1] = 0x4E + ((ch->gn - mb_G2) & 0x01);
    e = 2;
    break;
  case mb_SSR:
    off = 0xA0;
    buf[0] = 0x8E + ((ch->gn - mb_G2) & 0x01);
    e = 1;
    break;
  default:
    off = 0x20 | (((ch->sn - mb_SL) & 0x01) << 7);
    e = 0;
    break;
  }

  switch (ch->set & MB_ESC_SET_MASK) {
  case mb_94x94:
    buf[e++] = ch->c / 0x60 + off;
    buf[e++] = ch->c % 0x60 + off;
    break;
  case mb_96_0:
  case mb_96_1:
    buf[e++] = ch->c + off;
    break;
  case mb_94_0:
  case mb_94_1:
  case mb_94_2:
  case mb_94_3:
    buf[e++] = ch->c + off;
    break;
  default:
    switch (fc) {
    case 0x47 & MB_ESC_FC_MASK:
      e = mb_utf8_enc(ch->c, buf);
      break;
    case MB_UNKNOWN_FC & MB_ESC_FC_MASK:
      buf[0] = ch->c;
      e = 1;
      break;
    default:
      for (tb = 0, te = sizeof(mb_128decoderv) / sizeof(mb_128decoderv[0]) ; tb < te ;) {
	size_t i = (tb + te) / 2;

	if (fc < (unsigned char)mb_128decoderv[i].beg)
	  te = i;
	else if (fc > (unsigned char)mb_128decoderv[i].end)
	  tb = i + 1;
	else {
	  e = mb_128decoderv[i].func(ch->c, fc, buf);
	  break;
	}
      }

      break;
    }

    break;
  }

  return e;
}

void
mb_conv_to_xctext(mb_char_t *ch, mb_info_t *info)
{
  switch (ch->set) {
  case mb_128:
    break;
  case mb_94_0:
    switch (ch->fc & MB_ESC_FC_MASK) {
    case 0x49 & MB_ESC_FC_MASK:
      goto is_g1;
    default:
      break;
    }
  default:
    ch->gn = mb_G0;
    ch->sn = mb_SL;
    break;
  case mb_96_0:
  case mb_96_1:
  is_g1:
    ch->gn = mb_G1;
    ch->sn = mb_SR;
    break;
  }
}

void
mb_store_parsedchar_noconv(mb_char_t *ch, mb_info_t *info)
{
  char buf[MB_LEN_MAX];
  size_t e;

  if (info->GR == mb_FAKEUTF8)
    e = mb_nonutf8_enc(ch, buf);
  else {
    mb_store_esc(ch, info);
    e = mb_store_parsed_badutf8(ch, buf);
  }

  info->io_func.out(buf, e, info->io_arg);
}

void
mb_store_parsedchar(mb_char_t *ch, mb_info_t *info)
{
  mb_char_t copy;

  copy = *ch;
  mb_apply_convv(&copy, info);
  mb_store_parsedchar_noconv(&copy, info);
}

static void
mb_store_badutf8(int c, mb_info_t *info)
{
  mb_char_t ch;

  mb_parse_badutf8(c, &ch);
  mb_store_parsedchar(&ch, info);
}

size_t
mb_store_char_noconv(int c, mb_info_t *info)
{
  if (c == EOF) {
    size_t i;

    for (i = 0 ; i < mb_Gn ; ++i)
      if (info->G[i] != info->Gsave[i] && info->Gsave[i] != EOF) {
	mb_char_t ch;

	ch.set = MB_ESC_DEC_SET(info->Gsave[i]);
	ch.fc = MB_ESC_DEC_FC(info->Gsave[i]);
	ch.gn = i;
	ch.sn = i == info->GLsave ? mb_SL : i == info->GRsave ? mb_SR : mb_SSR;
	mb_store_esc(&ch, info);
      }

    return 0;
  }
  else {
    char buf[1];

    if (info->flag & MB_FLAG_ASCIIATCTL) {
      mb_char_t ch = {0, mb_94_0, 0x42, mb_G0, mb_SL};

      mb_store_esc(&ch, info);
    }

    buf[0] = c;
    return info->io_func.out(buf, 1, info->io_arg);
  }
}

static void
mb_store_sbc(int c, mb_info_t *info)
{
  if (c >= 0x21 && c <= 0x7E) {
    mb_char_t ch = {0, mb_94_0, 0x42, mb_G0, mb_SL};

    ch.c = c - 0x20;
    mb_store_parsedchar(&ch, info);
  }
  else
    mb_store_char_noconv(c, info);
}

size_t
mb_store_char(const char *s, size_t e, mb_info_t *info)
{
  size_t b = 0;
  int c;

  if ((c = mb_char_dec(s, &b, e)) != EOF) {
    if (b > 0)
      switch (b) {
      case 2:
	if (c < 0x80)
	  mb_store_badutf8(c, info);
	else
	  goto utf8;

	break;
      case 3:
	if (c < 0x800)
	  mb_store_badutf8(c + 0x80, info);
	else
	  goto utf8;

	break;
      case 4:
	if (c < 0x10000)
	  mb_store_badutf8(c + 0x880, info);
	else
	  goto utf8;

	break;
      case 5:
	if (c < 0x200000)
	  mb_store_badutf8(c + 0x10880, info);
	else
	  goto utf8;

	break;
	/* case 6: */
      default:
	if (c < 0x4000000) {
	  mb_store_badutf8(c + 0x210880, info);
	  break;
	}
      utf8:
	{
	  mb_char_t ch = {0, mb_128, 0x47, mb_G1, mb_SR};

	  ch.c = c;
	  mb_store_parsedchar(&ch, info);
	}

	break;
      }
    else if (c >= -0xFD && c <= -0xC0)
      b = 0;
    else {
      mb_store_sbc(c, info);
      b = 1;
    }
  }
  else
    mb_store_char_noconv(c, info);

  return b;
}

int
mb_putc(int c, mb_info_t *info)
{
  c &= 0xFF;

  if (info->in_i > 0) {
    if ((c & ~0x3F) == 0x80) {
      info->inbuf[(info->in_i)++] = c;

      if (!(info->in_i < info->in_n)) {
	mb_store_char(info->inbuf, info->in_i, info);
	info->in_i = info->in_n = 0;
      }

      goto end;
    }

    info->io_func.out(info->inbuf, info->in_i, info->io_arg);
    info->in_i = info->in_n = 0;
  }

  if (c >= 0xC0 && c <= 0xFD) {
    int mask;
    size_t i;

    for (i = 2, mask = 0x20 ; mask & c ; mask >>= 1, ++i)
      ;

    info->inbuf[0] = c;
    info->in_i = 1;
    info->in_n = i;
  }
  else
    mb_store_sbc(c, info);

end:
  return c;
}

size_t
mb_putmem(const char *s, size_t n, mb_info_t *info)
{
  size_t i, csize;

  for (i = 0 ; i < n && info->in_i > 0 ;)
    mb_putc((unsigned char)s[i++], info);

  while (i < n)
    if ((csize = mb_store_char(&s[i], n - i, info)) > 0)
      i += csize;
    else {
      do {
	mb_putc((unsigned char)s[i++], info);
      } while (i < n);

      break;
    }

  return i;
}

size_t
mb_putstr(const char *s, mb_info_t *info)
{
  return mb_putmem(s, strlen(s), info);
}

void
mb_flush(mb_info_t *info)
{
  if (info->in_i > 0)
    info->io_func.out(info->inbuf, info->in_i, info->io_arg);

  info->in_i = info->in_n = 0;
}

struct mb_mem_st {char *d; size_t end, size;};

static int
mb_mem_expand(struct mb_mem_st *mem, size_t n)
{
  if (mem->end + n > mem->size) {
    size_t nsize = ((mem->end + n) / MB_MEM_DELTA + 1) * MB_MEM_DELTA;
    char *nd = realloc(mem->d, nsize);

    if (!nd)
      return 1;

    mem->d = nd;
    mem->size = nsize;
  }

  return 0;
}

static size_t
mb_mem_write(const char *s, size_t n, void *ap)
{
  struct mb_mem_st *mem = ap;

  if (mb_mem_expand(mem, n))
    return 0;

  memcpy(&mem->d[mem->end], s, n);
  mem->end += n;
  return n;
}

char *
mb_vmem2iso(const char *s, size_t *p_n, mb_setup_t *setup, const char *op, va_list ap)
{
  struct mb_mem_st mem = {};
  mb_info_t info;
  size_t n;

  if ((mem.d = malloc(n = *p_n))) {
    mem.size = n;
    mb_vinit_w(&info, &mem, mb_mem_write, setup, op, ap);
    mb_putmem(s, n, &info);
    mb_flush(&info);
    *p_n = mem.end;
    return mem.d;
  }
  else
    return NULL;
}

char *
mb_mem2iso(const char *s, size_t *p_n, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *iso;

  va_start(ap, op);
  iso = mb_vmem2iso(s, p_n, setup, op, ap);
  va_end(ap);
  return iso;
}

char *
mb_vstr2iso(const char *s, mb_setup_t *setup, const char *op, va_list ap)
{
  size_t n = strlen(s) + 1;

  return mb_vmem2iso(s, &n, setup, op, ap);
}

char *
mb_str2iso(const char *s, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *iso;

  va_start(ap, op);
  iso = mb_vstr2iso(s, setup, op, ap);
  va_end(ap);
  return iso;
}

#define MB_PRINTF_FLAG_MINUS (1 << 0)
#define MB_PRINTF_FLAG_PLUS (1 << 1)
#define MB_PRINTF_FLAG_SPACE (1 << 2)
#define MB_PRINTF_FLAG_SHARP (1 << 3)
#define MB_PRINTF_FLAG_ZERO (1 << 4)

int
mb_vprintf(mb_info_t *info, const char *format, va_list ap)
{
  const char *b, *f, *s;
  char *e;
  char tf[sizeof("%-+ #0.ld") + sizeof(int) * CHAR_BIT / 3 + 1] = "%";
  size_t fe, fs;
  char buf[3 + sizeof(double) * CHAR_BIT * 2 / 3 + 2 + sizeof(long) * CHAR_BIT / 3 + sizeof(void *) * CHAR_BIT / 3];
  int n, flag, width, precision, tn, i, j;
  mb_info_t copy = *info;
  struct mb_mem_st mem = {};

  copy.io_func.out = mb_mem_write;
  copy.io_arg = &mem;

  for (n = 0, b = format ; (f = strchr(b, '%')) ;) {
    if (b < f)
      n += mb_putmem(b, f - b, info);

    b = f++;
    flag = width = 0;
    fe = 1;

    for (;; ++f)
      switch ((unsigned char)*f) {
      case '-':
	flag |= MB_PRINTF_FLAG_MINUS;
	break;
      case '+':
	flag |= MB_PRINTF_FLAG_PLUS;
	break;
      case ' ':
	flag |= MB_PRINTF_FLAG_SPACE;
	break;
      case '#':
	flag |= MB_PRINTF_FLAG_SHARP;
	break;
      case '0':
	flag |= MB_PRINTF_FLAG_ZERO;
	break;
      default:
	goto for_width;
      }

  for_width:
    if ((unsigned char)*f == '*') {
      ++f;
      width = va_arg(ap, int);
    }
    else {
      width = strtol(f, &e, 10);

      if (e > f)
	f = e;
    }

    if (width < 0) {
      flag |= MB_PRINTF_FLAG_MINUS;
      width = -width;
    }

    if (flag & MB_PRINTF_FLAG_MINUS)
      tf[fe++] = '-';

    if (flag & MB_PRINTF_FLAG_PLUS)
      tf[fe++] = '+';

    if (flag & MB_PRINTF_FLAG_SPACE)
      tf[fe++] = ' ';

    if (flag & MB_PRINTF_FLAG_SHARP)
      tf[fe++] = '#';

    if (flag & MB_PRINTF_FLAG_ZERO)
      tf[fe++] = '0';

    precision = -1;

    if ((unsigned char)*f == '.') {
      tf[fe++] = '.';

      if ((unsigned char)*++f == '*') {
	precision = va_arg(ap, int);
	++f;
      }
      else {
	precision = strtol(f, &e, 10);
	f = e;
      }

      fe += sprintf(&tf[fe], "%d", precision);
    }

    switch ((unsigned char)f[fs = strcspn(f, "cdeEfgGinopsuxX%")]) {
    case 'c':
      if (fs)
	goto error;

      tf[fe++] = 'c';
      tf[fe++] = '\0';
      tn = sprintf(buf, tf, va_arg(ap, int));
      break;
    case 'd':
    case 'i':
    case 'o':
    case 'u':
    case 'x':
    case 'X':
      switch (fs) {
      case 1:
	if ((unsigned char)f[0] == 'h')
	  tf[fe++] = 'h';
	else if ((unsigned char)f[0] == 'l') {
	  tf[fe++] = 'l';
	  tf[fe++] = f[1];
	  tf[fe++] = '\0';
	  tn = sprintf(buf, tf, va_arg(ap, long));
	  break;
	}
	else
	  goto error;
      case 0:
	tf[fe++] = f[fs];
	tf[fe++] = '\0';
	tn = sprintf(buf, tf, va_arg(ap, int));
	break;
      default:
	goto error;
      }

      break;
    case 'e':
      switch (fs) {
      case 1:
	if (f[0] == 'l') {
	  tf[fe++] = 'l';
	  tf[fe++] = f[1];
	  tf[fe++] = '\0';
	  tn = sprintf(buf, tf, va_arg(ap, long double));
	  break;
	}
	else
	  goto error;
      case 0:
	tf[fe++] = f[0];
	tf[fe++] = '\0';
	tn = sprintf(buf, tf, va_arg(ap, double));
	break;
      default:
	goto error;
      }

      break;
    case 'E':
    case 'f':
    case 'g':
    case 'G':
      switch (fs) {
      case 1:
	if (f[0] == 'L') {
	  tf[fe++] = 'L';
	  tf[fe++] = f[1];
	  tf[fe++] = '\0';
	  tn = sprintf(buf, tf, va_arg(ap, long double));
	  break;
	}
	else
	  goto error;
      case 0:
	tf[fe++] = f[0];
	tf[fe++] = '\0';
	tn = sprintf(buf, tf, va_arg(ap, double));
	break;
      default:
	goto error;
      }

      break;
    case 'n':
      switch (fs) {
      case 1:
	switch ((unsigned char)f[0]) {
	case 'h':
	  *(va_arg(ap, short *)) = n;
	  break;
	case 'l':
	  *(va_arg(ap, long *)) = n;
	  break;
	default:
	  goto error;
	}

	break;
      case 0:
	*(va_arg(ap, int *)) = n;
	break;
      default:
	goto error;
      }

      b = f + fs + 1;
      continue;
    case 'p':
      if (fs)
	goto error;

      tf[fe++] = 'p';
      tf[fe++] = '\0';
      tn = sprintf(buf, tf, va_arg(ap, void *));
      break;
    case 's':
      if (fs)
	goto error;

      memcpy(copy.G, info->G, sizeof(copy.G));
      copy.in_i = copy.in_n = copy.out_i = copy.out_n = 0;
      mem.end = 0;

      for (s = va_arg(ap, char *) ; *s ;) {
	tn = mem.end;
	mb_putc((unsigned char)*s++, &copy);

	if (precision >= 0 && mem.end > precision) {
	  mem.end = tn;
	  break;
	}
      }

      memcpy(info->G, copy.G, sizeof(info->G));
      mb_flush(info);

      if (mem.d) {
	if (precision < 0 || precision > mem.end)
	  precision = mem.end;

	if (flag & MB_PRINTF_FLAG_MINUS) {
	  n += info->io_func.out(mem.d, precision, info->io_arg);

	  for (; width > precision ; --width)
	    n += info->io_func.out(" ", 1, info->io_arg);
	}
	else {
	  for (; width > precision ; --width)
	    n += info->io_func.out(" ", 1, info->io_arg);

	  n += info->io_func.out(mem.d, precision, info->io_arg);
	}
      }
      else if (width > 0)
	for (; width > 0 ; --width)
	  n += mb_putmem(" ", 1, info);

      b = f + 1;
      continue;
    case '%':
      if (fs)
	goto error;

      buf[0] = '%';
      buf[1] = '\0';
      tn = 1;
      break;
    default:
    error:
      n += mb_putmem(b++, 1, info);
      continue;
    }

    b = f + fs + 1;

    if (width > tn)
      if (flag & MB_PRINTF_FLAG_MINUS) {
	n += mb_putmem(buf, tn, info);

	for (; tn < width ; ++tn)
	  n += mb_putmem(" ", 1, info);
      }
      else if (flag & MB_PRINTF_FLAG_ZERO) {
	for (i = 0 ; i < tn && strchr("-+0 ", (unsigned char)buf[i]) ; ++i)
	  ;

	if (i < tn && strchr("xX", (unsigned char)buf[i]))
	  ++i;

	n += mb_putmem(buf, i, info);

	for (j = tn ; j < width ; ++j)
	  n += mb_putmem("0", 1, info);

	n += mb_putmem(&buf[i], tn - i, info);
      }
      else {
	for (j = tn ; j < width ; ++j)
	  n += mb_putmem(" ", 1, info);

	n += mb_putmem(buf, tn, info);
      }
    else
      n += mb_putmem(buf, tn, info);
  }

  if (*b)
    n += mb_putstr(b, info);

  if (mem.d)
    free(mem.d);

  return n;
}

int
mb_printf(mb_info_t *info, const char *format, ...)
{
  va_list ap;
  int n;

  va_start(ap, format);
  n = mb_vprintf(info, format, ap);
  va_end(ap);
  return n;
}

static const char mb_b64_alpha[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

static void
mb_b64_flush_encode(mb_b64_t *b64)
{
  switch (b64->se) {
  case 1:
    b64->d[0] = mb_b64_alpha[(int)(unsigned char)b64->d[0]];
    b64->d[1] = mb_b64_alpha[(int)(unsigned char)b64->d[1]];
    b64->d[2] = b64->d[3] = '=';
    goto flush;
  case 2:
    b64->d[0] = mb_b64_alpha[(int)(unsigned char)b64->d[0]];
    b64->d[1] = mb_b64_alpha[(int)(unsigned char)b64->d[1]];
    b64->d[2] = mb_b64_alpha[(int)(unsigned char)b64->d[2]];
    b64->d[3] = '=';
    goto flush;
  case 3:
    b64->d[0] = mb_b64_alpha[(int)(unsigned char)b64->d[0]];
    b64->d[1] = mb_b64_alpha[(int)(unsigned char)b64->d[1]];
    b64->d[2] = mb_b64_alpha[(int)(unsigned char)b64->d[2]];
    b64->d[3] = mb_b64_alpha[(int)(unsigned char)b64->d[3]];
  flush:
    if (!(b64->dn)) {
      if (b64->opt->pre)
	b64->orig_writer(b64->opt->pre, b64->opt->pree, b64->orig_arg);
    }
    else if (b64->opt->llen > 0 && !(b64->dn % b64->opt->llen)) {
      b64->orig_writer(b64->opt->eol, b64->opt->eole, b64->orig_arg);
      b64->dn = 0;
    }

    b64->orig_writer(b64->d, 4, b64->orig_arg);
    b64->d[0] = b64->d[1] = b64->d[2] = b64->d[3] = b64->se = 0;
    b64->dn += 4;
  default:
    break;
  }
}

static size_t
mb_b64_write_encode(const char *s, size_t se, void *ap)
{
  mb_b64_t *b64 = ap;
  size_t sb, n;

  for (n = sb = 0 ; sb < se ; ++sb) {
    ++n;

    switch (b64->se) {
    case 0:
      b64->d[0] = ((unsigned char)s[sb] & 0xFC) >> 2;
      b64->d[1] = ((unsigned char)s[sb] & 0x03) << 4;
      break;
    case 1:
      b64->d[1] |= ((unsigned char)s[sb] & 0xF0) >> 4;
      b64->d[2] = ((unsigned char)s[sb] & 0x0F) << 2;
      break;
    default:
      b64->d[2] |= ((unsigned char)s[sb] & 0xC0) >> 6;
      b64->d[3] = (unsigned char)s[sb] & 0x3F;
      ++(b64->se);
      mb_b64_flush_encode(b64);
      continue;
    }

    ++(b64->se);
  }

  return n;
}

static size_t
mb_b64_write_through(const char *s, size_t se, void *ap)
{
  mb_b64_t *b64 = ap;

  return b64->orig_writer(s, se, b64->orig_arg);
}

static size_t
mb_putmem_b64encode_flush(const char *s, size_t b, size_t e, mb_b64_t *b64, mb_info_t *info)
{
  size_t n = 0;

  if (b < e) {
    n += mb_putmem(&s[b], e - b, info);

    if (info->io_func.out == mb_b64_write_encode) {
      mb_flush(info);
      n += mb_store_char_noconv(EOF, info);
      mb_b64_flush_encode(b64);

      if (b64->opt->post)
	n += b64->orig_writer(b64->opt->post, b64->opt->poste, b64->orig_arg);

      b64->dn = 0;
    }
  }

  return n;
}

size_t
mb_putmem_b64encode(const char *s, size_t se, mb_b64opt_t *b64opt, mb_info_t *info)
{
  size_t wb, sb, cb, ce, n;
  int c;
  mb_b64_t b64 = {};

  mb_flush(info);
  b64.opt = b64opt;
  b64.se = b64.dn = 0;
  b64.orig_writer = info->io_func.out;
  b64.orig_arg = info->io_arg;
  info->io_func.out = mb_b64_write_through;
  info->io_arg = &b64;

  for (wb = sb = n = 0 ; sb < se ;) {
    cb = 0;
    ce = se - sb;
    c = mb_find_char(&s[sb], &cb, &ce);

    if (ce > 1 || (c < 0x21 && c > 0x7E))
      info->io_func.out = mb_b64_write_encode;
    else if (strchr("\x09\x0A\x0D\x20", c)) {
      n += mb_putmem_b64encode_flush(s, wb, sb, &b64, info);
      info->io_func.out = mb_b64_write_through;
      mb_putc(c, info);
      wb = sb + ce;
    }

    sb += ce;
  }

  mb_flush(info);
  n += mb_putmem_b64encode_flush(s, wb, se, &b64, info);
  info->io_func.out = b64.orig_writer;
  info->io_arg = b64.orig_arg;
  return n;
}

char *
mb_vmem2b64(const char *s, size_t *p_n, const char *title, mb_setup_t *setup, const char *op, va_list ap)
{
  struct mb_mem_st mem = {};
  size_t n;

  if ((mem.d = malloc(n = *p_n))) {
    mb_info_t info;
    size_t cslen = strlen(title);
    char *pre;

    mem.size = n;
    mb_vinit_w(&info, &mem, mb_mem_write, setup, op, ap);

    if ((pre = malloc(cslen + sizeof("=??b?")))) {
      mb_b64opt_t b64opt;

      memcpy(pre, "=?", sizeof("=?") - 1);
      memcpy(&pre[sizeof("=?") - 1], title, cslen);
      memcpy(&pre[sizeof("=?") - 1 + cslen], "?b?", sizeof("?b?"));
      b64opt.pre = pre;
      b64opt.pree = cslen + sizeof("=??b?") - 1;
      b64opt.post = "?=";
      b64opt.poste = sizeof("?=") - 1;
      b64opt.llen = 76;
      b64opt.eol = "\n\t";
      b64opt.eole = sizeof("\n\t") - 1;
      *p_n = mb_putmem_b64encode(s, n, &b64opt, &info);
    }
    else {
      free(mem.d);
      mem.d = NULL;
    }
  }

  return mem.d;
}

char *
mb_mem2b64(const char *s, size_t *p_n, const char *title, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *b64;

  va_start(ap, op);
  b64 = mb_vmem2b64(s, p_n, title, setup, op, ap);
  va_end(ap);
  return b64;
}

char *
mb_vstr2b64(const char *s, const char *title, mb_setup_t *setup, const char *op, va_list ap)
{
  size_t n = strlen(s) + 1;

  return mb_vmem2b64(s, &n, title, setup, op, ap);
}

char *
mb_str2b64(const char *s, const char *title, mb_setup_t *setup, const char *op, ...)
{
  va_list ap;
  char *b64;

  va_start(ap, op);
  b64 = mb_vstr2b64(s, title, setup, op, ap);
  va_end(ap);
  return b64;
}
