#include <ctype.h>
#include <stddef.h>
#include <string.h>
#include "mb.h"
#ifdef USE_CN
#include "cn.h"
#endif
#ifdef USE_JA
#include "ja.h"
#endif
#ifdef USE_KR
#include "kr.h"
#endif
#ifdef USE_UCS
#include "ucs.h"
#endif
#include "mb128misc.h"

static int
mb_nc_memcmp(const char *a, size_t alen, const char *b, size_t blen)
{
  int cmp;

  if (alen < blen)
    cmp = -1;
  else if (alen > blen)
    cmp = 1;
  else {
    size_t i;

    for (cmp = 0, i = 0 ; i < alen ; ++i)
      if ((cmp = tolower((unsigned char)a[i]) - tolower((unsigned char)b[i])))
	break;
  }

  return cmp;
}

void *
mb_nc_bsearch(const char *key, size_t keylen, void *tab,
	      size_t key_off, size_t keylen_off, size_t elem_size, size_t tab_size)
{
  size_t b, e, i;
  char *elem;
  int cmp;

  for (b = 0, e = tab_size / elem_size ; b < e ;) {
    i = (b + e) / 2;
    elem = (char *)tab + elem_size * i;

    if (!(cmp = mb_nc_memcmp(key, keylen, *(const char **)(elem + key_off), *(size_t *)(elem + keylen_off))))
      return (void *)elem;
    else if (cmp < 0)
      e = i;
    else
      b = i + 1;
  }

  return NULL;
}

static void
mb_nc_swap(char *a, char *b, size_t size)
{
  char x;

  while (size > 0) {
    x = a[--size];
    a[size] = b[size];
    b[size] = x;
  }
}

static void
mb_nc_pushdown(void *tab, size_t key_off, size_t keylen_off, size_t elem_size, size_t b, size_t e)
{
  size_t l, r;

  while ((l = 2 * b + 1) < e) {
    char *paren = (char *)tab + elem_size * b;
    char *lchld = (char *)tab + elem_size * l;

    if ((r = 2 * b + 2) < e) {
      char *rchld = (char *)tab + elem_size * r;

      if (mb_nc_memcmp(*(char **)(paren + key_off), *(size_t *)(paren + keylen_off),
		       *(char **)(lchld + key_off), *(size_t *)(lchld + keylen_off)) >= 0)
	if (mb_nc_memcmp(*(char **)(paren + key_off), *(size_t *)(paren + keylen_off),
			 *(char **)(rchld + key_off), *(size_t *)(rchld + keylen_off)) >= 0)
	  break;
	else {
	  mb_nc_swap(paren, rchld, elem_size);
	  b = r;
	}
      else if (mb_nc_memcmp(*(char **)(lchld + key_off), *(size_t *)(lchld + keylen_off),
			    *(char **)(rchld + key_off), *(size_t *)(rchld + keylen_off)) >= 0) {
	mb_nc_swap(paren, lchld, elem_size);
	b = l;
      }
      else {
	mb_nc_swap(paren, rchld, elem_size);
	b = r;
      }
    }
    else if (mb_nc_memcmp(*(char **)(paren + key_off), *(size_t *)(paren + keylen_off),
			  *(char **)(lchld + key_off), *(size_t *)(lchld + keylen_off)) >= 0)
      break;
    else {
      mb_nc_swap(paren, lchld, elem_size);
      b = l;
    }
  }
}

void
mb_nc_sort(void *tab, size_t key_off, size_t keylen_off, size_t elem_size, size_t tab_size)
{
  size_t n, i;

  for (n = tab_size / elem_size, i = n / 2 ; i > 0 ;)
    mb_nc_pushdown(tab, key_off, keylen_off, elem_size, --i, n);

  for (i = n ; i > 0 ;) {
    --i;
    mb_nc_swap(tab, (char *)tab + elem_size * i, elem_size);
    mb_nc_pushdown(tab, key_off, keylen_off, elem_size, 0, i);
  }
}

static mb_conv_t to_utf8[] = {
#ifdef USE_UCS
  mb_conv_allg0sl,
  mb_conv_iso_to_ucs,
#endif
  NULL,
};

#ifdef USE_CN
static mb_conv_t to_big5[] = {
  mb_conv_allg0sl,
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_big5,
#endif
  mb_conv_cn_big5,
  NULL,
};
#endif

static mb_conv_t to_cn[] = {
  mb_conv_allg0sl,
#ifdef USE_CN
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_gb,
#endif
  mb_conv_big5_to_cns,
  mb_conv_iso2022cn,
#endif
  NULL,
};

#ifdef USE_CN
static mb_conv_t to_euctw[] = {
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_cn,
#endif
  mb_conv_big5_to_cns,
  mb_conv_euc_tw,
  NULL,
};
#endif

static mb_conv_t to_gb[] = {
  mb_conv_allg0sl,
#ifdef USE_CN
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_gb,
#endif
  mb_conv_cn_gb,
#endif
  NULL,
};

static mb_conv_t to_gb_isoir165[] = {
  mb_conv_allg0sl,
#ifdef USE_CN
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_gb_isoir165,
#endif
  mb_conv_cn_gb_isoir165,
#endif
  NULL,
};

static mb_conv_t to_eucjp[] = {
  mb_conv_allg0sl,
#ifdef USE_JA
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_ja,
#endif
  mb_conv_euc_jp,
#endif
  NULL,
};

static mb_conv_t to_iso2022jp[] = {
#ifdef USE_JA
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_ja,
#endif
#endif
  mb_conv_allg0sl,
  NULL,
};

#ifdef USE_JA
static mb_conv_t to_sjis[] = {
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_ja,
#endif
  mb_conv_sjis,
  NULL,
};
#endif

#ifdef USE_KR
static mb_conv_t to_uhang[] = {
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_uhang,
#endif
  mb_conv_uhang,
  NULL,
};
#endif

static mb_conv_t to_euckr[] = {
  mb_conv_allg0sl,
#ifdef USE_KR
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_kr,
#endif
  mb_conv_euc_kr,
#endif
  NULL,
};

static mb_conv_t to_iso2022kr[] = {
  mb_conv_allg0sl,
#ifdef USE_KR
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_kr,
#endif
  mb_conv_iso2022kr,
#endif
  NULL,
};

#ifdef USE_KR
static mb_conv_t to_johab[] = {
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_johab,
#endif
  NULL,
};
#endif

#if defined(USE_KOI8R) || defined(USE_KOI8U) || defined(USE_WIN125X)
static mb_conv_t to_mb128misc[] = {
#ifdef USE_UCS
  mb_conv_iso_to_ucs,
  mb_conv_ucs_to_misc,
#endif
  NULL,
};
#endif

static mb_conv_t to_xctext[] = {
  mb_conv_allg0sl,
  mb_conv_to_xctext,
  NULL,
};

static mb_cs2esc_t default_cs2esc[] = {
#ifdef USE_CN
  {MB_KEY_DEF("big5"), mb_flag_plus, 0, mb_G0, mb_BIG5,
   {{mb_94_0, 0x42}}, to_big5},
#endif
  {MB_KEY_DEF("cn-gb"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x41}}, to_gb},
  {MB_KEY_DEF("utf-8"), mb_flag_plus, 0, mb_G0, mb_UTF8,
   {{mb_94_0, 0x42}, {mb_96_0, 0x41}}, to_utf8},
  {MB_KEY_DEF("euc-jp"), mb_flag_plus, MB_FLAG_DONTPREFER_JISX0213, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x42}, {mb_94_0, 0x49}, {mb_94x94, 0x44}}, to_eucjp},
  {MB_KEY_DEF("euc-kr"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x43}}, to_euckr},
#ifdef USE_CN
  {MB_KEY_DEF("euc-tw"), mb_flag_plus, 0, mb_G0, mb_EUCTW,
   {{mb_94_0, 0x42}, {mb_94x94, 0x47}}, to_euctw},
#endif
  {MB_KEY_DEF("gb2312"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x41}}, to_gb},
#ifdef USE_KOI8R
  {MB_KEY_DEF("koi8-r"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_KOI8R << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
#endif
#ifdef USE_KOI8U
  {MB_KEY_DEF("koi8-u"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_KOI8U << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("x-sjis"), mb_flag_plus, MB_FLAG_DONTPREFER_JISX0213, mb_G0, mb_SJIS,
   {{mb_94_0, 0x42}}, to_sjis},
#endif
#ifdef USE_CN
  {MB_KEY_DEF("cn-big5"), mb_flag_plus, 0, mb_G0, mb_BIG5,
   {{mb_94_0, 0x42}}, to_big5},
#endif
  {MB_KEY_DEF("x-ctext"), mb_flag_plus, MB_FLAG_ASCIIATCTL, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x41}}, to_xctext},
#ifdef USE_KR
  {MB_KEY_DEF("x-johab"), mb_flag_plus, 0, mb_G0, mb_JOHAB,
   {{mb_94_0, 0x42}}, to_johab},
#endif
  {MB_KEY_DEF("x-euc-jp"), mb_flag_plus, MB_FLAG_DONTPREFER_JISX0213, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x42}, {mb_94_0, 0x49}, {mb_94x94, 0x44}}, to_eucjp},
#ifdef USE_CN
  {MB_KEY_DEF("x-euc-tw"), mb_flag_plus, 0, mb_G0, mb_EUCTW,
   {{mb_94_0, 0x42}, {mb_94x94, 0x47}}, to_euctw},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("shift_jis"), mb_flag_plus, MB_FLAG_DONTPREFER_JISX0213, mb_G0, mb_SJIS,
   {{mb_94_0, 0x42}}, to_sjis},
#endif
  {MB_KEY_DEF("iso-8859-1"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x41}}, to_xctext},
  {MB_KEY_DEF("iso-8859-2"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x42}}, to_xctext},
  {MB_KEY_DEF("iso-8859-3"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x43}}, to_xctext},
  {MB_KEY_DEF("iso-8859-4"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x44}}, to_xctext},
  {MB_KEY_DEF("iso-8859-5"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x4C}}, to_xctext},
  {MB_KEY_DEF("iso-8859-6"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x47}}, to_xctext},
  {MB_KEY_DEF("iso-8859-7"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x46}}, to_xctext},
  {MB_KEY_DEF("iso-8859-8"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x48}}, to_xctext},
  {MB_KEY_DEF("iso-8859-9"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x4D}}, to_xctext},
  {MB_KEY_DEF("iso-2022-cn"), mb_flag_plus, MB_FLAG_ASCIIATCTL, mb_G0, mb_Gn,
   {{mb_94_0, 0x42}}, to_cn},
  {MB_KEY_DEF("iso-2022-jp"), mb_flag_plus, MB_FLAG_ASCIIATCTL | MB_FLAG_DONTPREFER_JISX0213, mb_G0, mb_Gn,
   {{mb_94_0, 0x42}}, to_iso2022jp},
  {MB_KEY_DEF("iso-2022-kr"), mb_flag_plus, MB_FLAG_ASCIIATCTL, mb_G0, mb_Gn,
   {{mb_94_0, 0x42}}, to_iso2022kr},
  {MB_KEY_DEF("iso-8859-10"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_96_0, 0x56}}, to_xctext},
  {MB_KEY_DEF("euc-jisx0213"), mb_flag_plus, MB_FLAG_PREFER_JISX0213, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x4F}, {mb_94_0, 0x49}, {mb_94x94, 0x50}}, to_eucjp},
#ifdef USE_WIN125X
  {MB_KEY_DEF("windows-1250"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1250 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1251"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1251 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1252"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1252 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1253"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1253 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1254"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1254 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1255"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1255 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1256"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1256 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1257"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1257 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
  {MB_KEY_DEF("windows-1258"), mb_flag_plus, 0, mb_G0, mb_MISC | (mb128misc_WIN1258 << MB_ESC_LEN),
   {{mb_94_0, 0x42}}, to_mb128misc},
#endif
  {MB_KEY_DEF("x-fake-utf-8"), mb_flag_plus, 0, mb_G0, mb_FAKEUTF8,
   {{mb_94_0, 0x42}}, NULL},
  {MB_KEY_DEF("iso-2022-jp-2"), mb_flag_plus, MB_FLAG_ASCIIATCTL | MB_FLAG_DONTPREFER_JISX0213, mb_G0, mb_Gn,
   {{mb_94_0, 0x42}}, to_iso2022jp},
  {MB_KEY_DEF("iso-2022-jp-3"), mb_flag_plus, MB_FLAG_ASCIIATCTL | MB_FLAG_PREFER_JISX0213, mb_G0, mb_Gn,
   {{mb_94_0, 0x42}}, to_iso2022jp},
  {MB_KEY_DEF("cn-gb-isoir165"), mb_flag_plus, 0, mb_G0, mb_G1,
   {{mb_94_0, 0x42}, {mb_94x94, 0x45}}, to_gb_isoir165},
#ifdef USE_JA
  {MB_KEY_DEF("shift_jisx0213"), mb_flag_plus, MB_FLAG_PREFER_JISX0213, mb_G0, mb_SJIS,
   {{mb_94_0, 0x42}}, to_sjis},
#endif
  {MB_KEY_DEF("iso-2022-cn-ext"), mb_flag_plus, MB_FLAG_ASCIIATCTL, mb_G0, mb_Gn,
   {{mb_94_0, 0x42}}, to_cn},
#ifdef USE_KR
  {MB_KEY_DEF("x-unified-hangul"), mb_flag_plus, 0, mb_G0, mb_UHANG,
   {{mb_94_0, 0x42}, {mb_94x94, 0x43}}, to_uhang},
#endif
};

static mb_cs2esc_t *cs2esc = default_cs2esc;
static size_t cs2esc_size = sizeof(default_cs2esc);

size_t
mb_charset_tab_merge(mb_cs2esc_t *tab, size_t tab_end, size_t tab_size)
{
  size_t nend = tab_end + sizeof(default_cs2esc);

  if (nend > tab_size)
    return nend;

  memcpy((char *)tab + tab_end, default_cs2esc, sizeof(default_cs2esc));
  mb_nc_sort(tab, offsetof(mb_cs2esc_t, cs), offsetof(mb_cs2esc_t, cs_len), sizeof(mb_cs2esc_t), nend);
  cs2esc = tab;
  return cs2esc_size = nend;
}

size_t
mb_charset_cname_merge(mb_cs2esc_t *tab, const char **cnamev, size_t cnamec)
{
  size_t ncname, i;
  const char *s;

  for (ncname = i = 0 ; i < cnamec ; ++i)
    if ((s = strchr(cnamev[i], '='))) {
      mb_cs2esc_t *p = mb_nc_bsearch(cnamev[i], s - cnamev[i], cs2esc,
				     offsetof(mb_cs2esc_t, cs), offsetof(mb_cs2esc_t, cs_len),
				     sizeof(mb_cs2esc_t), cs2esc_size);

      if (p) {
	size_t j;

	for (++s ;; s += j + 1) {
	  if ((j = strcspn(s, ",")) > 0) {
	    if (tab) {
	      tab[ncname] = *p;
	      tab[ncname].cs = s;
	      tab[ncname].cs_len = j;
	    }

	    ++ncname;
	  }

	  if (!s[j]) break;
	}
      }
    }

  if (tab) {
    size_t nend = sizeof(mb_cs2esc_t) * ncname + sizeof(default_cs2esc);

    memcpy(&tab[ncname], default_cs2esc, sizeof(default_cs2esc));
    mb_nc_sort(tab, offsetof(mb_cs2esc_t, cs), offsetof(mb_cs2esc_t, cs_len), sizeof(mb_cs2esc_t), nend);
    cs2esc = tab;
    return cs2esc_size = nend;
  }
  else
    return sizeof(mb_cs2esc_t) * ncname + sizeof(default_cs2esc);
}

void
mb_charset_to_esc(const char *cs, mb_info_t *info)
{
  info->GL = info->GLsave = mb_G0;
  info->GR = info->GRsave = mb_Gn;
  info->G[mb_G0] = info->Gsave[mb_G0] = MB_ESC_ASCII;
  info->G[mb_G1] = info->G[mb_G2] = info->G[mb_G3]
    = info->Gsave[mb_G1] = info->Gsave[mb_G2] = info->Gsave[mb_G3]
    = info->GRB4UTF8 = EOF;
  info->cs2esc = NULL;
  info->flag |= MB_FLAG_UNKNOWNCS;

  if (cs && *cs) {
    mb_cs2esc_t *p = mb_nc_bsearch(cs, strlen(cs), cs2esc,
				   offsetof(mb_cs2esc_t, cs), offsetof(mb_cs2esc_t, cs_len),
				   sizeof(mb_cs2esc_t), cs2esc_size);

    if (p) {
      size_t j;

      switch (p->flag_op) {
      case mb_flag_set:
	info->flag = p->flag;
	break;
      case mb_flag_minus:
	info->flag &= ~(p->flag);
	break;
      default:
	info->flag |= p->flag;
      }

      info->GL = info->GLsave = p->GL;
      info->GR = info->GRsave = p->GR;

      for (j = mb_G0 ; j <= mb_G3 ; ++j)
	if (p->G[j].fc)
	  info->G[j] = info->Gsave[j] = MB_ESC_ENC(p->G[j].set, p->G[j].fc);

      info->convv = p->convv;
      info->cs2esc = p;
      info->flag &= ~MB_FLAG_UNKNOWNCS;
    }
  }
}

void
mb_vsetsetup(mb_setup_t *setup, const char *op, va_list ap)
{
  while (*op)
    switch ((unsigned char)*op++) {
    case '=':
      setup->flag_op = mb_flag_set;
      setup->flag = va_arg(ap, int);
      break;
    case '|':
      setup->flag_op = mb_flag_plus;
      setup->flag = va_arg(ap, int);
      break;
    case '-':
      setup->flag_op = mb_flag_minus;
      setup->flag = va_arg(ap, int);
      break;
    case '@':
      setup->cs = va_arg(ap, const char *);
      break;
    case '&':
      setup->convv = va_arg(ap, mb_conv_t *);
      break;
    case '!':
      *setup = *(va_arg(ap, mb_setup_t *));
    default:
      break;
    }
}

void
mb_setsetup(mb_setup_t *setup, const char *op, ...)
{
  va_list ap;

  va_start(ap, op);
  mb_vsetsetup(setup, op, ap);
  va_end(ap);
}

void
mb_vsetup(mb_info_t *info, mb_setup_t *dflt, const char *op, va_list ap)
{
  mb_setup_t setup = {};

  if (dflt)
    setup = *dflt;

  mb_vsetsetup(&setup, op, ap);

  if (setup.cs)
    mb_charset_to_esc(setup.cs, info);

  switch (setup.flag_op) {
  case mb_flag_set:
    info->flag = setup.flag;
    break;
  case mb_flag_plus:
    info->flag |= setup.flag;
    break;
  case mb_flag_minus:
    info->flag &= setup.flag;
  default:
    break;
  }

  if (setup.convv)
    info->convv = setup.convv;
}

void
mb_setup(mb_info_t *info, mb_setup_t *dflt, const char *op, ...)
{
  va_list ap;

  va_start(ap, op);
  mb_vsetup(info, dflt, op, ap);
  va_end(ap);
}

void
mb_vinit(mb_info_t *info, void *arg, mb_setup_t *dflt, const char *op, va_list ap)
{
  info->flag = 0;
  info->convv = NULL;
  mb_charset_to_esc(NULL, info);
  info->io_arg = arg;
  info->in_i = info->in_n = info->out_i = info->out_n = 0;
  mb_vsetup(info, dflt, op, ap);
}

void
mb_init(mb_info_t *info, void *arg, mb_setup_t *dflt, const char *op, ...)
{
  va_list ap;

  va_start(ap, op);
  mb_vinit(info, arg, dflt, op, ap);
  va_end(ap);
}

void
mb_vinit_r(mb_info_t *info, void *arg, int (*func)(void *), mb_setup_t *dflt, const char *op, va_list ap)
{
  static mb_conv_t noconv[] = {NULL};
  mb_setup_t copy = *dflt;

  if (!copy.convv)
    copy.convv = noconv;

  mb_vinit(info, arg, &copy, op, ap);
  info->io_func.in = func;
}

void
mb_init_r(mb_info_t *info, void *arg, int (*func)(void *), mb_setup_t *dflt, const char *op, ...)
{
  va_list ap;

  va_start(ap, op);
  mb_vinit_r(info, arg, func, dflt, op, ap);
  va_end(ap);
}

void
mb_vinit_w(mb_info_t *info, void *arg, size_t (*func)(const char *, size_t, void *), mb_setup_t *dflt, const char *op, va_list ap)
{
  mb_vinit(info, arg, dflt, op, ap);
  info->io_func.out = func;
}

void
mb_init_w(mb_info_t *info, void *arg, size_t (*func)(const char *, size_t, void *), mb_setup_t *dflt, const char *op, ...)
{
  va_list ap;

  va_start(ap, op);
  mb_vinit_w(info, arg, func, dflt, op, ap);
  va_end(ap);
}

static struct conv_tab_st {
  const char *name;
  size_t len;
  mb_conv_t value;
} conv_tab[] = {
#ifdef USE_CN
  {MB_KEY_DEF("b"), mb_conv_cn_big5},
#endif
#ifdef USE_CN
  {MB_KEY_DEF("c"), mb_conv_iso2022cn},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("j"), mb_conv_allg0sl},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("k"), mb_conv_iso2022kr},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("s"), mb_conv_sjis},
#endif
  {MB_KEY_DEF("a0"), mb_conv_allg0sl},
#ifdef USE_CN
  {MB_KEY_DEF("b2c"), mb_conv_big5_to_cns},
#endif
#ifdef USE_UCS
  {MB_KEY_DEF("i2u"), mb_conv_iso_to_ucs},
#ifdef USE_CN
  {MB_KEY_DEF("u2b"), mb_conv_ucs_to_big5},
  {MB_KEY_DEF("u2c"), mb_conv_ucs_to_cn},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("u2j"), mb_conv_ucs_to_ja},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("u2k"), mb_conv_ucs_to_kr},
#endif
  {MB_KEY_DEF("misc"), mb_conv_ucs_to_misc},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("sjis"), mb_conv_sjis},
#endif
#ifdef USE_UCS
#ifdef USE_CN
  {MB_KEY_DEF("u2gb"), mb_conv_ucs_to_gb},
#endif
#endif
  {MB_KEY_DEF("ascii"), mb_conv_ascii},
#ifdef USE_CN
  {MB_KEY_DEF("cn-gb"), mb_conv_cn_gb},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("euc-jp"), mb_conv_euc_jp},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("euc-kr"), mb_conv_euc_kr},
#endif
#ifdef USE_CN
  {MB_KEY_DEF("euc-tw"), mb_conv_euc_tw},
#endif
  {MB_KEY_DEF("charset"), mb_conv_for_charset},
#ifdef USE_CN
  {MB_KEY_DEF("cn-big5"), mb_conv_cn_big5},
#endif
#ifdef USE_UCS
  {MB_KEY_DEF("ms-latin1"), mb_conv_ms_latin1},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("shift_jis"), mb_conv_sjis},
#endif
#ifdef USE_UCS
#ifdef USE_CN
  {MB_KEY_DEF("ucs-to-cn"), mb_conv_ucs_to_cn},
  {MB_KEY_DEF("ucs-to-gb"), mb_conv_ucs_to_gb},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("ucs-to-ja"), mb_conv_ucs_to_ja},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("ucs-to-kr"), mb_conv_ucs_to_kr},
#endif
  {MB_KEY_DEF("iso-to-ucs"), mb_conv_iso_to_ucs},
#endif
#ifdef USE_CN
  {MB_KEY_DEF("big5-to-cns"), mb_conv_big5_to_cns},
#endif
#ifdef USE_UCS
#ifdef USE_CN
  {MB_KEY_DEF("ucs-to-big5"), mb_conv_ucs_to_big5},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("ucs-to-johab"), mb_conv_ucs_to_johab},
#endif
#endif
#ifdef USE_CN
  {MB_KEY_DEF("cn-gb-isoir165"), mb_conv_cn_gb_isoir165},
#endif
#ifdef USE_UCS
#ifdef USE_CN
  {MB_KEY_DEF("ucs-to-gb-isoir165"), mb_conv_ucs_to_gb_isoir165},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("ucs-to-unified-hangul"), mb_conv_ucs_to_uhang},
#endif
#endif
  {NULL, 0, NULL},
};

size_t
mb_namev_to_converterv(const char *s, mb_conv_t *v, size_t max, void (*ehook)(const char *, size_t))
{
  struct conv_tab_st *p;
  size_t i, j;

  for (i = 0 ; i + 1 < max && *s ; s += j + 1) {
    j = strcspn(s, ",");

    if ((p = mb_nc_bsearch(s, j, conv_tab,
			   offsetof(struct conv_tab_st, name), offsetof(struct conv_tab_st, len),
			   sizeof(conv_tab[0]), sizeof(conv_tab) - sizeof(conv_tab[0]))))
      v[i++] = p->value;
    else if (ehook)
      ehook(s, j);

    if (!s[j])
      break;
  }

  v[i] = NULL;
  return i;
}

static struct flag_tab_st {
  const char *name;
  size_t len;
  int value;
  int mask;
} flag_tab[] = {
  {MB_KEY_DEF("28"), MB_FLAG_28FOR94X94G0, ~0},
  {MB_KEY_DEF("ac"), MB_FLAG_ASCIIATCTL, ~0},
  {MB_KEY_DEF("uc"), MB_FLAG_UTF8_CHECK, ~0},
  {MB_KEY_DEF("nossl"), MB_FLAG_NOSSL, ~0},
  {MB_KEY_DEF("check-utf-8"), MB_FLAG_UTF8_CHECK, ~0},
  {MB_KEY_DEF("prefer-jisx0213"), MB_FLAG_PREFER_JISX0213, ~MB_FLAG_JISX_PREFERENCE},
  {MB_KEY_DEF("ascii-at-control"), MB_FLAG_ASCIIATCTL, ~0},
  {MB_KEY_DEF("dont-prefer-jisx0213"), MB_FLAG_DONTPREFER_JISX0213, ~MB_FLAG_JISX_PREFERENCE},
  {MB_KEY_DEF("use-0x28-for-94x94inG0"), MB_FLAG_28FOR94X94G0, ~0},
  {MB_KEY_DEF("ignore-7bit-single-shift"), MB_FLAG_NOSSL, ~0},
};

int
mb_namev_to_flag(const char *s, int flag, void (*ehook)(const char *, size_t))
{
  struct flag_tab_st *p;
  size_t j;

  for (; *s ; s += j + 1) {
    j = strcspn(s, ",");

    if ((p = mb_nc_bsearch(s, j, flag_tab,
			   offsetof(struct flag_tab_st, name), offsetof(struct flag_tab_st, len),
			   sizeof(flag_tab[0]), sizeof(flag_tab)))) {
      flag &= p->mask;
      flag |= p->value;
    }
    else if (ehook)
      ehook(s, j);

    if (!s[j])
      break;
  }

  return flag;
}

struct {
  char *cs;
  mb_cs_judge_encoding_t func;
} cjk_detector[] = {
  {"iso-8859-1", mb_cs_judge_latin1},
#ifdef USE_CN
  {"cn-gb", mb_cs_judge_gb2312},
  {"cn-big5", mb_cs_judge_big5},
  {"x-euc-tw", mb_cs_judge_euc_tw},
#endif
#ifdef USE_JA
  {"euc-jp", mb_cs_judge_euc_jp},
  {"shift_jis", mb_cs_judge_shift_jis},
#endif
#ifdef USE_KR
  {"euc-kr", mb_cs_judge_euc_kr},
  {"x-johab", mb_cs_judge_johab},
  {"x-unified-hangul", mb_cs_judge_uhang},
#endif
  {"utf-8", mb_cs_judge_utf8},
};

#define CJK_NDETECTOR (sizeof(cjk_detector) / sizeof(cjk_detector[0]))

size_t
mb_cs_judge_cjk(mb_cs_detector_t *p)
{
  size_t i;

  for (i = 0 ; i < CJK_NDETECTOR ; ++i)
    cjk_detector[i].func(&p->stat[i], p->bag, p->end);

  return MB_CS_DETECT_CHOICEMAX;
}

char *
mb_cs_setup_cjk(mb_cs_detector_t *p, size_t i, size_t same)
{
  char *cs = p->private;

  if ((p->stat[i].by_encode || p->stat[i].by_char) && (!same || !cs))
    cs = cjk_detector[i].cs;

  mb_charset_to_esc(cs, p->orig);
  return cs;
}

static struct detector_tab_st {
  const char *name;
  size_t len;
  mb_cs_judge_t judge;
  mb_cs_setup_t setup;
  size_t nstats;
} detector_tab[] = {
#ifdef USE_CN
  {MB_KEY_DEF("c"), mb_cs_judge_cn, mb_cs_setup_cn, mb_cs_detect_cn_N},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("j"), mb_cs_judge_ja, mb_cs_setup_ja, mb_cs_detect_ja_N},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("k"), mb_cs_judge_kr, mb_cs_setup_kr, mb_cs_detect_kr_N},
#endif
#ifdef USE_CN
  {MB_KEY_DEF("cn"), mb_cs_judge_cn, mb_cs_setup_cn, mb_cs_detect_cn_N},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("ja"), mb_cs_judge_ja, mb_cs_setup_ja, mb_cs_detect_ja_N},
  {MB_KEY_DEF("jp"), mb_cs_judge_ja, mb_cs_setup_ja, mb_cs_detect_ja_N},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("ko"), mb_cs_judge_kr, mb_cs_setup_kr, mb_cs_detect_kr_N},
  {MB_KEY_DEF("kr"), mb_cs_judge_kr, mb_cs_setup_kr, mb_cs_detect_kr_N},
#endif
  {MB_KEY_DEF("cjk"), mb_cs_judge_cjk, mb_cs_setup_cjk, CJK_NDETECTOR},
#ifdef USE_CN
  {MB_KEY_DEF("china"), mb_cs_judge_cn, mb_cs_setup_cn, mb_cs_detect_cn_N},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("japan"), mb_cs_judge_ja, mb_cs_setup_ja, mb_cs_detect_ja_N},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("korea"), mb_cs_judge_kr, mb_cs_setup_kr, mb_cs_detect_kr_N},
#endif
#ifdef USE_KR
  {MB_KEY_DEF("korean"), mb_cs_judge_kr, mb_cs_setup_kr, mb_cs_detect_kr_N},
#endif
#ifdef USE_CN
  {MB_KEY_DEF("chinese"), mb_cs_judge_cn, mb_cs_setup_cn, mb_cs_detect_cn_N},
#endif
#ifdef USE_JA
  {MB_KEY_DEF("japanese"), mb_cs_judge_ja, mb_cs_setup_ja, mb_cs_detect_ja_N},
#endif
  {NULL, 0, NULL, NULL, 0},
};

int
mb_lang_to_detector(const char *s, mb_cs_judge_t *p_judge, mb_cs_setup_t *p_setup, size_t *p_nstats)
{
  struct detector_tab_st *p = mb_nc_bsearch(s, strlen(s), detector_tab,
					    offsetof(struct detector_tab_st, name), offsetof(struct detector_tab_st, len),
					    sizeof(detector_tab[0]), sizeof(detector_tab) - sizeof(detector_tab[0]));

  if (p) {
    *p_judge = p->judge;
    *p_setup = p->setup;
    *p_nstats = p->nstats;
    return 1;
  }
  else
    return 0;
}
