use strict;

my $mb_esc_set_len = 3;
my $mb_esc_set_mask = (1 << $mb_esc_set_len) - 1;
my $mb_esc_fc_len = 5;
my $mb_esc_fc_mask = (1 << $mb_esc_fc_len) - 1;

my $mb_set = 0;
my $mb_94x94 = $mb_set++;
my $mb_96_0 = $mb_set++;
my $mb_96_1 = $mb_set++;
my $mb_94_0 = $mb_set++;
my $mb_94_1 = $mb_set++;
my $mb_94_2 = $mb_set++;
my $mb_94_3 = $mb_set++;
my $mb_128 = $mb_set++;

my $mb_fc = 0x40;
my $mb_sjis_fc = $mb_fc++;
my $mb_sjis0213_fc = $mb_fc++;
my $mb_big5l_fc = $mb_fc++;
my $mb_big5r_fc = $mb_fc++;
my $mb_johabll_fc = $mb_fc++;
my $mb_johablr_fc = $mb_fc++;
my $mb_johabh_fc = $mb_fc++;

++$mb_fc; # for UTF-8

my $mb_euctwb_fc = $mb_fc++;

$mb_fc += 15; # for CNS 11643 plane 2 -- 16

my $mb_uhangll_fc = $mb_fc++;
my $mb_uhanglr_fc = $mb_fc++;
my $mb_uhangh_fc = $mb_fc++;

my $mb_fc_upper = 0x5F;
my $mb_unknown_fc = $mb_fc_upper--;
my $mb_misc_fc = $mb_fc_upper--;
my $misc_esc = &mb_esc_enc($mb_128, $mb_misc_fc);

my @mb_misc_to_ucs =
  (
   'koi8-r',
   [
    0x2500,0x2502,0x250c,0x2510,0x2514,0x2518,0x251c,0x2524,
    0x252c,0x2534,0x253c,0x2580,0x2584,0x2588,0x258c,0x2590,
    0x2591,0x2592,0x2593,0x2320,0x25a0,0x2219,0x221a,0x2248,
    0x2264,0x2265,0x00a0,0x2321,0x00b0,0x00b2,0x00b7,0x00f7,
    0x2550,0x2551,0x2552,0x0451,0x2553,0x2554,0x2555,0x2556,
    0x2557,0x2558,0x2559,0x255a,0x255b,0x255c,0x255d,0x255e,
    0x255f,0x2560,0x2561,0x0401,0x2562,0x2563,0x2564,0x2565,
    0x2566,0x2567,0x2568,0x2569,0x256a,0x256b,0x256c,0x00a9,
    0x044e,0x0430,0x0431,0x0446,0x0434,0x0435,0x0444,0x0433,
    0x0445,0x0438,0x0439,0x043a,0x043b,0x043c,0x043d,0x043e,
    0x043f,0x044f,0x0440,0x0441,0x0442,0x0443,0x0436,0x0432,
    0x044c,0x044b,0x0437,0x0448,0x044d,0x0449,0x0447,0x044a,
    0x042e,0x0410,0x0411,0x0426,0x0414,0x0415,0x0424,0x0413,
    0x0425,0x0418,0x0419,0x041a,0x041b,0x041c,0x041d,0x041e,
    0x041f,0x042f,0x0420,0x0421,0x0422,0x0423,0x0416,0x0412,
    0x042c,0x042b,0x0417,0x0428,0x042d,0x0429,0x0427,0x042a,
    ],
   'koi8-u',
   [
    0x2500,0x2502,0x250C,0x2510,0x2514,0x2518,0x251C,0x2524,
    0x252C,0x2534,0x253C,0x2580,0x2584,0x2588,0x258C,0x2590,
    0x2591,0x2592,0x2593,0x2320,0x25A0,0x2219,0x221A,0x2248,
    0x2264,0x2265,0x00A0,0x2321,0x00B0,0x00B2,0x00B7,0x00F7,
    0x2550,0x2551,0x2552,0x0451,0x0454,0x2554,0x0456,0x0457,
    0x2557,0x2558,0x2559,0x255A,0x255B,0x0491,0x255D,0x255E,
    0x255F,0x2560,0x2561,0x0401,0x0404,0x2563,0x0406,0x0407,
    0x2566,0x2567,0x2568,0x2569,0x256A,0x0490,0x256C,0x00A9,
    0x044E,0x0430,0x0431,0x0446,0x0434,0x0435,0x0444,0x0433,
    0x0445,0x0438,0x0439,0x043A,0x043B,0x043C,0x043D,0x043E,
    0x043F,0x044F,0x0440,0x0441,0x0442,0x0443,0x0436,0x0432,
    0x044C,0x044B,0x0437,0x0448,0x044D,0x0449,0x0447,0x044A,
    0x042E,0x0410,0x0411,0x0426,0x0414,0x0415,0x0424,0x0413,
    0x0425,0x0418,0x0419,0x041A,0x041B,0x041C,0x041D,0x041E,
    0x041F,0x042F,0x0420,0x0421,0x0422,0x0423,0x0416,0x0412,
    0x042C,0x042B,0x0417,0x0428,0x042D,0x0429,0x0427,0x042A,
    ],
   );

my $mb_misc_subfc = @mb_misc_to_ucs / 2;

my @mb_win125x_subfc =
  (
   $mb_misc_subfc++, # 1250
   $mb_misc_subfc++, # 1251
   $mb_misc_subfc++, # 1252
   $mb_misc_subfc++, # 1253
   $mb_misc_subfc++, # 1254
   $mb_misc_subfc++, # 1255
   $mb_misc_subfc++, # 1256
   $mb_misc_subfc++, # 1257
   $mb_misc_subfc++, # 1258
   );

my @supplement =
  (
   [0xFF3C, $mb_94x94, 0x42, 0x21, 0x40, 'jix0208'], # FULLWIDTH REVERSE SOLIDUS
   );

sub mb_esc_enc {($_[0] & $mb_esc_set_mask) | (($_[1] & $mb_esc_fc_mask) << $mb_esc_set_len);}
sub mb_word_enc {($_[0] & ((1 << 14) - 1)) | ($_[1] << 14);}
sub mb_word_enc_3 {&mb_word_enc($_[0], &mb_esc_enc($_[1], $_[2]));}

sub domestic_ascii {
# Based on <URL:http://kanji.zinbun.kyoto-u.ac.jp/~yasuoka/CJK.html>
  (
# mb_94_0
   &mb_esc_enc($mb_94_0, 0x40),
   &mb_esc_enc($mb_94_0, 0x41),
   &mb_esc_enc($mb_94_0, 0x42),
   &mb_esc_enc($mb_94_0, 0x43),
   &mb_esc_enc($mb_94_0, 0x47),
   &mb_esc_enc($mb_94_0, 0x48),
   &mb_esc_enc($mb_94_0, 0x4A),
   &mb_esc_enc($mb_94_0, 0x4B),
   &mb_esc_enc($mb_94_0, 0x4C),
   &mb_esc_enc($mb_94_0, 0x52),
   &mb_esc_enc($mb_94_0, 0x54),
   &mb_esc_enc($mb_94_0, 0x59),
   &mb_esc_enc($mb_94_0, 0x5A),

# mb_94_1
   &mb_esc_enc($mb_94_1, 0x60),
   &mb_esc_enc($mb_94_1, 0x61),
   &mb_esc_enc($mb_94_1, 0x66),
   &mb_esc_enc($mb_94_1, 0x67),
   &mb_esc_enc($mb_94_1, 0x68),
   &mb_esc_enc($mb_94_1, 0x69),
   &mb_esc_enc($mb_94_1, 0x6E),
   &mb_esc_enc($mb_94_1, 0x75),
   &mb_esc_enc($mb_94_1, 0x77),
   &mb_esc_enc($mb_94_1, 0x78),
   &mb_esc_enc($mb_94_1, 0x7A),

# mb_94_2
   &mb_esc_enc($mb_94_2, 0x41),
   &mb_esc_enc($mb_94_2, 0x42),
   );
}

sub dbc2c {
  my ($c1, $c2) = @_;

  defined($c2) ? (($c1 & 0x7F) - 0x20) * 0x60 + ($c2 & 0x7F) - 0x20 : ($c1 & 0x7F) - 0x20;
}

my (@map, @noascii_map, %jis1tab, %jis1map, %jis2tab, %jis2map);

sub set_map {
  my ($ucs, $set, $fc, $c1, $c2, $cmt) = @_;
  my $esc = &mb_esc_enc($fc & (1 << $mb_esc_fc_len) ? $set + 1 : $set, $fc);
  my $c = &dbc2c($c1, $c2);

  &do_set_map($ucs, $esc, $c, $cmt);

  if ($ucs >= 0xA0 &&
      !defined($c2) &&
      $set >= $mb_94_0 && $set <= $mb_94_3 && ($fc & $mb_esc_fc_mask) != (0x42 & $mb_esc_fc_mask) &&
      $c1 >= 0x21 && $c1 <= 0x7E) {
    my $key = sprintf('%X,%X', $esc, $c);

    $noascii_map[$ucs] = +{} if (ref($noascii_map[$ucs]) ne 'HASH');

    if (ref($noascii_map[$ucs]->{$key}) ne 'ARRAY') {
      $noascii_map[$ucs]->{$key} = [$esc, $c, $cmt];
    }
  }
}

sub do_set_map {
  my ($ucs, $esc, $c, $cmt) = @_;

  if ($ucs >= 0xA0) {
    my $key = sprintf("%X,%X", $esc, $c);

    $map[$ucs] = +{} if (ref($map[$ucs]) ne 'HASH');

    if (ref($map[$ucs]->{$key}) ne 'ARRAY') {
      $map[$ucs]->{$key} = [$esc, $c, $cmt];
    }
  }
}

my @iso8859fc = (0x41, 0x42, 0x43, 0x44, 0x4C, 0x47, 0x46, 0x48, 0x4D, 0x56);

sub sbc96 {
  my ($iso, $ucs) = map{hex($_)} split(/\s+/, $_[0]);

  $iso >= 0xA0 && &set_map($ucs, $mb_96_0, $_[1], $iso, undef, @_[2 .. $#_]);
}

sub dbc {
  my ($ucs, $gb, $big5, $cns, $jis, $ksc) = split(/\s+/, $_[0]);

  $ucs = hex($ucs);
  $gb =~ /^0-(\d\d)(\d\d)$/ && &set_map($ucs, $mb_94x94, 0x41, $1 + 0x20, $2 + 0x20, 'gb2312');

  if ($big5 =~ /^([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})$/) {
    my ($c1, $c2) = (hex($1), hex($2));

    &set_map($ucs, $mb_128, $c2 < 0x7F ? $mb_big5l_fc : $mb_big5r_fc, $c1, $c2, 'big5');
  }

  $cns =~ /^([12])-([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})$/ &&
    &set_map($ucs, $mb_94x94, 0x47 + $1 - 1, hex($2), hex($3), "cns11643-$1");
  $jis =~ /^([01])-(\d{2})(\d{2})$/ &&
    &set_map($ucs, $mb_94x94, $1 eq '0' ? 0x42 : 0x44, $2 + 0x20, $3 + 0x20, 'jisx02' . ($1 eq '0' ? '08' : '12'));
  $ksc =~ /^0-(\d{2})(\d{2})$/ && &set_map($ucs, $mb_94x94, 0x43, $1 + 0x20, $2 + 0x20, 'ksx1001');
}

sub uni2cns {
  my ($ucs, @cns) = grep(!/^\#/, split(/\s+/, $_[0]));
  my %plane;

  $ucs = hex($ucs);

  foreach (grep(/^[1-7]-/, @cns)) {
    my ($p, $c) = split(/-/, $_, 2);

    unless ($plane{$p}) {
      $plane{$p} = 1;
      &set_map($ucs, $mb_94x94, 0x47 + $p - 1, (map {hex($_)} ($c =~ /^([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})$/)), "cns11643-$p");
    }
  }
}

sub uni2gb {
  my ($ucs, @gb) = grep(!/^\#/, split(/\s+/, $_[0]));
  my $c;

  foreach $c (grep(/^0-/, @gb)) {
    &set_map(hex($ucs), $mb_94x94, 0x41, (map {$_ + 0x20} ($c =~ /^0-(\d{2})(\d{2})$/)), 'gb2312');
  }
}

my %fc = ('0' => 0x40, '2' => 0x42);
my $mb_in_jisc6226 = 1 << 0;
my $mb_in_jisx0208 = 1 << 1;
my $mb_in_jisx0212 = 1 << 2;
my $mb_in_jisx0213_1 = 1 << 3;
my $mb_in_jisx0213_2 = 1 << 4;
my $mb_in_jis1common = ($mb_in_jisc6226 | $mb_in_jisx0208 | $mb_in_jisx0213_1);
my $mb_in_jis2common = ($mb_in_jisx0212 | $mb_in_jisx0213_2);

sub jisreg {
  my ($tab, $flag, $row, $col, $ucs) = @_;
  my $key = sprintf('%X-%X', $row * 0x60 + $col, $ucs);

  $tab->{$key} |= $flag;
}

sub uni2jis {
  my ($ucs, @jis) = grep(!/^\#/, split(/\s+/, $_[0]));
  my ($fc, $c1, $c2);

  $ucs = hex($ucs);

  foreach (@jis) {
    if (/^([02])-(\d{2})(\d{2})$/) {
      &jisreg(\%jis1tab, $1 eq '0' ? $mb_in_jisc6226 : $mb_in_jisx0208, $2, $3, $ucs);
    }
    elsif (/^S-(\d{2})(\d{2})$/) {
      &jisreg(\%jis2tab, $mb_in_jisx0212, $1, $2, $ucs);
    }
    elsif (m%^([KR])-(\d+)/(\d+)$%) {
      &set_map($ucs, $mb_94_0, $1 eq 'K' ? 0x49 : 0x4A, $2 * 0x10 + $3, undef, $1 eq 'K' ? 'jisx0201-r' : 'jisx0201-l');
    }
  }
}

sub uni2jisx0213 {
  if ($_[0] =~ /^u-([0-9A-Fa-f]{4}|0)\s+(\d)-(\d+)-(\d+)/) {
    my $ucs = hex($1);

    if ($2 eq '1') {
      &jisreg(\%jis1tab, $mb_in_jisx0213_1, $3, $4, $ucs);
    }
    else {
      &jisreg(\%jis2tab, $mb_in_jisx0213_2, $3, $4, $ucs);
    }
  }
}

sub mkjismap {
  my ($tab, $map, $common_flag, $common_fc, $common_cmt) = @_;
  my ($jis, $ucs, $key, $flag);

  while (($key, $flag) = each %$tab) {
    ($jis, $ucs) = map {hex($_)} split(/-/, $key);

    if ($flag == $common_flag) {
      &do_set_map($ucs, &mb_esc_enc($mb_94x94, $common_fc), $jis, $common_cmt);
    }
    else {
      &do_set_map($ucs, &mb_esc_enc($mb_94x94, 0x40), $jis, 'jisc6226') if ($flag & $mb_in_jisc6226);
      &do_set_map($ucs, &mb_esc_enc($mb_94x94, 0x42), $jis, 'jisx0208') if ($flag & $mb_in_jisx0208);
      &do_set_map($ucs, &mb_esc_enc($mb_94x94, 0x44), $jis, 'jisx0212') if ($flag & $mb_in_jisx0212);
      &do_set_map($ucs, &mb_esc_enc($mb_94x94, 0x4F), $jis, 'jisx0213-1') if ($flag & $mb_in_jisx0213_1);
      &do_set_map($ucs, &mb_esc_enc($mb_94x94, 0x50), $jis, 'jisx0213-2') if ($flag & $mb_in_jisx0213_2);
      $map->{$jis} |= $flag;
    }
  }
}

sub uni2ksx {
  my ($c1, $c2, $ucs) = $_[0] =~ /^0x([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})\s+0x([0-9A-Fa-f]{4})/;

  &set_map(hex($ucs), $mb_94x94, 0x43, hex($c1), hex($c2), 'ksx1001') if (defined($ucs));
}

my $uhangll_esc = &mb_esc_enc($mb_128, $mb_uhangll_fc);
my $uhanglr_esc = &mb_esc_enc($mb_128, $mb_uhanglr_fc);

sub uni2uhang {
  if ($_[0] =~ /^0x([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})\s+0x([0-9A-Fa-f]{4})/) {
    my ($c1, $c2, $ucs) = (hex($1), hex($2), hex($3));

    if ($c1 >= 0x81 && $c1 <= 0xA0) {
      if (($2 >= 0x41 && $c2 <= 0x5A) || ($c2 >= 0x61 && $c2 <= 0x7A)) {
	&do_set_map($ucs, $uhangll_esc, ($c1 - 0x80) * 0x60 + $c2 - 0x40, 'uhang');
      }
      elsif ($c2 >= 0x81 && $c2 <= 0xA0) {
	&do_set_map($ucs, $uhangll_esc, ($c1 - 0x80) * 0x60 + $c2 - 0x81 + 0x7B - 0x40, 'uhang');
      }
      elsif ($c2 >= 0xA1 && $c2 <= 0xFE) {
	&do_set_map($ucs, $uhangll_esc, ($c1 - 0x81 + 0xA1 - 0x80) * 0x60 + $c2 - 0xA0, 'uhang');
      }
    }
    elsif ($c1 >= 0xA1 && $c1 <= 0xC6) {
      if (($c2 >= 0x41 && $c2 <= 0x5A) || ($c2 >= 0x61 && $c2 <= 0x7A)) {
	&do_set_map($ucs, $uhanglr_esc, ($c1 - 0xA0) * 0x60 + $c2 - 0x40, 'uhang');
      }
      elsif ($c2 >= 0x81 && $c2 <= 0xA0) {
	&do_set_map($ucs, $uhanglr_esc, ($c1 - 0xA0) * 0x60 + $c2 - 0x81 + 0x7B - 0x40, 'uhang');
      }
      elsif ($c2 >= 0xA1 && $c2 <= 0xFE) {
	&do_set_map($ucs, $uhanglr_esc, ($c1 - 0xA0) * 0x60 + $c2 - 0xA0, 'uhang');
      }
    }
  }
}

sub uni2big5 {
  if ($_[0] =~ /^0x([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})\s+0x([0-9A-Fa-f]{4})/) {
    my ($c1, $c2, $ucs) = (hex($1), hex($2), hex($3));

    &set_map($ucs, $mb_128, $c2 < 0x7F ? $mb_big5l_fc : $mb_big5r_fc, $c1, $c2, 'big5');
  }
}

my $johabll_esc = &mb_esc_enc($mb_128, $mb_johabll_fc);
my $johablr_esc = &mb_esc_enc($mb_128, $mb_johablr_fc);
my $johabh_esc = &mb_esc_enc($mb_128, $mb_johabh_fc);
my $johabh_hi = 0xF9 - 0xE0 + 1 + 0xDE - 0xD8 + 1;

sub uni2johab {
  if ($_[0] =~ /^0x([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})\s+0x([0-9A-Fa-f]{4})/) {
    my ($c1, $c2, $ucs) = (hex($1), hex($2), hex($3));

    if ($c1 >= 0x84 && $c1 <= 0xD3) {
      if ($c2 >= 0x41 && $c2 <= 0x7E) {
	&do_set_map($ucs, $johabll_esc, ($c1 - 0x84) * 0x60 + ($c2 - 0x41), 'johab');
      }
      elsif ($c2 >= 0x81 && $c2 <= 0xA0) {
	&do_set_map($ucs, $johabll_esc, ($c1 - 0x84) * 0x60 + ($c2 - 0x81 + 0x7F - 0x41), 'johab');
      }
      elsif ($c2 >= 0xA1 && $c2 <= 0xFE) {
	&do_set_map($ucs, $johablr_esc, ($c1 - 0x84) * 0x60 + ($c2 - 0xA0), 'johab');
      }
    }
    elsif ($c1 >= 0xD8 && $c1 <= 0xDE) {
      if ($c2 >= 0x31 && $c2 <= 0x7E) {
	&do_set_map($ucs, $johabh_esc, ($c1 - 0xD7) * 0x60 + ($c2 - 0x30), 'johab');
      }
      elsif ($c2 >= 0x91 && $c2 <= 0xA0) {
	&do_set_map($ucs, $johabh_esc, ($c1 - 0xD7) * 0x60 + ($c2 - 0x91 + 0x7F - 0x30), 'johab');
      }
      elsif ($c2 >= 0xA1 && $c2 <= 0xFE) {
	&do_set_map($ucs, $johabh_esc, ($c1 - 0xD7 + $johabh_hi) * 0x60 + ($c2 - 0xA0), 'johab');
      }
    }
    elsif ($c1 >= 0xE0 && $c1 <= 0xF9) {
      if ($c2 >= 0x31 && $c2 <= 0x7E) {
	&do_set_map($ucs, $johabh_esc, ($c1 - 0xE0 + 0xDF - 0xD7) * 0x60 + ($c2 - 0x30), 'johab');
      }
      elsif ($c2 >= 0x91 && $c2 <= 0xA0) {
	&do_set_map($ucs, $johabh_esc, ($c1 - 0xE0 + 0xDF - 0xD7) * 0x60 + ($c2 - 0x91 + 0x7F - 0x30), 'johab');
      }
      elsif ($c2 >= 0xA1 && $c2 <= 0xFE) {
	&do_set_map($ucs, $johabh_esc, ($c1 - 0xE0 + 0xDF - 0xD7 + $johabh_hi) * 0x60 + ($c2 - 0xA0), 'johab');
      }
    }
  }
}

sub win125x {
  if ($_[1] =~ /^([8-9A-Fa-f][0-9A-Fa-f])\s*=\s*U\+([0-9A-Fa-f]{4})\s*:/i) {
    &do_set_map(hex($2), $misc_esc, (hex($1) & ((1 << 7) - 1)) | ($mb_win125x_subfc[$_[0]] << 7), sprintf("windows-125%d", $_[0]));
  }
}

sub make_map {
  my ($line, $i, $func);

  while (defined($line = <>)) {
    if ($line =~ s/^\s*\#\s*Name:\s*//i) {
      if ($line =~ m%ISO.*8859-(\d+)%i) {
	if (($i = $1 - 1) < @iso8859fc) {
	  my $cmt = "iso8859-$1";
	  my $fc = $iso8859fc[$i];

	  $func = sub {&sbc96($_[0], $fc, $cmt);};
	}
	else {
	  $func = undef;
	}
      }
      elsif ($line =~ /^The\s+Unicode\s+Han\s+Character\s+Cross-Reference/i) {
	$func = undef;
      }
      elsif ($line =~ /\bCNS/i) {
	$func = \&uni2cns;
      }
      elsif ($line =~ /\bGB\b/i) {
	$func = \&uni2gb;
      }
      elsif ($line =~ /\bJIS\s*X\s*0213\b/i) {
	$func = \&uni2jisx0213;
      }
      elsif ($line =~ /\bJIS\b/i) {
	$func = \&uni2jis;
      }
      elsif ($line =~ /\bKS\s*X\s*1001\b/i) {
	$func = \&uni2ksx;
      }
      elsif ($line =~ /\bBig\s*(5|Five)\b/i) {
	$func = \&uni2big5;
      }
      elsif ($line =~ /\bJohab\b/i) {
	$func = \&uni2johab;
      }
      elsif ($line =~ /\bUnified\s*Hange?ul\b/i) {
	$func = \&uni2uhang;
      }
      else {
	$func = undef;
      }
    }
    elsif ($line =~ /^Korean\s+Hangul\s+Encoding\s+Conversion\s+Table/i) {
      $func = undef;
    }
    elsif ($line =~ /^Microsoft\s+Windows\s+Codepage\s*:\s*(125[0-8])\s/i) {
      my $cpoff = $1 - 1250;

      $func = sub {&win125x($cpoff, @_)};
    }
    elsif ($line !~ /^\#/ && ref($func) eq 'CODE') {
      $func->($line);
    }
  }

  my $argv;

  foreach $argv (@supplement) {
    &set_map(@$argv);
  }

  my ($subfc, $c, $cmt, $arr);

  for ($subfc = 0 ; $subfc * 2 < @mb_misc_to_ucs ; ++$subfc) {
    for (($cmt, $arr) = @mb_misc_to_ucs[$subfc * 2, $subfc * 2 + 1], $c = 0 ; $c < @$arr ; ++$c) {
      &do_set_map($arr->[$c], $misc_esc, $c | ($subfc << 7), $cmt);
    }
  }

  &mkjismap(\%jis1tab, \%jis1map, $mb_in_jis1common, 0x4F, 'jisx0213-1');
  &mkjismap(\%jis2tab, \%jis2map, $mb_in_jis2common, 0x50, 'jisx0213-2');
  (\@map, \@noascii_map, \%jis1map, \%jis2map);
}

sub mkpreconvtab {
  (
   [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_94x94, 0x40),
    &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_94x94, 0x40),
    &mb_esc_enc($mb_94x94, 0x4F)],
   [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_94x94, 0x42),
    &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_94x94, 0x42),
    &mb_esc_enc($mb_94x94, 0x4F)],
   [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_94x94, 0x44),
    &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_94x94, 0x44),
    &mb_esc_enc($mb_94x94, 0x50)],
   [&mb_word_enc_3(&dbc2c(0x21), $mb_128, $mb_sjis_fc),
    &mb_word_enc_3(&dbc2c(0x7E), $mb_128, $mb_sjis_fc),
    &mb_esc_enc($mb_94_0, 0x49)],
   [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_128, $mb_sjis_fc),
    &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_128, $mb_sjis_fc),
    &mb_esc_enc($mb_94x94, 0x4F)],
   [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_128, $mb_sjis0213_fc),
    &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_128, $mb_sjis0213_fc),
    &mb_esc_enc($mb_94x94, 0x50)],
   (
    map {
      [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_128, $mb_euctwb_fc + $_),
       &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_128, $mb_euctwb_fc + $_),
       &mb_esc_enc($mb_94x94, 0x47 + $_)];
    } (0 .. 6)
    ),
   [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_128, $mb_uhangh_fc),
    &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_128, $mb_uhangh_fc),
    &mb_esc_enc($mb_94x94, 0x43)],
   );
}

sub is_space {1 << 0;}
sub never_bol {1 << 1;}
sub never_eol {1 << 2;}
sub may_break {1 << 3;}
sub eol_to_null {1 << 4;}

my @iso_cjk =
  sort {
    $a->[0] <=> $b->[0];
  } (
     (
      map {
	[&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_94x94, $_),
	 &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_94x94, $_),
	 &may_break | &eol_to_null];
      } (0x40 .. 0x42, 0x44 .. 0x5F)
      ),
     [&mb_word_enc_3(&dbc2c(0x21, 0x21), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x23, 0x7E), $mb_94x94, 0x43),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(&dbc2c(0x24, 0x21), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x24, 0x7E), $mb_94x94, 0x43),
      &may_break],
     [&mb_word_enc_3(&dbc2c(0x25, 0x21), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x28, 0x30), $mb_94x94, 0x43),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(&dbc2c(0x28, 0x31), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x28, 0x4C), $mb_94x94, 0x43),
      &may_break],
     [&mb_word_enc_3(&dbc2c(0x28, 0x4D), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x29, 0x30), $mb_94x94, 0x43),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(&dbc2c(0x29, 0x31), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x29, 0x4C), $mb_94x94, 0x43),
      &may_break],
     [&mb_word_enc_3(&dbc2c(0x29, 0x4D), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x2F, 0x7E), $mb_94x94, 0x43),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(&dbc2c(0x30, 0x21), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x48, 0x7E), $mb_94x94, 0x43),
      &may_break],
     [&mb_word_enc_3(&dbc2c(0x49, 0x21), $mb_94x94, 0x43),
      &mb_word_enc_3(&dbc2c(0x7E, 0x7E), $mb_94x94, 0x43),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(&dbc2c(0xA1, 0x40), $mb_128, $mb_big5l_fc),
      &mb_word_enc_3(&dbc2c(0xFE, 0x7E), $mb_128, $mb_big5l_fc),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(&dbc2c(0xA1, 0xA1), $mb_128, $mb_big5r_fc),
      &mb_word_enc_3(&dbc2c(0xFE, 0xFE), $mb_128, $mb_big5r_fc),
      &may_break | &eol_to_null],
     [&mb_word_enc_3(0, $mb_128, $mb_johabll_fc),
      &mb_word_enc_3(0xFFFF, $mb_128, $mb_johabll_fc),
      &may_break],
     [&mb_word_enc_3(0, $mb_128, $mb_johablr_fc),
      &mb_word_enc_3(0xFFFF, $mb_128, $mb_johablr_fc),
      &may_break],
     [&mb_word_enc_3(0, $mb_128, $mb_johabh_fc),
      &mb_word_enc_3(0xFFFF, $mb_128, $mb_johabh_fc),
      &may_break | &eol_to_null],
     (
      map {
	[&mb_word_enc_3(&dbc2c(0xA1, 0xA1), $mb_128, $mb_euctwb_fc + $_),
	 &mb_word_enc_3(&dbc2c(0xFE, 0xFE), $mb_128, $mb_euctwb_fc + $_),
	 &may_break | &eol_to_null];
      } (7 .. 15)
      ),
     [&mb_word_enc_3(0, $mb_128, $mb_uhangll_fc),
      &mb_word_enc_3(0xFFFF, $mb_128, $mb_uhangll_fc),
      &may_break],
     [&mb_word_enc_3(0, $mb_128, $mb_uhanglr_fc),
      &mb_word_enc_3(0xFFFF, $mb_128, $mb_uhanglr_fc),
      &may_break],
     );

sub isocjk {@iso_cjk;}

sub is94x94 {
  my ($c, $esc, $b, $e, $i, @yes);

  while (@_ >= 2) {
    ($c, $esc) = splice(@_, 0, 2);
    $c = &mb_word_enc($c, $esc);

    for ($b = 0, $e = @iso_cjk ; $b < $e ;) {
      $i = int(($b + $e) / 2);

      if ($c < $iso_cjk[$i]->[0]) {
	$e = $i;
      }
      elsif ($c > $iso_cjk[$i]->[1]) {
	$b = $i + 1;
      }
      else {
	push(@yes, $esc);
	last;
      }
    }
  }

  @yes;
}

my @mb_94_to_GR;

$mb_94_to_GR[0x49 & $mb_esc_fc_mask] = 1;
   
sub ucs2isov {
  my $ucs = shift;
  my $map = @_ ? $_[0] : \@map;

  if (ref($map->[$ucs]) eq 'HASH') {
    map {
      my $set = $_->[0] & $mb_esc_set_mask;
      my $fc = (($_->[0] >> $mb_esc_set_len) & $mb_esc_fc_mask) | 0x40;

      if ($set == $mb_128) {
	();
      }
      else {
	my $c1 = $_->[1] / 0x60 + 0x20;
	my $c2 = $_->[1] % 0x60 + 0x20;

	($_->[2],
	 $set == $mb_94x94 ? pack('C*', 0x1B, 0x24, 0x28, $fc, $c1, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X%02X)', $c1, $c2) :
	 $set == $mb_96_0 ? (($fc == 0x41 ? pack('C', $c2 | 0x80) : pack('C*', 0x1B, 0x2D, $fc, $c2 | 0x80, 0x1B, 0x2D, 0x41))
			     . sprintf(' (0x%02X)', $c2 | 0x80)):
	 $set == $mb_96_1 ? pack('C*', 0x1B, 0x2D, $fc | 0x20, $c2 | 0x80, 0x1B, 0x2D, 0x41) . sprintf(' (0x%02X)', $c2 | 0x80) :
	 $set == $mb_94_0 ? ($mb_94_to_GR[$fc & $mb_esc_fc_mask] ?
			     pack('C*', 0x1B, 0x29, $fc, $c2 | 0x80, 0x1B, 0x2D, 0x41) . sprintf(' (0x%02X)', $c2 | 0x80) :
			     pack('C*', 0x1B, 0x28, $fc, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X)', $c2)) :
	 $set == $mb_94_1 ? pack('C*', 0x1B, 0x28, $fc | 0x20, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X)', $c2) :
	 $set == $mb_94_2 ? pack('C*', 0x1B, 0x28, 0x21, $fc, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X)', $c2) :
	 pack('C*', 0x1B, 0x28, 0x21, $fc | 0x20, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X)', $c2),
	 ($_->[2] eq 'jisx0213-1' && !$jis1map{$_->[1]}) ?
	 ('jisc6226',
	  pack('C*', 0x1B, 0x24, 0x28, 0x40, $c1, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X%02X)', $c1, $c2),
	  'jisx0208',
	  pack('C*', 0x1B, 0x24, 0x28, 0x42, $c1, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X%02X)', $c1, $c2))
	 : (),
	 ($_->[2] eq 'jisx0213-2' && !$jis2map{$_->[1]}) ?
	 ('jisx0212',
	  pack('C*', 0x1B, 0x24, 0x28, 0x44, $c1, $c2, 0x1B, 0x28, 0x42) . sprintf(' (0x%02X%02X)', $c1, $c2)) :
	 ());
      }
    } values %{$map->[$ucs]}; 
  }
  else {
    ();
  }
}

1;
