/*
 * libjingle
 * Copyright 2004--2005, Google Inc.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TALK_BASE_HTTPCLIENT_H__
#define TALK_BASE_HTTPCLIENT_H__

#include "talk/base/httpbase.h"
#include "talk/base/sigslot.h"
#include "talk/base/socketpool.h"

namespace cricket {

//////////////////////////////////////////////////////////////////////
// HttpClient
//////////////////////////////////////////////////////////////////////

class HttpClient;
class IPNetPool;

class HttpClient : private IHttpNotify {
public:
  HttpClient(const char* agent, StreamPool* pool);
  virtual ~HttpClient();

  void set_pool(StreamPool* pool) { pool_ = pool; }

  const char* agent() const { return agent_.c_str(); }
  
  HttpRequestData& request() { return request_; }
  HttpResponseData& response() { return response_; }
  
  void reset(); // will also abort an active request
  void call(const char* server, uint16 port);
  
  // convenience methods
  void get(const char* url);
  void post(const char* url, const char* content_type,
            StreamInterface* request_doc);
  void set_fail_redirect(bool fail_redirect) { 
    fail_redirect_ = fail_redirect; 
  }
  void set_transparent_proxy(bool transparent_proxy) {
    transparent_proxy_ = transparent_proxy;
  }

  // Signalled when the current 'call' finishes.  On success, err is 0.
  sigslot::signal2<HttpClient*,int> SignalHttpClientComplete;
  // Signalled when the network connection goes down while a call is not
  // in progress.
  sigslot::signal2<HttpClient*,int> SignalHttpClientClosed;

protected:
  void release();

  // IHttpNotify Interface
  virtual HttpError onHttpHeaderComplete(size_t& data_size);
  virtual void onHttpComplete(HttpMode mode, HttpError err);
  virtual void onHttpClosed(HttpError err);
  
private:
  std::string agent_;
  StreamPool* pool_;
  HttpBase base_;
  HttpRequestData request_;
  HttpResponseData response_;
  bool fail_redirect_, transparent_proxy_;
};

//////////////////////////////////////////////////////////////////////
// Default implementation of HttpClient
//////////////////////////////////////////////////////////////////////

class HttpClientDefault : public ReuseSocketPool, public HttpClient {
public:
  HttpClientDefault(SocketFactory* factory, const char* agent) 
      : ReuseSocketPool(factory), HttpClient(agent, NULL) { 
    set_pool(this);
  }
};

//////////////////////////////////////////////////////////////////////

}  // namespace cricket

#endif // TALK_BASE_HTTPCLIENT_H__
