# Representation of a single Duo group for the Admin API.
#
# This class wraps the Duo representation of a single Duo group, as returned
# by (for example) the Admin /groups REST endpoint.

package Net::Duo::Admin::Group 1.00;

use 5.014;
use strict;
use warnings;

use parent qw(Net::Duo::Object);

# Data specification for converting JSON into our object representation.  See
# the Net::Duo::Object documentation for syntax information.
## no critic (Subroutines::ProhibitUnusedPrivateSubroutines)
sub _fields {
    return {
        desc          => 'simple',
        group_id      => 'simple',
        name          => 'simple',
        push_enabled  => ['simple', 'boolean'],
        sms_enabled   => ['simple', 'boolean'],
        status        => 'simple',
        voice_enabled => ['simple', 'boolean'],
    };
}
## use critic

# Install our accessors.
Net::Duo::Admin::Group->install_accessors;

# Override the create method to add the appropriate URI.
#
# $class    - Class of object to create
# $duo      - Net::Duo object to use to create the object
# $data_ref - Data for new object as a reference to a hash
#
# Returns: Newly-created object
#  Throws: Net::Duo::Exception on any problem creating the object
sub create {
    my ($class, $duo, $data_ref) = @_;
    return $class->SUPER::create($duo, '/admin/v1/groups', $data_ref);
}

# Delete the group from Duo.  After this call, the object should be treated as
# read-only since it can no longer be usefully updated.
#
# $self - The Net::Duo::Admin::Group object to delete
#
# Returns: undef
#  Throws: Net::Duo::Exception on any problem deleting the object
## no critic (Subroutines::ProhibitBuiltinHomonyms)
sub delete {
    my ($self) = @_;
    $self->{_duo}->call_json('DELETE', "/admin/v1/groups/$self->{group_id}");
    return;
}
## use critic

1;
__END__

=for stopwords
Allbery MERCHANTABILITY NONINFRINGEMENT SMS passcodes sublicense desc

=head1 NAME

Net::Duo::Admin::Group - Representation of a Duo group

=head1 SYNOPSIS

    my $decoded_json = get_json();
    my $group = Net::Duo::Admin::Group->new($decoded_json);
    say $group->desc;

=head1 REQUIREMENTS

Perl 5.14 or later and the modules HTTP::Request and HTTP::Response (part
of HTTP::Message), JSON, LWP (also known as libwww-perl), Perl6::Slurp,
Sub::Install, and URI::Escape (part of URI), all of which are available
from CPAN.

=head1 DESCRIPTION

A Net::Duo::Admin::Group object is a Perl representation of a Duo group as
returned by the Duo Admin API, usually via the groups() method or nested
in a user returned by the users() method.  It contains various information
about a group, including the privileges it controls.

=head1 CLASS METHODS

=over 4

=item create(DUO, DATA)

Creates a new group in Duo and returns the resulting user as a new
Net::Duo::Admin::Group object.  DUO is the Net::Duo object that should be
used to perform the creation.  DATA is a reference to a hash with the
following keys:

=over 4

=item name

The name of the group to create.  Required.

=item desc

The description of the group.  Optional.

=item push_enabled

Whether users in the group will be able to use Duo Push to authenticate.
See the L</push_enabled()> method below for more information.  Optional.

=item sms_enabled

Whether users in the group will be able to use SMS to authenticate.  See
the L</sms_enabled()> method below for more information.  Optional.

=item status

The group's authentication status.  See the L</status()> method below for
the possible values.  Optional.

=item voice_enabled

Whether users in the group will be able to use voice callbacks to
authenticate.  See the L</voice_enabled()> method below for more
information.  Optional.

=back

=item new(DUO, DATA)

Creates a new Net::Duo::Admin::Group object from a full data set.  DUO is
the Net::Duo object that should be used for any further actions on this
object.  DATA should be the data structure returned by the Duo REST API
for a single user, after JSON decoding.

=back

=head1 INSTANCE ACTION METHODS

=over 4

=item delete()

Delete this group from Duo.  After successful completion of this call, the
Net::Duo::Admin::Group object should be considered read-only, since no
further changes to the object can be meaningfully sent to Duo.

=item json()

Convert the data stored in the object to JSON and return the results.  The
resulting JSON should match the JSON that one would get back from the Duo
web service when retrieving the same object (plus any changes made locally
to the object via set_*() methods).  This is primarily intended for
debugging dumps or for passing Duo objects to other systems via further
JSON APIs.

=back

=head1 INSTANCE DATA METHODS

=over 4

=item desc()

The description of the group.

=item group_id()

The unique ID of this group as generated by Duo on group creation.

=item name()

The name of this group.

=item push_enabled()

If true, users in the group will be able to use Duo Push to authenticate.
If false, users in the group will not be able to use Duo Push to
authenticate.  Note that this setting has no effect if Duo Push is
disabled globally.

=item sms_enabled()

If true, users in the group will be able to use SMS passcodes to
authenticate.  If false, users in the group will not be able to use SMS
passcodes to authenticate.  Note that this setting has no effect if SMS
passcodes are disabled globally.

=item status()

One of the following values:

=over 4

=item C<active>

The users in this group must complete secondary authentication.

=item C<bypass>

The users in this group will bypass secondary authentication after
completing primary authentication.

=item C<disabled>

The users in this group will not be able to authenticate.

=back

=item voice_enabled()

If true, users in the group will be able to authenticate with a voice
callback.  If false, users in the group will not be able to authenticate
with a voice callback.  Note that this setting has no effect if voice
callback is disabled globally.

=back

=head1 AUTHOR

Russ Allbery <rra@cpan.org>

=head1 COPYRIGHT AND LICENSE

Copyright 2014 The Board of Trustees of the Leland Stanford Junior
University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

=head1 SEE ALSO

L<Net::Duo::Admin>

L<Duo Admin API for groups|https://www.duosecurity.com/docs/adminapi#groups>

This module is part of the Net::Duo distribution.  The current version of
Net::Duo is available from CPAN, or directly from its web site at
L<http://www.eyrie.org/~eagle/software/net-duo/>.

=cut
