#ifndef _RECORD_H
#define _RECORD_H
#include "osl/record/searchInfo.h"
#include "osl/move.h"
#include "osl/state/numEffectState.h"
#include "osl/misc/carray.h"
#include "osl/misc/align16New.h"
#include <boost/ptr_container/ptr_vector.hpp>
#include <iosfwd>

namespace osl
{
  namespace record
  {
    class Record;
    class IRecordStream{
    public:
      virtual void load(Record*)=0;
      virtual ~IRecordStream();
    private:
    };
    class ORecordStream{
    public:
      virtual void save(Record*)=0;
      virtual ~ORecordStream();
    private:
    };
  
    enum NodeType{
      MOVE,
      TORYO,
      MATTA,
      CHUDAN,
      SENNICHITE,
      JISHOGI,
      TSUMI,
      FUZUMI,
      ND_ERROR, // ERROR conflicts with ERROR macro on windows
      KACHI,
      HIKIWAKE,
    };

    /**
     * とりあえず tree を表現できるようにする．
     * 合流は考えない．
     * 所詮，出すのは CSA かGCF
     */
    class MoveRecord{
    private:
      Move move;
      int nodeIndex;
      int time;
      std::string comment;
    public:
      SearchInfo info;

      MoveRecord(const Move& mv, int ni); 
      const Move getMove() const;
      int getNodeIndex() const;
      void setTime(int t);
      int getTime() const{ return time; }
      void setComment(const std::string& com){ comment=com; }
      void addComment(const std::string& com)
      {
	if (! comment.empty())
	  comment += "\n";
	comment += com; 
      }
      const std::string& getComment() const{ return comment; }
    };

    class NodeRecord{
    private:
      NodeType type;
      vector<int> moves;
      std::string comment;
    public:
      NodeRecord():type(MOVE){}
      NodeType getType() const{ return type; }
      int size() const { return moves.size(); }
      int at(int index) const{ return moves.at(index); }
      void setComment(const std::string& com){ comment=com; }
      const std::string& getComment() const{ return comment; }
      void addMoveRecord(int moveIndex);
    };
  
    class Record
#if OSL_WORDSIZE == 32
      : public misc::Align16New
#endif
    {
    public:
      enum ResultType {
	UNKNOWN=0,
	BLACK_WIN=1,
	WHITE_WIN=2,
	SENNNICHITE=3,
	JISHOGI=4,
      };
    private:
      SimpleState initialState;
      std::string version, initial_comment, tounament_name;
      CArray<std::string,2> playerNames;
      vector<NodeRecord> nrs;
      vector<MoveRecord> mrs;
      ResultType result;
    public:
      Record();
      void init();
      void setVersion(const std::string& str);
      const std::string getVersion() const { return version; }
      void addInitialComment(const std::string& comment)
      {
	if (! initial_comment.empty())
	  initial_comment += "\n";
	initial_comment += comment;
      }
      const std::string getInitialComment() const 
      {
	return initial_comment; 
      }
      void setPlayer(Player player,const std::string& str);
      const std::string& getPlayer(Player player) const;
      void setInitialState(const SimpleState& state);
      const NumEffectState getInitialState() const;
      int addNodeRecord();
      int addMoveRecord(const MoveRecord& moveRecord);
      NodeRecord* nodeOf(int index);
      NodeRecord& operator[](int index);
      MoveRecord* moveOf(int index);
      void load(IRecordStream&); 
      void save(ORecordStream&); 
      const vector<Move> getMoves() const;
      void getMoves(vector<Move>&, vector<int>&) const;
      void getMoves(vector<Move>&, vector<int>&, vector<std::string>&,
		    vector<SearchInfo>&) const;
      const NodeRecord* nodeOf(int index) const;
      const MoveRecord* moveOf(int index) const;
      void setResult(ResultType new_result) { result = new_result; }
      ResultType getResult() const { return result; }
      void setTounamentName(const std::string& name) { tounament_name = name; }
      const std::string& tounamentName() const { return tounament_name; }
    };

    class RecordVisitor;

    class RecordVisitorObserver {
    public:
        virtual ~RecordVisitorObserver() {}
        virtual void update(RecordVisitor* rv) = 0;
    };
  
    class RecordVisitor{
    private:
      Record* rec;
      SimpleState* state;
      int lastMoveIndex;
      int nodeIndex;
      boost::ptr_vector<RecordVisitorObserver> observers;
    public:
      RecordVisitor():rec(NULL),state(NULL),lastMoveIndex(0),nodeIndex(0){}
      RecordVisitor(Record& r);
      ~RecordVisitor();

      SimpleState *getState() const{ return state; }
      void setState(SimpleState *s){ state=s;}
      Record *getRecord() { return rec; }
      void setRecord(Record *r){ rec=r;}
      MoveRecord *getLastMove(){ return rec->moveOf(lastMoveIndex); }
      void addMoveAndAdvance(Move move);
      NodeRecord *getNode(){ return rec->nodeOf(nodeIndex); }
      void addObserver(RecordVisitorObserver *observer)
      { observers.push_back(observer); }
    };
  
    std::ostream& operator<<(std::ostream&,const MoveRecord &);
    std::ostream& operator<<(std::ostream&,Record &);

    int readInt(std::istream& is);
    void writeInt(std::ostream& os, int n);
  } // namespace record
  using record::Record;
} // namespace osl
#endif /* _RECORD_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
