#include "osl/move_generator/allMoves.h"
#include "osl/move_generator/promote_.h"

#include "osl/move_generator/promote_.tcc"
#include "osl/move_action/store.h"
#include "osl/move_action/voidAction.h"
#include "osl/state/simpleState.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/move.h"
#include "osl/container/moveVector.h"
#include "osl/state/numEffectState.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

typedef osl::NumEffectState test_state_t;
#include <fstream>
#include <iostream>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

using namespace osl;

class generatePromoteTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( generatePromoteTest );
  CPPUNIT_TEST( testMove );
  CPPUNIT_TEST( testFile );
  CPPUNIT_TEST_SUITE_END();
public:
  void testMove();
  void testFile();
};

CPPUNIT_TEST_SUITE_REGISTRATION(generatePromoteTest);

using namespace osl;
using namespace osl::move_generator;
using namespace osl::move_action;

void generatePromoteTest::testMove()
{
  typedef test_state_t State;
  {
    SimpleState state=CsaString(
    "P1+NY+TO * +HI * -KI-OU-KE-KY\n"
    "P2 *  *  *  *  * -GI *  *  *\n"
    "P3 *  * +GI * +UM * -KI-FU-FU\n"
    "P4 *  * +FU-FU * +FU *  *  *\n"
    "P5 *  * -KE+FU+FU *  * +FU *\n"
    "P6+KE *  *  *  * -FU *  * +FU\n"
    "P7 *  * -UM *  *  *  *  *  *\n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 * +OU * -GI *  *  *  * -NG\n"
    "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU\n"
    "P-00KI00KY00FU00FU\n"
    "P-00AL\n"
    "+\n"
    ).getInitialState();
    State eState(state);

    MoveVector moves;
    Promote<BLACK,true>::generate(eState,moves);

    // 銀
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,3),Position(8,4),PSILVER,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,3),Position(6,2),PSILVER,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // captureする手は生成しない
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(7,3),Position(6,4),PSILVER,PAWN,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // 歩
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),Position(4,3),PPAWN,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,5),Position(2,4),PTYPE_EMPTY,PAWN,false,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // 飛車
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(5,1),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(7,1),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(6,2),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(6,3),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // capture moveは生成しない．
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(6,1),Position(6,4),PROOK,PAWN,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT((size_t)9==moves.size()  ||
		   (std::cerr << eState << moves << std::endl,0));
		   
  }
  {
    SimpleState state=CsaString(
    "P1+NY+TO * +HI * -KI-OU-KE-KY\n"
    "P2 *  *  *  *  * -GI *  *  *\n"
    "P3 *  * +GI * +UM * -KI-FU-FU\n"
    "P4 *  * +FU-FU * +FU *  *  *\n"
    "P5 *  * -KE+FU+FU *  * +FU *\n"
    "P6+KE *  *  *  * -FU *  * +FU\n"
    "P7 *  * -UM *  *  *  *  *  *\n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 * +OU * -GI *  *  *  * -NG\n"
    "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU\n"
    "P-00KI00KY00FU00FU\n"
    "P-00AL\n"
    "+\n"
    ).getInitialState();
    State eState(state);

    MoveVector moves;
    Promote<BLACK,false>::generate(eState,moves);

    // 銀
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,3),Position(8,4),PSILVER,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,3),Position(6,2),PSILVER,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // captureする手も生成する
    CPPUNIT_ASSERT(moves.isMember(Move(Position(7,3),Position(6,4),PSILVER,PAWN,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // 歩
    CPPUNIT_ASSERT(moves.isMember(Move(Position(4,4),Position(4,3),PPAWN,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(!moves.isMember(Move(Position(2,5),Position(2,4),PTYPE_EMPTY,PAWN,false,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // 飛車
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(5,1),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(7,1),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(6,2),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(6,3),PROOK,PTYPE_EMPTY,true,BLACK)) ||
		 (std::cerr << eState << moves << std::endl,0));
    // capture moveも生成する．
    CPPUNIT_ASSERT(moves.isMember(Move(Position(6,1),Position(6,4),PROOK,PAWN,true,BLACK)) ||
		   (std::cerr << eState << moves << std::endl,0));
    CPPUNIT_ASSERT_EQUAL((size_t)12,moves.size());
  }
}


static void testFile(const std::string& fileName)
{
  Record rec=CsaFile(fileName).getRecord();
  SimpleState state=rec.getInitialState();
  typedef test_state_t State;
  State eState(state);
  vector<osl::Move> moves=rec.getMoves();
  for(unsigned int i=0;i<moves.size();i++)
  {
    MoveVector allMoves;
    {
      Store store(allMoves);
      AllMoves<Store>::
	generate(eState.getTurn(),eState,store);
    }
    { // check nocapture
      MoveVector promoteMoves;
      if(eState.getTurn()==BLACK)
	Promote<BLACK,true>::generate(eState,promoteMoves);
      else
	Promote<WHITE,true>::generate(eState,promoteMoves);
      {
	// uniqueなmove
	size_t origSize=promoteMoves.size();
	promoteMoves.unique();
	CPPUNIT_ASSERT(promoteMoves.size()==origSize);
      }
      for(size_t j=0;j<allMoves.size();j++){
	Move move=allMoves[j];
	if(move.isPromote() && move.capturePtype()==PTYPE_EMPTY){
	  CPPUNIT_ASSERT(promoteMoves.isMember(move) ||
			 (std::cerr << eState << promoteMoves << std::endl << move << std::endl,0)
			 );
	}
      }
      for(size_t j=0;j<promoteMoves.size();j++){
	Move move=promoteMoves[j];
	CPPUNIT_ASSERT(eState.isValidMove(move,true)  
		       && move.isPromote() 
		       && move.capturePtype()==PTYPE_EMPTY);
      }
    }
    { // check capture
      MoveVector promoteMoves;
      if(eState.getTurn()==BLACK)
	Promote<BLACK,false>::generate(eState,promoteMoves);
      else
	Promote<WHITE,false>::generate(eState,promoteMoves);
      {
	// uniqueなmove
	size_t origSize=promoteMoves.size();
	promoteMoves.unique();
	CPPUNIT_ASSERT(promoteMoves.size()==origSize);
      }
      for(size_t j=0;j<allMoves.size();j++){
	const Move move=allMoves[j];
	if (move.isPromote()){
	  CPPUNIT_ASSERT(promoteMoves.isMember(move));
	}
      }
      for(size_t j=0;j<promoteMoves.size();j++){
	Move move=promoteMoves[j];
	CPPUNIT_ASSERT(eState.isValidMove(move,true)  && move.isPromote());
      }
    }
    ApplyMoveOfTurn::doMove(eState, moves[i]);
  }
}

void generatePromoteTest::testFile()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=100;
  if (isShortTest) 
    count=10;
  std::string fileName;
  while((ifs >> fileName) && ++i<count)
  {
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    ::testFile(OslConfig::testCsaFile(fileName));
  }
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
