/* FileOpenAction.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.actions;

import java.awt.Window;
import java.io.File;
import javax.swing.filechooser.FileFilter;

/**
 * Abstract super class for actions that allow files to be loaded.<p>
 * Clients should at least override {@link #openFile}.
 */
public abstract class FileOpenAction extends FileAction {

    /**
     * Create an action of this type
     * @param parent Parent window for the file chooser dialog
     * @param multipleAllowed Indicates whether the user is allowed to select
     * more than a single file at the time
     * @param filters File filters for the file chooser dialog
     */
    public FileOpenAction(Window parent, boolean multipleAllowed, FileFilter... filters) {
        super (parent, filters);
        this.fileChooser.setMultiSelectionEnabled(multipleAllowed);
    }

    /**
     * Called for every file selected by the user. Must be overridden 
     * by clients.
     */
    protected abstract void openFile(File file);

    /**
     * Shows the file chooser open dialog.
     */
    protected int showDialog() {
        return fileChooser.showOpenDialog(parent);
    }

    /**
     * Invoke {@link #openFile} on all files chosen. 
     */
    public void dialogApproved() {
        if (fileChooser.isMultiSelectionEnabled()) {
            for (File file : fileChooser.getSelectedFiles()) {
                openFile(file);
            }
        } else {
            openFile(fileChooser.getSelectedFile());
        }
    }
}
