/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "shapefil.h"
#include "TeImportExport.h"
#include "TeTable.h"
#include "TeException.h"
#include "TeUtils.h"
#include "TeLayer.h"
#include "TeDatabase.h"
#include <vector>


DBFHandle TeCreateDBFFile (const string& dbfFilename, TeAttributeList& attList)
{
	DBFHandle hDBF = DBFCreate( dbfFilename.c_str() );
	if( hDBF == 0 )
		return 0;
	
	int i =0;
	TeAttributeList::iterator it=attList.begin();

	while ( it != attList.end() )
	{
		TeAttribute at = (*it);
		string atName = at.rep_.name_;

		if (atName.size() > 10)
		{
			int extra = (int)(atName.size() - 10)/2;
			int middle = (int)(atName.size()/2);
			string str = atName.substr(0,middle-extra-1);
			str += atName.substr(middle+extra);
			atName = str;
		}

		if (at.rep_.type_ == TeSTRING )
		{
			if (DBFAddField( hDBF, atName.c_str(), FTString, at.rep_.numChar_, 0 ) == -1 )
				return 0;
		}
		else if (at.rep_.type_ == TeINT)
		{
			if (DBFAddField( hDBF, atName.c_str(), FTInteger, 10, 0 ) == -1 )
				return 0;
		}
		else if (at.rep_.type_ == TeREAL)
		{
			if (DBFAddField( hDBF, atName.c_str(), FTDouble, 10, 5 ) == -1 )
				return 0;
		}
		// OBS: shapelib doesnt deal with xBase field type for Date 
		// we are transforming it to string
		else if (at.rep_.type_ == TeDATETIME)
		{
			if (DBFAddField( hDBF, atName.c_str(), FTDate, 8, 0 ) == -1 )
				return 0;
		}
		++i;
		++it;
	}
	return hDBF;
}

bool
TeExportShapefile(TeLayer* layer, const string& shpFileName, const string& tableName)
{
  	if (!layer || shpFileName.empty() || tableName.empty())
		return false;

	// check if layer has the attribute table
	TeTable tableE;
	if (!layer->getAttrTablesByName(tableName,tableE))
		return false;
	string linkName = tableE.linkName();

	TeDatabasePortal* portalObjs = layer->database()->getPortal();
	if (!portalObjs)
		return false;

	vector<int> reps;
	if (layer->hasGeometry(TePOINTS))
		reps.push_back(SHPT_POINT);

	if (layer->hasGeometry(TePOLYGONS) || layer->hasGeometry(TeCELLS) )
		reps.push_back(SHPT_POLYGON);

	if (layer->hasGeometry(TeLINES))
		reps.push_back(SHPT_ARC);

	if (reps.empty())
	{
		delete portalObjs;
		return false;	// there are no geometrical representations
	}

	DBFHandle	hDBF;
    SHPHandle	hSHP;

	for (unsigned int i=0; i<reps.size(); i++)
	{
		string sql1 = "SELECT * FROM " + tableName;
		if (!portalObjs->query(sql1) || !portalObjs->fetchRow())
		{
			delete portalObjs;
			return false;
		}
		
		string dbfFilename;
		string shpFilename;
		int nShapeType = reps[i];
		switch (nShapeType)
		{
		case SHPT_POINT:
			dbfFilename = shpFileName + "_point.dbf";
			shpFilename = shpFileName + "_point.shp";
			break;
		case SHPT_POLYGON:
			dbfFilename = shpFileName + "_pol.dbf";
			shpFilename = shpFileName + "_pol.shp";
			break;
		case SHPT_ARC:
			dbfFilename = shpFileName + "_lin.dbf";
			shpFilename = shpFileName + "_lin.shp";
		}

		// create files
		hSHP = SHPCreate( shpFilename.c_str(), nShapeType );
		if( hSHP == 0 )
		{
			delete portalObjs;
			return false;
		}

		TeAttributeList attList = portalObjs->AttributeList();
		hDBF = TeCreateDBFFile(dbfFilename, attList);
		if( hDBF == 0 )
		{
			delete portalObjs; 
			return false;
		}

		TeDatabasePortal* portalGeoms = layer->database()->getPortal();
		if (!portalGeoms)
		{
			delete portalObjs;
			return false;
		}

		int count = 0;
		if(TeProgress::instance())
			TeProgress::instance()->setTotalSteps(portalObjs->numRows());
		clock_t	ti, tf;

		int iRecord = 0;
		string geoid, sql2;
		bool ok;
		int	 nVertices, nParts, *panParts;
		double	*padfX, *padfY;
		SHPObject	*psObject;
		int shpRes = 0;
		ti = clock();
		do 
		{
			geoid = portalObjs->getData(linkName);
			nVertices = 0;
			bool flag = true;
			TePointSet pos;
			TeLineSet lis;
			TePolygonSet pols;
			TeCellSet cells;
			switch (nShapeType)
			{
			case SHPT_POINT:		// create a shapefile of points
				sql2 = "SELECT * FROM " + layer->tableName(TePOINTS) + " WHERE object_id='" + geoid + "'";
				ok = false;			
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					do 
					{
						TePoint point;
						flag  = portalGeoms->fetchGeometry(point);
						pos.add(point);
						ok = true;
					} while (flag);
					nVertices = nParts = pos.size();
					panParts = (int *) malloc(sizeof(int) * nParts);
					padfX = (double *) malloc(sizeof(double) * nParts);
					padfY = (double *) malloc(sizeof(double) * nParts);
					for (int k=0;k<nParts;k++)
					{
						panParts[k] = k;
						padfX[k] = pos[k].location().x();
						padfY[k] = pos[k].location().y();
					}
					psObject = SHPCreateObject( nShapeType, -1, nParts, panParts, NULL,
												nVertices, padfX, padfY, NULL, NULL );
					shpRes = SHPWriteObject( hSHP, -1, psObject );
					SHPDestroyObject( psObject );
					free( panParts );
					free( padfX );
					free( padfY );
					pos.clear();
				}
				portalGeoms->freeResult();
				break;  
			case SHPT_ARC:	// create a shapefile of lines
				sql2 = "SELECT * FROM " + layer->tableName(TeLINES) + " WHERE object_id='" + geoid + "'";
				ok = false;
				int npoints;
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					npoints = 0;
					do
					{
						TeLine2D line;
						flag = portalGeoms->fetchGeometry(line);
						lis.add(line);
						ok = true;
						npoints += line.size();
					} while (flag);
					nVertices = lis.size();
					panParts = (int *) malloc(sizeof(int) * nVertices);
					padfX = (double *) malloc(sizeof(double) * npoints);
					padfY = (double *) malloc(sizeof(double) * npoints);
					int posXY = 0;
					for (int k=0; k<nVertices; k++)
					{
						int lineSize = lis[k].size();
						panParts[k]=posXY;
						for (int l=0; l<lineSize; l++ )
						{
							padfX[posXY] = lis[k][l].x();
							padfY[posXY] = lis[k][l].y();
							posXY++;
						}
					}
					lis.clear();
					psObject = SHPCreateObject( nShapeType, -1, nVertices, panParts, NULL,
												posXY, padfX, padfY, NULL, NULL );
					SHPWriteObject( hSHP, -1, psObject );
					SHPDestroyObject( psObject );
					free( panParts );
					free( padfX );
					free( padfY );
				}
				portalGeoms->freeResult();
				break;
			case SHPT_POLYGON: // create a shapefile of polygons
				sql2 = "SELECT * FROM " + layer->tableName(TePOLYGONS) + " WHERE object_id='" + geoid + "'";
				ok = false;
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					int totpoints;
					nVertices = 0;
					totpoints = 0;
					do
					{
						TePolygon poly;
						flag = portalGeoms->fetchGeometry(poly);
						pols.add(poly);
						ok = true;
						nVertices += poly.size();
						unsigned int n;
						for (n=0; n<poly.size();n++)
							totpoints += poly[n].size();
					} while (flag);
					panParts = (int *) malloc(sizeof(int) * nVertices);
					padfX = (double *) malloc(sizeof(double) * totpoints);
					padfY = (double *) malloc(sizeof(double) * totpoints);
					
					int posXY = 0;
					int npolygons, nrings, npoints, nelem = 0;
					npolygons = pols.size();
					for (int k=0; k<npolygons; k++)
					{
						nrings = pols[k].size();

						for (int l=0; l<nrings; ++l)
						{
							if (l==0)
							{
								if (TeOrientation(pols[k][l]) == TeCOUNTERCLOCKWISE)
									TeReverseLine(pols[k][l]);
							}
							else
							{
								if (TeOrientation(pols[k][l]) == TeCLOCKWISE)
									TeReverseLine(pols[k][l]);

							}
							
							npoints = pols[k][l].size();
							panParts[nelem]=posXY;
							for (int m=0; m<npoints; m++ )
							{
								padfX[posXY] = pols[k][l][m].x();
								padfY[posXY] = pols[k][l][m].y();
								posXY++;
							}
							nelem++;
						}
					}
					pols.clear();
					psObject = SHPCreateObject( nShapeType, -1, nelem, panParts, NULL,
												posXY, padfX, padfY, NULL, NULL );
					shpRes = SHPWriteObject( hSHP, -1, psObject );
					SHPDestroyObject( psObject );
					free( panParts );
					free( padfX );
					free( padfY );
				}
				portalGeoms->freeResult();

				// --- Cells are exported as polygons

				sql2 = "SELECT * FROM " + layer->tableName(TeCELLS) + " WHERE object_id='" + geoid + "'";
				if (portalGeoms->query(sql2) && portalGeoms->fetchRow())
				{
					int totpoints;
					nVertices = 0;
					totpoints = 0;
					do
					{
						TeCell cell;
						flag = portalGeoms->fetchGeometry(cell);
						cells.add(cell);
						ok = true;
						nVertices += 1;
						totpoints += 5;
					} while (flag);
					panParts = (int *) malloc(sizeof(int) * nVertices);
					padfX = (double *) malloc(sizeof(double) * totpoints);
					padfY = (double *) malloc(sizeof(double) * totpoints);
					
					int posXY = 0;
					int ncells, nelem = 0;
					ncells = cells.size();
					for (int k=0; k<ncells; k++)
					{
						panParts[nelem]=posXY;
						TeCell cell = cells[k];
						TeCoord2D ll = cell.box().lowerLeft();
						TeCoord2D ur = cell.box().upperRight();
						padfX[posXY] = ll.x();
						padfY[posXY] = ll.y();
						posXY++;		
						padfX[posXY] = ur.x();
						padfY[posXY] = ll.y();
						posXY++;		
						padfX[posXY] = ur.x();
						padfY[posXY] = ur.y();
						posXY++;		
						padfX[posXY] = ll.x();
						padfY[posXY] = ur.y();
						posXY++;		
						padfX[posXY] = ll.x();
						padfY[posXY] = ll.y();
						posXY++;	
						nelem++;
					}
					cells.clear();
					psObject = SHPCreateObject( nShapeType, -1, nelem, panParts, NULL,
												posXY, padfX, padfY, NULL, NULL );
					SHPWriteObject( hSHP, -1, psObject );
					SHPDestroyObject( psObject );
					free( panParts );
					free( padfX );
					free( padfY );
				}
				portalGeoms->freeResult();
			} // end - case geometry representation
		
			if (!ok)	// no geometry associated to this object
				continue;

			// write attributes
			for (int j = 0; j < portalObjs->numFields(); j++)
			{
				int c = attList[j].rep_.type_;
				if (c == TeSTRING)
				{
					DBFWriteStringAttribute(hDBF, iRecord, j, portalObjs->getData(j) );
				}
				else if (c == TeDATETIME)
				{
					TeTime time = portalObjs->getDate(j);
					char dd[8];
					sprintf(dd,"%04d%02d%02d",time.year(),time.month(),time.day());
					DBFWriteDateAttribute(hDBF, iRecord, j, dd );
				}
				else if (c == TeINT)
				{
					DBFWriteIntegerAttribute(hDBF, iRecord, j, portalObjs->getInt(j) );
				}
				else if (c == TeREAL)
				{
					DBFWriteDoubleAttribute(hDBF, iRecord, j, portalObjs->getDouble(j));
				}
			}
			iRecord++;
			count++;
			if (TeProgress::instance())
			{
				tf = clock();
				if (((tf-ti)/CLOCKS_PER_SEC) > 3)
					TeProgress::instance()->setProgress(count);
				if (TeProgress::instance()->wasCancelled())
					break;
				ti = tf;
			}
		} while (portalObjs->fetchRow()); // end - for every object
		DBFClose( hDBF );
		SHPClose( hSHP );
		portalObjs->freeResult();
		delete portalGeoms;
	} // end - for each representation
	delete portalObjs;
	if(TeProgress::instance())
		TeProgress::instance()->reset();
    return true ;
}

