/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeDecoderDatabase.h"
#include "TeRasterRemap.h"
#include "TeGeometryAlgorithms.h"
#include "TeSpatialOperations.h"
#include "TeImportRaster.h"
#include "TeDatabase.h"

#include <stdio.h>

typedef map<int,TeNode> TeNodeMap;

static const int scales [] =
{	150000,
	80000,
	40000,
	20000,
	10000,
	3000,
	1500,
	800,
	400,
	200,
	100,
	30,
	15,
	8,
	4,
	2,
	1
};

//-*********************

TeDatabase::TeDatabase() :
	isConnected_ (false),
	host_(""),
	user_(""),
	password_(""),
	database_(""),
	portNumber_(0),
	errorNumber_(0),
	errorMessage_("")
{
}


TeDatabase::~TeDatabase()
{
}

bool TeDatabase::validTable (TeTable& table)
{
	int cont=0;
	bool change = false;
	bool changeTable = false;
	bool uniqueName, linkName; 
	
	TeAttributeList::iterator it = table.attributeList().begin();
	TeAttributeList::iterator it2;  
	while(it != table.attributeList().end())
	{
		uniqueName = false;
		linkName = false;

		if((*it).rep_.name_==table.uniqueName())
			uniqueName=true;
		
		if((*it).rep_.name_==table.linkName())
			linkName=true;

		string errorMess;
		string temp = TeCheckName((*it).rep_.name_, change, errorMess);
		
		if(change)
		{
			it2 = table.attributeList().begin();
			while(it2!=table.attributeList().end())
			{
				if(temp==(*it2).rep_.name_)
				{
					temp += Te2String(cont);
					it2 = table.attributeList().begin();
					++cont;
				}
				else
					++it2;
			}

			changeTable = true;
		}

		if(change && uniqueName)
			table.setUniqueName(temp);
		if(change && linkName)
			table.setLinkName (temp);

		(*it).rep_.name_ = temp;
		++it;
		++cont;
	}
	
	return changeTable;
}

string TeDatabase::getTableName(int tableId)
{
	string tableName;

	TeDatabasePortal* pt = getPortal();
	string q = "SELECT attr_table FROM te_layer_table";
	q += " WHERE table_id = " + Te2String(tableId);
	if (pt->query(q) == true && pt->fetchRow())
		tableName = pt->getData("attr_table");
	delete pt;
	return tableName;
}

bool TeDatabase::createTable(const string& table, TeAttributeList &attr)
{
	string	q ="CREATE TABLE " + table +" (";
	string	type;
	char	size[8];
	bool	first=true;

	string pkeys;

	TeAttributeList::iterator it;
	it = attr.begin();
	while ( it != attr.end() )
	{
		string name = (*it).rep_.name_;
		
		switch ((*it).rep_.type_)
		{
			case TeSTRING:
				type = "VARCHAR ";
				sprintf (size,"(%d)",(*it).rep_.numChar_);
				type += string (size);
				break;
			case TeREAL:
				type = "DOUBLE";
				sprintf (size,"(%d,%d)",(*it).rep_.numChar_, (*it).rep_.decimals_);
				type += string (size);
				break;
			case TeINT:
				type = "INT ";
				break;
			case TeCHARACTER:
				type = "CHAR ";
				break;
			case TeDATETIME:
				type = "DATETIME ";  //time
				break;
			case TeBLOB:
				type = "BLOB ";
				break;
			default:
				type = "VARCHAR ";
				sprintf (size,"(%d)",(*it).rep_.numChar_);
				type += string (size);
				break;
		}

		if((*it).rep_.isAutoNumber_ && (*it).rep_.isPrimaryKey_)
			type += " AUTO_INCREMENT ";
				
		if (!first)
			q += ",";
		else
			first = false;
		
		q += name + " " + type;

		// check if column is part of primary key
		if ( (*it).rep_.isPrimaryKey_ && (*it).rep_.type_ != TeBLOB )
		{
			if (!pkeys.empty())
				pkeys += ", ";
			pkeys += (*it).rep_.name_;
			q += " NOT NULL ";
		}
		
		++it;
	}

	if (!pkeys.empty())
		q += ", PRIMARY KEY (" + pkeys + ") ";
	q += ")";

	return execute(q);
}

bool
TeDatabase::deleteTable (const string& table)
{
	int f =	table.find ("te_collection", std::string::npos);

	if( table=="te_theme" ||
		table=="te_layer" ||
		table=="te_representation" ||
		table=="te_tables_relation" ||
		table=="te_layer_table" ||
		table=="te_raster_metadata" ||
		table=="te_projection" ||
		table=="te_view" ||
		table=="te_legend" ||
		table=="te_visual" ||
		table=="te_database" ||
		f == 0)
	{
		errorMessage_ = "No  possvel deletar tabelas do modelo!";
		return false;
	}

	string del = "DROP TABLE " + table;
	if(!execute(del))
		return false;

	return true;
}


bool 
TeDatabase::deleteColumn (const string& table, const string& colName)
{
	if (!tableExist(table))
		return false;
	TeAttribute attr;
	if (!columnExist(table,colName,attr))
		return true;
	string drop = "ALTER  TABLE " + table + " DROP COLUMN " + colName;
	if(execute(drop) == false)
		return false;

	string tableId;
	TeDatabasePortal* portal = getPortal();
	string sql = "SELECT table_id FROM te_layer_table WHERE attr_table = '" + table + "'";
	if(portal->query(sql) && portal->fetchRow())
		tableId = portal->getData(0);

	delete portal;
	if(tableId.empty() == false)
	{
		// delete relation
		sql = "DELETE FROM te_tables_relation WHERE (related_table_id = " + tableId;
		sql += " AND related_attr = '" + colName + "')";
		sql += " OR (external_table_name = '" + table + "'";
		sql += " AND external_attr = '" + colName + "')";
		if(execute(sql) == false)
			return false;

		// delete grouping
		TeDatabasePortal* portal = getPortal();
		sql = "SELECT theme_id FROM te_grouping WHERE grouping_attr = '" + table + "." + colName + "'";
		if(portal->query(sql) && portal->fetchRow())
		{
			string themeId = portal->getData(0);

			sql = "DELETE FROM te_legend WHERE theme_id = " + themeId + " AND group_id >= 0";
			if(execute(sql) == false)
			{
				delete portal;
				return false;
			}
		}
		delete portal;

		sql = "DELETE FROM te_grouping";
		sql += " WHERE grouping_attr = '" + table + "." + colName + "'";
		if(execute(sql) == false)
			return false;
	}
	return true;
}


bool 
TeDatabase::defineIntegrity(void)
{
	if (existRelation("te_layer","fk_layer_proj_id") == TeNoRelation )
		if (!createRelation("fk_layer_proj_id", "te_layer", "projection_id", "te_projection", "projection_id", false))
			return false;
        
	if (existRelation("te_representation","fk_rep_layer_id") == TeNoRelation )
		if (!createRelation("fk_rep_layer_id", "te_representation", "layer_id", "te_layer", "layer_id", true))
			return false;

	if (existRelation("te_view","fk_view_proj_id") == TeNoRelation )
		if (!createRelation("fk_view_proj_id", "te_view", "projection_id", "te_projection", "projection_id", false))
			return false;

	if (existRelation("te_theme","fk_theme_layer_id") == TeNoRelation )
		if (!createRelation("fk_theme_layer_id", "te_theme", "layer_id", "te_layer", "layer_id", true))
			return false;

	if (existRelation("te_theme","fk_theme_view_id") == TeNoRelation )
		if (!createRelation("fk_theme_view_id", "te_theme", "view_id", "te_view", "view_id", true))
			return false;

	if (existRelation("te_theme_table","fk_thmtable_theme_id") == TeNoRelation )
		if (!createRelation("fk_thmtable_theme_id", "te_theme_table", "theme_id", "te_theme", "theme_id", true))
			return false;

	if (existRelation("te_theme_table","fk_thmtable_lytable_id") == TeNoRelation )
		if (!createRelation("fk_thmtable_lytable_id", "te_theme_table", "table_id", "te_layer_table", "table_id", true))
			return false;

	if (existRelation("te_theme_table","fk_thmtable_relation_id") == TeNoRelation )
		if (!createRelation("fk_thmtable_relation_id", "te_theme_table", "relation_id", "te_tables_relation", "relation_id", true))
			return false;

	if (existRelation("te_grouping","fk_group_theme_id") == TeNoRelation )
		if (!createRelation("fk_group_theme_id", "te_grouping", "theme_id", "te_theme", "theme_id", true))
			return false;

	if (existRelation("te_legend","fk_legend_theme_id") == TeNoRelation )
		if (!createRelation("fk_legend_theme_id", "te_legend", "theme_id", "te_theme", "theme_id", true))
			return false;

	if (existRelation("te_visual","fk_visual_legend_id") == TeNoRelation )
		if (!createRelation("fk_visual_legend_id", "te_visual", "legend_id", "te_legend", "legend_id", true))
			return false;
        
	if (existRelation("te_layer_table","fk_laytable_layer_id") == TeNoRelation )
		if (!createRelation ("fk_laytable_layer_id", "te_layer_table", "layer_id", "te_layer", "layer_id", true))
			return false;
        
	if (existRelation("te_tables_relation","fk_tabrelation_laytable_id") == TeNoRelation )
		if (!createRelation("fk_tabrelation_laytable_id", "te_tables_relation", "related_table_id", "te_layer_table", "table_id", true))
			return false;

	if (existRelation("te_visual_raster","fk_visrast_theme_id") == TeNoRelation )
		if (!createRelation("fk_visrast_theme_id", "te_visual_raster", "theme_id", "te_theme", "theme_id", true))
			return false;

	return true;
}

bool 
TeDatabase::createConceptualModel(bool withIntegrity, bool newDatabase, bool /* createIndex */)
{
	bool status = true;
	bool createMainTables = false;
	
	if (!this->tableExist("te_projection"))
	{
		status = this->createProjectionTable();
		if (!status)
			return false;	
		createMainTables = true;
	}

	if (!this->tableExist("te_layer"))
	{
		status = this->createLayerTable();
		if (!status)
			return false;
		createMainTables = true;
	}

	if (!this->tableExist("te_layer_table"))
	{
		status = this->createLayerTableTable();
		if (!status)
			return false;
		createMainTables = true;
	}

	if (!this->tableExist("te_tables_relation"))
	{
		status = this->createTablesRelationTable();
		if (!status)
			return false;
	}
	
	if (!this->tableExist("te_representation"))
	{
		status = this->createRepresentationTable();
		if (!status)
			return false;
		createMainTables = true;
	}

	if (!this->tableExist("te_view"))
	{
		status = this->createViewTable();
		if (!status)
			return false;
		createMainTables = true;
	}

	if (!this->tableExist("te_theme"))
	{
		status = this->createThemeTable();
		if (!status)
			return false;
	}

	if (!this->tableExist("te_grouping"))
	{
		status = this->createGroupingTable();
		if (!status)
			return false;
	}

	if (!this->tableExist("te_theme_table"))
	{
		status = this->createThemeTablesTable();
		if (!status)
			return false;
	}

	if (!this->tableExist("te_legend"))
	{
		status = this->createLegendTable();
		if (!status)
			return false;	
	}

	if (!this->tableExist("te_visual"))
	{
		status = this->createVisualTable();
		if (!status)
			return false;	
	}

	if (!this->tableExist("te_visual_raster"))
	{
		status = this->createVisualRasterTable();
		if (!status)
			return false;	
	}

	if (!this->tableExist("te_database"))
	{
		status = this->createDatabaseTable();
		if (!status)
			return false;
	}

	if(newDatabase || createMainTables)
	{
		string ins = "INSERT INTO te_database (db_version) VALUES ('3.0.2')";
		if (!execute(ins))
			return false;
	}

	if (withIntegrity)
		status = defineIntegrity();

	return status;
}

bool
TeDatabase::createIndex(const string& tableName, const string& indexName, const string& columnsName)
{
	string sql = "CREATE INDEX " + indexName + " ON " + tableName + "(" + columnsName + ")";	
	return execute (sql);
}

bool 
TeDatabase::createDatabaseTable()
{
	string create = "CREATE TABLE te_database (";
	create += "db_version		VARCHAR(50) NOT NULL,";
	create += "db_creation		DATETIME,";
	create += "PRIMARY KEY (db_version))";
	return execute (create);
}


bool 
TeDatabase::createProjectionTable ()
{
	string create = "CREATE TABLE te_projection (";
	create += "projection_id	INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += "name				VARCHAR(50) NOT NULL,";
	create += "long0			DOUBLE(24,15) DEFAULT '0.0',";
	create += "lat0				DOUBLE(24,15) DEFAULT '0.0',";
	create += "offx				DOUBLE(24,15) DEFAULT '0.0',";
	create += "offy				DOUBLE(24,15) DEFAULT '0.0',";
	create += "stlat1			DOUBLE(24,15) DEFAULT '0.0',";
	create += "stlat2			DOUBLE(24,15) DEFAULT '0.0',";
	create += "unit				VARCHAR(50) NOT NULL,";
	create += "scale			DOUBLE(24,15) DEFAULT '0.0',";
	create += "hemis			INT(4) NOT NULL,";
	create += "datum			VARCHAR(50) NOT NULL,";
	create += "radius			DOUBLE(24,15) DEFAULT '0.0',";
	create += "flattening		DOUBLE(24,15) DEFAULT '0.0',";
	create += "dx				DOUBLE(24,15) DEFAULT '0.0',";
	create += "dy				DOUBLE(24,15) DEFAULT '0.0',";
	create += "dz				DOUBLE(24,15) DEFAULT '0.0',";
	create += "PRIMARY KEY (projection_id))";
	return execute (create);
}

bool TeDatabase::createLayerTable ()
{
	string create = "CREATE TABLE te_layer (";
	create += "layer_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += "projection_id	INT(10) UNSIGNED NOT NULL,";
	create += "name				VARCHAR(255) NOT NULL,";
	create += "lower_x			DOUBLE(24,15) DEFAULT '0.0' NOT NULL,";
	create += "lower_y			DOUBLE(24,15) DEFAULT '0.0' NOT NULL,";
	create += "upper_x			DOUBLE(24,15) DEFAULT '0.0' NOT NULL,";
	create += "upper_y			DOUBLE(24,15) DEFAULT '0.0' NOT NULL,";
	create += "initial_time		DATETIME,";
	create += "final_time		DATETIME, ";
	create += "PRIMARY KEY (layer_id))";
	return execute (create);
}

bool TeDatabase::createLayerTableTable()
{
	string create = "CREATE TABLE te_layer_table (";
	create += "table_id				INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += "layer_id				INT(10) UNSIGNED,";
	create += "attr_table			VARCHAR(255) NOT NULL,";
	create += "unique_id			VARCHAR(255), ";
	create += "attr_link			VARCHAR(255), ";
	create += "attr_initial_time	VARCHAR(255), ";
	create += "attr_final_time		VARCHAR(255), ";
	create += "attr_time_unit		INT(4) UNSIGNED, ";
	create += "attr_table_type		INT(4) UNSIGNED, ";
	create += "user_name			VARCHAR(255),";
	create += "initial_time			DATETIME,";
	create += "final_time			DATETIME, ";
	create += "PRIMARY KEY (table_id))";
	return execute (create);
}

bool 
TeDatabase::updateTableInfo(int layerId, TeTable &table, const string user)
{
	if (table.id() <= 0 )  // table information doesnt exist in database yet
		return this->insertTableInfo(layerId,table,user);
	string sql;
	sql = "UPDATE te_layer_table SET ";
	sql += "unique_id='" + table.uniqueName() + "', ";
	sql += "attr_link='" + table.linkName() + "', ";
	sql += "attr_initial_time='" + table.attInitialTime() + "', ";
	sql += "attr_final_time='" + table.attFinalTime() + "', ";
	sql += "attr_time_unit=" + Te2String(table.attTimeUnit()) + ", ";
	sql += "attr_table_type="+ Te2String(table.tableType()) + ", ";
	sql += "user_name='" + user + "' WHERE ";
	sql += "layer_id=" + Te2String(layerId) + " AND ";
	sql += "attr_table_name='" + table.name() + "'";
	return execute (sql);
}

bool 
TeDatabase::loadTableInfo(TeTable& table)
{
	if (table.name().empty())
		return false;

	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
	{
		this->errorMessage_ = "No foi possvel abrir portal para o banco";
		return false;
	}
	
	string get =  "SELECT * FROM te_layer_table WHERE attr_table = '" + table.name() + "'";
	if (!portal->query(get) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	table.setId(portal->getInt("table_id"));
	table.setLinkName(portal->getData("attr_link"));
	table.setUniqueName(portal->getData("unique_id"));
	table.setTableType(TeAttrTableType(portal->getInt("attr_table_type")));
	TeAttributeList attrList;
	getAttributeList(table.name(), attrList);
	table.setAttributeList(attrList);
	table.attInitialTime(portal->getData("attr_initial_time"));
	table.attFinalTime(portal->getData("attr_final_time"));
	table.attTimeUnit(TeChronon(portal->getInt("attr_time_unit")));
	delete portal;
	return true;
}


bool TeDatabase::createLUTTable(const string& name)
{
	string create = "CREATE TABLE " + name + "(";
	create += "index_id		INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += "r_val		INT(10) UNSIGNED NOT NULL,";
	create += "g_val		INT(10) NOT NULL,";
	create += "b_val		INT(10) NOT NULL,";
	create += "PRIMARY KEY (index_id))";
	return execute (create);
}

bool TeDatabase::createTablesRelationTable()
{
	string create = "CREATE TABLE te_tables_relation (";
	create += "relation_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += "related_table_id		INT(10) NOT NULL,";
	create += "related_attr			VARCHAR(255) NOT NULL,";
	create += "external_table_name	VARCHAR(255) NOT NULL,";
	create += "external_attr		VARCHAR(255) NOT NULL,";
	create += "PRIMARY KEY (relation_id))";
	return execute (create);
}

bool TeDatabase::createRepresentationTable ()
{
	string create = "CREATE TABLE te_representation (";
	create += "repres_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT, ";
	create += "layer_id				INT(10) UNSIGNED NOT NULL, ";
	create += "geom_type			INT(10) NOT NULL, ";
	create += "geom_table			VARCHAR(255) NOT NULL, ";
	create += "description			VARCHAR(255), ";
	create += "lower_x				DOUBLE(24,15) DEFAULT '0.0', ";
	create += "lower_y				DOUBLE(24,15) DEFAULT '0.0', ";
	create += "upper_x				DOUBLE(24,15) DEFAULT '0.0', ";
	create += "upper_y				DOUBLE(24,15) DEFAULT '0.0', ";
	create += "res_x				DOUBLE(24,15) DEFAULT '0.0', ";
	create += "res_y				DOUBLE(24,15) DEFAULT '0.0', ";
	create += "num_cols				INT(10), ";
	create += "num_rows				INT(10), ";
	create += "initial_time			DATETIME, ";
	create += "final_time			DATETIME, ";
	create += "PRIMARY KEY (repres_id), ";
	create += "INDEX representation_index (layer_id,geom_type))";
	return execute (create);
}

bool TeDatabase::createRasterMetadataTable(const string& tableName)
{
	if (tableName.empty())
		return false;

	string create = "CREATE TABLE " + tableName + " (";
	create += " geom_id				INT(10) UNSIGNED NOT NULL, ";
	create += " band_id				INT(10) UNSIGNED NOT NULL, ";
	create += " min_value			DOUBLE(24,15) DEFAULT '0.0', ";
	create += " max_value			DOUBLE(24,15) DEFAULT '0.0', ";
	create += " num_bits			INT(10), ";
	create += " data_type			INT(4), ";
	create += " photometric_type	INT(4), ";
	create += " compression_type	INT(4), ";
	create += " dummy				DOUBLE(24,15), ";
	create += " PRIMARY KEY (geom_id, band_id))";
	return execute (create);
}


bool TeDatabase::createRasterGeometry(const string& tableName)
{
	if (tableName.empty())
		return false;

	string create = "CREATE TABLE " + tableName + " (";
	create += "geom_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT, ";
	create += "object_id		VARCHAR(255) NOT NULL, ";
	create += "raster_table		VARCHAR(255) NOT NULL, ";
	create += "lut_table		VARCHAR(255), ";
	create += "res_x			DOUBLE(24,15) DEFAULT '0.0', ";
	create += "res_y			DOUBLE(24,15) DEFAULT '0.0', ";
	create += "num_bands		INT(10), ";
	create += "num_cols			INT(10), ";
	create += "num_rows			INT(10), ";
	create += "block_height		INT(10), ";
	create += "block_width		INT(10), ";
	create += "lower_x			DOUBLE(24,15) DEFAULT '0.0', ";
	create += "lower_y			DOUBLE(24,15) DEFAULT '0.0', ";
	create += "upper_x			DOUBLE(24,15) DEFAULT '0.0', ";
	create += "upper_y			DOUBLE(24,15) DEFAULT '0.0', ";
	create += "tiling_type      INT(4), ";
	create += "PRIMARY KEY (geom_id), ";
	create += "INDEX " + tableName + "_index(object_id))";
	return execute (create);
}

bool TeDatabase::createViewTable ()
{
	string create = "CREATE TABLE te_view (";
	create += "view_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += "projection_id	INT(10) UNSIGNED NOT NULL,";
	create += "name				VARCHAR(255) NOT NULL,";
	create += "user_name		VARCHAR(255),";
	create += "visibility		INT(4),";
	create += "PRIMARY KEY (view_id))";
	return execute (create);
}

bool TeDatabase::createThemeTable ()
{
	string create = "CREATE TABLE te_theme (";
	create += " theme_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += " layer_id			INT(10) UNSIGNED,";
	create += " view_id				INT(10) UNSIGNED NOT NULL,";
	create += " name				VARCHAR(255) NOT NULL,";
	create += " parent_id			INT(10) UNSIGNED NOT NULL,";
	create += " priority			INT(10) UNSIGNED NOT NULL,";
	create += " node_type			INT(4) UNSIGNED NOT NULL,";
	create += " min_scale			DOUBLE(24,15),";
	create += " max_scale			DOUBLE(24,15),";
	create += " generate_attribute_where	VARCHAR(255),";
	create += " generate_spatial_where		VARCHAR(255),";
	create += " generate_temporal_where		VARCHAR(255),";
	create += " collection_table			VARCHAR(255),";
	create += " visible_rep					INT(10) UNSIGNED,";
	create += " enable_visibility			INT(10) UNSIGNED,";
	create += " PRIMARY KEY (theme_id),";
	create += " INDEX view_id_idx(view_id))";
	return execute (create);
}

bool TeDatabase::createGroupingTable()
{
	string create = "CREATE TABLE te_grouping (";
	create += " theme_id			INT(10) UNSIGNED NOT NULL,";
	create += " grouping_number		INT(10) UNSIGNED,";
	create += " grouping_attr		VARCHAR(255),";
	create += " grouping_attr_type	INT(4) UNSIGNED,";
	create += " grouping_mode		INT(4) UNSIGNED,";
	create += " grouping_norm_attr	VARCHAR(255),";
	create += " grouping_std_dev	DOUBLE(24,15) DEFAULT '0.0',";
	create += " grouping_precision	INT(10) UNSIGNED,";
	create += " grouping_function	VARCHAR(20),";
	create += " PRIMARY KEY (theme_id))";
	return execute (create);
}

bool TeDatabase::createThemeTablesTable()
{
	string create = "CREATE TABLE te_theme_table  (";
	create += " theme_table_id		INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += " theme_id			INT(10) UNSIGNED NOT NULL,";
	create += " table_id			INT(10) UNSIGNED NOT NULL,";
	create += " relation_id			INT(10) UNSIGNED,";
	create += " table_order			INT(10) UNSIGNED,";
	create += " PRIMARY KEY (theme_table_id), ";
	create += " INDEX theme_table_index(theme_id))";

	return execute (create);
}

bool TeDatabase::createLegendTable ()
{
	string create = "CREATE TABLE te_legend (";
	create += " legend_id		INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	create += " theme_id		INT(10) UNSIGNED NOT NULL,";
	create += " group_id		INT(10) ,";
	create += " num_objs		INT(10) UNSIGNED,";
	create += " lower_value		VARCHAR(255),";
	create += " upper_value		VARCHAR(255),";
	create += " label			VARCHAR(255), ";
	create += " PRIMARY KEY (legend_id),";
	create += " INDEX legend_index (theme_id))";
	return execute (create);
}


bool TeDatabase::createVisualTable()
{
	string create = "CREATE TABLE te_visual (";
	create += " legend_id		INT(10) UNSIGNED NOT NULL,";
	create += " geom_type		INT(4) UNSIGNED NOT NULL,";
	create += " lib_name		VARCHAR(255), ";
	create += " symb_id			INT(10), ";
	create += " red				INT(4) UNSIGNED,";
	create += " green			INT(4) UNSIGNED,";
	create += " blue			INT(4) UNSIGNED,";
	create += " transparency	INT(4) UNSIGNED,";
	create += " width			INT(4) UNSIGNED,";
	create += " contour_lib_name	VARCHAR(255), ";
	create += " contour_symb_id		INT(10), ";
	create += " contour_red			INT(4) UNSIGNED,";
	create += " contour_green		INT(4) UNSIGNED,";
	create += " contour_blue		INT(4) UNSIGNED,";
	create += " contour_transp		INT(4) UNSIGNED,";
	create += " contour_width		INT(10) UNSIGNED,";
	create += " size_value			INT(10) UNSIGNED,";
	create += " pt_angle			INT(4) UNSIGNED,";
	create += " family				VARCHAR(255),";
	create += " bold				INT(4) UNSIGNED,";
	create += " italic				INT(4) UNSIGNED,";
	create += " alignment_vert		DOUBLE(24,15),";
	create += " alignment_horiz		DOUBLE(24,15),";
	create += " tab_size			INT(4) UNSIGNED,";
	create += " line_space			INT(4) UNSIGNED,";
	create += " fixed_size			INT(4) UNSIGNED,";
	create += " PRIMARY KEY (legend_id, geom_type))";
	return execute (create);
}

bool TeDatabase::createVisualRasterTable()
{
	string create = "CREATE TABLE te_visual_raster (";
	create += " theme_id		INT(10) UNSIGNED NOT NULL,";
	create += " band_in			INT(4) UNSIGNED NOT NULL,";
	create += " band_out		INT(4) UNSIGNED, ";
	create += " transf_type		INT(4), ";
	create += " param1			DOUBLE(24,15),";
	create += " param2			DOUBLE(24,15),";
	create += " lut_table		VARCHAR(255), ";
	create += " PRIMARY KEY (theme_id, band_in))";
	return execute (create);
}

bool TeDatabase::insertVisual (TeLegendEntry *legend)
{
	TeGeomRepVisualMap mapVis = legend->getVisualMap();
	TeGeomRepVisualMap::iterator it =  mapVis.begin();
	while ( it != mapVis.end())
	{ 
		TeGeomRep rep = it->first;
		TeVisual vis = it->second;
		
		string style, contourStyle, sizeValue, width;

		if(rep == TePOLYGONS || rep == TeCELLS)
		{
			sizeValue = Te2String(0);
			contourStyle = Te2String(vis.contourStyle());
			width =  Te2String(0);
			style = Te2String(vis.style());
		}
		else if(rep == TeLINES)
		{
			sizeValue = Te2String(0);
			contourStyle = Te2String(0);
			width = Te2String(vis.width());
			style = Te2String(vis.style());
		}
		else if(rep == TePOINTS)
		{
			sizeValue = Te2String(vis.size());
			contourStyle = Te2String(0);
			width = Te2String(0);
			style = Te2String(vis.style());
		}
		else if(rep == TeTEXT)
		{
			sizeValue = Te2String(vis.size());
			contourStyle = Te2String(0);
			width = Te2String(0);
			style = Te2String(0);
		}
		
		string insert = "INSERT INTO te_visual(legend_id, geom_type, ";
		insert += "lib_name, symb_id, "; 
		insert += "red, green, blue, transparency, width, ";
		insert += "contour_lib_name, contour_symb_id, "; 
		insert += "contour_red, contour_green,";
		insert += "contour_blue, contour_transp, contour_width, size_value,";
		insert += "pt_angle, family, bold, italic, ";
		insert += "alignment_vert, alignment_horiz, tab_size, line_space, fixed_size) ";
		insert += " VALUES (";	
		insert += Te2String(legend->id()) + ", ";
		insert += Te2String(rep)+ ", ";

		TeColor cor = vis.color();				// filling color
		TeColor contourCor = vis.contourColor();// contour color
		
		insert += "'"+ vis.libName() + "', ";
		insert += style + ", ";
		insert += Te2String(cor.red_) + ", ";
		insert += Te2String(cor.green_) + ", ";
		insert += Te2String(cor.blue_) + ", ";
		insert += Te2String(vis.transparency()) + ", ";
		insert += width +",";

		insert += "'"+ vis.contourLibName() + "', ";
		insert += contourStyle + ", ";
		insert += Te2String(contourCor.red_) + ", ";
		insert += Te2String(contourCor.green_) + ", ";
		insert += Te2String(contourCor.blue_) + ", ";
		insert += Te2String(vis.contourTransparency()) + ", ";	
		insert += Te2String(vis.contourWidth()) + ", ";

		insert += sizeValue +",";
		insert += Te2String(vis.ptAngle()) +", ";

		insert += "'" + vis.family() + "', ";
		if (vis.bold())
			insert += "1, ";
		else
			insert += "0, ";
		if (vis.italic())
			insert += "1, ";
		else
			insert += "0, ";
		insert += Te2String(vis.alignmentVert()) + ",";
		insert += Te2String(vis.alignmentHoriz()) + ",";
		insert += Te2String(vis.tabSize()) + ",";
		insert += Te2String(vis.lineSpace()) +",";
		if (vis.fixedSize())
			insert += "1 ";
		else
			insert += "0";			
		insert += ")";

		if (!execute(insert))
			return false;
		++it;
	}
	return true;
}

bool TeDatabase::createCollectionTable(const string& tableName)
{
	string create = "CREATE TABLE " + tableName + " (";
	create += " c_object_id			VARCHAR(255) NOT NULL, ";
	create += " c_legend_id			INT(10), ";
	create += " label_x				DOUBLE(24,9), ";
	create += " label_y				DOUBLE(24,9), ";
	create += " c_legend_own		INT(10), ";
	create += " c_object_status		INT(10), ";
	create += " PRIMARY KEY (c_object_id))";
	return execute (create);
}

bool TeDatabase::createCellGeometry (const string& table)
{
	string q ="CREATE TABLE " + table +"(";
	q += "geom_id				INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	q += "object_id				VARCHAR(255) NOT NULL ,";
	q += "lower_x				double(24,15) NOT NULL ,";
	q += "lower_y				double(24,15) NOT NULL ,";
	q += "upper_x				double(24,15) NOT NULL ,";
	q += "upper_y				double(24,15) NOT NULL ,";
	q += "col_number			INT(10) NOT NULL ,";
	q += "row_number			INT(10) NOT NULL ,";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX box_idx (lower_x, lower_y,upper_x, upper_y),";
	q += "INDEX lc (row_number, col_number))";
	return execute(q);
}

bool TeDatabase::createTextGeometry(const string& table)
{
	string q ="CREATE TABLE " + table +"(";
	q += " geom_id			INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	q += " object_id		VARCHAR(255) NOT NULL ,";
	q += " x				DOUBLE(24,9) DEFAULT '0.0',";
	q += " y				DOUBLE(24,9) DEFAULT '0.0',";
	q += " text_value		VARCHAR(255),";
	q += " angle			DOUBLE(24,9) DEFAULT '0.0',";
	q += " height			DOUBLE(24,9) DEFAULT '0.0',";
	q += " alignment_vert	DOUBLE(24,9),";
	q += " alignment_horiz	DOUBLE(24,9),";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX pos_idx(x,y))";
	return execute(q);
}

bool TeDatabase::createNodeGeometry(const string& table)
{
	string q ="CREATE TABLE " + table +"(";
	q += "geom_id		INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	q += "object_id		VARCHAR(255) NOT NULL ,";
	q += "x				DOUBLE(24,15) DEFAULT '0.0',";
	q += "y				DOUBLE(24,15) DEFAULT '0.0',";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX pos_idx(x,y))";
	return execute(q);
}

bool TeDatabase::createPointGeometry(const string& table)
{
	string q ="CREATE TABLE " + table +"(";
	q += "geom_id		INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	q += "object_id		VARCHAR(255) NOT NULL ,";
	q += "x				DOUBLE(24,15) DEFAULT '0.0',";
	q += "y				DOUBLE(24,15) DEFAULT '0.0',";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX pos_idx(x,y))";
	return execute(q);
}

bool TeDatabase::createArcGeometry(const string& table)
{
	string q ="CREATE TABLE " + table +"(";
	q += "geom_id		INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,";
	q += "object_id		VARCHAR(255) NOT NULL ,";
	q += "from_node		INT(10),";
	q += "to_node		INT(10),";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX (from_node),";
	q += "INDEX (to_node))";
	return execute(q);
}


bool
TeDatabase::insertTable	(TeTable &table)
{
	string tableName = table.name();
	TeAttributeList att = table.attributeList();
	TeAttributeList::iterator it = att.begin();

	TeTableRow row;
	unsigned int i;
	unsigned int j;
	for ( i = 0; i < table.size(); i++  )
	{
		row = table[i];
		it = att.begin();
		string q = 	"INSERT INTO "+tableName+" values(";
		j = 1;
		int jj = 0;
		string dt = "";
		while ( it != att.end() )
		{
			if(((*it).rep_.type_==TeDATETIME) && (!row[jj].empty()))
			{
				const string temp_dt = string(row[jj].c_str());
				TeTime t(temp_dt, (*it).dateChronon_, (*it).dateTimeFormat_, (*it).dateSeparator_, (*it).timeSeparator_, (*it).indicatorPM_);
				dt=t.getDateTime("YYYYsMMsDDsHHsmmsSS", "-");
			}

  			switch ((*it).rep_.type_)
  			{
  				case TeSTRING:
					q += "'"+this->escapeSequence(row[jj])+"'";
  				break;
  				case TeREAL:
					q += "'"+row[jj]+"'";
  				break;
  				case TeINT:
					q += "'"+row[jj]+"'";
  				break;
				case TeDATETIME:
					q += "'"+dt+"'";   
  				break;
				default:
					q += "'"+this->escapeSequence(row[jj])+"'";
  				break;
  			}
			if (j<att.size())
				q+= ",";
			++it;
			j++;
			jj++;
		}
		q += ")";
		if (!this->execute(q))
//			return false;
			continue;
	}
	return true;
}

bool 
TeDatabase::alterTable(const string& tableName, TeAttributeRep& rep, const string& oldColName)
{
	if(!tableExist(tableName))
		return false;

	string tab;

	if(oldColName.empty())
	{
		tab = " ALTER TABLE " + tableName + " MODIFY ";
		tab += rep.name_ +"  ";
	}
	else
	{
		tab = " ALTER TABLE " + tableName + " CHANGE ";
		tab += oldColName +" "+ rep.name_ +"  ";
	}

	switch (rep.type_)
	{
		case TeSTRING:
			tab += "VARCHAR(" + Te2String(rep.numChar_) + ") ";
			break;
			
		case TeREAL:
			tab += "DOUBLE";	
			break;
			
		case TeINT:
			tab += "INT";
			break;

		case TeDATETIME:
			tab += "DATETIME";
			break;

		case TeCHARACTER:
			tab += "CHAR";
			break;

		case TeBLOB:
			tab += "LONGBLOB";
			break; 
		
		default:
			tab += "VARCHAR(" + Te2String(rep.numChar_) + ") ";
			break;
	}

	tab += " NULL ";

	if(!execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error alter table " + tableName + " !";
		return false;
	}

	string tableId;
	TeDatabasePortal* portal = getPortal();
	string sql = "SELECT table_id FROM te_layer_table WHERE attr_table = '" + tableName + "'";
	if(portal->query(sql) && portal->fetchRow())
		tableId = portal->getData(0);

	delete portal;

	if(tableId.empty() == false)
	{
		if(oldColName.empty() == false) // column name changed
		{
			 // update relation
			sql = "UPDATE te_tables_relation SET related_attr = '" + rep.name_ + "'";
			sql += " WHERE related_table_id = " + tableId;
			sql += " AND related_attr = '" + oldColName + "'";
			if(execute(sql) == false)
				return false;

			sql = "UPDATE te_tables_relation SET external_attr = '" + rep.name_ + "'";
			sql += " WHERE external_table_name = '" + tableName + "'";
			sql += " AND external_attr = '" + oldColName + "'";
			if(execute(sql) == false)
				return false;

			 // update grouping
			sql = "UPDATE te_grouping SET grouping_attr = '" + tableName + "." + rep.name_ + "'";
			sql += " WHERE grouping_attr = '" + tableName + "." + oldColName + "'";
			if(execute(sql) == false)
				return false;
		}
		else // column type changed
		{
			// delete relation
			sql = "DELETE FROM te_tables_relation WHERE (related_table_id = " + tableId;
			sql += " AND related_attr = '" + rep.name_ + "')";
			sql += " OR (external_table_name = '" + tableName + "'";
			sql += " AND external_attr = '" + rep.name_ + "')";
			if(execute(sql) == false)
				return false;

			// delete grouping
			TeDatabasePortal* portal = getPortal();
			sql = "SELECT theme_id FROM te_grouping WHERE grouping_attr = '" + tableName + "." + oldColName + "'";
			if(portal->query(sql) && portal->fetchRow())
			{
				string themeId = portal->getData(0);

				sql = "DELETE FROM te_legend WHERE theme_id = " + themeId + " AND group_id >= 0";
				if(execute(sql) == false)
				{
					delete portal;
					return false;
				}
			}
			delete portal;

			sql = "DELETE FROM te_grouping";
			sql += " WHERE grouping_attr = '" + tableName + "." + oldColName + "'";
			if(execute(sql) == false)
				return false;
		}
	}
	return true;
}

bool 
TeDatabase::getAttrTables(TeAttrTableVector& atts, TeAttrTableType attType)
{
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
	{
		this->errorMessage_ = "No foi possvel abrir portal para o banco";
		return false;
	}
	
	// Get layer tables	
	string get =  " SELECT * FROM te_layer_table"; 
	if (attType != TeAllAttrTypes)
		get += " WHERE attr_table_type = " + Te2String(attType);
	if (!portal->query(get))
	{	
		delete portal;
		return false;
	}

	while (portal->fetchRow())
	{
		string tableName = portal->getData("attr_table");
		int    tableId = portal->getInt("table_id");
		string indexName = portal->getData("attr_link");

		TeAttributeList attrList;
		getAttributeList(tableName, attrList);

		TeTable attTable(tableName,attrList,
				         portal->getData("unique_id"), indexName,
						 TeAttrTableType(portal->getInt("attr_table_type")));

		attTable.setId(tableId);
		attTable.attInitialTime(portal->getData("attr_initial_time"));
		attTable.attFinalTime(portal->getData("attr_final_time"));
		attTable.attTimeUnit(TeChronon(portal->getInt("attr_time_unit")));
		atts.push_back(attTable);
	}
	delete portal;
	return (atts.size() > 0);
}

bool
TeDatabase::updateTable	(TeTable &table)
{
	string tableName = table.name();
	TeAttributeList att = table.attributeList();
	TeAttributeList::iterator it = att.begin();

	TeTableRow row;
	unsigned int i;
	string uniqueName = table.uniqueName();
	string uniqueVal;
	bool isUniqueValString = false;

	for ( i = 0; i < table.size(); i++  )
	{
		row = table[i];
		it = att.begin();
		string q = 	"UPDATE "+tableName+" SET ";
		unsigned int jj = 0;
		while ( it != att.end() )
		{
			string val = row[jj];
			if(row[jj].empty())
				val = " NULL ";
			
			if ((*it).rep_.name_ != uniqueName)
			{
				q += (*it).rep_.name_ + "=";
  				switch ((*it).rep_.type_)
  				{
  					case TeSTRING:
						q += "'"+escapeSequence(row[jj])+"'";
  					break;
  					case TeREAL:
						q += val;
  					break;
  					case TeINT:
						q += val;
  					break;
  					default:
						q += "'"+escapeSequence(row[jj])+"'";
  					break;
  				}
				if (jj < att.size()-1)
					q+= ",";
			}
			else
			{
				if((*it).rep_.type_ == TeSTRING)
					isUniqueValString = true;
				uniqueVal = row[jj];
			}

			++it;
			jj++;
		}
		
		if(isUniqueValString)
			q += " WHERE " + uniqueName + " = '" + uniqueVal +"'";
		else
			q += " WHERE " + uniqueName + " = " + uniqueVal;
		
		if (!this->execute(q))
			continue;
	}
	return true;
}


bool 
TeDatabase::loadTable(const string& tableName, TeTable &table)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	string q ="SELECT * FROM " + tableName;
	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}
	table.name(tableName);
	table.setAttributeList(portal->AttributeList());

	while (portal->fetchRow())
	{
		TeTableRow row;
		for (int j = 0; j < portal->numFields(); j++)
			row.push_back (portal->getData (j)); 
		table.add(row);
	}
	delete portal;
	return true;
}

bool 
TeDatabase::selectTable (const string& tableName, const string& criteria, TeTable &table)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	string q ="SELECT * FROM " + tableName;
	if (!criteria.empty())
		q += " WHERE " + criteria;

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}
	table.name(tableName);
	table.setAttributeList(portal->AttributeList());

	while (portal->fetchRow())
	{
		TeTableRow row;
		for(int i = 0; i < portal->numFields(); i++)
			row.push_back(portal->getData(i));
		table.add(row);
	}
	delete portal;
	return true;
}

bool 
TeDatabase::updateView (TeView *view)
{
	TeProjection* proj = view->projection();
	if (proj)
	{
		if (proj->id() <= 0)
			this->insertProjection(view->projection());
		else
			this->updateProjection(view->projection());
	}
	else
	{
		errorMessage_ = "No  possvel atualizar vista sem projeo!";
		return false;
	}

	string sql = "UPDATE te_view SET projection_id=" + Te2String(proj->id());
	sql+= ", name='" + view->name() + "'";
	sql+= ", user_name='" + view->user() + "'";
	sql+= ", visibility= " + Te2String((int)view->isVisible());
	sql +=" WHERE view_id = " + Te2String(view->id());

	if (!this->execute (sql))
		return false;

	TeViewTree* tree = view->root();
	if (tree) 
		return	updateViewTree(tree);
	return true;
}


bool 
TeDatabase::loadViewSet (const string& user)
{
	TeViewMap::iterator it = viewMap_.begin();
	while (it != viewMap_.end())
	{
		delete it->second;
		++it;
	}
	viewMap_.clear();
	
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	string sql;
	if (!user.empty())
		sql = "SELECT * FROM te_view WHERE user_name = '" + user + "'";
	else
		sql = "SELECT * FROM te_view WHERE user_name = '" + this->user() + "'";

	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	while (portal->fetchRow())
	{
		TeView *view = new TeView();
		view->id (portal->getInt("view_id"));
		loadView (view);
	}
	delete portal;
	return true;
}

TeViewTree* 
TeDatabase::loadViewTree(TeView* view, int id)
{
	if( view == 0) 
		return 0;
	
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return 0;

	string q;
	TeViewTree *node = 0;

	if (id != 0)
	{
		q = "SELECT * FROM te_theme";
		q += " WHERE view_id = " + Te2String (view->id());
		q += " AND theme_id = " + Te2String(id);

		if (!portal->query(q) || !portal->fetchRow())
		{
			delete portal;
			return 0;
		}

		TeViewNodeType type = (TeViewNodeType)portal->getInt("node_type");
		if(type != TeTREE)
		{
			portal->freeResult();
			delete portal;
			return NULL;
		}

		node = portal->getViewTree();
		portal->freeResult();
	}

	q ="SELECT * FROM te_theme";
	q += " WHERE view_id = " + Te2String (view->id());
	q += " AND parent_id = " + Te2String(id);
	q += " ORDER BY priority ASC";

	if (!portal->query(q))
	{
		delete portal;
		return node;
	}

	while (portal->fetchRow())
	{
		TeViewNodeType childType = (TeViewNodeType)portal->getInt("node_type");
		TeViewNode *childNode;
		if (childType == TeTHEME)
		{
			childNode = new TeTheme();
			childNode->id(portal->getInt(0));
			this->loadTheme((TeTheme*)childNode);
		} 
		else
		{
			childNode = loadViewTree(view, portal->getInt("theme_id"));
		}

		if(id == 0)
		{
			view->add(childNode);
		} 
		else
		{
//			view->addTheme(childNode);
			node->add(childNode);
		}
	}
	delete portal;
	return node;
}

bool 
TeDatabase::loadView (TeView* view)
{
	string q;
	if (view->id() > 0)
		q = "SELECT * FROM te_view WHERE view_id=" + Te2String(view->id());
	else if (!view->name().empty())
	{
		q = "SELECT * FROM te_view WHERE name='" + view->name() + "'";
		
		if(!view->user().empty())
			q += " AND user_name='" + view->user() + "'"; 
	}
	else
		return false;
		
	TeDatabasePortal* portal = getPortal();
	if (!portal)
	{
		this->errorMessage_ = "Erro ao tentar abrir um portal";
		return false;
	}

	if (!portal->query(q))
	{
		delete portal;
		this->errorMessage_ = "Erro ao submeter query: \"" + q + "\"";
		return false;
	}

	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	int projId = portal->getInt("projection_id");
	TeProjection* proj = loadProjection(projId);
	if (proj != 0)
		view->projection(proj);
	view->name(portal->getData("name"));
	view->user(portal->getData("user_name"));
	view->isVisible (portal->getBool("visibility"));
	view->id(portal->getInt("view_id"));
	portal->freeResult();

	loadViewTree(view, 0);

	delete portal;

	vector<TeViewNode*> &themes = view->themes();
	for (unsigned int i = 0; i < themes.size() ; i++)
	{
		TeTheme *theme = (TeTheme*)themes[i];
		int id = theme->layerId();
		TeLayerMap::iterator it = layerMap_.find(id);
		if (it == layerMap_.end())
		{
			layerMap_.clear();
			loadLayerSet();
		}
		theme->layer(layerMap_[id]);
		if (!loadLegend (theme))	// retrieve associated legend
			return false;
	}
	viewMap_[view->id()] = view;
	return true;
}


void
TeDatabase::clear()
{
	TeViewMap::iterator viewIt;
	for (viewIt = viewMap_.begin(); viewIt != viewMap_.end(); ++viewIt)
		delete viewIt->second;

	TeLayerMap::iterator layerIt;
	for (layerIt = layerMap_.begin(); layerIt != layerMap_.end(); ++layerIt)
		delete layerIt->second;

	legendMap_.clear();
	themeMap_.clear();
	viewMap_.clear();
	layerMap_.clear();
}

bool 
TeDatabase::deleteView (int viewId)
{
	TeDatabasePortal* portal = this->getPortal();

	// view projection should be deleted manually
	string sql =  "SELECT projection_id FROM te_view WHERE view_id=" + Te2String(viewId);
	portal->freeResult();
	if (!portal->query(sql))
	{
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	string projId = portal->getData("projection_id");
	portal->freeResult();
	
	// delete themes belonging to this view 
	sql = "SELECT theme_id FROM te_theme WHERE view_id=" + Te2String(viewId);
	if (!portal->query(sql))
	{
		delete portal;
		return false;
	}
	while (portal->fetchRow())
	{
		int id = atoi(portal->getData(0));
		if(deleteTheme(id) == false)
		{	
			delete portal;
			return false;
		}
	}
	// delete view
	sql = "DELETE FROM te_view WHERE view_id = " + Te2String(viewId);
	if (!this->execute (sql))
	{
		delete portal;
		return false;
	}

	sql = "DELETE FROM te_projection WHERE  projection_id = " + projId;
	if (!this->execute (sql))
	{
		delete portal;
		return false;
	}

	// Delete the view and its themes
	TeView* view = viewMap_[viewId];
	viewMap_.erase(viewId);
	delete view;
	delete portal;
	return true;
}

bool
TeDatabase::updateViewTree (TeViewTree *tree)
{
	if(tree->id() > 0)
	{
		string sql;
		sql = "UPDATE te_theme SET ";
		sql += "name='" + tree->name()+"'";
		sql += ", parent_id=" + Te2String (tree->parentId());
		sql += ", node_type=" + Te2String (tree->type());
		sql += " ,priority=" + Te2String (tree->priority());
		sql += " WHERE theme_id = " + Te2String(tree->id());

		if(!this->execute (sql)) return false;
	}

	for (unsigned int th=0; th<tree->size(); th++)
	{
		TeViewNode* node = tree->retrieve(th);
		if (node->type() == TeTHEME)
		{
			TeTheme *theme = (TeTheme*)node;
			if(!updateTheme(theme)) return false;
		}
		else
		{
			TeViewTree* tree = (TeViewTree*)node;
			if(!updateViewTree(tree)) return false;
		}
	}
	return true;
}

bool 
TeDatabase::viewExist(string viewName)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	viewName = TeConvertToUpperCase(viewName);

	string sql = "SELECT name FROM te_view";
	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	while (portal->fetchRow())
	{
		string name = portal->getData(0);
		name = TeConvertToUpperCase(name);
		if (viewName == name)
		{
			delete portal;
			return true;
		}
	}
	delete portal;
	return false;
}


bool
TeDatabase::updateTheme (TeTheme *theme)
{
	string sql;

	if (theme->id() <= 0 )  // theme doesnt exist in the database yet
	{
		return this->insertTheme(theme);
	}
	
	// update theme metadata
	sql = "UPDATE te_theme SET ";
	sql += " layer_id=" + Te2String (theme->layerId());
	sql += ", view_id=" + Te2String (theme->view());
	sql += ", name='" + escapeSequence(theme->name())+"'";
	sql += ", parent_id=" + Te2String (theme->parentId());
	sql += ", priority=" + Te2String (theme->priority());
	sql += ", node_type=" + Te2String (theme->type());
	sql += ", min_scale=" + Te2String (theme->minScale(),5);
	sql += ", max_scale=" + Te2String (theme->maxScale(),5);
	sql += ", generate_attribute_where='" + escapeSequence(theme->attributeRest())+"'";
	sql += ", generate_spatial_where='" + escapeSequence(theme->spatialRest())+"'";
	sql += ", generate_temporal_where='" + escapeSequence(theme->temporalRest())+"'";
	sql += ", collection_table='" + theme->collectionTable() + "'";
	sql += ", visible_rep= " + Te2String(theme->visibleRep ());
	sql += ", enable_visibility= " + Te2String(theme->visibility()); 
	sql += " WHERE theme_id=" + Te2String (theme->id());

	if (!this->execute (sql))
		return false;
         
	//delete grouping 
	sql = "DELETE FROM te_grouping WHERE theme_id= "+ Te2String(theme->id());
	this->execute (sql);
		
	if(theme->grouping())
	{
		if(!insertGrouping(theme->id(), theme->grouping()))
			return false;
	}
	
	// update each of its legends
	bool status = true;
	
	if(theme->legend().size() == 0)
	{
		if(!deleteLegend(theme->id()))
			return false;
	}
	else
	{
		for (unsigned int i = 0; i < theme->legend().size() ; i++)
		{
			status = updateLegend(&(theme->legend()[i]));
			if (!status)
				return status;
		}
	}

	status = updateLegend(&(theme->withoutDataConnectionLegend()));
	if (!status)
		return status;

	status = updateLegend(&(theme->outOfCollectionLegend()));
	if (!status)
		return status;

	status = updateLegend(&(theme->defaultLegend()));
	if (!status)
		return status;

	status = updateLegend(&(theme->pointingLegend()));
	if (!status)
		return status;

	status = updateLegend(&(theme->queryLegend()));
	if (!status)
		return status;

	status = updateLegend(&(theme->queryAndPointingLegend()));
	if (!status)
		return status;

	// theme tables
	return updateThemeTable (theme);
}

bool
TeDatabase::loadTheme (TeTheme* theme)
{
	if (theme == 0)
		return false;

	string get = "SELECT te_theme.* FROM te_view INNER JOIN te_theme ON te_view.view_id = te_theme.view_id WHERE ";
	if (theme->id() > 0)
		get += " te_theme.theme_id = "+  Te2String(theme->id());
	else if (!theme->name().empty())
		get += " te_theme.name = '"+  theme->name() + "'";
	else
	{
		this->errorMessage_ = "Theme procurado no possui nem id nem nome";
		return false;
	}
	get += " AND te_view.user_name = \'" + this->user() +"\'";

	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
	{
		this->errorMessage_ = "No foi possvel abrir portal para o banco";
		return false;
	}

	if (!portal->query (get))
	{	
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	//theme information
	theme->id(atoi (portal->getData(0)));
	theme->layerId (atoi(portal->getData(1)));
	theme->view (atoi(portal->getData (2)));
	theme->name(string (portal->getData("name")));
	theme->parentId( atoi(portal->getData ("parent_id")));
	theme->type (TeViewNodeType(atoi(portal->getData ("node_type"))));
	theme->priority(portal->getInt("priority"));
	theme->minScale (portal->getDouble ("min_scale"));
	theme->maxScale (portal->getDouble ("max_scale"));
	theme->attributeRest(string(portal->getData ("generate_attribute_where")));
	theme->spatialRest(string(portal->getData ("generate_spatial_where")));
	theme->temporalRest(string(portal->getData ("generate_temporal_where")));
	theme->collectionTable(string(portal->getData ("collection_table")));
	theme->collectionAuxTable(theme->collectionTable() + "_aux");
	theme->visibleRep(atoi(portal->getData ("visible_rep")));
	theme->visibility(atoi(portal->getData ("enable_visibility")));
	
	//load layer
	int id = theme->layerId();
	TeLayerMap::iterator it = layerMap_.find(id);
	if (it == layerMap_.end())
	{
		layerMap_.clear();
		loadLayerSet();
	}
	theme->layer(layerMap_[id]);
	portal->freeResult();

	// load them grouping
	get = "SELECT * FROM te_grouping  WHERE theme_id = " + Te2String(theme->id());
	if (portal->query (get) && portal->fetchRow())
	{	
		TeAttributeRep atRep;
		string attname = portal->getData ("grouping_attr");
		string norname = portal->getData ("grouping_norm_attr");
		int f = attname.find("(");
		if(f >= 0)
		{
			string alias = attname;
			attname.erase(f, attname.size()-f);
			alias.erase(0, f+1);
			alias.erase(alias.size()-1, 1);
			map<string, string>& m = mapThemeAlias_[theme->id()];
			m[attname] = alias;
		}
		f = norname.find("(");
		if(f >= 0)
		{
			string alias = norname;
			norname.erase(f, norname.size()-f);
			alias.erase(0, f+1);
			alias.erase(alias.size()-1, 1);
			map<string, string>& m = mapThemeAlias_[theme->id()];
			m[norname] = alias;
		}

		if(attname=="NONE")
			attname = "";
		
		if(norname=="NONE")
			norname = "";
		
		atRep.name_ = attname;
		atRep.type_ = TeAttrDataType(atoi(portal->getData ("grouping_attr_type")));
		
		TeGrouping* g = new TeGrouping ( atRep, norname, 
						TeGroupingMode(atoi(portal->getData ("grouping_mode"))),
						atoi(portal->getData ("grouping_number")),
						atoi(portal->getData ("grouping_precision")),
						portal->getDouble("grouping_std_dev"), 
						portal->getData("grouping_function") );

		theme->grouping(g);
	}
	portal->freeResult();

	// load theme tables
	if(!loadThemeTable(theme))
	{
		delete portal;
		return false;
	}

	get = "SELECT * FROM te_visual_raster  WHERE theme_id = " + Te2String(theme->id());
	if (portal->query (get) && portal->fetchRow())
	{
		TeRasterTransform* vis = new TeRasterTransform();
		vis->setSrcBand(portal->getInt(1));
		vis->setDestBand(portal->getInt(2));
		vis->setTransfFunction(static_cast<TeRasterTransform::TeRasterTransfFunctions>(portal->getInt(3)));
		if (vis->getTransfFunction() == TeRasterTransform::TeExtractRGB || 
			vis->getTransfFunction() == TeRasterTransform::TeExtractBands)
		{
			do
			{
				vis->setBChannelMapping(portal->getInt(1),static_cast<TeRasterTransform::TeRGBChannels>(portal->getInt(2)));
			}while (portal->fetchRow());
		}
		theme->rasterVisual(vis);
	}

	// load theme legends  
	loadLegend(theme);

	themeMap_[theme->id()] = theme;

	delete portal;
	return true;
}


bool
TeDatabase::loadThemeTable (TeTheme* theme)
{
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
	{
		this->errorMessage_ = "No foi possvel abrir portal para o banco";
		return false;
	}
	
	//theme tables
	string sel = "SELECT te_theme_table.*, te_tables_relation.*, te_layer_table.*";
	sel += " FROM (te_theme_table LEFT JOIN te_tables_relation";
	sel += " ON te_theme_table.relation_id = te_tables_relation.relation_id)";
	sel += " LEFT JOIN te_layer_table ON te_theme_table.table_id = te_layer_table.table_id";
	sel += " WHERE  te_theme_table.theme_id = " + Te2String(theme->id());
	sel += " ORDER BY table_order";
	
	if (!portal->query (sel))
	{	
		delete portal;
		return false;
	}

	while(portal->fetchRow ())
	{
		string tableName = portal->getData("attr_table");
		TeAttributeList attrList;
		getAttributeList(tableName, attrList);
		string uniqueId = portal->getData("unique_id");
		string linkName = portal->getData("attr_link");

		TeTable table(tableName, attrList, uniqueId, linkName);
		table.setId(portal->getInt("te_theme_table.table_id"));
		table.setOrder(portal->getInt("table_order"));
		table.attInitialTime(portal->getData("attr_initial_time"));
		table.attFinalTime(portal->getData("attr_final_time"));
		table.attTimeUnit(TeChronon(portal->getInt("attr_time_unit")));

		int tableType = portal->getInt("attr_table_type");
		if (tableType == TeAttrExternal)
		{
			int relatedTableId = portal->getInt("related_table_id");
			table.relatedTableName(getTableName(relatedTableId));
			int relationId = portal->getInt("te_theme_table.relation_id");
			relationMSet_.insert(relationId);

			string relatedAttr = portal->getData("related_attr");
			table.setTableType(TeAttrExternal, relatedTableId, relatedAttr);

			table.setLinkName(portal->getData("external_attr"));
		}
		else
			table.setTableType((TeAttrTableType)tableType);

		theme->addThemeTable(table);
	}

	delete portal;
	return true;
}


bool
TeDatabase::insertThemeTable(TeTheme *theme, TeTable& inputTable)
{
	int themeId = theme->id();
	int	tableOrder = 0;
	int	relationId;
	bool status;
	string qString;

	// Get the order of the last theme table
	qString = "SELECT MAX(table_order) FROM te_theme_table";
	qString += " WHERE theme_id = " + Te2String(themeId);

	TeDatabasePortal* portal = getPortal();
	if (portal->query(qString) == false || portal->fetchRow() == false)
		return false;

	string data = portal->getData(0);
	if (data.empty())
		tableOrder = 0;
	else
	{
		tableOrder = atoi(portal->getData(0));
		++tableOrder;
	}
	inputTable.setOrder(tableOrder);
	delete portal;

	if (inputTable.tableType() == TeAttrExternal)
	{
		status = insertRelationInfo(inputTable.relatedTableId(),inputTable.relatedAttribute(),
									inputTable.name(),inputTable.linkName(),relationId);
		if (status == false)
			return false;
		relationMSet_.insert(relationId);

		status = insertThemeTable(themeId, inputTable.id(), relationId, tableOrder);
	}
	else
	{
		// Insert the table in the te_theme_table
		status = insertThemeTable(themeId, inputTable.id(), 0, tableOrder);
	}
	return status;
}

bool
TeDatabase::removeThemeTable(TeTheme *theme, int tableOrder)
{
	if (tableOrder < 0)
		return false;

	int relationId = -1;
	string qString;

	TeAttrTableVector attrTableVector; 
	theme->getAttTables(attrTableVector);

	// If the table is external, find the relation id
	qString = "SELECT relation_id FROM te_theme_table";
	qString += " WHERE theme_id = " + Te2String(theme->id());
	qString += " AND relation_id IS NOT NULL";
	qString += " AND table_order = " + Te2String(tableOrder);

	TeDatabasePortal* portal = getPortal();
	if (portal->query(qString) && portal->fetchRow())
		relationId = portal->getInt("relation_id");
	else
	{
		delete portal;
		return false;
	}
	delete portal;

	// Remove the relation from the te_tables_relation
	// table if only this theme is using it
	if (relationId > 0)
	{
		if (relationMSet_.count(relationId) == 1)
		{
			qString = "DELETE FROM te_tables_relation WHERE relation_id = " + Te2String(relationId);
			if (execute(qString) == false)
				return false;
			relationMSet_.erase(relationId);
		}
		else
			relationMSet_.erase(relationMSet_.find(relationId));
	}

	// Remove the table from the te_theme_table
	qString = "DELETE FROM te_theme_table WHERE theme_id = " + Te2String(theme->id());
	qString += " AND table_order = " + Te2String(tableOrder);

	if (execute(qString) == false)
		return false;

	return true;
}


bool 
TeDatabase::updateThemeTable(TeTheme *theme)
{
	//Initially remove from te_theme_table all the records of this theme
	string q = "SELECT * FROM te_theme_table WHERE theme_id = " + Te2String(theme->id());
	TeDatabasePortal *portal = getPortal();

	if (portal->query(q) == false)
	{
		delete portal;
		return false;
	}

	while (portal->fetchRow())
	{
		int relationId;
		int themeTableId = portal->getInt("theme_table_id");

		string data = portal->getData("relation_id");
		if (data.empty())
			relationId = -1;
		else
			relationId = atoi(data.c_str());

		// Remove the relation from the te_tables_relation
		// table if only this theme is using it
		if (relationId > 0)
		{
			if (relationMSet_.count(relationId) == 1)
			{
				q = "DELETE FROM te_tables_relation WHERE relation_id = " + Te2String(relationId);
				if (execute(q) == false)
				{
					delete portal;
					return false;
				}
				relationMSet_.erase(relationId);
			}
			else
				relationMSet_.erase(relationMSet_.find(relationId));
		}

		// Remove the table from the te_theme_table
		q = "DELETE FROM te_theme_table WHERE theme_id = " + Te2String(theme->id());
		q += " AND theme_table_id = " + Te2String(themeTableId);
		if (execute(q) == false)
		{
			delete portal;
			return false;
		}
	}
	delete portal;

	// Update te_theme_table and te_tables_relation(if there are any external tables)
	// with the information provided from the new vector of theme tables
	TeAttrTableVector tablesVector; 
	theme->getAttTables(tablesVector);
	for (unsigned i = 0; i < tablesVector.size(); ++i)
		insertThemeTable(theme, tablesVector[i]);

	return true;
}


bool 
TeDatabase::insertGrouping (int themeId, TeGrouping* grouping)
{
	if((themeId < 1) || (!grouping) )
		return false;

	string ins = " INSERT INTO te_grouping (theme_id, grouping_number, ";
	ins += " grouping_attr, grouping_attr_type, grouping_mode, "; 
	ins += " grouping_norm_attr, grouping_std_dev, grouping_precision, grouping_function )"; 
	ins += " VALUES ( ";
	ins += Te2String(themeId);
	ins += ", "+ Te2String(grouping->groupNumSlices_);

	string attname = grouping->groupAttribute_.name_;
	if(attname.empty() || (attname=="NONE") )
		attname = "";
	
	string norname = grouping->groupNormAttribute_;
	if(norname.empty() || (norname=="NONE"))
		norname = "";

	map<int, map<string, string> >::iterator it = mapThemeAlias_.find(themeId);
	if(it != mapThemeAlias_.end())
	{
		map<string, string>::iterator tit = it->second.find(attname);
		if(tit != it->second.end())
		{
			string alias = tit->second;
			attname += "(" + alias + ")";
		}
		if(norname.empty() == false)
		{
			map<string, string>::iterator tit = it->second.find(norname);
			if(tit != it->second.end())
			{
				string nalias = tit->second;
				norname += "(" + nalias + ")";
			}
		}
	}

	ins += ", '"+ attname +"'";
	ins += ", "+ Te2String(grouping->groupAttribute_.type_);
	ins += ", "+ Te2String(grouping->groupMode_);
	ins += ", '"+ norname +"'";
	ins += ", "+ Te2String(grouping->groupStdDev_); 
	ins += ", "+ Te2String(grouping->groupPrecision_);
	ins += ", '"+ grouping->groupFunction_ +"'";
	ins += ")";

	return (execute(ins));
}

bool 
TeDatabase::updateGrouping (int themeId, TeGrouping* grouping)
{
	if((themeId < 1) || (!grouping))
		return false;

	string up = " UPDATE te_grouping SET ";
	up += "  grouping_number = "+ Te2String(grouping->groupNumSlices_);

	string attname = grouping->groupAttribute_.name_;
	if(attname.empty() || (attname=="NONE"))
		attname = "";

	string norname = grouping->groupNormAttribute_;
	if(norname.empty()|| (norname=="NONE"))
		norname = "";

	map<int, map<string, string> >::iterator it = mapThemeAlias_.find(themeId);
	if(it != mapThemeAlias_.end())
	{
		map<string, string>::iterator tit = it->second.find(attname);
		if(tit != it->second.end())
		{
			string alias = tit->second;
			attname += "(" + alias + ")";
		}
		if(norname.empty() == false)
		{
			map<string, string>::iterator tit = it->second.find(norname);
			if(tit != it->second.end())
			{
				string nalias = tit->second;
				norname += "(" + nalias + ")";
			}
		}
	}

	up += ", grouping_attr = '"+ attname +"'";
	up += ", grouping_attr_type = "+ Te2String(grouping->groupAttribute_.type_);
	up += ", grouping_mode = "+ Te2String(grouping->groupMode_);
	up += ", grouping_norm_attr = '"+ norname +"'";
	up += ", grouping_std_dev = "+ Te2String(grouping->groupStdDev_); 
	up += ", grouping_precision = "+ Te2String(grouping->groupPrecision_);
	up += ", grouping_function = '"+ grouping->groupFunction_ +"'";
	up += " WHERE theme_id = "+ Te2String(themeId);

	return (execute(up));
}

bool 
TeDatabase::generateLabelPositions(TeTheme *theme)
{
	string	piebar, geomTable, upd;
	string	collTable = theme->collectionTable();
	
	if((collTable.empty()) || (!tableExist(collTable)))
		return false;

	if (theme->layer()->hasGeometry(TeCELLS) )
	{
		geomTable = theme->layer()->tableName(TeCELLS);

		piebar = "SELECT label_x, label_y, lower_x, lower_y, upper_x, upper_y";
		piebar += " FROM " + collTable + " LEFT JOIN " + geomTable;
		piebar += " ON " + collTable + ".c_object_id = " + geomTable + ".object_id";

		upd = " UPDATE (" + piebar + ") SET";
		upd += " label_x = lower_x + (upper_x-lower_x)/2,";
		upd += " label_y = lower_y + (upper_y-lower_y)/2";

		if(!execute(upd))
			return false;
	}

	if( theme->layer()->hasGeometry(TePOLYGONS))
	{
		geomTable = theme->layer()->tableName(TePOLYGONS);

		piebar = "SELECT label_x, label_y, lower_x, lower_y, upper_x, upper_y";
		piebar += " FROM " + collTable + " LEFT JOIN " + geomTable;
		piebar += " ON " + collTable + ".c_object_id = " + geomTable + ".object_id";
		piebar += " WHERE label_x is null OR label_y is null";
		piebar += " ORDER BY c_object_id ASC, ext_max ASC";
		
		upd = " UPDATE (" + piebar + ") SET";
		upd += " label_x = lower_x + (upper_x-lower_x)/2,";
		upd += " label_y = lower_y + (upper_y-lower_y)/2";

		if(!execute(upd))
			return false;
	}
	
	if (theme->layer()->hasGeometry(TePOINTS))
	{
		geomTable = theme->layer()->tableName(TePOINTS);

		piebar = "SELECT label_x, label_y, x, y";
		piebar += " FROM " + collTable + " LEFT JOIN " + geomTable;
		piebar += " ON " + collTable + ".c_object_id = " + geomTable + ".object_id";
		piebar += " WHERE label_x is null OR label_y is null";
		upd = " UPDATE (" + piebar + ") SET";
		upd += " label_x = x,";
		upd += " label_y = y";

		if(!execute(upd))
			return false;
	}
	
	if(theme->layer()->hasGeometry(TeLINES))
	{
		geomTable = theme->layer()->tableName(TeLINES);

		piebar = "SELECT label_x, label_y, lower_x, lower_y, upper_x, upper_y";
		piebar += " FROM " + collTable + " LEFT JOIN " + geomTable;
		piebar += " ON " + collTable + ".c_object_id = " + geomTable + ".object_id";
		piebar += " WHERE label_x is null OR label_y is null";
		piebar += " ORDER BY c_object_id ASC, ext_max ASC";
		
		upd = " UPDATE (" + piebar + ") SET";
		upd += " label_x = lower_x + (upper_x-lower_x)/2,";
		upd += " label_y = lower_y + (upper_y-lower_y)/2";

		if(!execute(upd))
			return false;
	}

	return true;
}

bool 
TeDatabase::themeExist(string themeName)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	themeName = TeConvertToUpperCase(themeName);

	string sql = "SELECT name FROM te_theme";
	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	while (portal->fetchRow())
	{
		string name = portal->getData(0);
		name = TeConvertToUpperCase(name);
		if (themeName == name)
		{
			delete portal;
			return true;
		}
	}
	delete portal;
	return false;
}

bool
TeDatabase::deleteTheme(int themeId)
{
	string sql;

	// delete the collection table associated to this theme
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return false;

	sql = "SELECT collection_table FROM te_theme WHERE theme_id = " + Te2String(themeId);
	if(!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	string colTab = portal->getData("collection_table");
	//delete collection table
	if (this->tableExist(colTab))	
	{
		sql = "DROP TABLE " + colTab;
		if (!this->execute(sql) )
		{	
			delete portal;
			return false;
		}
	}
	//delete auxiliar collection table
	if (this->tableExist(colTab +"_aux"))	
	{
		sql = "DROP TABLE " +colTab +"_aux";
		if (!this->execute(sql) )
		{	
			delete portal;
			return false;
		}
	}

	portal->freeResult();

	//delete the visual associated to this theme
	if (existRelation("te_visual","fk_visual_legend_id") != TeRICascadeDeletion)
	{
		sql = "SELECT legend_id FROM te_legend WHERE theme_id = " + Te2String(themeId);
		if(!portal->query(sql))
		{	
			delete portal;
			return false;
		}
		string wherec;
		int c = 0;
		while (portal->fetchRow())
		{
			if (c)
				wherec += ",";
			c++;
			wherec += portal->getData(0);
		}
		portal->freeResult();
		sql = "DELETE FROM te_visual WHERE legend_id IN (" + wherec + ")";
		if (!this->execute(sql))
		{	
			delete portal;
			return false;
		}
	}

	//delete all visuals of raster associated to this theme
	if (existRelation("te_visual_raster","fk_visrast_theme_id") != TeRICascadeDeletion)
	{
		sql = "DELETE FROM te_visual_raster WHERE theme_id =" + Te2String(themeId);
		if (!this->execute (sql))
		{	
			delete portal;
			return false;
		}
	}
	
	//delete all legends associated to this theme
	if (existRelation("te_legend","fk_legend_theme_id") != TeRICascadeDeletion)
	{
		sql = "DELETE FROM te_legend WHERE theme_id =" + Te2String(themeId);
		if (!this->execute (sql))
		{	
			delete portal;
			return false;
		}
	}
	
	//select the view of this theme
	sql = "SELECT view_id FROM te_theme WHERE theme_id = " + Te2String(themeId);
	portal->freeResult();
	if(!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}

	int viewId = portal->getInt("view_id");
	TeView* view = viewMap_[viewId];
	TeTheme* tema = (TeTheme*)view->remove(themeId); 
	if(!tema)
	{
		delete portal;
		return false;
	}

	unsigned int i;
	TeLegendEntryVector& legendVector = tema->legend();
	for (i = 0; i < legendVector.size(); ++i)
		legendMap_.erase(legendVector[i].id());
	themeMap_.erase(themeId);

	delete tema;
	delete portal;

	//delete the tables associated to this theme
	if (existRelation("te_theme_table","fk_thmtable_theme_id") != TeRICascadeDeletion)
	{
		sql = "DELETE FROM te_theme_table WHERE theme_id =" + Te2String(themeId);
		if (!this->execute (sql))
			return false;
	}
		
	//delete the grouping
	if (existRelation("te_grouping","fk_group_theme_id")  != TeRICascadeDeletion)
	{
		sql = "DELETE FROM te_grouping WHERE theme_id =" + Te2String(themeId);
		if (!this->execute (sql))
			return false;
	}

	// delete raster visual
	if (existRelation("te_visual_raster","fk_visrast_theme_id") == TeNoRelation )
	{
		sql = "DELETE FROM te_visual_raster WHERE theme_id =" + Te2String(themeId);
		if (!this->execute (sql))
			return false;
	}

	// delete the theme
	sql = "DELETE FROM te_theme WHERE theme_id = " + Te2String(themeId);
	if (!this->execute (sql))
		return false;

	return true;
}

bool
TeDatabase::deleteThemeGroup(int themeId)
{
	string sql;
	// delete the theme
	sql = "DELETE FROM te_theme WHERE theme_id = " + Te2String(themeId);
	if (!this->execute (sql))
		return false;
	return true;
}

bool
TeDatabase::deleteLegend(int themeId)
{
	// If there is a collection table update legend of the objects in it
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return false;
	string sel = "SELECT collection_table FROM te_theme WHERE theme_id = " + Te2String(themeId);
	string TC;
	if (portal->query(sel) && portal->fetchRow())
		TC = portal->getData(0);
	delete portal;

	if (!TC.empty() && this->tableExist(TC))
	{
		string up = "UPDATE " + TC + " SET c_legend_id = 0";
		if (!execute(up))
			return false;
	}

	// Delete visual of the legends
	string del = "DELETE FROM te_visual WHERE legend_id IN ";
	del += "(SELECT legend_id FROM te_legend WHERE theme_id = " + Te2String(themeId);
	del += " AND group_id > -1)";
	if (!execute(del))
		return false;

	del = "DELETE FROM te_legend WHERE theme_id = " + Te2String(themeId);
	del += " AND group_id > -1";
	if (!execute(del))
		return false;

	// Delete from memory the legends of the theme
	unsigned int i;
	TeTheme *theme = themeMap_[themeId];
	TeLegendEntryVector& legendVector = theme->legend();
	for (i = 0; i < legendVector.size(); ++i)
		legendMap_.erase(legendVector[i].id());
	legendVector.clear();

	//delete grouping
	del = "DELETE FROM te_grouping WHERE theme_id =" + Te2String(themeId);
	if (!execute (del))
		return false;
		
	return true;
}

bool
TeDatabase::updateLayer(TeLayer *layer)
{
	if (!layer)
		return false;

	string sql;
	sql = "UPDATE te_layer SET ";
	sql += "name = '" + layer->name() + "' ";
	sql += ", lower_x = " + Te2String(layer->box().x1(),15) + " ";
	sql += ", lower_y = " + Te2String(layer->box().y1(),15) + " ";
	sql += ", upper_x = " + Te2String(layer->box().x2(),15) + " ";
	sql += ", upper_y = " + Te2String(layer->box().y2(),15) + " ";
	sql += " WHERE layer_id = " + Te2String(layer->id());

	if (layer->projection())
		updateProjection(layer->projection());
	return (this->execute (sql));
}

bool
TeDatabase::loadLayerSet()
{
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return false;

	string get = "SELECT * FROM te_layer";
	if (!portal->query(get))
	{	
		layerMap_.clear();
		delete portal;
		return false;
	}

	int lid;
	while (portal->fetchRow())
	{
		TeLayer* layer = 0;
		lid = portal->getInt("layer_id");
		
		TeLayerMap::iterator it = layerMap_.find(lid);
		// if layer is not in the cache yet load it in a new pointer
		if (it == layerMap_.end())
		{
			layer = new TeLayer();
			layer->id(lid);
 			loadLayer(layer);
		}
		// else just refresh the pointer
		else
		{
			it->second->id(lid);
			loadLayer(it->second);
		}
	}
	delete portal;
	return true;
}

string
TeDatabase::getRasterTable(int layerId, const string& objectId)
{
	if (layerId <=0 )
		return "";

	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return "";

	string get;
	// get the raster representation
	get = "SELECT geom_table FROM te_representation WHERE layer_id = "+Te2String(layerId);
	get += " AND geom_type= " + Te2String((int)TeRASTER);
	
	// error executing query or no there is no raster representation 
	if (!portal->query(get) || !portal->fetchRow())
	{
		delete portal;
		return "";
	}

	string tableName = portal->getData(0);
	portal->freeResult();
	if (tableName.empty())
	{
		delete portal;
		return "";
	}

	// check if a specific object is being looked for
	get = "SELECT raster_table FROM " + tableName + " WHERE object_id='" + objectId + "'";
	if (!portal->query(get) || !portal->fetchRow())
	{
		delete portal;
		return "";
	}
	tableName = portal->getData(0);
	delete portal;
	return tableName;
}


TeRaster*
TeDatabase::loadLayerRaster(int layerId, const string& objectId, const char& mode)
{
	if (layerId <=0 )
		return 0;

	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return 0;

	TeRaster* raster = 0;
	string get;
	// get the raster representation
	get = "SELECT geom_table FROM te_representation WHERE layer_id = "+Te2String(layerId);
	get += " AND geom_type= " + Te2String((int)TeRASTER);
	
	// error executing query or no there is no raster representation 
	if (!portal->query(get) || !portal->fetchRow())
	{
		delete portal;
		return 0;
	}

	string tableName = portal->getData(0);
	portal->freeResult();
	if (tableName.empty())
	{
		delete portal;
		return 0;
	}

	//--- this check is made for compatibility reasons with older terralib databases
	TeAttributeRep	attrRep;
	attrRep.name_ = "tiling_type";
	attrRep.type_ = TeINT;
		
	TeAttribute att;
	if(!columnExist(tableName, attrRep.name_,att))
	{
		addColumn (tableName, attrRep);
		string sql = "UPDATE " + tableName + " SET tiling_type = " + Te2String(static_cast<int>(TeExpansible));
		this->execute(sql);
	}

	// ---

	// check if a specific object is being looked for
	get = "SELECT * FROM " + tableName;
	if (!objectId.empty())
		get += " WHERE object_id='" + objectId + "'"; 
	if (!portal->query(get) || !portal->fetchRow())
	{
		delete portal;
		return 0;
	}
		
	string oid = portal->getData("object_id");
	int geomId = portal->getInt("geom_id"); // get raster representation id

	// get raster parameters from te_raster_table table
	TeRasterParams params;
	params.fileName_ = portal->getData("raster_table");
	params.lutName_ = portal->getData("lut_table");
	params.nBands(portal->getInt("num_bands"));
	params.boundingBoxResolution(portal->getDouble("lower_x"),portal->getDouble("lower_y"),
								 portal->getDouble("upper_x"),portal->getDouble("upper_y"),
							     portal->getDouble("res_x"),portal->getDouble("res_y"));
	params.blockHeight_ = portal->getInt("block_height");
	params.blockWidth_ = portal->getInt("block_width");
	params.tiling_type_ = static_cast<TeRasterTilingType>(portal->getInt("tiling_type"));

	portal->freeResult();

	// get extra information from te_raster_metadata table
	string metadatatable = tableName + "_metadata"; 
	unsigned int nb = params.nBands();
	unsigned int i;
	for (i=0; i<nb; i++)
	{
		get = "SELECT * FROM " + metadatatable + " WHERE geom_id=" + Te2String(geomId);
		get += " AND band_id=" + Te2String(i);
		if (portal->query(get) && portal->fetchRow()) 
		{
			params.vmax_[i] = portal->getDouble("max_value");
			params.vmin_[i] = portal->getDouble("min_value");
			params.nbitsperPixel_[i] = portal->getInt("num_bits");
			params.dataType_[i] = TeDataType(portal->getInt("data_type"));
			params.compression_[i] = TeCompressionMode(portal->getInt("compression_type"));
			params.photometric_[i] = portal->getInt("photometric_type");
		}
		portal->freeResult();
	}

	// if raster is pallete get the associated LUT
	if ((params.photometric_[0] == TeRASTERPALETTE) ||  
    (params.photometric_[0] == TeRASTERKERNEL))
		this->loadRasterLUT(&params);

	// raster has the same projection as its layer
	get = "SELECT projection_id FROM te_layer WHERE layer_id = " + Te2String(layerId);
	int projId; 
	TeProjection* proj=0;
	if (portal->query(get) && portal->fetchRow())
	{
		projId = portal->getInt("projection_id");
		proj = this->loadProjection(projId);
	}
	portal->freeResult();

	params.projection(proj);
	if (proj)
		delete proj;
	params.nTilesInMemory_ = 0;
	params.database_ = this;

	bool hasDummy = false;

	get = "SELECT band_id, dummy FROM " + metadatatable + " WHERE geom_id=" + Te2String(geomId);
	get += " AND NOT (dummy IS NULL)";
	if (portal->query(get))
	{
		while (portal->fetchRow())
		{
			int b = portal->getInt(0);
			double d = portal->getDouble("dummy");
			params.setDummy(d,b);
			hasDummy = true;
		}
	}
	params.useDummy_ = hasDummy;
	params.mode_ = mode;
	delete portal;

	// builds a raster from the parameters
	TeDecoderDatabase* dec = new TeDecoderDatabase(params);
	dec->init();
	raster = new TeRaster();
	raster->setDecoder(dec);
	raster->objectId(oid);
	return raster;
}

bool 
TeDatabase::loadRasterLUT(TeRasterParams* par)
{
	if (par->lutName_.empty())		
		return false;
	
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	string get = "SELECT COUNT(index_id) FROM " + par->lutName_;
	if (!portal->query(get) || !portal->fetchRow())	 // if there is no table, or table is empty
	{
		delete portal;
		return false;
	}
	int nentries = atoi(portal->getData(0));
	if (nentries <= 0)
	{
		delete portal;
		return false;
	}

	par->setNumberPalleteEntries(nentries);
	portal->freeResult();

	get = "SELECT * FROM " + par->lutName_ + " ORDER BY index_id ASC ";

	if (!portal->query(get) || !portal->fetchRow())	 // if there is no table, or table is empty
	{
		delete portal;
		return false;
	}
	do
	{
		int index = atoi(portal->getData(0));
		par->lutr_[index] = atoi(portal->getData(1));
		par->lutg_[index] = atoi(portal->getData(2));
		par->lutb_[index] = atoi(portal->getData(3));
	}while (portal->fetchRow());

	delete portal;
	return true;
}


bool
TeDatabase::loadLayer(TeLayer* layer)
{
	if (layer == 0)
		return false;

	string get;
	if (layer->id() > 0)
		get = "SELECT * FROM te_layer WHERE layer_id = "+  Te2String(layer->id());
	else if (!layer->name().empty())
		get = "SELECT * FROM te_layer WHERE name = '"+  layer->name() + "'";
	else
	{
		this->errorMessage_ = "Layer procurado no possui nem id nem nome";
		return false;
	}

	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
	{
		this->errorMessage_ = "No foi possvel abrir portal para o banco";
		return false;
	}

	if (!portal->query (get))
	{	
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	
	layer->id(atoi (portal->getData("layer_id")));
	layer->name(string (portal->getData("name")));
	layer->setLayerBox(TeBox(portal->getDouble("lower_x"),
						portal->getDouble("lower_y"),
						portal->getDouble("upper_x"),
						portal->getDouble("upper_y")));
	
	int projId = portal->getInt("projection_id");
	portal->freeResult();

    layer->setDatabase(this);

	// Load layer projection
	TeProjection* proj = loadProjection (projId);
	if (proj)
		layer->setProjection (proj);

	// Load layer representations
	get = "SELECT * FROM te_representation WHERE layer_id = "+Te2String(layer->id());
	if (portal->query (get))
	{
		while (portal->fetchRow())
		{
			TeRepresentation* repp = new TeRepresentation();
			repp->id_ = atoi(portal->getData(0));
			TeGeomRep g = (TeGeomRep)atoi (portal->getData(2));
			repp->geomRep_ = g;
			repp->tableName_ = portal->getData(3);
			repp->description_ = portal->getData(4);
			repp->box_ = TeBox(portal->getDouble(5),
						portal->getDouble(6),
						portal->getDouble(7),
						portal->getDouble(8));
			repp->resX_ = portal->getDouble(9);
			repp->resY_ = portal->getDouble(10);
			repp->nCols_ = portal->getInt(11);
			repp->nLins_ = portal->getInt(12);
			layer->addVectRepres(repp);
		}
	}
	
	delete portal;
	if(!loadLayerTable(layer))
		return false;

	layerMap_[layer->id()] = layer;
	return true; 
}

bool
TeDatabase::loadLayerTable(TeLayer* layer)
{
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
	{
		this->errorMessage_ = "No foi possvel abrir portal para o banco";
		return false;
	}
	
	// Get layer tables	
	string get =  " SELECT * FROM te_layer_table"; 
	get += " WHERE layer_id = " + Te2String(layer->id());
	get += " ORDER BY attr_table_type, table_id";

	if (!portal->query (get))
	{	
		delete portal;
		return false;
	}

	while (portal->fetchRow())
	{
		string tableName = portal->getData("attr_table");
		int    tableId = portal->getInt("table_id");
		string indexName = portal->getData("attr_link");

		TeAttributeList attrList;
		getAttributeList(tableName, attrList);

		TeTable attTable(tableName,attrList,
				         portal->getData("unique_id"), indexName,
						 TeAttrTableType(portal->getInt("attr_table_type")));

		attTable.setId(tableId);
		attTable.attInitialTime(portal->getData("attr_initial_time"));
		attTable.attFinalTime(portal->getData("attr_final_time"));
		attTable.attTimeUnit(TeChronon(portal->getInt("attr_time_unit")));

		layer->addAttributeTable(attTable);
	}

	delete portal;
	return true;
}

bool 
TeDatabase::layerExist(int layerId)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	string sql = "SELECT layer_id FROM te_layer WHERE layer_id = " + Te2String(layerId);
	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	delete portal;
	return true;
}

bool 
TeDatabase::layerExist(string layerName)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	layerName = TeConvertToUpperCase(layerName);

	string sql = "SELECT name FROM te_layer";
	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}
	while (portal->fetchRow())
	{
		string name = portal->getData(0);
		name = TeConvertToUpperCase(name);
		if (layerName == name)
		{
			delete portal;
			return true;
		}
	}
	delete portal;
	return false;
}


bool
TeDatabase::deleteLayer(int layerId)
{
	//Delete attributes tables
	if(!deleteLayerTable(layerId))
		return false;

	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	
	string sql = "SELECT projection_id FROM te_layer WHERE layer_id = ";
	sql += Te2String(layerId);

	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}

	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	string projId = portal->getData("projection_id");
	portal->freeResult();

	// Get all representations that are associated to this layer
	sql = "SELECT * FROM te_representation WHERE layer_id = "+ Te2String(layerId);
	if (!portal->query (sql))
	{
		delete portal;
		return false;
	}

	while (portal->fetchRow())
	{	
		// Save the name of the geometry table
		string geomTable = portal->getData("geom_table");

		// Delete lut table
		TeGeomRep rep = TeGeomRep(portal->getInt("geom_type"));
		if (rep == TeRASTER)
		{
			TeDatabasePortal* portal2 = this->getPortal();
			sql = "SELECT lut_table, raster_table FROM " + geomTable;
			string tabName;
			if (!portal2->query (sql))
			{
				delete portal2;
				continue;
			}

			while (portal2->fetchRow())
			{
				// remove lut table
				tabName = portal2->getData(0);
				if (!tabName.empty() && this->tableExist(tabName))
				{
					sql = "DROP TABLE " + tabName;
					this->execute(sql);
				}
				// remove raster table
				tabName = portal2->getData(1);
				if (!tabName.empty() && this->tableExist(tabName))
				{
					sql = "DROP TABLE " + tabName;
					this->execute(sql);
				}
			}
			delete portal2;
			// remove raster metadata table
			tabName = geomTable + "_metadata";
			if (!tabName.empty() && this->tableExist(tabName))
			{
				sql = "DROP TABLE " + tabName;
				this->execute(sql);
			}
		}
		if (this->tableExist(geomTable))
		{
			sql = "DROP TABLE " + geomTable;
			if (!this->execute(sql) )
			{
				delete portal;
				return false;
			}
		}
	}
	portal->freeResult();

	if (existRelation("te_representation","fk_rep_layer_id") != TeRICascadeDeletion)
	{
		// Delete entries into representations table
		sql = "DELETE FROM te_representation WHERE layer_id = " +Te2String(layerId);
		if (!this->execute(sql) )
		{
			delete portal;
			return false;
		}
	}

	// delete layer themes
	sql = "SELECT theme_id FROM te_theme WHERE layer_id=" + Te2String(layerId);
	if (!portal->query (sql))
	{
		delete portal;
		return false;
	}
	
	int themeId;
	while (portal->fetchRow())
	{	
		themeId = portal->getInt("theme_id");
		this->deleteTheme(themeId);
	}
	
	sql = "DELETE FROM te_layer WHERE layer_id=" + Te2String(layerId);
	if (!this->execute(sql))
	{
		delete portal;
		return false;
	}

	// delete layer projection
	sql = "DELETE FROM te_projection WHERE projection_id = "+ projId;
	if (!this->execute(sql))
	{	
		delete portal;
		return false;
	}

	// remove all the items themes associated to the layer to be removed
	TeThemeMap::iterator it;
	for (it = themeMap_.begin(); it != themeMap_.end(); ++it)
	{
		TeTheme *theme = it->second;
		if (theme && theme->layer() && (theme->layer()->id() == layerId))
		{
			themeMap_.erase(theme->id());
			delete theme;
		}
	}

	// delete layer and its entry in the layer map
	TeLayer* layer = layerMap_[layerId];
	layerMap_.erase(layerId);
	delete layer;

	delete portal;
	return true;
}


bool
TeDatabase::deleteLayerTable (int layerId, TeAttrTableType ttype)
{
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return false;

	//tables of the type 1 can be relationed with other layer ??? Conferir
	string query = "SELECT attr_table, table_id FROM te_layer_table WHERE layer_id = " + Te2String(layerId);
	query += " AND attr_table_type = " + Te2String(static_cast<int>(ttype));
	if(!portal->query(query))
	{
		delete portal;
		return false;
	}
		
	vector<int> tableIds;
	string attrTable;
	string tableId;
	string drop;
	while (portal->fetchRow())
	{
		attrTable = portal->getData(0);
		tableId = portal->getData(1);
		drop = "DROP TABLE " + attrTable;
		if(!execute(drop))
		{
			delete portal;
			return false;
		}
		tableIds.push_back(atoi(tableId.c_str()));

		string del  = "DELETE FROM te_address_locator WHERE table_id = "+ tableId;
		execute(del);
	}

	delete portal;
	string del;
	if (existRelation("te_tables_relation","fk_tabrelation_laytable_id") != TeRICascadeDeletion)
	{
		for (unsigned int i=0; i<tableIds.size();i++)
		{
			del = "DELETE FROM te_tables_relation WHERE relation_id = " + Te2String(tableIds[i]);
			if (!execute (del))
				return false;
		}
	}
	del = "DELETE FROM te_layer_table WHERE layer_id = " + Te2String(layerId);
	if (!execute (del))
		return false;
	return true;
}

bool 
TeDatabase::updateRepresentation (int layerId, TeRepresentation& rep)
{
	if (layerId <= 0)
		return false;

	string sql;
	sql  = "UPDATE te_representation SET ";
	sql += " lower_x= " + Te2String(rep.box_.x1(),15);
	sql += ", lower_y= " + Te2String(rep.box_.y1(),15);
	sql += ", upper_x= " + Te2String(rep.box_.x2(),15);
	sql += ", upper_y= " + Te2String(rep.box_.y2(),15);
	sql += ", description= '" + rep.description_ + "'";
	sql += ", res_x= " + Te2String(rep.resX_);
	sql += ", res_y= " + Te2String(rep.resY_);
	sql += ", num_cols=" + Te2String(rep.nCols_);
	sql += ", num_rows=" + Te2String(rep.nLins_);

	if (rep.geomRep_ != TeTEXT)
		sql += ", geom_table='" + rep.tableName_ + "'";

	sql += " WHERE layer_id=" + Te2String(layerId);
	sql += " AND geom_type= " + Te2String(rep.geomRep_);

	if (rep.geomRep_ == TeTEXT)
		sql += " AND geom_table='" + rep.tableName_ + "'";

	return this->execute(sql);
}

bool 
TeDatabase::insertRasterGeometry(const string& tableName, TeRasterParams& par, const string& objectId)
{
	if (tableName.empty())
		return false;
	
	string objId; 
	if (objectId.empty())
		objId = "O1";
	else
		objId = objectId;

	//------ this check is made for compatibility reasons with old versions of TerraLib databases
	TeAttributeRep	attrRep;
	attrRep.name_ = "tiling_type";
	attrRep.type_ = TeINT;
		
	TeAttribute att;
	if(!columnExist(tableName, attrRep.name_,att))
	{
		addColumn (tableName, attrRep);
		string sql = "UPDATE " + tableName + " SET tiling_type = " + Te2String(static_cast<int>(TeExpansible));
		this->execute(sql);
	}
	//------

	// finds the name of the raster geometry table
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return false;

	TeBox box = par.boundingBox();

	string ins = "INSERT INTO " + tableName + " (object_id, raster_table, lut_table, ";
	ins += "res_x, res_y, num_bands, num_cols, num_rows, block_height, block_width, ";
	ins += "lower_x, lower_y, upper_x, upper_y, tiling_type) ";
	ins += " VALUES ('" + objId + "', '" + par.fileName_+ "', '" +  par.lutName_ + "', ";
	ins += Te2String(par.resx_) + ", " + Te2String(par.resy_) + ", ";
	ins += Te2String(par.nBands()) + ", " + Te2String(par.ncols_) + ", " + Te2String(par.nlines_) + ", ";
	ins += Te2String(par.blockHeight_) + ", " + Te2String(par.blockWidth_) + ", ";
	ins += Te2String(box.x1_,15) +", " + Te2String(box.y1_,15) + ", ";
	ins += Te2String(box.x2_,15) +", " + Te2String(box.y2_,15) + ", ";
	ins	+= Te2String(par.tiling_type_) + ")";
	if (!this->execute(ins))
	{
		delete portal;
		return false;
	}

	// save the pallete associated to the raster
	// if it doesnt exist yet
	if ((par.photometric_[0] == TeRASTERPALETTE ||
         par.photometric_[0] == TeRASTERKERNEL) && !par.lutName_.empty()) 
	{

		 if (!this->tableExist(par.lutName_))
		 {
			 if (this->createLUTTable(par.lutName_))
			{
				for (unsigned int i=0; i<par.lutb_.size(); i++)
				{
					string sql = "INSERT INTO " + par.lutName_ + " VALUES(";
					sql += Te2String(i) + ", ";
					sql += Te2String(par.lutr_[i]) + ", ";
					sql += Te2String(par.lutg_[i]) + ", ";
					sql += Te2String(par.lutb_[i]) + ")";
					this->execute(sql);
				}
			 }
		 }
	}
	
	ins = "SELECT geom_id FROM " + tableName + " WHERE object_id='" + objId + "'";
	ins += " AND raster_table='" + par.fileName_+ "'";
	if(!portal->query(ins) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	int geomId = atoi(portal->getData(0));
	delete portal;
	string 	metadataTableName = tableName+"_metadata";
	insertRasterMetadata(metadataTableName, geomId,par);
	 return true;
}

bool 
TeDatabase::updateRasterRepresentation(int layerId, TeRasterParams& par, const string& objectId)
{
	TeDatabasePortal* portal = this->getPortal();
	if(!portal)
		return false;

	string sql = "SELECT repres_id, lower_x, lower_y, upper_x, upper_y, geom_table ";
	sql += " FROM te_representation WHERE layer_id= " + Te2String(layerId);
	sql += " AND geom_type= " + Te2String(TeRASTER);

	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	TeBox box (portal->getDouble(1),portal->getDouble(2),
		       portal->getDouble(3),portal->getDouble(4));
	int represId = atoi(portal->getData(0));
	string rasterrep = portal->getData(5);
	portal->freeResult();

	updateBox(box,par.boundingBox());
	sql = "UPDATE te_representation SET lower_x = " + Te2String(box.x1_,15);
	sql += ", lower_y = " + Te2String(box.y1_,15) + ", upper_x = " + Te2String(box.x2_,15);
	sql += ", upper_y = " + Te2String(box.y2_,15) + "  WHERE repres_id=" + Te2String(represId);

	if(!execute(sql))
	{
		delete portal;
		return false;
	}	 

	string objId; 
	if (objectId.empty())
		objId = "O1";
	else
		objId = objectId;

	box = par.boundingBox();

	sql = "UPDATE " + rasterrep + " SET lut_table ='" + par.lutName_ + "'";
	sql += ", res_x= " + Te2String(par.resx_) + ", res_y=" + Te2String(par.resy_);
	sql += ", num_bands=" + Te2String(par.nBands()) + ", num_cols=" + Te2String(par.ncols_);
	sql += ", num_rows=" + Te2String(par.nlines_) + ", block_height=" +  Te2String(par.blockHeight_);
	sql += ", block_width= " + Te2String(par.blockWidth_) + ", lower_x = " + Te2String(box.x1_,15);
	sql += ", lower_y = " + Te2String(box.y1_,15) +  ", upper_x = " + Te2String(box.x2_,15);
	sql += ", upper_y = " + Te2String(box.y2_,15);
	sql += "  WHERE object_id='" + objId + "' AND raster_table='" + par.fileName_ + "'";
	if (!this->execute(sql))
	{
		delete portal;
		return false;
	}
	
	sql = "SELECT geom_id FROM " + rasterrep + " WHERE object_id='" + objId + "'";
	sql+= " AND raster_table='" + par.fileName_+ "'";
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	// save the pallete associated to the raster
	// if it doesnt exist yet
	if ((par.photometric_[0] == TeRASTERPALETTE ||
         par.photometric_[0] == TeRASTERKERNEL) && !par.lutName_.empty()) 
	{

		 if (!this->tableExist(par.lutName_))
		 {
			 if (this->createLUTTable(par.lutName_))
			{
				for (unsigned int i=0; i<par.lutb_.size(); i++)
				{
					string sql = "INSERT INTO " + par.lutName_ + " VALUES(";
					sql += Te2String(i) + ", ";
					sql += Te2String(par.lutr_[i]) + ", ";
					sql += Te2String(par.lutg_[i]) + ", ";
					sql += Te2String(par.lutb_[i]) + ")";
					this->execute(sql);
				}
			 }
		 }
	}

	int geomId = atoi(portal->getData(0));
	delete portal;
	string metadatatabel = rasterrep + "_metadata";
	return updateRasterMetadata(metadatatabel,geomId,par);
}

bool 
TeDatabase::insertRasterMetadata (const string& tableName, int geomId, TeRasterParams& par)
{
	if (geomId <= 0)
		return false;
	
	string ins;
	unsigned int i;

	unsigned int nb = par.nBands();
	ins = "INSERT INTO " + tableName + " (geom_id, band_id, min_value, max_value, ";
	ins += " num_bits, data_type, photometric_type, compression_type ) VALUES (";
	string vals;
	for (i=0; i<nb; i++)
	{
		vals = Te2String(geomId) + ", " + Te2String(i) + ", ";
		vals += Te2String(par.vmin_[i],15) + ", " +  Te2String(par.vmax_[i],15) + ", ";
		vals += Te2String(par.nbitsperPixel_[i]) + ", " +  Te2String(par.dataType_[i]) + ", " ;
		vals += Te2String(par.photometric_[i]) + ", " +  Te2String(par.compression_[i]) + " )" ;
		string sql = ins + vals;
		if (!this->execute(sql))
			return false;
	}

	// update dummy value
	if (par.useDummy_)
	{
		ins = "UPDATE " + tableName + " SET dummy = ";
		for (i=0; i<nb; i++)
		{
			vals = Te2String(par.dummy_[i]) + " WHERE geom_id = " +  Te2String(geomId);
			vals += " AND band_id=" + Te2String(i);
			string sql = ins + vals;
			if (!this->execute(sql))
				return false;
		}
	}
	return true;
}

bool 
TeDatabase::updateRasterMetadata (const string& tableName, int geomId, TeRasterParams& par)
{
	if (geomId <= 0)
		return false;

	string sql = "DELETE FROM " + tableName + " WHERE geom_id = " + Te2String(geomId);
	if (!this->execute (sql))
		return false;
	return insertRasterMetadata(tableName,geomId,par);
}

bool 
TeDatabase::updateLegend (TeLegendEntry *legend)
{
	if (!legend)
		return false;

	string sql;
	if (legend->id() > 0 )
	{
		sql = "UPDATE te_legend SET ";
		sql += " theme_id=" + Te2String (legend->theme());
		sql += ",group_id=" + Te2String (legend->group());
		sql += ",num_objs=" + Te2String (legend->count());
		sql += ",lower_value='" + escapeSequence(legend->from())+"'";
		sql += ",upper_value='" + escapeSequence(legend->to())+"'";
		sql += ",label='" + escapeSequence(legend->label())+"'";
		sql += " WHERE legend_id=" + Te2String (legend->id());

		if (execute(sql) == false)
			return false;
	}
	else
	{
		if (!insertLegend(legend))
			return false;
	}
	legendMap_[legend->id()] = legend;

	return updateVisual(legend);
}

bool
TeDatabase::updateVisual(TeLegendEntry *legend)
{
	if (!legend)
		return false;

	TeGeomRepVisualMap mapVis = legend->getVisualMap();
	TeGeomRepVisualMap::iterator it =  mapVis.begin();
	while ( it != mapVis.end())
	{ 
			
		TeGeomRep rep = it->first;
		TeVisual vis = it->second;

		TeColor cor = vis.color();				// filling color
		TeColor contourCor = vis.contourColor();// contour color

		string update = "UPDATE te_visual SET ";
		update += " lib_name = '"+ vis.libName() +"', ";
		update += "red = "+ Te2String(cor.red_) + ", ";
		update += "green =" + Te2String(cor.green_) + ", ";
		update += "blue =" + Te2String(cor.blue_) + ", ";
		update += "transparency =" + Te2String(vis.transparency()) + ", ";

		update += "contour_lib_name='"+ vis.contourLibName() +"', ";
		update += "contour_red=" + Te2String(contourCor.red_) + ", ";
		update += "contour_green=" + Te2String(contourCor.green_) + ", ";
		update += "contour_blue=" + Te2String(contourCor.blue_) + ", "; 
		update += "contour_transp=" + Te2String(vis.contourTransparency()) + ", ";
		update += "contour_width=" + Te2String(vis.contourWidth()) + ", ";

		if(rep == TePOLYGONS)
		{
			update += "width=" + Te2String(vis.contourWidth()) + ", ";
			update += "contour_symb_id=" + Te2String(vis.contourStyle()) + ", ";
			update += "symb_id=" + Te2String(vis.style()) + ", ";
		}
		else if(rep == TeLINES)
		{
			update += "width=" + Te2String(vis.width()) + ", ";
			update += "symb_id=" + Te2String(vis.style()) + ", ";
		}
		else if(rep == TePOINTS)
		{
			update += "size_value=" + Te2String(vis.size()) + ", ";
			update += "symb_id=" + Te2String(vis.style ()) + ", ";
		}
		else if(rep == TeTEXT)
		{
			update += "size_value=" + Te2String(vis.size()) + ", ";
			update += "pt_angle=" + Te2String(vis.ptAngle()) + ", ";
		}

		update += "family='" + vis.family() + "', ";
		if (vis.bold())
			update += "bold=1, ";
		else
			update += "bold=0, ";

		if (vis.italic())
			update += "italic=1, ";
		else
			update += "italic=0, ";

		update += "fixed_size=" + Te2String(vis.fixedSize())+ ", ";
		update += "alignment_vert=" + Te2String(vis.alignmentVert())+ ", ";
		update += "alignment_horiz=" + Te2String(vis.alignmentHoriz())+ ", ";
		update += "tab_size=" + Te2String(vis.tabSize())+ ", ";
		update += "line_space=" + Te2String(vis.lineSpace());

		update += " WHERE legend_id= " + Te2String(legend->id()) ;
		update += " AND geom_type= " + Te2String(rep);

		if (!execute(update))
			return false;
		++it;
	}
	return true;
}

bool 
TeDatabase::loadLegend (TeTheme *theme)
{
	if (!theme)
		return false;

	unsigned int i;
	int legId, gId, lastLegId;
	TeGeomRep rep;
	TeVisual visual;
	vector<string> legIdVec;
	map<int, TeLegendEntry> legMap;
	map<int, TeLegendEntry>::iterator it;
	map<int, TeLegendEntry*> ownLegMap;

	theme->cleanLegend();

	TeDatabasePortal* portal = getPortal();

	string q = "SELECT * FROM te_legend WHERE theme_id = " + Te2String(theme->id());
	q += " ORDER BY legend_id";

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}

	while(portal->fetchRow())
	{
		legId = portal->getInt("legend_id");
		legIdVec.push_back(Te2String(legId));

		TeLegendEntry& legEntry = legMap[legId];
		legEntry.id(legId);
		legEntry.theme(theme->id());
		TeSlice& slice = legEntry.slice();
		
		gId = portal->getInt("group_id");
		legEntry.group(gId);
		if (gId == -10)
		{
			TeLegendEntry* ownLeg = new TeLegendEntry;
			ownLegMap[legId] = ownLeg;
		}

		string lb = portal->getData("label");
		legEntry.label(lb);
		slice.count_ = portal->getInt("num_objs");
		slice.from_ = portal->getData("lower_value");
		slice.to_ = portal->getData("upper_value");
	}

	string inClause;
	for(i = 0; i < legIdVec.size(); ++i)
	{
		inClause += legIdVec[i];
		if (i != legIdVec.size() - 1)
			inClause += ",";
	}

	portal->freeResult();
	q = "SELECT * FROM te_visual WHERE legend_id IN (";
	q += inClause + ") ORDER BY legend_id";

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}

	while(portal->fetchRow())
	{
		legId = portal->getInt("legend_id");
		rep = TeGeomRep(portal->getInt("geom_type"));
		visual = portal->getVisual();

		TeLegendEntry& legEntry = legMap[legId];
		gId = legEntry.group();
		legEntry.setVisual(visual, rep);

		if (gId == -10)		// own visual
		{
			TeLegendEntry* ownLegend = ownLegMap[legId];
			*ownLegend = legEntry;

			if (rep == TePOLYGONS)
				ownLegend->setVisual(visual, TePOLYGONS);
			else if (rep == TeLINES)
				ownLegend->setVisual(visual, TeLINES);
			else if (rep == TePOINTS)
				ownLegend->setVisual(visual, TePOINTS);
			else if (rep == TeTEXT)
				ownLegend->setVisual(visual, TeTEXT);

			legendMap_[legId] = ownLegend;
		}
		else if(gId == -6)
		{
			TeLegendEntry& pointingQueryLegend = theme->queryAndPointingLegend();
			pointingQueryLegend = legEntry;

			pointingQueryLegend.setVisual(visual, TeLINES); 
			pointingQueryLegend.setVisual(visual, TePOINTS);
			pointingQueryLegend.setVisual(visual, TeCELLS);
			pointingQueryLegend.setVisual(visual, TeTEXT);  

			legendMap_[legId] = &pointingQueryLegend;
		}
		else if(gId == -5)
		{
			TeLegendEntry& queryLegend = theme->queryLegend();
			queryLegend = legEntry;

			queryLegend.setVisual(visual, TeLINES); 
			queryLegend.setVisual(visual, TePOINTS);
			queryLegend.setVisual(visual, TeCELLS);
			queryLegend.setVisual(visual, TeTEXT); 

			legendMap_[legId] = &queryLegend;
		}
		else if(gId == -4)
		{
			TeLegendEntry& pointingLegend = theme->pointingLegend();
			pointingLegend = legEntry;

			pointingLegend.setVisual(visual, TeLINES); 
			pointingLegend.setVisual(visual, TePOINTS);
			pointingLegend.setVisual(visual, TeCELLS);
			pointingLegend.setVisual(visual, TeTEXT);  

			legendMap_[legId] = &pointingLegend;
		}
		else if(gId == -3)
		{
			TeLegendEntry& defaultLegend = theme->defaultLegend();
			defaultLegend = legEntry;
			legendMap_[legId] = &defaultLegend;

			// Make the pointing visual equal to the default visual, excluding
			// the pointing color 
			TeColor color;
			TeLegendEntry& pointingLegend = theme->pointingLegend();
			TeVisual& pointingVisual = pointingLegend.visual(rep);
			color = pointingVisual.color();
			pointingLegend.setVisual(visual, rep);
			pointingVisual.color(color);

			// Make the query visual equal to the default visual, excluding
			// the query color 
			TeLegendEntry& queryLegend = theme->queryLegend();
			TeVisual& queryVisual = queryLegend.visual(rep);
			color = queryVisual.color();
			queryLegend.setVisual(visual, rep);
			queryVisual.color(color);

			// Make the pointingQuery visual equal to the default visual, excluding
			// the pointingQuery color 
			TeLegendEntry& pointingQueryLegend = theme->queryAndPointingLegend();
			TeVisual& pointingQueryVisual = pointingQueryLegend.visual(rep);
			color = pointingQueryVisual.color();
			pointingQueryLegend.setVisual(visual, rep);
			pointingQueryVisual.color(color);
		}
		else if(gId == -2)
		{
			TeLegendEntry& withoutDataConnectionLegend = theme->withoutDataConnectionLegend();
			withoutDataConnectionLegend = legEntry;
			legendMap_[legId] = &withoutDataConnectionLegend;
		}
		else if(gId == -1)
		{
			TeLegendEntry& outOfCollectionLegend = theme->outOfCollectionLegend();
			outOfCollectionLegend = legEntry;
			legendMap_[legId] = &outOfCollectionLegend;
		}
		else if (gId > -1)
		{
			TeLegendEntryVector& legendEntryVector = theme->legend();
			if (legendEntryVector.empty() == true)
				legendEntryVector.push_back(legMap[legId]);
			else
			{
				if (legId != lastLegId)
					legendEntryVector.push_back(legMap[legId]);
				else
				{
					legendEntryVector.pop_back();
					legendEntryVector.push_back(legMap[legId]);				
				}
			}
			lastLegId = legId;
		}
	}

	TeLegendEntryVector& legVector = theme->legend();
	for (i = 0; i < legVector.size(); ++i)
		legendMap_[legVector[i].id()] = &legVector[i];

	delete portal;
	return true;
}


bool 
TeDatabase::updateProjection (TeProjection *proj)
{
	if (proj->id() <= 0)
		return false;
	string sql;
	sql = "UPDATE te_projection SET ";
	sql += "name='" + proj->name() + "',";
	sql += " long0=" + Te2String(proj->params().lon0*TeCRD,15)+ ",";
	sql += " lat0=" + Te2String(proj->params().lat0*TeCRD,15) + ",";
	sql += " offx=" +Te2String(proj->params().offx,15) + ",";
	sql += " offy=" +Te2String(proj->params().offy,15) + ",";
	sql += " stlat1="+ Te2String(proj->params().stlat1*TeCRD,15) + ",";
	sql += " stlat2=" +Te2String(proj->params().stlat2*TeCRD,15) + ",";
	sql += " unit='" + proj->params().units + "',";
	sql += " scale=" + Te2String(proj->params().scale) + ",";
	sql += " hemis=" + Te2String(proj->params().hemisphere) + ",";
	sql += " datum='" + proj->datum().name() + "',";
	sql += " radius=" + Te2String(proj->datum().radius(),15) + ",";
	sql += " flattening=" + Te2String(proj->datum().flattening(),15) + ",";
	sql += " dx=" + Te2String(proj->datum().xShift(),15) + ",";
	sql += " dy=" + Te2String(proj->datum().yShift(),15) + ",";
	sql += " dz=" + Te2String(proj->datum().zShift(),15) ;
	sql += " WHERE projection_id = " + Te2String(proj->id());
	return this->execute(sql);
}

TeProjection* 
TeDatabase::loadProjection (int projId)
{
	TeDatabasePortal* portal = this->getPortal();

	string sql ="SELECT * FROM te_projection WHERE projection_id = " + Te2String (projId);

	if (!portal->query(sql))
	{
		delete portal;
		return 0;
	}

	// Look for the projection
	if (!portal->fetchRow())
	{
		delete portal;
		return 0;
	}

	TeDatum datum (	portal->getData("datum"),
					portal->getDouble("radius"),
					portal->getDouble("flattening"),
					portal->getDouble("dx"),
					portal->getDouble("dy"),
					portal->getDouble("dz"));

	TeProjectionParams mProjPars;
	mProjPars.datum = datum;
	mProjPars.name = portal->getData("name");
	mProjPars.lat0 = portal->getDouble("lat0")*TeCDR;
	mProjPars.lon0 = portal->getDouble("long0")*TeCDR;
	mProjPars.offx = portal->getDouble("offx");
	mProjPars.offy = portal->getDouble("offy");
	mProjPars.stlat1 = portal->getDouble("stlat1")*TeCDR;
	mProjPars.stlat2 = portal->getDouble("stlat2")*TeCDR;
	mProjPars.units = portal->getData("unit");
	mProjPars.scale = portal->getDouble("scale");
	mProjPars.hemisphere = (TeHemisphere)portal->getInt("hemis");

	TeProjection* proj = TeProjectionFactory::make(mProjPars);
	delete portal;
	proj->id(projId);
	return proj;
}

bool
TeDatabase::insertPolygonSet(const string& table, TePolygonSet &ps)
{
	for (unsigned int i = 0; i < ps.size(); i++ )
	{
		TePolygon poly = ps [i];
		if (!insertPolygon (table,poly))
			return false;
	}
	return true;
}

bool 
TeDatabase::updatePolygonSet (const string& table, TePolygonSet &ps)
{
 	for (unsigned int i = 0; i < ps.size(); i++ )
	{
		TePolygon poly = ps [i];
		if (!updatePolygon (table,poly))
			return false;
	}
	return true;
}

bool 
TeDatabase::selectPolygonSet (const string& table, const string& criteria, TePolygonSet &ps)
{
	TeDatabasePortal *portal = this->getPortal();
	string sql ="SELECT * FROM " + table;
	if (!criteria.empty())
		sql += " WHERE " + criteria;
	sql += " ORDER BY object_id ASC, parent_id, num_holes DESC, ext_max ASC";
	 
	if (!portal->query(sql))
	{
		delete portal;
		return false;
	}
	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::loadPointSet(TeTheme* theme, TePointSet &ps)
{
	string collTable = theme->collectionTable();
	if (collTable.empty())
		return false;

	TeLayer* themeLayer = theme->layer();
	if (!themeLayer->hasGeometry(TePOINTS))
		return false;
	
	string pointTable = themeLayer->tableName(TePOINTS);
	if (pointTable.empty())
		return false;

	string sql = "SELECT * FROM (" + pointTable + " RIGHT JOIN " + collTable;
	sql = sql + " ON " + pointTable + ".object_id = " + collTable + ".object_id)";

	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	if (!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag = true;
	do {
		TePoint pt;
		flag =  portal->fetchGeometry(pt);
		ps.add(pt);
	}
	while (flag);
	delete portal;
	return true;
}

bool 
TeDatabase::loadLineSet(TeTheme* theme, TeLineSet &ls)
{
	string collTable = theme->collectionTable();
	if (collTable.empty())
		return false;

	TeLayer* themeLayer = theme->layer();
	if (!themeLayer->hasGeometry(TeLINES))
		return false;
	
	string lineTable = themeLayer->tableName(TeLINES);
	if (lineTable.empty())
		return false;

	string sql = "SELECT * FROM (" + lineTable + " RIGHT JOIN " + collTable;
	sql = sql + " ON " + lineTable + ".object_id = " + collTable + ".object_id)";

	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	if (!portal->query(sql)  || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag;
	do 
	{
		TeLine2D lin;
		flag = portal->fetchGeometry(lin);
		ls.add(lin);
	}while(flag);
	delete portal;
	return true;
}

bool 
TeDatabase::loadPolygonSet (const string& table, const string& geoid, TePolygonSet &ps)
{
	
	TeDatabasePortal *portal = this->getPortal();
	string q ="SELECT * FROM " + table;

	if (!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";
	q += " ORDER BY parent_id, num_holes DESC, ext_max ASC";

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}

	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);
	delete portal;
	return true;
}

bool 
TeDatabase::loadPolygonSet (const string& table, TeBox &bb, TePolygonSet &polSet)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (bb, TePOLYGONS);
	q += " ORDER BY parent_id, num_holes DESC, ext_max ASC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		polSet.add(poly);
	}
	while (flag);
	delete portal;
	return true;
}

TeDatabasePortal* 
TeDatabase::loadPolygonSet(const string& table, TeBox &box)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return 0;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (box, TePOLYGONS);
	q += " ORDER BY parent_id, num_holes DESC, ext_max ASC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return 0;
	}
	else 
		return portal;
}


bool 
TeDatabase::locatePolygon (const string& table, TeCoord2D &pt, TePolygon &polygon, const double& tol)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);

	string q;
	q = "SELECT * FROM " + table + " WHERE lower_x < " + Te2String(box.x2(),15);
	q += " AND upper_x > " + Te2String(box.x1(),15);
	q += " AND lower_y < " + Te2String(box.y2(),15);
	q += " AND upper_y > " + Te2String(box.y1(),15);
	q += " ORDER BY parent_id, num_holes DESC, ext_max ASC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		if (TeWithin (TePoint(pt), poly))
		{
			polygon = poly;
			delete portal;
			return true;
		}
	}
	while (flag);
	delete portal;
	return false;
}


bool 
TeDatabase::locatePolygonSet (const string& table, TeCoord2D &pt, double tol, TePolygonSet &polygons)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);

	string q;
	q = "SELECT * FROM " + table + " WHERE lower_x < " + Te2String(box.x2(),6);
	q += " AND upper_x > " + Te2String(box.x1(),6);
	q += " AND lower_y < " + Te2String(box.y2(),6);
	q += " AND upper_y > " + Te2String(box.y1(),6);
	q += " ORDER BY parent_id, num_holes DESC, ext_max ASC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	bool flag = true;
	polygons.clear();
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		if (TeWithin (TePoint(pt), poly))
			polygons.add(poly);
	}
	while (flag);
	delete portal;

	if(polygons.size())
		return (true);
	return false;
}


bool 
TeDatabase::loadPolygonSet(TeTheme* theme, TePolygonSet &ps)
{
	string collTable = theme->collectionTable();
	if (collTable.empty())
		return false;

	TeLayer* themeLayer = theme->layer();
	if (!themeLayer->hasGeometry(TePOLYGONS))
		return false;
	
	string polygonTable = themeLayer->tableName(TePOLYGONS);
	if (polygonTable.empty())
		return false;

	string sql = "SELECT * FROM (" + polygonTable + " RIGHT JOIN " + collTable;
	sql = sql + " ON " + polygonTable + ".object_id = " + collTable + ".object_id)";
	sql += " ORDER BY " + polygonTable + ".parent_id, ";
	sql += polygonTable + ".num_holes DESC, " + polygonTable + ".ext_max ASC";

	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	if (!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		ps.add ( poly );
	}
	while (flag);		
	delete portal;
	return true;
}

bool 
TeDatabase::insertLineSet	(const string& table, TeLineSet &ls)
{
	for (unsigned int i = 0; i < ls.size(); i++ )
	{
		TeLine2D line = ls [i];
		if (!insertLine (table,line))
			return false;
	}
	return true;
}

bool 
TeDatabase::updateLineSet	(const string& table, TeLineSet &ls)
{
	for (unsigned int i = 0; i < ls.size(); i++ )
	{
		TeLine2D line = ls [i];
		if (!updateLine (table,line))
			return false;
	}
	return true;
}

bool 
TeDatabase::loadLineSet (const string& table, const string& geoid, TeLineSet &ls)
{
	TeDatabasePortal *portal = this->getPortal();

	string q ="SELECT * FROM " + table;

	if (!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";

	q += " ORDER BY ext_max DESC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;
	do 
	{
		TeLine2D line;
		flag = portal->fetchGeometry(line);
		ls.add ( line );
	}while(flag);

	delete portal;
	return true;
}

bool 
TeDatabase::loadLineSet (const string& table, TeBox &bb, TeLineSet &linSet)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (bb, TeLINES);
	q += " ORDER BY ext_max DESC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	bool flag = true;
	do
	{
		TeLine2D lin;
		flag = portal->fetchGeometry(lin);
		linSet.add(lin);
	}
	while (flag);
	delete portal;
	return true;
}

TeDatabasePortal* 
TeDatabase::loadLineSet (const string& table, TeBox &box)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return 0;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (box, TeLINES);
	q += " ORDER BY ext_max DESC";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return 0;
	}
	return portal;
}

bool 
TeDatabase::selectLineSet (const string& table, const string& criteria, TeLineSet &ls)
{
	TeDatabasePortal *portal = this->getPortal();
	string q ="SELECT * FROM " + table;
	if (!criteria.empty())
		q += " WHERE " + criteria;

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}

	bool flag = true;
	do 
	{
		TeLine2D line;
		flag = portal->fetchGeometry(line);
		ls.add ( line );
	}while(flag);

	delete portal;
	return true;
}

bool 
TeDatabase::locateLine (const string& table, TeCoord2D &pt, TeLine2D &line, const double& tol)
{
	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE lower_x < %f AND upper_x > %f AND lower_y < %f AND upper_y > %f",
		box.x2(),box.x1(),box.y2(),box.y1());
	q += buf;
	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	// Get all lines
	TeLineSet ls;
	int k;
	bool flag = true;
	do 
	{
		TeLine2D l;
		flag = portal->fetchGeometry( l );
		ls.add ( l );
	} while (flag);

	delete portal;

	TeCoord2D paux;

	if (TeNearest (pt, ls, k, paux, tol))
	{
		line = ls[k];
		return true;
	}
	return false;
}

bool 
TeDatabase::insertPointSet	(const string& table, TePointSet &ps)
{
	for (unsigned int i = 0; i < ps.size(); i++ )
	{
		TePoint point = ps [i];
		if (!insertPoint (table,point))
			return false;
	}
	return true;
}

bool 
TeDatabase::updatePointSet (const string& table, TePointSet &ps)
{
	for (unsigned int i = 0; i < ps.size(); i++ )
	{
		TePoint point = ps [i];
		if (!updatePoint (table,point))
			return false;
	}
	return true;
}

bool 
TeDatabase::loadPointSet (const string& table, const string& geoid, TePointSet &ps)
{
	TeDatabasePortal* portal = this->getPortal();
	string q ="SELECT * FROM " + table;

	if (!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	bool flag = true;
	do 
	{
		TePoint point;
		flag = portal->fetchGeometry (point);
		ps.add ( point );
	}while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::loadPointSet (const string& table, TeBox &bb, TePointSet &ps)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return false;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (bb, TePOINTS);

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	bool flag = true;
	do
	{
		TePoint pt;
		flag = portal->fetchGeometry(pt);
		ps.add(pt);
	}
	while (flag);
	delete portal;
	return true;
}

TeDatabasePortal* 
TeDatabase::loadPointSet(const string& table, TeBox &box)
{
	TeDatabasePortal *portal = this->getPortal();
	if (!portal)
		return 0;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (box, TePOINTS);

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return 0;
	}
	return portal;
}

bool 
TeDatabase::selectPointSet (const string& table, const string& criteria, TePointSet &ps)
{
	TeDatabasePortal* portal = this->getPortal();
	string q ="SELECT * FROM " + table;
	if (!criteria.empty())
		q += " WHERE " + criteria;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	bool flag = true;
	do 
	{
		TePoint point;
		flag = portal->fetchGeometry (point);
		ps.add ( point );
	}while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::locatePoint (const string& table, TeCoord2D &pt, TePoint &point, const double& tol)
{
	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE x < %f AND x > %f AND y < %f AND y > %f",
		box.x2(),box.x1(),box.y2(),box.y1());
	q += buf;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	TePointSet ps;
	bool flag = true;
	do 
	{
		TePoint point;
		flag = portal->fetchGeometry (point);
		ps.add ( point );
	}while (flag);

	delete portal;
	int k;
	if (TeNearest (pt, ps, k, tol))
	{
		point = ps[k];
		return true;
	}
	return false;
}

bool 
TeDatabase::insertTextSet	(const string& table, TeTextSet &ts)
{
	for (unsigned int i = 0; i < ts.size(); i++ )
	{
		TeText text = ts [i];
		if (!insertText (table,text))
			return false;
	}
	return true;
}

bool 
TeDatabase::updateTextSet	(const string& table, TeTextSet &ts)
{
	for ( unsigned int i = 0; i < ts.size(); i++ )
	{
		TeText text = ts [i];
		if (!updateText (table,text))
			return false;
	}
	return true;
}

bool 
TeDatabase::loadTextSet (const string& table, const string& geoid, TeTextSet &ts)
{
	TeDatabasePortal *portal = this->getPortal();
	
	string q ="SELECT * FROM " + table;
	if (!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	// Look for all texts
	bool flag  = true;
	do
	{
		TeText p;
		flag = portal->fetchGeometry(p);
		ts.add ( p );
	} while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::selectTextSet (const string& table, const string& criteria, TeTextSet &ts)
{
	TeDatabasePortal* portal = this->getPortal();
	string q ="SELECT * FROM " + table;
	if (!criteria.empty())
		q += " WHERE " + criteria;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	// Look for all texts
	bool flag = true;
	do
	{
		TeText p;
		flag = portal->fetchGeometry(p);
		ts.add ( p );
	} while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::updateText(const string& table, TeText &t)
{
	string sql;
	sql = "UPDATE " + table + " SET ";
    sql += "x=" + Te2String(t.location().x(),10) + ", ";
	sql += "y=" + Te2String(t.location().y(),10) + ", ";
	sql += "text_value='" + t.textValue() + "', ";
	sql += "angle=" + Te2String(t.angle(),5) + ", ";
	sql += "height=" + Te2String(t.height(),5) + ",";
	sql += "alignment_vert=" + Te2String(t.alignmentVert(),5) + ",";
	sql += "alignment_horiz=" + Te2String(t.alignmentHoriz(),5);
    sql += " WHERE geom_id=" + Te2String(t.geomId());
	return (this->execute(sql));
}

bool 
TeDatabase::updateNode(const string& table, TeNode &node)
{
	string sql;
	sql = "UPDATE " + table + " SET ";
    sql += "x=" + Te2String(node.location().x(),10) + ", ";
	sql += "y=" + Te2String(node.location().y(),10) + ", ";
    sql += " WHERE geom_id = " + Te2String(node.geomId());
	return (this->execute(sql));
}

bool 
TeDatabase::updatePoint(const string& table, TePoint &p)
{
	string sql;
	sql = "UPDATE " + table + " SET ";
    sql += "x=" + Te2String(p.location().x(),10) + ", ";
	sql += "y=" + Te2String(p.location().y(),10) + ", ";
    sql += " WHERE geom_id = " + Te2String(p.geomId());
	return (this->execute(sql));
}

bool TeDatabase::locateText (const string& table, TeCoord2D &pt, TeText &text, const double& tol)
{
	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE x < %f AND x > %f AND y < %f AND y > %f",
		box.x2(),box.x1(),box.y2(),box.y1());
	q += buf;
	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	// Look for all texts
	bool flag = true;
	TeTextSet ts;
	do
	{
		TeText p;
		flag = portal->fetchGeometry(p);
		ts.add ( p );
	} while (flag);

	delete portal;

	int k;
	if (TeNearest (pt, ts, k, tol))
	{
		text = ts[k];
		return true;
	}
	return false;
}

bool 
TeDatabase::insertArcSet	(const string& table, TeArcSet &as)
{
	for (unsigned int i = 0; i < as.size(); i++ )
	{
		TeArc arc = as [i];
		if (!insertArc (table,arc))
			return false;
	}
	return true;
}

bool 
TeDatabase::updateArcSet	(const string& table, TeArcSet &as)
{
	for ( unsigned int i = 0; i < as.size(); i++ )
	{
		TeArc arc = as [i];
		if (!updateArc (table,arc))
			return false;
	}
	return true;
}

bool 
TeDatabase::loadArcSet (const string& table, const string& geoid, TeArcSet &as)
{

	TeDatabasePortal* portal = this->getPortal();
	string q ="SELECT * FROM " + table;
	if (!geoid.empty())
		q += " WHERE object_id = " + geoid;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	// Look for all nodes
	bool flag;
	do 
	{
		TeArc arc;
		flag = portal->fetchGeometry(arc);
		as.add (arc);
	} while (flag);
	delete portal;
	return true;
}

bool 
TeDatabase::updateArc	(const string& /*table */, TeArc &arc)
{
	string sql;
	sql = "UPDATE table  SET ";
    sql += "from_node=" + Te2String(arc.fromNode().geomId()) + ", ";
	sql += "to_node=" + Te2String(arc.toNode().geomId()) + ", ";
    sql += " WHERE object_id = " + arc.geomId();
	return (this->execute(sql));
}

bool 
TeDatabase::insertNodeSet	(const string& table, TeNodeSet &ns)
{
	for (unsigned int i = 0; i < ns.size(); i++ )
	{
	     TeNode no = ns [i];
	     if (!insertNode (table,no))
		return false;
	}
	return true;
}

bool 
TeDatabase::updateNodeSet	(const string& table, TeNodeSet &ns)
{
	for (unsigned int i = 0; i < ns.size(); i++ )
	{
		TeNode no = ns [i];
		if (!updateNode (table,no))
			return false;
	}
	return true;
}

bool 
TeDatabase::loadNodeSet (const string& table, const string& geoid, TeNodeSet &ns)
{
	TeDatabasePortal* portal = this->getPortal();
	string q ="SELECT * FROM " + table;
	if (!geoid.empty())
		q += " WHERE object_id = " + geoid;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag = true;
	do 
	{
		TeNode n;
		flag = portal->fetchGeometry(n);
		ns.add ( n );
	}while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::insertCellSet	(const string& table, TeCellSet &cs)
{
	for (unsigned int i = 0; i < cs.size(); i++ )
	{
		TeCell cell = cs [i];
		if (!insertCell (table,cell))
			return false;
	}
	return true;
}

bool
TeDatabase::updateCellSet	(const string& table, TeCellSet &cs)
{
	for (unsigned int i = 0; i < cs.size(); i++ )
	{
		TeCell cell = cs [i];
		if (!updateCell (table,cell))
			return false;
	}
	return true;
}

bool 
TeDatabase::loadCellSet (const int& layerId, const string& table, const string& geoid, TeCellSet &cs)
{

	TeDatabasePortal *portal = this->getPortal();

	// Get the cell set resolution
	string q  = "SELECT * FROM te_representation WHERE layer_id = " + Te2String(layerId);
	q += " AND geom_type = " + Te2String(TeCELLS);

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	cs.resX(portal->getDouble("res_x"));
	cs.resY(portal->getDouble("res_y"));
	
	portal->freeResult();
		
	q = "SELECT * FROM " + table;
	if (!geoid.empty())
		q += " WHERE object_id = '" + geoid +"'";
	q += " ";

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag;
	do 
	{
		TeCell cell;
		flag = portal->fetchGeometry(cell);
		cs.add ( cell );
	} while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::selectCellSet (const int& layerId, const string& table, const string& criteria, TeCellSet &cs)
{
	TeDatabasePortal* portal = this->getPortal();
	string q = "SELECT * FROM te_representation WHERE layer_id = " ;
	q += Te2String(layerId) + " AND geom_type = " + Te2String(TeCELLS);

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	cs.resX(portal->getDouble("res_x"));
	cs.resY(portal->getDouble("res_y"));
	portal->freeResult();
	
	q ="SELECT * FROM " + table;
	if (!criteria.empty())
		q += " WHERE " + criteria;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag;
	do 
	{
		TeCell cell;
		flag = portal->fetchGeometry(cell);
		cs.add ( cell );
	} while (flag);

	delete portal;
	return true;
}

bool 
TeDatabase::updateCell(const string& /* table */, TeCell &c)
{
	TeBox b = c.box();

	string sql;
	sql = "UPDATE table  SET ";
	sql += "lower_x=" + Te2String(b.lowerLeft().x(),15) + ", ";
	sql += "lower_y=" + Te2String(b.lowerLeft().y(),15) + ", ";
	sql += "upper_x=" + Te2String(b.upperRight().x(),15) + ", ";
	sql += "upper_y=" + Te2String(b.upperRight().y(),15) + ", ";
	sql += "col_number=" + Te2String(c.column()) + ", ";
	sql += "row_number=" + Te2String(c.line());
	sql += " WHERE geom_id = " + c.geomId();
	return (this->execute(sql));
}

bool 
TeDatabase::locateCell (const string& table, TeCoord2D &pt, TeCell &cell, const double& /* tol */)
{
	TeDatabasePortal* portal = this->getPortal();
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE lower_x < %f AND upper_x > %f AND lower_y < %f AND upper_y > %f",
		pt.x(),pt.x(),pt.y(),pt.y());
	q += buf;
	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	portal->fetchGeometry(cell);
	delete portal;
	return true;
}

bool
TeDatabase::inClauseValues(const string& query, const string& /* attribute */, vector<string>& inClauseVector)
{
	inClauseVector.push_back( "(" + query + ")" );
	return true;
}

//Spatial query
//retornam um portal

bool 
TeDatabase::spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, TeDatabasePortal *portal, int relate, const string& actCollTable)
{
	return (TeTopologicalRelation(actGeomTable, actRep, actIdsIn, portal, relate, actCollTable));
}

bool 
TeDatabase::spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, const string& visGeomTable, TeGeomRep visRep, TeDatabasePortal *portal, int relate, const string& visCollTable)
{
	return (TeTopologicalRelation(actGeomTable, actRep, actIdsIn, visGeomTable, visRep, portal, relate, visCollTable));
}

bool 
TeDatabase::spatialRelation(const string& actGeomTable, TeGeomRep actRep, TeGeometry* geom, TeDatabasePortal *portal, int relate, const string& actCollTable)
{
	return (TeTopologicalRelation(actGeomTable, actRep, geom, portal, relate, actCollTable));
}

//retornam um vetor de object_ids resultantes da consulta
bool 
TeDatabase::spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, Keys& actIdsOut, int relate, const string& actCollTable)
{
	return (TeTopologicalRelation(actGeomTable, actRep, actIdsIn, actIdsOut, this, relate, actCollTable));
}

bool 
TeDatabase::spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, const string& visGeomTable, TeGeomRep visRep, Keys& visIdsOut, int relate, const string& visCollTable)
{
	return (TeTopologicalRelation(actGeomTable, actRep, actIdsIn, visGeomTable, visRep, visIdsOut, this, relate, visCollTable));
}

bool 
TeDatabase::spatialRelation(const string& actGeomTable, TeGeomRep actRep, TeGeometry* geom, Keys& actIdsOut, int relate, const string& actCollTable)
{
	return (TeTopologicalRelation(actGeomTable, actRep, geom, actIdsOut, this, relate, actCollTable));
}

//metric functions
bool
TeDatabase::calculateArea(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, double &area)
{
	return (TeGetArea(actGeomTable, actRep, actIdsIn, this, area));
}

bool 
TeDatabase::calculateLength(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn , double& length )
{
	return (TeGetLength(actGeomTable, actRep, actIdsIn, this, length));
}

bool 
TeDatabase::calculateDistance(const string& actGeomTable, TeGeomRep actRep, Keys& Ids, double& distance)
{
	return (TeGetDistance(actGeomTable, actRep, Ids, this, distance));
}

bool 
TeDatabase::calculateDistance(const string& /* actGeomTable */, TeGeomRep /*actRep */, const string& /* objId1 */, const string& /* visGeomTable */, TeGeomRep /* visRep */, const string& /* objId2 */, double& /* distance */)
{
	return false;
}

bool 
TeDatabase::withinDistance(const string& actGeomTable, TeGeomRep actRep, const TeCoord2D& coord, KeysToDist& IdsDistOut, const double& max_distance, const string& actCollTable)
{
	return (TeGetWithinDistance(actGeomTable, actRep, coord, IdsDistOut, this, max_distance, actCollTable));
}

// functions that generate new geometry
bool 
TeDatabase::Buffer(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TePolygonSet& bufferSet, double dist)
{
	return (TeGetBuffer(actGeomTable, actRep, actIds, this, bufferSet, dist));
}

bool 
TeDatabase::Centroid(const string&  actGeomTable , TeGeomRep actRep, TePointSet& centroidSet, Keys actIds, const string& actCollTable)
{
	return (TeGetCentroid(actGeomTable, actRep, this, centroidSet, actIds, actCollTable));
}

bool 
TeDatabase::ConvexHull(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TePolygonSet& convexHullSet)
{
	return (TeGetConvexHull(actGeomTable, actRep, actIds, this, convexHullSet));
}

bool 
TeDatabase::nearestNeighbors(const string& /* actGeomTable */, const string& /* actCollTable */, TeGeomRep /* actRep */, const string& /* objId1 */, Keys& /* actIdsOut */, int /* numRes */)
{
	return false;
}

bool 
TeDatabase::nearestNeighbors(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* visGeomTable */, const string& /* visCollTable */, TeGeomRep /* visRep */, Keys& /* visIdsOut */, int /* numRes */)
{
	return false;
}

bool 
TeDatabase::nearestNeighbors(const string& /* actGeomTable */, const string& /* actCollTable */, TeGeomRep /* actRep */, const string& /* objId1 */, TeDatabasePortal* /* portal */, int /* numRes */)
{
	return false;
}
	
bool 
TeDatabase::nearestNeighbors(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* visGeomTable */, const string& /* visCollTable */, TeGeomRep /* visRep */, TeDatabasePortal* /* portal */, int /* numRes */)
{
	return false;
}

bool 
TeDatabase::geomIntersection(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TeGeometryVect& geomVect)
{
	return (TeGetOverlay(actGeomTable, actRep, actIds, this, geomVect, TeINTERSECTION));
}
	
bool 
TeDatabase::geomIntersection(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* visGeomTable */, TeGeomRep /* visRep */, const string& /* objId2 */, TeGeometryVect& /* geomVect */)
{
	return false;
}

bool 
TeDatabase::geomDifference(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& objId2, TeGeometryVect& geomVect)
{
	Keys actIds;
	actIds.push_back(objId1);
	actIds.push_back(objId2);

	return (TeGetOverlay(actGeomTable, actRep, actIds, this, geomVect, TeDIFFERENCE));

}

bool 
TeDatabase::geomDifference(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* visGeomTable */, TeGeomRep /* visRep */, const string& /* objId2 */, TeGeometryVect& /* geomVect */)
{
	return false;
}

bool 
TeDatabase::geomUnion(const string&  actGeomTable, TeGeomRep actRep, Keys& actIds, TeGeometryVect& geomVect)
{
    return (TeGetOverlay(actGeomTable, actRep, actIds, this, geomVect, TeUNION));
}

bool 
TeDatabase::geomUnion(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* visGeomTable */, TeGeomRep /* visRep */, const string& /* objId2 */, TeGeometryVect& /* geomVect */)
{
	return false;
}

bool 
TeDatabase::geomXOr(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* objId2 */, TeGeometryVect& /* geomVect */)
{
	return false;
}

bool 
TeDatabase::geomXOr(const string& /* actGeomTable */, TeGeomRep /* actRep */, const string& /* objId1 */, const string& /* visGeomTable */, TeGeomRep /* visRep */, const string& /* objId2 */, TeGeometryVect& /* geomVect */)
{
	return false;
}

// Operation with Image

bool 
TeDatabase::Zonal(const string& rasterTable, const string& actGeomTable, Keys& Ids, TeObjectStatistics& result)
{
	TeDatabasePortal* portal=getPortal();
	if (!portal)
		return false;
	
	//recuperar o raster!!!
	string sql = "SELECT layer_id FROM te_representation";
	sql += " WHERE geom_table = '" + rasterTable +"'";
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	int layerId = atoi(portal->getData(0));
	TeRaster* raster = loadLayerRaster(layerId);

	//recuperar as geometrias
	portal->freeResult();
	
	string objIds = getStringIds(Ids);
	sql = "SELECT * FROM "+ actGeomTable;
	sql+= " WHERE object_id IN ("+ objIds +")";

	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);

		TeStatisticsDimensionVect st;

		TeRaster::iteratorPoly itBegin = raster->begin(poly, TeBoxPixelIn);
		TeRaster::iteratorPoly itEnd = raster->end(poly, TeBoxPixelIn);
	
		if(!TeCalculateStatistics (itBegin, itEnd, st))
		{
			delete portal;
			return false;
		}
	
		result[poly.objectId()] = st;

	}while (flag);

	
	delete raster;
	delete portal;
	return true;
}
	
bool 
TeDatabase::Zonal(const string& rasterTable, const string& actGeomTable, const string& actCollTable, TeObjectStatistics& result)
{
	TeDatabasePortal* portal=getPortal();
	if (!portal)
		return false;
	
	//recuperar o raster
	string sql = "SELECT layer_id FROM te_representation";
	sql += " WHERE geom_table = '" + rasterTable +"'";
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	int layerId = atoi(portal->getData(0));
	TeRaster* raster = loadLayerRaster(layerId);

	//recuperar as geometrias
	portal->freeResult();
	
	sql = "SELECT * FROM "+ actGeomTable;

	if(!actCollTable.empty())
	{
		sql += " ,"+ actCollTable;
		sql += " WHERE object_id = c_object_id ";
	}
	
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);

		TeStatisticsDimensionVect st;
		
		TeRaster::iteratorPoly itBegin = raster->begin(poly, TeBoxPixelIn);
		TeRaster::iteratorPoly itEnd = raster->end(poly, TeBoxPixelIn);
	
		if(!TeCalculateStatistics (itBegin, itEnd, st))
		{
			delete portal;
			return false;
		}
		result[poly.objectId()] = st;

	}while (flag);

	
	delete raster;
	delete portal;
	return true;
}

bool 
TeDatabase::Zonal(const string& rasterTable, TePolygon& poly, TeStatisticsDimensionVect& result)
{
	TeDatabasePortal* portal=getPortal();
	if (!portal)
		return false;
	
	//recuperar o raster!!!
	string sql = "SELECT layer_id FROM te_representation";
	sql += " WHERE geom_table = '" + rasterTable +"'";
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	int layerId = atoi(portal->getData(0));
	TeRaster* raster = loadLayerRaster(layerId);

	delete portal;

	TeRaster::iteratorPoly itBegin = raster->begin(poly, TeBoxPixelIn);
	TeRaster::iteratorPoly itEnd = raster->end(poly, TeBoxPixelIn);
	
	if(!TeCalculateStatistics (itBegin, itEnd, result))
		return false;

	delete raster;
	return true;
}


bool 
TeDatabase::Mask(const string& rasterTable, const string& actGeomTable, const string& objId, const string& nameLayerOut, TeStrategicIterator st)
{
	TeDatabasePortal* portal=getPortal();
	if (!portal)
		return false;
	
	//recuperar o raster!!!
	string sql = "SELECT layer_id FROM te_representation";
	sql += " WHERE geom_table = '" + rasterTable +"'";
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	int layerId = atoi(portal->getData(0));
	TeRaster* raster = loadLayerRaster(layerId);

	//recuperar a geometria
	portal->freeResult();
	
	sql = "SELECT * FROM "+ actGeomTable;
	sql+= " WHERE object_id = '" + objId + "'"; 
	
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	TePolygon poly;
	portal->fetchGeometry(poly);

	TeRaster* rasterOut = TeMask (raster, poly, st);
	TeRasterParams param = rasterOut->params();	

	TeBox b = param.boundingBox();
	TeLayer* layer = new TeLayer(nameLayerOut,this, b , rasterOut->projection());
	bool status = TeImportRaster(layer,rasterOut, param.blockWidth_, param.blockHeight_, param.compression_[0]);

	delete raster;
	delete rasterOut;
	delete portal;
	return status;
}



bool
TeDatabase::Mask(const string& rasterTable, TePolygon& poly, const string& nameLayerOut, TeStrategicIterator st)
{
	TeDatabasePortal* portal=getPortal();
	if (!portal)
		return false;
	
	//recuperar o raster!!!
	string sql = "SELECT layer_id FROM te_representation";
	sql += " WHERE geom_table = '" + rasterTable +"'";
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	int layerId = atoi(portal->getData(0));
	TeRaster* raster = loadLayerRaster(layerId);

	delete portal;

	TeRaster* rasterOut = TeMask (raster, poly, st);
	TeRasterParams param = rasterOut->params();	

	TeBox b = param.boundingBox();
	TeLayer* layer = new TeLayer(nameLayerOut,this, b , rasterOut->projection());
	bool status = TeImportRaster(layer,rasterOut, param.blockWidth_, param.blockHeight_, param.compression_[0]);
	
	delete raster;
	delete rasterOut;
	return status;
}


string
TeDatabase::getSQLBoxWhere (TeBox &box, TeGeomRep rep)
{
	string wherebox;
	string lowerX, lowerY, upperX, upperY;
	if(rep == TePOLYGONS || rep == TeLINES || rep == TeCELLS || rep == TeRASTER)
	{
		lowerX = "lower_x";
		lowerY = "lower_y";
		upperX = "upper_x";
		upperY = "upper_y";
	}
	else if(rep == TePOINTS || rep == TeTEXT)
	{
		lowerX = "x";
		lowerY = "y";
		upperX = "x";
		upperY = "y";
	}

	wherebox = "NOT("+ lowerX +" >= " + Te2String(box.x2_, 12) + " OR ";
	wherebox += upperX +" <= " + Te2String(box.x1_, 12) + " OR ";
	wherebox += lowerY +" >= " + Te2String(box.y2_, 12) + " OR ";
	wherebox += upperY +" <= " + Te2String(box.y1_, 12) + ")";
		
	return wherebox;
}


string 
TeDatabase::getSQLBoxWhere (const string& table1, const string& table2, TeGeomRep rep2, TeGeomRep /* rep1 */)
{
	string wherebox;

	if(rep2 == TePOLYGONS || rep2 == TeLINES || rep2 == TeCELLS || rep2 == TeRASTER)
	{
		wherebox = " NOT( ";
		wherebox += table2 +".lower_x > "+ table1 +".upper_x  OR ";
		wherebox += table2 +".upper_x < "+ table1 +".lower_x  OR ";
		wherebox += table2 +".lower_y > "+ table1 +".upper_y  OR ";
		wherebox += table2 +".upper_y < "+ table1 +".lower_y )";
	}
	else if(rep2 == TePOINTS || rep2 == TeTEXT)
	{
		wherebox = " NOT( ";
		wherebox += table2 +".x > "+ table1 +".upper_x  OR ";
		wherebox += table2 +".x < "+ table1 +".lower_x  OR ";
		wherebox += table2 +".y > "+ table1 +".upper_y  OR ";
		wherebox += table2 +".y < "+ table1 +".lower_y )";
	}
	
	return wherebox;
}


string 
TeDatabase::getSQLBoxSelect (const string& tableName, TeGeomRep /* rep */)
{
	string sql = tableName +".* ";
	return sql;
}


string
TeDatabase::getSQLStatistics (TeGroupingAttr& attrs)
{
	string sql = "";
	string virg = "";

	TeGroupingAttr::iterator it = attrs.begin();
	int count = 0;
	while(it != attrs.end())
	{
		if(count>0)
			virg = ",";

		switch ((*it).second)
		{
			case TeSUM:
				sql += virg +" SUM( "+ (*it).first.name_ +") AS SUM_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMAXVALUE:
				sql += virg +" MAX( "+ (*it).first.name_ +") AS MAX_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC; 
				++count;
				break;
			case TeMINVALUE:
				sql += virg +" MIN( "+ (*it).first.name_ +") AS MIN_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeCOUNT:
				sql += virg +" COUNT( "+ (*it).first.name_ +") AS COUNT_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMEAN:
				sql += virg +" AVG( "+ (*it).first.name_ +") AS AVG_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			default:
				break;
		}
		++it;
	}
	return sql;
}

string 
TeDatabase::getSQLAutoNumber(const string& /* table */)
{
	return "";
}

string 
TeDatabase::getSQLTemporalWhere (TeTimeInterval& timeInterval, TeTemporalRelation timeOperator, const string& initialTime, const string& finalTime)
{
	string sql;
	string t1 = getSQLTime(timeInterval.getT1()); 
	string t2 = getSQLTime(timeInterval.getT2());
	
	switch(timeOperator)
	{
		case TeTIMEBEFORE:  
			sql = finalTime +" < "+ t1;
			break;
		case TeTIMEAFTER:  
			sql = initialTime +" > "+ t2;
			break;
		case TeTIMEEQUAL:             
			sql = "( "+ initialTime +" >= "+ t1;
			sql += " AND "+ initialTime +" <= "+ t2 +" )";
			if (initialTime != finalTime)
			{
				sql += " AND ";
				sql += "( "+ finalTime +" >= "+ t1;
				sql += " AND "+ finalTime +" <= "+ t2 +" )";
			}
			break;

		case TeTIMEMEETS:             
			sql = finalTime +" = "+ t1;
			sql += " OR "+ initialTime +" = "+ t2;
			break;

		case TeTIMEDURING: 
			sql = initialTime +" >= "+ t1;
			sql += " AND "+ initialTime +" <= "+ t2;
			if (initialTime != finalTime)
			{
				sql += " AND "+ finalTime +" >= "+ t1;
				sql += " AND "+ finalTime +" <= "+ t2;
			}
			break;

		case TeTIMEOVERLAPS:             
			sql = "( "+ initialTime +" < "+ t1;
			sql += " AND "+ finalTime +" > "+ t1;
			sql += " AND "+ finalTime +" < "+ t2 +" )";
			sql += " OR ";
			sql += "( "+ initialTime +" > "+ t1;
			sql += " AND "+ initialTime +" < "+ t2;
			sql += " AND "+ finalTime +" > "+ t2 +" )";
			break;

		case TeTIMEENDS:
			sql = finalTime +" = "+ t2;
			break;

		case TeTIMESTARTS:
			sql = initialTime +" = "+ t1;
			break;

        default:
            break;
	}

	return sql; 
}

string
TeDatabase::getSQLTemporalWhere(int time1, int time2, TeChronon chr, TeTemporalRelation rel, 
						   const string& initialTime, const string& finalTime)
{
	//rever os chronons definidos - alterar o parser para restrio temporal
	string func, sql;
	switch(chr)
	{
		case TeSECONDOFMINUTE:  
		   func = " second"; 
		break;
		
		case TeMINUTEOFHOUR:  
		   func = " minute";
		break;
		
		case TeHOUROFDAY:             
			func = " hour";
		break;
		
		case TeDAYOFMONTH:             
			func = " day";
		break;

		case TeDAYOFWEEK:
			func = " weekday";
		break;
		
		case TeMONTHOFYEAR:
		   func = " month";
		break;

		case TeYEAR:             
		   func = " year";
		break;

		default:
			return "";
	}

	switch(rel)
	{
		case TeTIMEBEFORE:  
			sql = func +"("+ finalTime +") < "+ Te2String(time1);
		break;
		
		case TeTIMEAFTER:  
			sql = func +"("+ initialTime +") > "+ Te2String(time2);
		break;
		
		case TeTIMEEQUAL:  
			sql = func +"("+ initialTime +") = "+ Te2String(time1);
			sql += " AND "+ func +"(" + finalTime +") = "+ Te2String(time2);
		break;

		case TeTIMEMEETS:     
			sql = func +"("+ finalTime +") = "+ Te2String(time1);
			sql += " OR "+ func +"(" + initialTime +") = "+ Te2String(time2);
		break;

		case TeTIMEDURING: 
			sql = func +"("+ initialTime +") >= "+ Te2String(time1);
			sql += " AND "+ func +"("+ initialTime +") <= "+ Te2String(time2);
			sql += " AND "+ func +"("+ finalTime +") >= "+ Te2String(time1);
			sql += " AND "+ func +"("+ finalTime +") <= "+ Te2String(time2);
		break;

		case TeTIMEOVERLAPS:         
			sql =  "("+ func +"("+ initialTime +") <= "+ Te2String(time1);
			sql += " AND "+ func +"("+ finalTime +") >= "+ Te2String(time1);
			sql += " AND "+ func +"("+ finalTime +") <= "+ Te2String(time2) +")";
			sql += " OR ";
			sql += "("+ func +"("+ initialTime +") >= "+ Te2String(time1);
			sql += " AND "+ func +"("+ initialTime +") <= "+ Te2String(time2);
			sql += " AND "+ func +"("+ finalTime +") >= "+ Te2String(time2) +")";
		break;

		case TeTIMEENDS:
			sql = func +"("+ finalTime +") = "+ Te2String(time2);
		break;

		case TeTIMESTARTS:
			sql = func +"("+ initialTime +") = "+ Te2String(time1);

		default:
			break;
	}
	
	return sql;
}

string 
TeDatabase::getSQLTemporalWhere (const string& temporalRest)
{
	string result, tableName, initialCol, finalCol, time1, time2, mask; 
	TeTemporalRelation rel;
	TeChronon chronon; 
	
	string temp = temporalRest;
	bool flag = true;
	int cont = 0;
		
	while(flag)
	{
		string element;
		int pos = temp.find (";");
		if(pos<0)
		{
			flag = false;
			element = temp;
		}
		else
		{
			element = temp.substr(0, pos);
			temp = temp.substr (pos+1);
		}

		if(cont==0) //table name
			tableName = element;
		else if(cont==1) //column name (initial time) 
			initialCol = element; 
		else if(cont==2) //column name (final time)
			finalCol = element;
		else if(cont==3) //TeTemporalRelation 
		{
			if(element=="TeTIMEEQUAL")
				rel = TeTIMEEQUAL;
			else if (element=="TeTIMEBEFORE")
				rel = TeTIMEBEFORE;
			else if (element=="TeTIMEAFTER")
				rel = TeTIMEAFTER;
			else if (element=="TeTIMEMEETS")
				rel = TeTIMEMEETS;
			else if (element=="TeTIMEDURING")
				rel = TeTIMEDURING;
			else if (element=="TeTIMEOVERLAPS")
				rel = TeTIMEOVERLAPS;
			else if (element=="TeTIMEENDS")
				rel = TeTIMEENDS;
			else if (element=="TeTIMESTARTS")
				rel = TeTIMESTARTS;
		}
		else if(cont==4) //time 1
			time1 = element;
		else if(cont==5) //time 2
		{
			if(element.empty())
				time1 = time2;
			else
				time2 = element;
		}
		else if(cont==6)
			mask = element;
		else if(cont==7) //TeChronon  
		{
			if (element=="TeSECOND")
				chronon = TeSECOND;
			else if (element=="TeMINUTE")
				chronon = TeMINUTE;
			else if (element=="TeHOUR")
				chronon = TeHOUR;
			else if (element=="TeDAY")
				chronon = TeDAY;
			else if (element=="TeMONTH")
				chronon = TeMONTH;
			else if (element=="TeYEAR")
				chronon = TeYEAR;
			else if (element=="TeDAYOFWEEK")
				chronon = TeDAYOFWEEK;
			else if (element=="TeMONTHOFYEAR")
				chronon = TeMONTHOFYEAR;
			else if (element=="TeSEASON")
				chronon = TeSEASON;
			else if (element=="TeWEEKOFYEAR")
				chronon = TeWEEKOFYEAR;
		}

		++cont;
	}

	int posMask = mask.find ("s");
	if(posMask<0) 
	{
		result = " "+ getSQLTemporalWhere(atoi(time1.c_str()), atoi(time2.c_str()), chronon, rel, (tableName+"."+initialCol), (tableName+"."+finalCol)) +" ";
	}
	else
	{
		TeTimeInterval interval(time1, time2, chronon, mask);
		result = " "+ getSQLTemporalWhere(interval, rel, (tableName+"."+initialCol), (tableName+"."+finalCol)) + " "; 
	}

	return result;
}


string 
TeDatabase::getSQLTemporalFunction (TeChronon chr, const string& colName)
{
	string func;
	switch(chr)
	{
		case TeYEAR:             
			func = " year ("+ colName +")";
		break;

		case TeMONTH:			// Fev/2003 != Fev/2004
			func = " year ("+ colName +"), month ("+ colName +") ";
		break;

		case TeMONTHOFYEAR:		// Fev/2003 == Fev/2004
			func = " month ("+ colName +")";
		break;

		case TeDAY:				// 01/01/2003 != 01/01/2004 != 01/02/2004             
			func = " year ("+ colName +"), month ("+ colName +"), day("+ colName +") ";
		break;

		case TeDAYOFMONTH:		// 01/01/2003 == 01/01/2004 == 01/02/2004             
			func = " day("+ colName +") ";
		break;

		case TeDAYOFYEAR:		// 01/01/2003 == 01/01/2004 != 01/02/2004             
			func = " month ("+ colName +"), day("+ colName +") ";
		break;

		case TeDAYOFWEEK:		// 01/01/2003 != 01/01/2004 != 01/02/2004             
			func = " weekday ("+ colName +") ";
		break;

		case TeHOUR:			// 01/01/2003 10:00 != 01/01/2004 10:00 != 01/02/2004 10:00             
			func = " year ("+ colName +"), month ("+ colName +"), day("+ colName +"), hour("+ colName +") ";
		break;

		case TeHOUROFDAY:       // 01/01/2003 10:00 == 01/01/2004 10:00 == 01/02/2004 10:00    
			func = " hour ("+ colName +")";
		break;

		case TeMINUTE:			// 01/01/2003 10:30 != 01/01/2004 10:30 != 01/02/2004 10:30 
		   func = " year ("+ colName +"), month ("+ colName +"), day("+ colName +"), hour("+ colName +"), minute("+ colName +") ";
		break;

		case TeMINUTEOFHOUR:   // 01/01/2003 10:30 == 01/01/2004 10:30 == 01/02/2004 10:30 != 01/02/2004 10:31
			func = " minute("+ colName +") ";
		break;

		case TeSECOND:  // 01/01/2003 10:30:04 != 01/01/2004 10:30:04 != 01/02/2004 10:30:04 
		   func = " year ("+ colName +"), month ("+ colName +"), day("+ colName +"), hour("+ colName +"), minute("+ colName +"), second("+ colName +") ";
		break;

		case TeSECONDOFMINUTE:  // 01/01/2003 10:30:04 == 01/01/2004 10:30:04 == 01/02/2004 10:30:04 != 01/02/2004 10:30:07
		   func = " second("+ colName +") "; 
		break;
		
		default:
			return "";
	}
	
	return func;
}
 

bool
TeDatabase::getMBRGeom(string tableGeom, string object_id, TeBox& box, string /* colGeom */)
{
	double xmin = TeMAXFLOAT;
	double xmax = -TeMAXFLOAT;
	double ymin = TeMAXFLOAT;
	double ymax = -TeMAXFLOAT;
			
	TeDatabasePortal* portal = getPortal();
	if(!portal)
		return false;

	string sel = "SELECT lower_x, upper_x, lower_y, upper_y FROM " + tableGeom;
	sel += " WHERE object_id = '" + object_id + "'";

	if(!portal->query(sel))
	{
		delete portal;
		return false;
	}
		
	bool b = portal->fetchRow();
	if(!b)
	{
		delete portal;
		return false;
	}
	
	while(b)
	{
		xmin = MIN(xmin, portal->getDouble(0));
		xmax = MAX(xmax, portal->getDouble(1));
		ymin = MIN(ymin, portal->getDouble(2));
		ymax = MAX(ymax, portal->getDouble(3));
		b = portal->fetchRow();	
	}
			
	TeBox bb(xmin, ymin, xmax, ymax);
	box = bb;
	delete portal;
	return true;
}

bool 
TeDatabase::getMBRSelectedObjects(string /* geomTable */,string /* colGeom */, string fromClause, string whereClause, string afterWhereClause, TeGeomRep repType,TeBox &bout, const double& tol)
{
	string	fields;
	string	query;
	bool	status = false;

	TeBox	box;
	bout = box;

	TeDatabasePortal* portal = this->getPortal();

	switch(repType)
	{
		case TePOLYGONS:
		case TeLINES:

			fields = "MIN(lower_x), MIN(lower_y), MAX(upper_x), MAX(upper_y)";
			query =  " SELECT " + fields;
			query += " FROM " + fromClause; 
			if (!whereClause.empty())
				query += " WHERE " + whereClause;
			if (!afterWhereClause.empty())
				query += afterWhereClause;

			if (portal->query (query))
			{
				bool b = portal->fetchRow();
				while(b)
				{
					string vxmin = portal->getData(0);
					string vymin = portal->getData(1);
					string vxmax = portal->getData(2);
					string vymax = portal->getData(3);
					if(vxmin.empty() || vymin.empty() || vxmax.empty() || vymax.empty())
					{
						b = portal->fetchRow();
						continue;
					}
					double xmin = atof(vxmin.c_str());
					double ymin = atof(vymin.c_str());
					double xmax = atof(vxmax.c_str());
					double ymax = atof(vymax.c_str());
					TeBox	ibox(xmin, ymin, xmax, ymax);
					updateBox (bout, ibox);
					b = portal->fetchRow();
					status = true;
				}
			}
			break;

		case TePOINTS:

			fields = "MIN(x), MIN(y), MAX(x), MAX(y)";
			query =  " SELECT " + fields;
			query += " FROM " + fromClause; 
			if (!whereClause.empty())
				query += " WHERE " + whereClause;
			if (!afterWhereClause.empty())
				query += afterWhereClause;
			
			if (portal->query (query))
			{
				bool b = portal->fetchRow();
				while(b)
				{
          string vxmin = portal->getData(0);
          string vymin = portal->getData(1);
          string vxmax = portal->getData(2);
          string vymax = portal->getData(3);
          if(vxmin.empty() || vymin.empty() || vxmax.empty() || vymax.empty())
          {
            b = portal->fetchRow();
            continue;
          }
          double xmin = atof(vxmin.c_str());
          double ymin = atof(vymin.c_str());
          double xmax = atof(vxmax.c_str());
          double ymax = atof(vymax.c_str());
          
          TeBox ibox;
          if (xmin == xmax) {
            ibox.x1_ = xmin - tol;
            ibox.x2_ = xmax + tol;
          }
          else{
            ibox.x1_ = xmin;
            ibox.x2_ = xmax;
          }
          if (ymin == ymax) {
            ibox.y1_ = ymin - tol;
            ibox.y2_ = ymax + tol;
          }
          else {
            ibox.y1_ = ymin;
            ibox.y2_ = ymax;
          }
					updateBox (bout, ibox);
					b = portal->fetchRow();
					status = true;
				}
			}
			break;


		default:
			status = false;
			break;
	}

	delete portal;

	return status;

}

bool 
TeDatabase::getAttributeList(const string& tableName,TeAttributeList& attList)
{
	TeDatabasePortal* portal = this->getPortal();
	if (!portal)
		return false;

	string sql = "SELECT * FROM " + tableName + " WHERE 1=2";
	if (!portal->query(sql))
	{
		delete portal;
		return false;
	}
	else
		attList = portal->AttributeList();
	delete portal;
	return true;
}

TeBox 
TeDatabase::getThemeBox(TeTheme* theme)
{
	TeBox bb;
	if (!theme)
		return bb;

	string colTabName = theme->collectionTable();
	if (colTabName.empty())
		return bb;

	string sqlfrom;
	TeLayer* layer = theme->layer();
	string geomTable;
	if (layer->hasGeometry(TePOINTS))
	{
		geomTable = layer->tableName(TePOINTS);
		sqlfrom = colTabName + " LEFT JOIN " + geomTable;
		sqlfrom += " ON " + colTabName + ".c_object_id = " + geomTable + ".object_id";
		TeBox bpt;
		if (getMBRSelectedObjects(geomTable,"spatial_data", sqlfrom, "","",TePOINTS,bpt))
			updateBox(bb,bpt);
	}
	if (layer->hasGeometry(TeLINES))
	{
		geomTable = layer->tableName(TeLINES);
		sqlfrom = colTabName + " LEFT JOIN " + geomTable;
		sqlfrom += " ON " + colTabName + ".c_object_id = " + geomTable + ".object_id";
		TeBox bln;
		if (getMBRSelectedObjects(geomTable,"spatial_data", sqlfrom, "","",TeLINES,bln))
			updateBox(bb,bln);
	}
	if (layer->hasGeometry(TePOLYGONS))
	{
		geomTable = layer->tableName(TePOLYGONS);
		sqlfrom = colTabName + " LEFT JOIN " + geomTable;
		sqlfrom += " ON " + colTabName + ".c_object_id = " + geomTable + ".object_id";
		TeBox bpol;
		if (getMBRSelectedObjects(geomTable,"spatial_data", sqlfrom, "","",TePOLYGONS,bpol))
			updateBox(bb,bpol);
	}

	if (layer->hasGeometry(TeCELLS))
	{
		geomTable = layer->tableName(TeCELLS);
		sqlfrom = colTabName + " LEFT JOIN " + geomTable;
		sqlfrom += " ON " + colTabName + ".c_object_id = " + geomTable + ".object_id";
		TeBox bpol;
		if (getMBRSelectedObjects(geomTable,"spatial_data", sqlfrom, "","",TeCELLS,bpol))
			updateBox(bb,bpol);
	}
	return bb;
}

bool TeDatabase::insertRasterVisual (int themeId , TeRasterVisual* vis)
{
	TeRasterTransform::TeRasterTransfFunctions tf = vis->getTransfFunction();
		
	if (tf == TeRasterTransform::TeNoTransf)
		return true;

	string sql = "DELETE FROM te_visual_raster WHERE theme_id = " + Te2String(themeId);
	if (!this->execute(sql))
		return false;

	if (tf == TeRasterTransform::TeMono2Three ||
		tf == TeRasterTransform::TePall2Three || 
		tf == TeRasterTransform::TeLUT2Three)
	{
		sql = "INSERT INTO te_visual_raster (theme_id, band_in, transf_type) VALUES (";
		sql += Te2String(themeId) + "," + Te2String(vis->getSrcBand()) + ",";
		sql += Te2String(static_cast<short>(tf)) + ")";
		if (!this->execute(sql))
			return false;
		return true;
	}

	if (tf == TeRasterTransform::TeExtractBand)
	{
		sql = "INSERT INTO te_visual_raster (theme_id, band_in, band_out, transf_type) VALUES (";
		sql += Te2String(themeId) + "," + Te2String(vis->getSrcBand()) + ",";
		sql += Te2String(vis->getDestBand()) + "," + Te2String(static_cast<short>(tf))+ ")";
		if (!this->execute(sql))
			return false;
		return true;
	}

	if (tf == TeRasterTransform::TeExtractBands || tf == TeRasterTransform::TeExtractRGB)
	{
		map<TeRasterTransform::TeRGBChannels,short>& RGBmap = vis->getRGBMap();
		map<TeRasterTransform::TeRGBChannels,short>::iterator it = RGBmap.begin();
		while (it != RGBmap.end())
		{
			sql = "INSERT INTO te_visual_raster (theme_id, band_in, band_out, transf_type) VALUES (";
			sql += Te2String(themeId) + "," + Te2String(static_cast<short>(it->second)) + ",";
			sql += Te2String(static_cast<short>(it->first)) + "," + Te2String(static_cast<short>(tf))+ ")";
			if (!this->execute(sql))
				return false;
			++it;
		}
		return true;
	}
	return false;
}


bool 
TeDatabase::locateLineSet (const string& table, TeCoord2D &pt, TeLineSet &ls, const double& tol)
{
	bool located=false;
	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE lower_x < %f AND upper_x > %f AND lower_y < %f AND upper_y > %f",
		box.x2(),box.x1(),box.y2(),box.y1());
	q += buf;
	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	// Get all lines
	bool flag = true;
	do 
	{
		TeLine2D l;
		TeLineSet tmp;
		int index;
		double dist;
		TeCoord2D pout;

		flag = portal->fetchGeometry( l );
		tmp.add(l);
		if(TeNearest(pt,tmp,index,pout,dist,tol))
			{
				ls.add ( l );
				located=true;
			}
	} while (flag);

	delete portal;
	return located;
}


bool 
TeDatabase::locatePointSet (const string& table, TeCoord2D &pt, TePointSet &pointSet, const double& tol)
{
	bool located=false;

	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE x < %f AND x > %f AND y < %f AND y > %f",
		box.x2(),box.x1(),box.y2(),box.y1());
	q += buf;

	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	
	bool flag = true;
	do 
	{
		TePoint point;
		flag = portal->fetchGeometry (point);
		pointSet.add ( point );
		located=true;
	}while (flag);

	delete portal;
	return located;
}


bool 
TeDatabase::locateTextSet (const string& table, TeCoord2D &pt, TeTextSet& textSet, const double& tol)
{
	bool located=false;

	TeDatabasePortal* portal = this->getPortal();

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	string q ="SELECT * FROM " + table;
	char buf [1024];
	sprintf (buf," WHERE x < %f AND x > %f AND y < %f AND y > %f",
		box.x2(),box.x1(),box.y2(),box.y1());
	q += buf;
	if (!portal->query(q) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}
	// Look for all texts
	bool flag = true;
	
	do
	{
		TeText p;
		flag = portal->fetchGeometry(p);
		textSet.add ( p );
		located=true;
	} while (flag);

	delete portal;
	return located;
}


// End TeDatabase Methods

// Begin TeDatabasePortal Methods

TeDatabasePortal::TeDatabasePortal():
	db_(0),			
	numRows_(0),
	numFields_ (0),	
	errorMessage_(""),
	errorNumber_(0)
	{}

TeDatabasePortal::~TeDatabasePortal ()
{
}

double 
TeDatabasePortal::getDouble (int i)
{  
	char* val = getData(i); 
	return atof(val); 
}

double 
TeDatabasePortal::getDouble (const string& s)
{  
	char* val = getData(s); 
	return atof(val); 
}

int 
TeDatabasePortal::getInt (int i)
{  
	char* val = getData(i); 
	return atoi(val); 
}

int 
TeDatabasePortal::getInt (const string& s)
{  
	char* val = getData(s); 
	return atoi(val); 
}

bool
TeDatabasePortal::fetchGeometry (TeCell& cell)
{
	int index = atoi(getData("geom_id"));
	string object_id = getData("object_id");
	TeBox b (atof(getData("lower_x")),atof(getData("lower_y")),atof(getData("upper_x")),atof(getData("upper_y")));
	cell.geomId(index);
	cell.objectId(object_id);
	cell.setBox (b);
	cell.column(atoi(getData("col_number")));
	cell.line(atoi(getData("row_number")));
	return (fetchRow());
}

TeAttribute TeDatabasePortal::getAttribute (int i)
{
	TeAttributeList::iterator it = attList_.begin();
	int j = 0;
	while ( it != attList_.end() )
	{
		if (i == j)
			return (*it);
		++it;
		j++;
	}
	return TeAttribute();
}

TeAttribute TeDatabasePortal::getAttribute (const string& s)
{
	TeAttributeList::iterator it = attList_.begin();
	while ( it != attList_.end() )
	{
		if (TeConvertToUpperCase(s) == TeConvertToUpperCase((*it).rep_.name_))
			return (*it);
		++it;
	}
	return TeAttribute();
}

int 
TeDatabasePortal::getColumnIndex (const string& s)
{
	TeAttributeList::iterator it = attList_.begin();
	int j = 0;
	while ( it != attList_.end() )
	{
		if (TeConvertToUpperCase(s) == TeConvertToUpperCase((*it).rep_.name_))
			return j;
		++it;
		j++;
	}
	return -1;
}

string 
TeDatabasePortal::getColumnName (int i)
{
	TeAttributeList::iterator it = attList_.begin();
	int j=0;
	while ( it != attList_.end() )
	{
		if (j==i)
			return (*it).rep_.name_;
		++it;
		j++;
	}
	return "";
}


TeViewTree*
TeDatabasePortal::getViewTree ()
{
	TeViewTree *tree = new TeViewTree();
	tree->id (atoi(getData("theme_id")));
	tree->name (getData("name"));
	tree->priority(atoi(getData("priority")));
	return tree;
}

TeLegendEntry 
TeDatabasePortal::getLegend ()
{
	TeLegendEntry leg;
	leg.id (atoi(getData("legend_id")));
	leg.theme (atoi(getData("theme_id")));
	leg.group (atoi(getData("group_id")));

	string data;
	data = getData("num_objs");
	leg.count(atoi(data.c_str()));
	data = getData("lower_value");
	leg.from(data);
	data = getData("upper_value");
	leg.to(data);
	data = getData("label");
	leg.label(data);
	return leg;
}

TeVisual
TeDatabasePortal::getVisual()
{
	TeVisual vis;
	TeGeomRep rep = (TeGeomRep)atoi (getData("geom_type"));

	TeColor cor(atoi(getData("red")),atoi(getData("green")),atoi(getData("blue")));
	vis.color(cor);
	vis.libName (string(getData("lib_name")));
	vis.transparency(atoi(getData("transparency")));

	TeColor ccor(atoi(getData("contour_red")),atoi(getData("contour_green")),atoi(getData("contour_blue")));
	vis.contourColor(ccor);
	vis.contourLibName(string(getData("contour_lib_name")));
	vis.contourWidth(atoi(getData("contour_width")));
	vis.contourTransparency(atoi(getData("contour_transp")));

	if(rep == TePOLYGONS || rep == TeCELLS)
	{
		vis.contourWidth(atoi(getData("width")));
		vis.contourStyle(atoi(getData("contour_symb_id")));
		vis.style(atoi(getData("symb_id")));
	}
	else if(rep == TeLINES)
	{
		vis.width(atoi(getData("width")));
		vis.style(atoi(getData("symb_id")));
	}
	else if(rep == TePOINTS)
	{
		vis.size(atoi(getData("size_value")));
		vis.style(atoi(getData("symb_id")));
	}
	else if(rep == TeTEXT)
		vis.size(atoi(getData("size_value")));

	vis.family(getData("family"));
	vis.bold (getBool("bold"));
	vis.italic (getBool("italic"));
	vis.fixedSize (getBool("fixed_size"));

	vis.alignmentVert(getDouble("alignment_vert"));
	vis.alignmentHoriz(getDouble("alignment_horiz"));

	vis.tabSize(atoi(getData("tab_size")));
	vis.lineSpace(atoi(getData("line_space")));
	return vis;
}

bool 
TeDatabasePortal::fetchGeometry (TeText& t)
{
	TeCoord2D c(getDouble("x"), getDouble("y"));
	string txt = getData ("text_value");
	TeText t2(c,txt);
	t2.geomId(atoi(getData("geom_id")));
	t2.objectId(string(getData("object_id")));
	t2.setAngle (getDouble("angle"));
	t2.setHeight (getDouble("height"));
	t2.setAlignmentVert(getDouble("alignment_vert"));
	t2.setAlignmentHoriz(getDouble("alignment_horiz"));
	t = t2;
	return (fetchRow());
}

TeColor 
TeDatabasePortal::getColor ()
{
	TeColor c(atoi(getData("red")), atoi(getData("green")), atoi(getData("blue")));
	return c;
}

bool 
TeDatabasePortal::fetchGeometry (TeArc& arc)
{
	arc.fromId(atoi(getData(2)));
	arc.toId(atoi(getData(3)));
	arc.geomId(atol(getData(0)));
	arc.objectId(string(getData(1)));
	return fetchRow();
}



// End TeDatabasePortal Methods

