------------------------------------------------------------------------------
-- COMMON                                                                   --
--                                                                          --
-- Part of TextTools                                                        --
-- Designed and Programmed by Ken O. Burtch                                 --
--                                                                          --
------------------------------------------------------------------------------
--                                                                          --
--                 Copyright (C) 1999-2007 Ken O. Burtch                    --
--                                                                          --
-- This is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  This is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with this;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- This is maintained at http://www.pegasoft.ca/tt.html                     --
--                                                                          --
------------------------------------------------------------------------------
with Gen_List;
  pragma Elaborate( Gen_List ); -- remind Ada to elaborate generics first
with Ada; use Ada;
with Ada.Strings; use Ada.Strings; -- overrides our "strings" package
with Ada.Strings.Bounded;
with Ada.Strings.Maps;

with Interfaces.C;
use  Interfaces.C;
-- C language support

package Common is

type C_String is new char_array(0..255 );
-- C-style string for C language support

--- Compile Flags
--
-- Differentiate between compiling the finder (ie. the server that first
-- runs and establishes the environment and contains routines to answer
-- all questions) from the normal applications running under it (ie. the
-- clients which must negotiate with the finder for control of the screen.

IsFinder : constant boolean := true; -- true if compiling for a server
pragma export( C, IsFinder, "is_finder" );

---> Time types

subtype ATimeStamp is long_long_integer;

---> Standard Math functions
--
-- There are standard Ada packages for some of these, but they're all
-- built-in so who cares?

subtype APercent is integer range 1..100;
function Sin( angle : float ) return float;
  pragma Import( C, Sin, "CSin" );
function Cos( angle : float ) return float;
  pragma Import( C, Cos, "CCos" );
function RND( limit : integer ) return integer;
  pragma Import( C, RND, "Crnd" );
function NormalRND( limit : integer ) return integer;
  pragma Import( C, NormalRND, "Cnormalrnd" );
function Odds( percent : APercent ) return boolean;
  pragma Import( C, Odds, "COdds" );
procedure SetRNDSeed( seed : integer );
  pragma Import( C, SetRNDSeed, "Csetrndseed" );

---> Basic Rectangles
--
-- Rectangles are used all over.

subtype Points is positive;

type ARect is record
   left, top, right, bottom : integer;
end record;

nullRect : constant ARect := (0, 0, -1, -1);
pragma export( C, nullRect, "null_rect" );

procedure SetRect( r : out ARect; left, top, right, bottom : integer );
pragma export( C, SetRect, "set_rect" );

procedure OffsetRect( r : in out ARect; dx, dy : integer );
function  OffsetRect( r : in ARect; dx, dy : integer ) return ARect;
pragma export( C, offsetRect, "offset_rect" );

procedure InsetRect( r : in out ARect; dx, dy : integer );
function  InsetRect( r : in ARect; dx, dy : integer ) return ARect;
pragma export( C, insetRect, "inset_rect" );

function  InsideRect( Inner, Outer : in ARect ) return boolean;
pragma export( C, insideRect, "inside_rect" );

function  InRect( x, y : integer; r : ARect ) return boolean;
pragma export( C, InRect, "in_rect" );

function  IsEmptyRect( r : ARect ) return boolean;
pragma export( C, IsEmptyRect, "is_empty_rect" );


---> Bounded Strings
--
-- 80 character limit

--package Strings80  is new Ada.Strings.Bounded.Generic_Bounded_Length
--                          ( Max =>  80 );
--subtype Str80 is Strings80.Bounded_String;
--NullStr80 : str80 renames Strings80.Null_Bounded_String;
--
--function To80( Source : in String;
--               Drop   : in Strings.Truncation := Strings.Error)
--               return Str80 renames Strings80.To_Bounded_String;
--function ToString( source : in str80 ) return String
--               renames Strings80.To_String;

---> 255 character limit

package Strings255 is new Ada.Strings.Bounded.Generic_Bounded_Length
                          ( Max => 255 );
subtype Str255 is Strings255.Bounded_String;
NullStr255 : str255 renames Strings255.Null_Bounded_String;
pragma export( C, NullStr255, "null_str255" );

function To255( Source : in String;
                Drop   : in Strings.Truncation := Strings.Error )
                return Str255 renames Strings255.To_Bounded_String;
  pragma Pure_Function( To255 );
function ToString( source : in str255 ) return String
                renames Strings255.To_String;
  pragma Pure_Function( ToString );

-- Renamings here
--
-- 1. Renaming these functions makes them automatically present with common.
-- 2. Gnat 2.0 has some of these incomplete, so I'll provide my own
--    definitions

function Length (Source : in Str255) return Strings255.Length_Range
                renames Strings255.Length;
-- this one doesn't work in gnat 2.0
function Append (Left, Right : in Str255;
                Drop   : in Strings.Truncation  := Strings.Error)
                return Str255;
                -- return Str255 renames Strings255.Append;
function Append (Left : in Str255;
                Right : in String;
                Drop  : in Strings.Truncation := Strings.Error)
                return  Str255 renames Strings255.Append;
function Append (Left : in String;
                Right : in Str255;
                Drop  : in Strings.Truncation := Strings.Error)
                return  Str255 renames Strings255.Append;
function Append (Left  : in Str255;
                Right : in Character;
                Drop  : in Strings.Truncation := Strings.Error)
                return  Str255 renames Strings255.Append;
function Append (Left  : in Character;
                Right : in Str255;
                Drop  : in Strings.Truncation := Strings.Error)
                return  Str255 renames Strings255.Append;
-- this one doesn't work in gnat 2.0
procedure Append (Source   : in out Str255;
                New_Item : in Str255;
                Drop     : in Strings.Truncation  := Strings.Error);
                --renames Strings255.Append;
procedure Append (Source   : in out Str255;
                New_Item : in String;
                Drop     : in Strings.Truncation  := Strings.Error)
                renames Strings255.Append;
procedure Append (Source   : in out Str255;
                New_Item : in Character;
                Drop     : in Strings.Truncation  := Strings.Error)
                renames Strings255.Append;
-- this one doesn't work in gnat 2.0
function "&" (Left, Right : in Str255) return Str255;
             -- renames Strings255."&";
-- Adding this one because it's natural!
function "&" (Left  : in Str255; Right : in String)
             return  Str255 renames Strings255."&";
-- this one doesn't work in gnat 2.0
function "&" (Left  : in String; Right : in Str255)
             return  Str255; -- renames Strings255."&";
function "&" (Left  : in Str255; Right : in Character)
             return  Str255 renames Strings255."&";
function "&" (Left  : in Character; Right : in Str255)
             return  Str255 renames Strings255."&";
function Element (Source : in Str255; Index  : in Positive)
             return   Character renames Strings255.Element;
procedure Replace_Element (Source : in out Str255;
             Index  : in Positive;
             By     : in Character)
             renames Strings255.Replace_Element;
function Slice (Source : in Str255;
             Low    : in Positive;
             High   : in Natural)
             return   String
             renames Strings255.Slice;
function "=" (Left, Right : in Str255) return Boolean
             renames Strings255."=";
function "=" (Left  : in Str255; Right : in String)
             return  Boolean renames Strings255."=";
function "=" (Left  : in String; Right : in Str255)
             return  Boolean renames Strings255."=";
function "<" (Left, Right : in Str255) return Boolean
             renames Strings255."<";
function "<" (Left  : in Str255; Right : in String)
             return  Boolean renames Strings255."<";
function "<" (Left  : in String; Right : in Str255)
             return  Boolean renames Strings255."<";
function "<=" (Left, Right : in Str255) return Boolean
             renames Strings255."<=";
function "<=" (Left  : in Str255; Right : in String)
             return  Boolean renames Strings255."<=";
function "<=" (Left  : in String; Right : in Str255)
             return  Boolean renames Strings255."<=";
function ">"  (Left, Right : in Str255) return Boolean
             renames Strings255.">";
function ">" (Left  : in Str255; Right : in String)
             return  Boolean renames Strings255.">";
function ">" (Left  : in String; Right : in Str255)
             return  Boolean renames Strings255.">";
function ">=" (Left, Right : in Str255) return Boolean
             renames Strings255.">=";
function ">=" (Left  : in Str255; Right : in String)
             return  Boolean renames Strings255.">=";
function ">=" (Left  : in String; Right : in Str255)
             return  Boolean renames Strings255.">=";
function Index (Source  : in Str255;
             Pattern : in String;
             Going   : in Strings.Direction := Strings.Forward;
             Mapping : in Maps.Character_Mapping := Maps.Identity)
             return    Natural
             renames Strings255.Index;
function Index (Source  : in Str255;
             Pattern : in String;
             Going   : in Strings.Direction := Strings.Forward;
             Mapping : in Maps.Character_Mapping_Function)
             return    Natural
             renames Strings255.Index;
function Index (Source : in Str255;
             Set    : in Maps.Character_Set;
             Test   : in Strings.Membership := Strings.Inside;
             Going  : in Strings.Direction  := Strings.Forward)
             return   Natural
             renames Strings255.Index;
function Index_Non_Blank (Source : in Str255;
             Going  : in Strings.Direction := Strings.Forward)
             return   Natural
             renames Strings255.Index_Non_Blank;
function Count (Source  : in Str255;
             Pattern : in String;
             Mapping : in Maps.Character_Mapping := Maps.Identity)
             return    Natural
             renames Strings255.Count;
function Count (Source  : in Str255;
             Pattern : in String;
             Mapping : in Maps.Character_Mapping_Function)
             return    Natural
             renames Strings255.Count;
 function Count (Source : in Str255;
             Set    : in Maps.Character_Set)
             return   Natural
             renames Strings255.Count;
 procedure Find_Token (Source : in Str255;
             Set    : in Maps.Character_Set;
             Test   : in Strings.Membership;
             First  : out Positive;
             Last   : out Natural)
             renames Strings255.Find_Token;
 function Translate (Source   : in Str255;
             Mapping  : in Maps.Character_Mapping)
             return Str255 renames Strings255.Translate;
 procedure Translate (Source   : in out Str255;
             Mapping  : in Maps.Character_Mapping)
             renames Strings255.Translate;
 function Translate (Source  : in Str255;
             Mapping : in Maps.Character_Mapping_Function)
             return  Str255 renames Strings255.Translate;
 procedure Translate (Source  : in out Str255;
             Mapping : in Maps.Character_Mapping_Function)
             renames Strings255.Translate;
 function Replace_Slice (Source   : in Str255;
             Low      : in Positive;
             High     : in Natural;
             By       : in String;
             Drop     : in Strings.Truncation := Strings.Error)
             return   Str255
             renames Strings255.Replace_Slice;
 procedure Replace_Slice (Source   : in out Str255;
             Low      : in Positive;
             High     : in Natural;
             By       : in String;
             Drop     : in Strings.Truncation := Strings.Error)
             renames Strings255.Replace_Slice;
 function Insert (Source   : in Str255;
             Before   : in Positive;
             New_Item : in String;
             Drop     : in Strings.Truncation := Strings.Error)
             return     Str255
             renames Strings255.Insert;
 procedure Insert (Source   : in out Str255;
             Before   : in Positive;
             New_Item : in String;
             Drop     : in Strings.Truncation := Strings.Error)
             renames Strings255.Insert;
function Overwrite (Source    : in Str255;
             Position  : in Positive;
             New_Item  : in String;
             Drop      : in Strings.Truncation := Strings.Error)
             return      Str255
             renames Strings255.Overwrite;
procedure Overwrite (Source    : in out Str255;
             Position  : in Positive;
             New_Item  : in String;
             Drop      : in Strings.Truncation := Strings.Error)
             renames Strings255.Overwrite;
function Delete (Source  : in Str255;
             From    : in Positive;
             Through : in Natural)
             return    Str255
             renames Strings255.Delete;
procedure Delete (Source  : in out Str255;
             From    : in Positive;
             Through : in Natural)
             renames Strings255.Delete;
function Trim (Source : in Str255;
             Side   : in Strings.Trim_End)
             return   Str255
             renames Strings255.Trim;
procedure Trim (Source : in out Str255;
             Side   : in Strings.Trim_End)
             renames Strings255.Trim;
function Trim (Source  : in Str255;
             Left   : in Maps.Character_Set;
             Right  : in Maps.Character_Set)
             return   Str255
             renames Strings255.Trim;
procedure Trim (Source : in out Str255;
             Left   : in Maps.Character_Set;
             Right  : in Maps.Character_Set)
             renames Strings255.Trim;
-- This can also error with core dump
-- my version ignores Drop
function Head (Source : in Str255;
             Count  : in Natural;
             Pad    : in Character := Strings.Space;
             Drop   : in Strings.Truncation := Strings.Error)
             return   Str255;
             --renames Strings255.Head;
procedure Head (Source : in out Str255;
             Count  : in Natural;
             Pad    : in Character  := Strings.Space;
             Drop   : in Strings.Truncation := Strings.Error)
             renames Strings255.Head;
function Tail (Source : in Str255;
             Count  : in Natural;
             Pad    : in Character  := Strings.Space;
             Drop   : in Strings.Truncation := Strings.Error)
             return Str255
             renames Strings255.Tail;
 procedure Tail (Source : in out Str255;
             Count  : in Natural;
             Pad    : in Character  := Strings.Space;
             Drop   : in Strings.Truncation := Strings.Error)
             renames Strings255.Tail;
 function "*" (Left  : in Natural; Right : in Character)
             return  Str255 renames Strings255."*";
 function "*" (Left  : in Natural; Right : in String)
             return  Str255 renames Strings255."*";
 function "*" (Left  : in Natural; Right : in Str255)
             return  Str255 renames Strings255."*";
 function Replicate (Count : in Natural;
             Item  : in Character;
             Drop  : in Strings.Truncation := Strings.Error)
             return  Str255
             renames Strings255.Replicate;
 function Replicate (Count : in Natural;
             Item  : in String;
             Drop  : in Strings.Truncation := Strings.Error)
             return  Str255
             renames Strings255.Replicate;
 function Replicate (Count : in Natural;
             Item  : in Str255;
             Drop  : in Strings.Truncation := Strings.Error)
             return  Str255
             renames Strings255.Replicate;

--function To255( s : in str80 ) return str255;
--function To80( s : in str255 ) return str80;
function ToInteger( s : in str255 ) return integer;
function ToInteger( s : in string ) return integer;
function ToLongInteger( s : in str255 ) return long_integer;
function ToLongInteger( s : in string ) return long_integer;

-- Lists of Bounded Strings

package Str255List is new gen_List( Str255, Strings255."=", Strings255.">=" );

---> Lists of Booleans

package BooleanList is new Gen_List( boolean, "=", ">=" );

---> Lists of Rectangles

function RectOrder( left, right : ARect ) return boolean;
package RectList is new Gen_List( ARect, "=", RectOrder );

---> Various Pointer Types
--
-- These are listed for convenience.

type BooleanPtr       is access all Boolean;
type IntegerPtr       is access all Integer;
type Short_IntegerPtr is access all Short_Integer;
type Long_IntegerPtr  is access all Long_Integer;
--type Str80Ptr         is access all Str80;
type Str255Ptr        is access all Str255;
type FloatPtr         is access all Float;
type RectPtr          is access all ARect;

---> Error Handling
---
--- To get the best of all worlds, I'm providing the following vars/
--- procs.  If RaiseErrors is used, all calls to Error will result
--- in a GeneralError being raised (the error code is in LastError).
--- If TrapErrors (default) is used, all calls to Error will save the
--- error code in LastError and return and it's up to the program to
--- check to see if an error occurred.  If you need to save the error
--- handling method of the caller, save the value of RaisingErrors. 
--- (I was going to create a stack, but that's slower and more work.
--- This is better when memory is low, and often the flag doesn't need
--- saving.)  If your subprogram uses these routines, call NoError first
--- to clear any outstanding error codes.
---
--- Core Error Codes
---
--- Listed here for convenience and to ensure they are unique
---
--- Core System Errors

type AnErrorCode is new short_integer range -1..short_integer'last;
subtype ACoreErrorCode is AnErrorCode range -1..499;
subtype AnUserErrorCode is AnErrorCode range 500..AnErrorCode'last;

TT_NotYetWritten : constant ACoreErrorCode :=  -1; -- routine not yet written
TT_OK            : constant ACoreErrorCode :=   0; -- no error
TT_MemoryLeak    : constant ACoreErrorCode :=  10; -- memory not deallocated
TT_LowMemory     : constant ACoreErrorCode :=  11; -- low on memory

pragma export( C, TT_NotYetWritten, "TT_not_yet_written" );
pragma export( C, TT_OK, "TT_ok" );
pragma export( C, TT_MemoryLeak, "TT_memory_leak" );
pragma export( C, TT_LowMemory, "TT_low_memory" );

--- Core System and related

TT_SystemError   : constant ACoreErrorCode := 100; -- command failed
TT_ParamError    : constant ACoreErrorCode := 101; -- param too long
TT_FileExistance : constant ACoreErrorCode := 110; -- file found/not found
TT_PathExistance : constant ACoreErrorCode := 111; -- path found/not found
TT_VolExistance  : constant ACoreErrorCode := 112; -- volume found/not found
TT_DevExistance  : constant ACoreErrorCode := 113; -- device found/not found
TT_FileStatus    : constant ACoreErrorCode := 114; -- open / not open
TT_FileLocking   : constant ACoreErrorCode := 115; -- file is locked/unlocked
TT_FileAccess    : constant ACoreErrorCode := 116; -- file is un/accessible
TT_VolLocking    : constant ACoreErrorCode := 117; -- volume readonly or not
TT_VolAccess     : constant ACoreErrorCode := 118; -- volume is un/accessible
TT_VolFull       : constant ACoreErrorCode := 119; -- no space on disk
TT_DevSequential : constant ACoreErrorCode := 120; -- tape device
TT_IOError       : constant ACoreErrorCode := 121; -- hardware or media error
TT_PathError     : constant ACoreErrorCode := 122; -- bad path for file sys
TT_FileBounds    : constant ACoreErrorCode := 123; -- position out of bounds

TT_OSOld         : constant ACoreErrorCode := 130; -- UNIX too old
TT_OSService     : constant ACoreErrorCode := 131; -- UNIX service missing

TT_Integrity     : constant ACoreErrorCode := 140; -- integrity test failure
TT_TestData      : constant ACoreErrorCode := 141; -- test data in operation

pragma export( C, TT_SystemError, "TT_system_error" );
pragma export( C, TT_ParamError, "TT_param_error" );
pragma export( C, TT_FileExistance, "TT_file_existance" );
pragma export( C, TT_PathExistance, "TT_path_existance" );
pragma export( C, TT_VolExistance, "TT_vol_existance" );
pragma export( C, TT_DevExistance, "TT_dev_existance" );
pragma export( C, TT_FileStatus, "TT_file_status" );
pragma export( C, TT_FileLocking, "TT_file_locking" );
pragma export( C, TT_FileAccess, "TT_file_access" );
pragma export( C, TT_VolLocking, "TT_vol_locking" );
pragma export( C, TT_VolAccess, "TT_vol_access" );
pragma export( C, TT_VolFull, "TT_vol_full" );
pragma export( C, TT_DevSequential, "TT_dev_sequential" );
pragma export( C, TT_IOError, "TT_io_error" );
pragma export( C, TT_PathError, "TT_path_error" );
pragma export( C, TT_FileBounds, "TT_file_bounds" );
pragma export( C, TT_OSOld, "TT_os_old" );
pragma export( C, TT_OSService, "TT_os_service" );
pragma export( C, TT_Integrity, "TT_integrity" );
pragma export( C, TT_TestData, "TT_test_data" );

---> Interpreter Errors
-- (not used)

TT_UnexpErr        : constant ACoreErrorCode := 200; -- unexpected character
TT_ParanErr        : constant ACoreErrorCode := 201; -- Bad paranthesis
TT_OperandErr      : constant ACoreErrorCode := 202; -- missing operand
TT_SyntaxErr       : constant ACoreErrorCode := 203; -- bad syntax
TT_TooCompErr      : constant ACoreErrorCode := 204; -- formula too complex
TT_ClashErr        : constant ACoreErrorCode := 205; -- type clash
TT_NotDeclErr      : constant ACoreErrorCode := 206; -- ident not declared
TT_EOProgErr       : constant ACoreErrorCode := 207; -- end of prog encountered
TT_QuoteErr        : constant ACoreErrorCode := 208; -- bad quote marks
TT_DivZeroErr      : constant ACoreErrorCode := 209; -- divide by zero

---> Core Userio Errors

---> Core Control Errors

---> Core Window Errors

TT_WindowExistance : constant ACoreErrorCode := 160; --window found/not
TT_NoControls      : constant ACoreErrorCode := 161; --no controls in window
TT_ControlExistance: constant ACoreErrorCode := 162;
TT_NoDialogTaskCB  : constant ACoreErrorCode := 163; --no manual handler

pragma export( C, TT_WIndowExistance,  "TT_window_existance" );
pragma export( C, TT_NoControls,       "TT_no_controls" );
pragma export( C, TT_ControlExistance, "TT_control_existance" );
pragma export( C, TT_NoDialogTaskCB,   "TT_no_dialog_task_cb" );

---> Error Variables/Functions

GeneralError     : exception;       -- exception raised by Error();
LastError        : AnErrorCode;     -- last Error error code
--LastErrorDetails : Str255List.List; -- details of last error (if any)
RaisingErrors    : boolean;         -- TRUE if GeneralError will be raised

procedure NoError;  -- clear LastError
  pragma Inline( NoError );
  pragma Export( C, NoError, "no_error" );
procedure Error( ErrorCode : AnErrorCode ); -- log an error
  pragma Inline( Error );
  pragma Export( C, Error, "error" );
procedure RaiseErrors; -- cause Error to raise a GeneralError
  pragma Inline( RaiseErrors );
procedure TrapErrors; -- cause Error to return normally
  pragma Inline( TrapErrors );
function  RaiseErrors return boolean;
function  TrapErrors return boolean;
procedure RestoreRaising( oldflag : boolean );
  pragma Inline( RestoreRaising );

--- Housekeeping
--
-- These are set by StartupCommon.

ProgramName      : str255 := NullStr255;
pragma export( C, ProgramName, "program_name" );

ShortProgramName : str255 := NullStr255;
pragma export( C, ShortProgramName, "short_program_name" );

-- Short program name is used for $SYS directory in os package.
-- and (when I get to it) temp file name prefix.

procedure StartupCommon( theProgramName, theShortProgramName : string );

procedure StartupCommonCPP( theProgramName, theShortProgramName : C_string );
pragma export( CPP, StartupCommonCPP, "startup_common" );

procedure IdleCommon( IdlePeriod : ATimeStamp );
pragma export( C, IdleCommon, "idle_common" );

procedure ShutdownCommon;
pragma export( C, ShutdownCommon, "shutdown_common" );

end Common;
