/***************************************************************************
 *
 * This file is covered by a dual licence. You can choose whether you
 * want to use it according to the terms of the GNU GPL version 2, or
 * under the terms of Zorp Professional Firewall System EULA located
 * on the Zorp installation CD.
 *
 * $Id: streamgzip.c,v 1.5 2004/04/27 15:12:47 bazsi Exp $
 *
 * Author  : SaSa
 * Auditor :
 * Last audited version:
 * Notes:
 *
 ***************************************************************************/

#include <zorp/stream.h>
#include <zorp/log.h>

#include <zlib.h>

#include <glib.h>

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <assert.h>

#ifdef G_OS_WIN32
#  include <winsock2.h>
#else
#  include <sys/socket.h>
#  include <sys/poll.h>
#endif

extern ZClass ZStreamGzip__class;

typedef struct _ZStreamGzip
{
  ZStream super;

  guint flags;
  z_stream encode;
  z_stream decode;
  gsize buffer_length;
  gchar *buffer_encode_out;
  gchar *buffer_encode_out_p;
  gchar *buffer_decode_in;
  gint flush;
  gint err;
  gboolean eof_before;
  gboolean eof_after;
  gint shutdown;
  GIOCondition child_cond;

} ZStreamGzip;

/*
 * It's function flush the zlib buffer.
 * After flush it, try to write it out.
 * In nonblocking mode, it's not trt to
 * write all data, z_stream_gzip_write_callback
 * will do the rest.
 * If output buffer is full, return with
 * G_IO_STATUS_AGAIN immediately.
 */
GIOStatus
z_stream_gzip_flush(ZStream *stream)
{
  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  GIOStatus res = G_IO_STATUS_NORMAL;
  gint rc;
  
  z_enter();

  if (self->err & Z_STREAM_FLAG_READ)
    {
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  
  if (self->encode.avail_out == 0)
    {
      z_leave();
      return G_IO_STATUS_AGAIN;
    }

  self->encode.avail_in = 0;
  self->encode.next_in = NULL;
  rc = deflate(&self->encode, Z_SYNC_FLUSH);
  
  if (rc != Z_OK)
    {
      self->err |= Z_STREAM_FLAG_READ;
      z_leave();
      return G_IO_STATUS_ERROR;
    }

  if (self->encode.avail_out < self->buffer_length)
    {
      gint length = (gchar *)self->encode.next_out - (gchar *)self->buffer_encode_out_p;
      gsize writted_length;
      
      res = z_stream_write(self->super.child, self->buffer_encode_out_p, length, &writted_length, NULL);
      if (res == G_IO_STATUS_AGAIN)
        {
          z_leave();
          return G_IO_STATUS_AGAIN;
        }
      else if (res != G_IO_STATUS_NORMAL)
        {
          self->err |= Z_STREAM_FLAG_WRITE;
          z_leave();
          return G_IO_STATUS_ERROR;
        }
      
      self->buffer_encode_out_p += writted_length;
      
      if ((gchar *)self->buffer_encode_out_p != (gchar *)self->encode.next_out)
        {
          z_leave();
          return G_IO_STATUS_AGAIN;
        }
    }
  
  z_leave();
  return G_IO_STATUS_NORMAL;
}

/* I/O callbacks for stacked stream */

static gboolean
z_stream_gzip_read_callback(ZStream *stream G_GNUC_UNUSED, GIOCondition poll_cond G_GNUC_UNUSED, gpointer s)
{
  ZStreamGzip *self = Z_CAST(s, ZStreamGzip);

  z_enter();
  self->child_cond |= Z_STREAM_FLAG_READ;
  z_leave();
  return TRUE;
}

static gboolean
z_stream_gzip_write_callback(ZStream *stream G_GNUC_UNUSED, GIOCondition poll_cond G_GNUC_UNUSED, gpointer s)
{
  ZStreamGzip *self = Z_CAST(s, ZStreamGzip);
  GIOStatus res;

  z_enter();
  
  /*
   * If some data in output buffer try to write it out.
   * If write is not success leave it, and doesn't set
   * it writeable.
   */
  if (self->encode.avail_out < self->buffer_length)
    {
      gint length = (gchar *)self->encode.next_out - (gchar *)self->buffer_encode_out_p;
      gsize writted_length;
      
      res = z_stream_write(self->super.child, self->buffer_encode_out_p, length, &writted_length, NULL);
      if (res == G_IO_STATUS_AGAIN)
        {
          z_leave();
          return TRUE;
        }
      else if (res != G_IO_STATUS_NORMAL)
        {
          self->err |= Z_STREAM_FLAG_WRITE;
          z_leave();
          return TRUE;
        }
      
      self->buffer_encode_out_p += writted_length;
      
      if ((gchar *)self->buffer_encode_out_p != (gchar *)self->encode.next_out)
        {
          z_leave();
          return TRUE;
        }
    }
  self->child_cond |= Z_STREAM_FLAG_WRITE;
  

  z_leave();
  return TRUE;
}

/* virtual methods */

static GIOStatus
z_stream_gzip_read_method(ZStream *stream,
                          gchar   *buf,
                          gsize   count,
                          gsize   *bytes_read,
                          GError  **error G_GNUC_UNUSED)
{
  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  GIOStatus res;
  gint ret;
  
  z_enter();

  self->child_cond &= ~Z_STREAM_FLAG_READ;

  if (self->shutdown & Z_STREAM_FLAG_READ)
    {
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  
  if (self->eof_after)
    {
      z_leave();
      return G_IO_STATUS_EOF;
    }

  if (self->err & Z_STREAM_FLAG_READ)
    {
      z_leave();
      return G_IO_STATUS_ERROR;
    }

  self->decode.next_out = buf;
  self->decode.avail_out = count;

  if (self->decode.avail_in == 0 && !self->eof_before)
    {
      gsize length;
          
      self->decode.next_in = self->buffer_decode_in;

      res = z_stream_read(self->super.child, self->decode.next_in, self->buffer_length, &length, NULL);
      
      self->decode.avail_in = length;
          
      if (res == G_IO_STATUS_AGAIN)
        {
          z_leave();
          return G_IO_STATUS_AGAIN;
        }
      else if (res == G_IO_STATUS_EOF)
        {
          self->eof_before = TRUE;
        }
      else if (res != G_IO_STATUS_NORMAL)
        {
          self->err |= Z_STREAM_FLAG_READ;
          z_leave();
          return G_IO_STATUS_ERROR;
        }
    }
      
  ret = inflate(&self->decode, Z_NO_FLUSH);
      
  if (ret != Z_OK)
    {
      if (ret == Z_STREAM_END)
        {
          self->eof_after = TRUE;
          z_leave();
          return G_IO_STATUS_EOF;
        }
      self->err |= Z_STREAM_FLAG_READ;
      z_leave();
      return G_IO_STATUS_ERROR;
    }
  
  *bytes_read = count - self->decode.avail_out;

  z_leave();
  return G_IO_STATUS_NORMAL;
}

static GIOStatus
z_stream_gzip_write_method(ZStream     *stream,
                           const gchar *buf,
                           gsize       count,
                           gsize       *bytes_written,
                           GError      **error G_GNUC_UNUSED)
{
  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  GIOStatus res = G_IO_STATUS_NORMAL;
  gint rc;
  gsize length;
  gsize writted_length;

  z_enter();

  self->child_cond &= ~Z_STREAM_FLAG_WRITE;

  if (self->shutdown & Z_STREAM_FLAG_WRITE)
    {
      z_leave();
      return G_IO_STATUS_ERROR;
    }

  if (self->err & Z_STREAM_FLAG_WRITE)
    {
      z_leave();
      return G_IO_STATUS_ERROR;
    }

  if (self->encode.avail_out < self->buffer_length)
    {
      length = (gchar *)self->encode.next_out - (gchar *)self->buffer_encode_out_p;
      res = z_stream_write(self->super.child, self->buffer_encode_out_p, length, &writted_length, NULL);
      if (res == G_IO_STATUS_AGAIN)
        {
          z_leave();
          return G_IO_STATUS_AGAIN;
        }
      else if (res != G_IO_STATUS_NORMAL)
        {
          self->err |= Z_STREAM_FLAG_WRITE;
          z_leave();
          return G_IO_STATUS_ERROR;
        }
      
      self->buffer_encode_out_p += writted_length;
      
      if ((gchar *)self->buffer_encode_out_p != (gchar *)self->encode.next_out)
        {
          z_leave();
          return G_IO_STATUS_AGAIN;
        }
    }
  
  self->encode.next_out = self->buffer_encode_out;
  self->encode.avail_out = self->buffer_length;
  
  self->encode.next_in = (gchar *)buf;
  self->encode.avail_in = count;
  
  self->buffer_encode_out_p = self->buffer_encode_out;

  while (res == G_IO_STATUS_NORMAL && self->encode.avail_in)
    {
      if (self->encode.avail_out)
        {
          rc = deflate(&self->encode, self->flush);
  
          if (rc != Z_OK)
            {
              self->err |= Z_STREAM_FLAG_READ;
              z_leave();
              return G_IO_STATUS_ERROR;
            }
        }
  
      length = self->buffer_length - self->encode.avail_out;
      res = z_stream_write(self->super.child, self->buffer_encode_out, length, &writted_length, NULL);
      if (res == G_IO_STATUS_NORMAL)
        {
          self->buffer_encode_out_p += writted_length;
  
          if ((gchar *)self->buffer_encode_out_p == (gchar *)self->encode.next_out)
            {
              self->encode.next_out = self->buffer_encode_out;
              self->encode.avail_out = self->buffer_length;
              self->buffer_encode_out_p = self->buffer_encode_out;
            }
        }
      else if (res != G_IO_STATUS_AGAIN)
        {
          self->err |= Z_STREAM_FLAG_WRITE;
          z_leave();
          return G_IO_STATUS_ERROR;
        }
    }
  
  *bytes_written = count - self->encode.avail_in;
  
  if (*bytes_written == 0)
    {
      z_leave();
      return G_IO_STATUS_AGAIN;
    }

  z_leave();
  return G_IO_STATUS_NORMAL;
}


static GIOStatus
z_stream_gzip_shutdown_method(ZStream *stream, int method, GError **error)
{
  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  GIOStatus res = G_IO_STATUS_ERROR, ret = G_IO_STATUS_NORMAL;
  gint rc;
  
  z_enter();
  
  if ((method == SHUT_RD || method == SHUT_RDWR) && (self->shutdown & Z_STREAM_FLAG_READ) == 0)
    {
      rc = inflateEnd(&self->decode);
      if (rc == Z_OK)
        res = G_IO_STATUS_NORMAL;
      self->shutdown |= Z_STREAM_FLAG_READ;
    }
  
  if ((method == SHUT_WR || method == SHUT_RDWR) && (self->shutdown & Z_STREAM_FLAG_WRITE) == 0)
    {
      gboolean i;
      gsize length;
      
      i = z_stream_get_nonblock(self->super.child);
      z_stream_set_nonblock(self->super.child, FALSE);
      
      while ((gchar *)self->buffer_encode_out_p != (gchar *)self->encode.next_out && ret == G_IO_STATUS_NORMAL)
        {
          ret = z_stream_write(self->super.child, self->buffer_encode_out_p, (gchar *)self->encode.next_out - (gchar *)self->buffer_encode_out_p, &length, NULL);
          if (ret == G_IO_STATUS_NORMAL)
            self->buffer_encode_out_p += length;
        }
      
      if (ret == G_IO_STATUS_NORMAL)
        {
          self->encode.avail_out = self->buffer_length;
          self->encode.next_out = self->buffer_encode_out;
          self->encode.avail_in = 0;
          self->encode.next_in = NULL;
          
          self->buffer_encode_out_p = self->buffer_encode_out;
          
          rc = deflate(&self->encode, Z_FINISH);
          if (rc == Z_STREAM_END)
            {
              if (self->encode.avail_out < self->buffer_length)
                {
                  while ((gchar *)self->buffer_encode_out_p != (gchar *)self->encode.next_out && ret == G_IO_STATUS_NORMAL)
                    {
                      ret = z_stream_write(self->super.child, self->buffer_encode_out_p, (gchar *)self->encode.next_out - (gchar *)self->buffer_encode_out_p, &length, NULL);
                      if (ret == G_IO_STATUS_NORMAL)
                        self->buffer_encode_out_p += length;
                    }
                }
            }
        }
      z_stream_set_nonblock(self->super.child, i);
        
      rc = deflateEnd(&self->encode);
      
      if (ret == G_IO_STATUS_NORMAL && rc == Z_OK)
        res = G_IO_STATUS_NORMAL;
      
      self->shutdown |= Z_STREAM_FLAG_WRITE;
    }

  ret = z_stream_shutdown(self->super.child, method, error);
  
  if (ret != G_IO_STATUS_NORMAL)
    res = ret;
  
  z_leave();
  return res;

}

static GIOStatus
z_stream_gzip_close_method(ZStream *stream, GError **error)
{
  GIOStatus res;
  GIOStatus ret;

  z_enter();

  res = z_stream_gzip_shutdown_method(stream, SHUT_RDWR, NULL);

  ret = z_stream_close(stream->child, error);
  
  if (ret != G_IO_STATUS_NORMAL)
    res = ret;
  
  z_leave();
  return res;
}

/*
 * Allithato dolgok:
 * - Tomorites erossege
 */
static gboolean
z_stream_gzip_ctrl_method(ZStream *stream, guint function, gpointer value, guint vlen)
{
//  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  gboolean ret = FALSE;
  
  z_enter();

  switch (ZST_CTRL_MSG(function))
    {
    case ZST_CTRL_SET_CALLBACK_READ:
    case ZST_CTRL_SET_CALLBACK_WRITE:
    case ZST_CTRL_SET_CALLBACK_PRI:
      ret = z_stream_ctrl_method(stream, function, value, vlen);
      break;
    default:
      ret = z_stream_ctrl_method(stream, ZST_CTRL_MSG_FORWARD | function, value, vlen);
      break;
    }
  z_leave();
  return ret;
}

static void
z_stream_gzip_attach_source_method(ZStream *stream, GMainContext *context)
{
  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  ZStream *p;
  gint stacked_count = 0;

  z_enter();

  z_stream_ref(stream);
  p = self->super.child;
  while (p)
    {
      stacked_count++;
      p = p->child;
    }
  z_stream_attach_source(self->super.child, context);
  if (!stream->source)
    {
      stream->source = z_stream_source_new(stream);
      g_source_set_priority(stream->source, G_PRIORITY_DEFAULT - stacked_count);
      g_source_attach(stream->source, context);
    }
  
  z_stream_unref(stream);
  z_leave();
}

static void
z_stream_gzip_detach_source_method(ZStream *stream)
{
  ZStreamGzip *self = Z_CAST(stream, ZStreamGzip);
  GSource *source;
  
  z_enter();
  if (stream->source)
    {
      source = stream->source;
      stream->source = NULL;
      /*
         NOTE Must be in this order because
         g_source_unref may drop the last
         reference to source. g_source_destroy
         doesn't destroy the source, only make it
         unusable.
       */
      g_source_destroy(source);
      g_source_unref(source);
    }

  z_stream_detach_source(self->super.child);
    
  z_leave();
}

static gboolean 
z_stream_gzip_watch_prepare(ZStream *s, GSource *src G_GNUC_UNUSED, gint *timeout)
{
  ZStreamGzip *self = Z_CAST(s, ZStreamGzip);
  gboolean ret = FALSE;
  gboolean child_readable, child_writeable, child_enable;

  z_enter();
  
  *timeout = -1;

  if (s->want_read)
    {
      child_readable = !!(self->child_cond & Z_STREAM_FLAG_READ);

      if (self->decode.avail_in == 0 && !child_readable)
        {
          child_enable = TRUE;
        }
      else
        {
          child_enable = FALSE;
          ret = TRUE;
        }
    }
  else
    child_enable = FALSE;

  z_stream_set_cond(s->child, Z_STREAM_FLAG_READ, child_enable);

  if (s->want_write)
    {
      child_writeable = !!(self->child_cond & Z_STREAM_FLAG_WRITE);
      if (self->encode.avail_out == self->buffer_length)
        ret = TRUE;
    }
  
  if (self->encode.avail_out != self->buffer_length)
    z_stream_set_cond(s->child, Z_STREAM_FLAG_WRITE, TRUE);
  else
    z_stream_set_cond(s->child, Z_STREAM_FLAG_WRITE, FALSE);

  z_leave();
  return ret;
}

static gboolean 
z_stream_gzip_watch_check(ZStream *s, GSource *src G_GNUC_UNUSED)
{
  ZStreamGzip *self = Z_CAST(s, ZStreamGzip);
  gboolean ret = FALSE;
  gboolean child_readable, child_writeable;

  z_enter();

  if (s->want_read)
    {
      child_readable = !!(self->child_cond & Z_STREAM_FLAG_READ);
      if (self->decode.avail_in || child_readable)
        {
          ret = TRUE;
        }
    }

  if (s->want_write)
    {
      child_writeable = !!(self->child_cond & Z_STREAM_FLAG_WRITE);
      if (self->encode.avail_out == self->buffer_length || child_writeable)
        {
          ret = TRUE;
        }
    }

  z_leave();
  return ret;
}

static
gboolean 
z_stream_gzip_watch_dispatch(ZStream *s, GSource *src G_GNUC_UNUSED)
{
  ZStreamGzip *self = Z_CAST(s, ZStreamGzip);
  gboolean rc = FALSE;
  gboolean child_readable, child_writeable;

  z_enter();

  if (s->want_read)
    {
      child_readable = !!(self->child_cond & Z_STREAM_FLAG_READ);
      if (self->decode.avail_in || child_readable)
        {
          rc = self->super.read_cb(s, G_IO_IN, self->super.user_data_read);
        }
    }

  if (s->want_write)
    {
      child_writeable = !!(self->child_cond & Z_STREAM_FLAG_WRITE);
      if (self->encode.avail_out == self->buffer_length || child_writeable)
        {
          rc = self->super.write_cb(s, G_IO_OUT, self->super.user_data_write);
        }
    }

  z_leave();
  return rc;
}


/* destructor */
static void
z_stream_gzip_free_method(ZObject *s)
{
  ZStreamGzip *self = Z_CAST(s, ZStreamGzip);

  z_enter();

  g_free(self->buffer_encode_out);
  g_free(self->buffer_decode_in);
      
  z_stream_free_method(s);
  z_leave();
}

ZStreamFuncs z_stream_gzip_funcs =
{
  {
    Z_FUNCS_COUNT(ZStream),
    z_stream_gzip_free_method,
  },
  z_stream_gzip_read_method,
  z_stream_gzip_write_method,
  NULL,
  NULL,
  z_stream_gzip_shutdown_method,
  z_stream_gzip_close_method,
  z_stream_gzip_ctrl_method,
  
  z_stream_gzip_attach_source_method,
  z_stream_gzip_detach_source_method,
  z_stream_gzip_watch_prepare,
  z_stream_gzip_watch_check,
  z_stream_gzip_watch_dispatch,
  NULL,
  NULL,
  NULL,
  NULL
};

ZClass ZStreamGzip__class = 
{
  Z_CLASS_HEADER,
  &ZStream__class,
  "ZStreamGzip",
  sizeof(ZStreamGzip),
  &z_stream_gzip_funcs.super
};

/* constructor */
/*
 * Flagek:
 * - Automatikus flush
 */
ZStream *
z_stream_gzip_new(ZStream *child, gboolean flush, guint level, guint buffer_length)
{
  ZStreamGzip *self;

  z_enter();

  self = Z_CAST(z_stream_new(Z_CLASS(ZStreamGzip), child->name, child, Z_STREAM_FLAG_READ | Z_STREAM_FLAG_WRITE), ZStreamGzip);
  self->super.timeout = self->super.child->timeout;
  
  z_stream_set_callback(self->super.child, Z_STREAM_FLAG_READ, z_stream_gzip_read_callback, self, NULL);
  z_stream_set_callback(self->super.child, Z_STREAM_FLAG_WRITE, z_stream_gzip_write_callback, self, NULL);

  deflateInit(&self->encode, level);
  inflateInit(&self->decode);
  
  self->buffer_length = buffer_length;
  
  self->buffer_encode_out = g_new(gchar, self->buffer_length);
  self->buffer_decode_in  = g_new(gchar, self->buffer_length);
  
  self->encode.next_out = self->buffer_encode_out;
  self->encode.avail_out = self->buffer_length;
  self->buffer_encode_out_p = self->buffer_encode_out;
  
  if (flush)
    self->flush = Z_SYNC_FLUSH;
  else
    self->flush = Z_NO_FLUSH;

  z_leave();
  return (ZStream *) self;
}
