#ifndef TAGCOLL_EXPERIMENTS_H
#define TAGCOLL_EXPERIMENTS_H

/** \file
 * Experimental code being tried out
 */

/*
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/CardinalityStore.h>
#include <tagcoll/Expression.h>

#include <vector>
#include <ostream>

namespace Tagcoll
{

template <typename TAG>
class Scores
{
protected:
	struct Score
	{
		Expression expr;
		float score;
		Score(const std::string& expr, float score) :
			expr(expr), score(score) {}
	};
	float defaultScore;
	std::vector<Score> scores;
public:
	Scores(float def = 1.0) : defaultScore(def) {}

	void add(const std::string& expr, float score)
	{
		scores.push_back(Score(expr, score));
	}

	float operator()(const TAG& tag) const
	{
		for (typename std::vector<Score>::const_iterator i = scores.begin();
				i != scores.end(); i++)
		{
			OpSet<TAG> tags;
			tags += tag;
			if (i->expr(tags))
				return i->score;
		}
		return defaultScore;
	}

	float distance(const OpSet<TAG>& ts1, const OpSet<TAG>& ts2) const
	{
		float res = 0;
		OpSet<TAG> diff = (ts1 - ts2) + (ts2 - ts1);
		for (typename OpSet<TAG>::const_iterator i = diff.begin();
				i != diff.end(); i++)
			res += (*this)(*i);
		return res;
	}
};

template <typename ITEM, typename TAG>
class Normalizer : public CardinalityStore<ITEM, TAG>
{
protected:
	unsigned int max_threshold;
	unsigned int merge_threshold;
	unsigned int min_threshold;
	float maxMergeDist;

	typedef CardinalityStore<ITEM, TAG> tagsets_t;

	typedef std::map< OpSet<TAG>, std::vector< OpSet<TAG> > > distgraph_t;
	distgraph_t distGraph;

	bool addToGraph(const Scores<TAG>& scores, const OpSet<TAG>& ts1, const OpSet<TAG>& ts2);
	bool mergeTagsets(const OpSet<TAG>& ts1, const OpSet<TAG>& ts2);
	void removeAfterMerge(const OpSet<TAG>& ts, const OpSet<TAG>& merged);


public:
	Normalizer(float mmd = 1.0) :
		max_threshold(14),
		merge_threshold(7),
		min_threshold(2),
		maxMergeDist(mmd) {}

	void buildGraph(const Scores<TAG>& scores);

	void normalize();
};

template <typename ITEM, typename TAG>
class Graph : public CardinalityStore<ITEM, TAG>
{
protected:
	typedef CardinalityStore<ITEM, TAG> tagsets_t;
	
	unsigned int seq;
	std::map< OpSet<TAG>, int > handles;

	int getHandle(const OpSet<TAG>& node);
	void buildSubGraph(std::ostream& out, const OpSet<TAG>& node, OpSet< OpSet<TAG> >& selected, int maxdist, int maxlev);

public:
	Graph() : seq(0) {}

	void buildGraph(std::ostream& out, const OpSet<TAG>& node, int maxdist = 3, int maxlev = 3);
	void buildGraphs(const std::string& dir, int maxdist = 3);
};

}

// vim:set ts=4 sw=4:
#endif
