/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test handle-mapper managers.
 *
 * ti-handle-mapper.h:
 * Maps handle numbers to their respective data (like their names)
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Copyright (C) 2007-2008 Collabora Ltd <http://www.collabora.co.uk/>
 * Copyright (C) 2007-2008 Nokia Corporation
 *
 * Original author:
 *  Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "handle-mapper.h"

#include <telepathy-glib/errors.h>

#include "constants.h"
#include "preferences.h"

struct _TIHandleMapperClass {
    GObjectClass parent;
};

G_DEFINE_TYPE (TIHandleMapper, ti_handle_mapper, G_TYPE_OBJECT);

struct _TIHandleMapperPrivate {

    TpConnection *connection;
    TIPreferences *preferences;

    GtkListStore *handles_list_store;
    GtkListStore *contact_handle_list_store;
};

static void _ti_handle_list_store_set_name (GtkListStore *list_store,
    guint handle_type, guint handle_number, const gchar *name,
    gboolean is_real);
static gboolean _ti_handle_list_store_get_row (GtkListStore *list_store,
    guint handle_type, guint handle_number, GtkTreeIter *iter);
static void _ti_handle_list_store_remove_row (GtkListStore *list_store,
    guint handle_type, guint handle_number);

static void
ti_handle_mapper_dispose (GObject *object)
{
  TIHandleMapper *self = TI_HANDLE_MAPPER (object);

  if (self->priv->connection != NULL)
    {
      g_object_unref (self->priv->connection);
      self->priv->connection = NULL;
    }

  if (self->priv->preferences != NULL)
    {
      g_object_unref (self->priv->preferences);
      self->priv->preferences = NULL;
    }

  if (self->priv->handles_list_store != NULL)
    {
      g_object_unref (self->priv->handles_list_store);
      self->priv->handles_list_store = NULL;
    }

  if (self->priv->contact_handle_list_store != NULL)
    {
      g_object_unref (self->priv->contact_handle_list_store);
      self->priv->contact_handle_list_store = NULL;
    }

  G_OBJECT_CLASS (ti_handle_mapper_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_handle_mapper_class_init (TIHandleMapperClass *cls)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (cls);

  gobject_class->dispose = ti_handle_mapper_dispose;

  g_type_class_add_private (cls, sizeof (TIHandleMapperPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_handle_mapper_init (TIHandleMapper *self)
{
  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self,
      TI_TYPE_HANDLE_MAPPER, TIHandleMapperPrivate);

  self->priv->preferences = ti_preferences_new ();

  self->priv->handles_list_store = gtk_list_store_new
      (TI_HANDLES_LIST_N_COLUMNS, G_TYPE_UINT /* type */,
       G_TYPE_UINT /* handle */, G_TYPE_STRING /* name */,
       G_TYPE_BOOLEAN /* is real name? */);
  g_object_ref_sink (self->priv->handles_list_store);

  self->priv->contact_handle_list_store = gtk_list_store_new
      (TI_HANDLES_LIST_N_COLUMNS, G_TYPE_UINT /* type (always CONTACT) */,
       G_TYPE_UINT /* handle */, G_TYPE_STRING /* name */,
       G_TYPE_BOOLEAN /* is real name? */);
  g_object_ref_sink (self->priv->contact_handle_list_store);
}

TIHandleMapper *
ti_handle_mapper_new (TpConnection *connection)
{
  TIHandleMapper *self = NULL;

  self = g_object_new (TI_TYPE_HANDLE_MAPPER, NULL);

  self->priv->connection = connection;
  g_object_ref (connection);

  return self;
}

/**
 * Set Handle
 * Sets an unnamed handle
 */
void
ti_handle_mapper_set_handle (TIHandleMapper *self,
                             guint handle_type,
                             guint handle_number)
{
  gchar *name;

  name = g_strdup_printf ("%u", handle_number);

  _ti_handle_list_store_set_name (self->priv->handles_list_store, handle_type,
      handle_number, name, FALSE);

  if (handle_type == TP_HANDLE_TYPE_CONTACT)
    {
      _ti_handle_list_store_set_name (self->priv->contact_handle_list_store,
          handle_type, handle_number, name, FALSE);
    }

  g_free (name);
}

void
ti_handle_mapper_set_handle_name (TIHandleMapper *self,
                                  guint handle_type,
                                  guint handle_number,
                                  const gchar *name)
{
  _ti_handle_list_store_set_name (self->priv->handles_list_store, handle_type,
      handle_number, name, TRUE);

  if (handle_type == TP_HANDLE_TYPE_CONTACT)
    {
      _ti_handle_list_store_set_name (self->priv->contact_handle_list_store,
          handle_type, handle_number, name, TRUE);
    }
}

static gchar *
ti_connection_inspect_single_handle (TpConnection *connection,
                                     TpHandleType handle_type,
                                     const GArray *arr,
                                     GError **error)
{
  gchar **inspected;
  gchar *name;

  if (!tp_cli_connection_run_inspect_handles (connection, -1,
        handle_type, arr, &inspected, error, NULL))
    {
      return NULL;
    }

  if (inspected[0] == NULL || inspected[1] != NULL)
    {
      g_strfreev (inspected);
      g_set_error (error, TP_ERRORS, TP_ERROR_INVALID_ARGUMENT,
          "InspectHandles() returned the wrong number of strings");
      return NULL;
    }

  name = inspected[0];
  g_free (inspected);

  return name;
}

gchar *
ti_handle_mapper_get_handle_name (TIHandleMapper *self,
                                  guint handle_type,
                                  guint handle_number)
{
  gboolean found;
  GtkTreeIter iter;
  gboolean is_real_name;
  gchar *name = NULL;
  GError *error = NULL;
  GArray *arr = g_array_sized_new (FALSE, FALSE, sizeof (guint), 1);

  g_array_append_val (arr, handle_number);

  found = _ti_handle_list_store_get_row (self->priv->handles_list_store,
      handle_type, handle_number, &iter);
  if (!found)
    {
      if (ti_preferences_get_auto_inspect_handles (self->priv->preferences))
        {
          if (!tp_cli_connection_run_hold_handles (self->priv->connection, -1,
                handle_type, arr, &error, NULL))
            goto CLEAN_UP;

          name = ti_connection_inspect_single_handle (self->priv->connection,
              handle_type, arr, &error);

          if (name == NULL)
            goto CLEAN_UP;

          ti_handle_mapper_set_handle_name (self, handle_type,
              handle_number, name);
        }

      goto CLEAN_UP;
    }

  gtk_tree_model_get (GTK_TREE_MODEL (self->priv->handles_list_store), &iter,
      TI_HANDLES_LIST_COLUMN_NAME, &name,
      TI_HANDLES_LIST_COLUMN_IS_REAL_NAME, &is_real_name,
      -1);

  if (ti_preferences_get_auto_inspect_handles (self->priv->preferences) &&
      !is_real_name)
    {
      name = ti_connection_inspect_single_handle (self->priv->connection,
          handle_type, arr, &error);
      if (error != NULL)
        goto CLEAN_UP;

      gtk_list_store_set (self->priv->handles_list_store, &iter,
          TI_HANDLES_LIST_COLUMN_NAME, &name,
          TI_HANDLES_LIST_COLUMN_IS_REAL_NAME, TRUE,
          -1);
    }

CLEAN_UP:

  g_array_free (arr, TRUE);

  if (error != NULL)
    g_error_free (error);

  return name;
}

void
ti_handle_mapper_invalidate_handle (TIHandleMapper *self,
                                    guint handle_type,
                                    guint handle_number)
{
  _ti_handle_list_store_remove_row (self->priv->handles_list_store,
      handle_type, handle_number);

  if (handle_type == TP_HANDLE_TYPE_CONTACT)
    {
      _ti_handle_list_store_remove_row (self->priv->contact_handle_list_store,
          handle_type, handle_number);
    }
}

void
ti_handle_mapper_invalidate_contact_handles (TIHandleMapper *self,
                                             GArray *handle_numbers)
{
  guint i;
  guint handle_number;

  for (i = 0; i < handle_numbers->len; i++)
    {
      handle_number = g_array_index (handle_numbers, guint, i);

      _ti_handle_list_store_remove_row (self->priv->handles_list_store,
          TP_HANDLE_TYPE_CONTACT, handle_number);
      _ti_handle_list_store_remove_row (self->priv->contact_handle_list_store,
          TP_HANDLE_TYPE_CONTACT, handle_number);
    }
}

/**
 * Invalidate List Handles
 */
void
ti_handle_mapper_invalidate_list_handles (TIHandleMapper *self,
                                          GArray *handle_numbers)
{
  guint i;
  guint handle_number;

  for (i = 0; i < handle_numbers->len; i++)
    {
      handle_number = g_array_index (handle_numbers, guint, i);

      _ti_handle_list_store_remove_row (self->priv->handles_list_store,
          TP_HANDLE_TYPE_LIST, handle_number);
    }
}

static void
_ti_handle_list_store_set_name (GtkListStore *list_store,
                                guint handle_type,
                                guint handle_number,
                                const gchar *name,
                                gboolean is_real)
{
  GtkTreeIter iter;
  gboolean found;

  found = _ti_handle_list_store_get_row (list_store, handle_type,
      handle_number, &iter);

  if (found)
    {
      gtk_list_store_set (list_store, &iter,
          TI_HANDLES_LIST_COLUMN_NAME, name,
          -1);
    }
  else
    {
      gtk_list_store_append (list_store, &iter);

      gtk_list_store_set (list_store, &iter,
          TI_HANDLES_LIST_COLUMN_TYPE, handle_type,
          TI_HANDLES_LIST_COLUMN_NUMBER, handle_number,
          TI_HANDLES_LIST_COLUMN_NAME, name,
          TI_HANDLES_LIST_COLUMN_IS_REAL_NAME, is_real,
          -1);
    }
}

static void
_ti_handle_list_store_remove_row (GtkListStore *list_store,
                                  guint handle_type,
                                  guint handle_number)
{
  GtkTreeIter iter;
  gboolean found;

  found = _ti_handle_list_store_get_row (list_store, handle_type,
      handle_number, &iter);

  if (found)
    {
      gtk_list_store_remove (list_store, &iter);
    }
}

GtkListStore *
ti_handle_mapper_get_handles_list_store (TIHandleMapper *self)
{
  return g_object_ref (self->priv->handles_list_store);
}

GtkListStore *
ti_handle_mapper_get_contact_handle_list_store (TIHandleMapper *self)
{
  return g_object_ref (self->priv->contact_handle_list_store);
}

/**
 * Handle List Store Get Row
 *
 * @return TRUE if the row was found and FALSE otherwise.
 */
static gboolean
_ti_handle_list_store_get_row (GtkListStore *list_store,
                               guint handle_type,
                               guint handle_number,
                               GtkTreeIter *iter)
{
  gboolean ok;
  gboolean found = FALSE;
  guint row_handle_type;
  guint row_handle_number;

  ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (list_store), iter);
  while (ok && !found)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (list_store), iter,
                          TI_HANDLES_LIST_COLUMN_TYPE, &row_handle_type,
                          TI_HANDLES_LIST_COLUMN_NUMBER, &row_handle_number,
                          -1);

      if (handle_type == row_handle_type && handle_number == row_handle_number)
        {
          found = TRUE;
        }
      else
        {
          ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (list_store), iter);
        }
    }

  return found;
}
