#include <stdlib.h>
#include <time.h>

#include "place.h"
#include "trandom.h"
#include "writer.h"

void
t_terrain_place_objects (TTerrain *terrain,
                         gchar    *name,
                         gfloat    elevation_min,
                         gfloat    elevation_max,
                         gfloat    density,
                         gfloat    variance,
                         gfloat    scale_x,
                         gfloat    scale_y,
                         gfloat    scale_z,
                         gfloat    variance_x,
                         gfloat    variance_y,
                         gfloat    variance_z,
                         gboolean  vary_direction,
                         gboolean  symmetrical,
                         gint      seed)
{
  gfloat x, y;
  gfloat cell_width, cell_height;
  gfloat pov_size_x, pov_size_y, pov_size_z;
  gfloat midelv;
  TRandom *rnd;

  g_return_if_fail (terrain != NULL);
  g_return_if_fail (elevation_min >= 0 && elevation_max <=1);
  g_return_if_fail (density >= 0 && density <= 1);
  g_return_if_fail (variance >= 0 && variance <=1);
  g_return_if_fail (scale_x > 0);
  g_return_if_fail (scale_y > 0);
  g_return_if_fail (scale_z > 0);
  g_return_if_fail (variance_x >= 0);
  g_return_if_fail (variance_y >= 0);
  g_return_if_fail (variance_z >= 0);

  if (seed == -1)
    seed = new_seed ();

  if (name == NULL)
    name = "placed_object";

  rnd = t_random_new (seed);

  /* Determine terrain size in POV-Ray space */
  t_terrain_get_povray_size (terrain, &pov_size_x, &pov_size_y, &pov_size_z);

  /* calculate cell size */
  cell_width = 1.0 / (pov_size_x / scale_x * density);
  cell_height = 1.0 / (pov_size_z / scale_z * density); 
  midelv = elevation_min + (elevation_max-elevation_min)/2;

  /* loop through the squares on Pov and Hf offsets */
  for (y = cell_height * 0.5; y <= 1.0; y += cell_height)
    for (x = cell_width * 0.5; x <= 1.0; x += cell_width)
      {
        gfloat angle;
        gfloat position_x, position_z;
        gfloat size_x, size_y, size_z;
	gfloat dist;

        /* calculate the object position in the scene */
        position_x =
          x + cell_width * variance * (t_random_rnd1 (rnd) - 0.5);
        position_z =
          y + cell_height * variance * (t_random_rnd1 (rnd) - 0.5);

        /* calculate the object scaling factors */
        if (symmetrical)
          {
            gfloat f;

            f = t_random_rnd1 (rnd) - 0.5;
            size_x = scale_x * (1.0 + variance_x * f);
            size_y = scale_y * (1.0 + variance_y * f);
            size_z = scale_z * (1.0 + variance_z * f);
          }
        else
          {
            size_x =
              scale_x * (1.0 + variance_x * (t_random_rnd1 (rnd) * 2.0 - 1.0));
            size_y =
              scale_y * (1.0 + variance_y * (t_random_rnd1 (rnd) * 2.0 - 1.0));
            size_z =
              scale_z * (1.0 + variance_z * (t_random_rnd1 (rnd) * 2.0 - 1.0));
          }

        if (position_x >= 0.0 && position_x <= 1.0 &&
            position_z >= 0.0 && position_z <= 1.0)
          {
            gfloat height;
            gfloat elvdiff;
            gint   hf_x, hf_y, pos;

            hf_x = position_x * (terrain->width - 1);
            hf_y = position_z * (terrain->height - 1);
            pos = hf_y * terrain->width + hf_x;
            height = terrain->heightfield[pos];
            elvdiff = ABS(midelv - (midelv-height))/midelv;

	    if (elvdiff > t_random_rnd1(rnd)) 
              {


            if (vary_direction)
              angle = rand () % 360;
            else
              angle = 0.0;

            if (height >= elevation_min && height <= elevation_max)
              if (terrain->selection == NULL ||
                  terrain->selection[pos] > 0.5)
                t_terrain_add_object (terrain, hf_x, hf_y,
                                      position_x, position_z,
                                      angle,
                                      size_x, size_y, size_z, name);
              }
          }
      }

  t_random_free (rnd);
}

void
t_terrain_rescale_placed_objects (TTerrain *terrain,
                                  gfloat    scale_x,
                                  gfloat    scale_y,
                                  gfloat    scale_z,
                                  gfloat    variance_x,
                                  gfloat    variance_y,
                                  gfloat    variance_z,
                                  gboolean  symetrical, 
				  gint      seed)
{
  gint i=0;
  TTerrainObject *obj; 
  TRandom *rnd;

  g_return_if_fail (terrain != NULL);
  g_return_if_fail (scale_x > 0);
  g_return_if_fail (scale_y > 0);
  g_return_if_fail (scale_z > 0);
  g_return_if_fail (variance_x > 0);
  g_return_if_fail (variance_y > 0);
  g_return_if_fail (variance_z > 0);

  if (seed == -1)
    seed = new_seed ();

  rnd = t_random_new (seed);
  srand (seed);

  while ((obj = &g_array_index (terrain->objects,TTerrainObject,i++)) != NULL)
    {
    if (terrain->selection == NULL || 
        terrain->selection[obj->oy*terrain->width+obj->ox] > 0.0)
      {
      if (symetrical)
        {
        gfloat f;

        f = t_random_rnd1 (rnd) - 0.5;
        obj->scale_x *= (scale_x + variance_x * f);
        obj->scale_y *= (scale_y + variance_y * f);
        obj->scale_z *= (scale_z + variance_z * f);
        }
      else
        {
        obj->scale_x *= (scale_x + variance_x * (t_random_rnd1(rnd) * 2.0 - 1.0));
        obj->scale_y *= (scale_y + variance_y * (t_random_rnd1(rnd) * 2.0 - 1.0));
        obj->scale_z *= (scale_z + variance_z * (t_random_rnd1(rnd) * 2.0 - 1.0));
        }
      }

    }

  t_random_free (rnd);
}

