
/******************************************************************************
* MODULE     : tm_config.gen.cc
* DESCRIPTION: Configuration routines for TeXmacs server
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tm_config.gen.h>

#module code_tm_config
#import tm_config

extern bool eval_scheme_bool (string s, bool& b);

/******************************************************************************
* Constructor and destructor
******************************************************************************/

tm_config_rep::tm_config_rep ():
  in_lan ("english"),
  out_lan ("english"),
  latex_cmds (tree ("undefined latex command")),
  latex_help (""),
  keymap (""),
  shorthands (""),
  help_message ("") {}

tm_config_rep::~tm_config_rep () {}

/******************************************************************************
* Set input and output language
******************************************************************************/

void
tm_config_rep::set_input_language (string s) {
  in_lan= s;
}

void
tm_config_rep::set_output_language (string s) {
  out_lan= s;
  get_display()->set_output_language (s);
}

string
tm_config_rep::get_input_language () {
  return in_lan;
}

string
tm_config_rep::get_output_language () {
  return out_lan;
}

/******************************************************************************
* Setup TeXmacs fonts
******************************************************************************/

void
tm_config_rep::set_font_rules (scheme_tree rules) {
  int i, n= arity (rules);
  for (i=0; i<n; i++)
    if (arity (rules [i]) == 2) {
      tree l= (tree) rules[i][0];
      tree r= (tree) rules[i][1];
      font_rule (l, r);
    }  
}

/******************************************************************************
* Latex (or user) commands
******************************************************************************/

void
tm_config_rep::set_latex_commands (scheme_tree cmds) {
  int i, n= arity (cmds);
  for (i=0; i<n; i++)
    if ((arity (cmds [i]) == 3) &&
	is_atomic (cmds[i][0]) && is_atomic (cmds[i][1]))
      {
	string  l     = unquote (cmds[i][0]->label);
	latex_help (l)= unquote (cmds[i][1]->label);
	latex_cmds (l)= cmds[i][2];
      }
}

void
tm_config_rep::set_latex_symbols (scheme_tree syms) {
  int i, n= arity (syms);
  for (i=0; i<n; i++)
    if (is_atomic (syms[i])) {
      string l= syms[i]->label;
      if ((N(l)>=2) && (l[0]=='\42') && (l[N(l)-1]=='\42'))
	l= l (1, N(l)-1);
      latex_help (l)= "insert#<" * l * ">";
      latex_cmds (l)= string_to_scheme_tree ("(insert-string \"<" * l * ">\"");
    }
}

bool
tm_config_rep::get_latex_command (string which, string& help, scheme_tree& p) {
  if (!latex_help->contains (which)) return FALSE;
  help= latex_help [which];
  p   = latex_cmds [which];
  return TRUE;
}

/******************************************************************************
* Server keyboard mappings and shorthands
******************************************************************************/

static bool
predicate_implies (string p1, string p2) {
  if (p1 == p2) return TRUE;
  if (p2 == "always?") return TRUE;
  if (p2 == "in-text?")
    return
      (p1 == "in-british?") ||
      (p1 == "in-czech?") ||
      (p1 == "in-dutch?") ||
      (p1 == "in-english?") ||
      (p1 == "in-french?") ||
      (p1 == "in-german?") ||
      (p1 == "in-hungarian?") ||
      (p1 == "in-italian?") ||
      (p1 == "in-polish?") ||
      (p1 == "in-portuguese?") ||
      (p1 == "in-romanian?") ||
      (p1 == "in-russian?") ||
      (p1 == "in-spanish?") ||
      (p1 == "in-swedish?") ||
      (p1 == "in-ukrainian?");
  return FALSE;
}

void
tm_config_rep::insert_key_binding (string pred, string key, scheme_tree im) {
  // cout << "Binding '" << key << "' when " << pred << " to " << im << "\n";
  if (!keymap->contains (key)) keymap (key)= tree (TUPLE);
  delete_key_binding (pred, key);
  keymap (key)= join (tuple (tuple (pred, im)), keymap [key]);
}

void
tm_config_rep::delete_key_binding (string key) {
  // cout << "Deleting binding '" << key << "'\n";
  keymap->reset (key);
}

void
tm_config_rep::delete_key_binding (string pred, string key) {
  //cout << "Deleting binding '" << key << "' when " << pred << "\n";
  tree t= keymap [key];
  int i, n= N(t);
  for (i=0; i<n; i++)
    if (t[i][0] == pred) {
      tree map= keymap [key];
      keymap (key)= join (map (0, i), map (i+1, n));
    }
}

scheme_tree
tm_config_rep::find_key_binding (string key) {
  // cout << "Find binding '" << key << "'\n";
  tree t= keymap [key];
  int i, n= N(t);
  for (i=0; i<n; i++) {
    bool found;
    (void) eval_scheme_bool ("(" * t[i][0]->label * ")", found);
    if (found) return t[i][1];
  }
  return tree (UNKNOWN);
}

scheme_tree
tm_config_rep::find_key_binding (string pred, string key) {
  // cout << "Find binding '" << key << "' when " << pred << "\n";
  tree t= keymap [key];
  int i, n= N(t);
  for (i=0; i<n; i++)
    if (predicate_implies (pred, t[i][0]->label))
      return t[i][1];
  return tree (UNKNOWN);
}

static string
unspace (string s) {
  int i, n= N(s);
  string r;
  for (i=0; i<n; i++)
    if (s[i] != ' ') r << s[i];
    else { i++; if (i<n) r << s[i]; }
  return r;
}

void
tm_config_rep::determine_sub_key_bindings (string pred, string s) {
  int i=0, j= 0;
  for (; i<N(s); j=i) {
    if (i!=0) i+=2;
    while ((i<N(s)) && (s[i]!=' ')) i++;
    if (i >= N(s)) break;
    if (find_key_binding (pred, s (0, i)) == tree (UNKNOWN)) {
      tree prev= find_key_binding (pred, s (0, j));
      if (is_tuple (prev) && (N(prev) == 2)) prev= prev[0];
      if ((prev == tree (UNKNOWN)) || (!is_atomic (prev))) prev= s (0, j);
      string im= prev->label * unspace (s (j, i));
      insert_key_binding (pred, s (0, i), tuple (im, ""));
    }
  }
}

void
tm_config_rep::set_keymap (scheme_tree preds, scheme_tree cmds) {
  int i, j, m= arity (cmds), n= arity (preds);
  for (j=0; j<n; j++) {
    if (!is_atomic (preds[j])) continue;
    string pred= preds[j]->label;
    if ((pred == "text") || (pred == "math") || (pred == "prog") ||
	(pred == "british") || (pred == "czech") || (pred == "dutch") ||
	(pred == "english") || (pred == "french") || (pred == "german") ||
	(pred == "hungarian") || (pred == "italian") || (pred == "polish") ||
	(pred == "portuguese") || (pred == "romanian") ||
	(pred == "russian") || (pred == "spanish") ||
	(pred == "swedish") || (pred == "ukrainian"))
      pred= "in-" * pred * "?";

    for (i=0; i<m; i++)
      if ((arity (cmds [i]) >= 2) && is_atomic (cmds[i][0])) {
	string l= cmds[i][0]->label;
	if (is_quoted (l)) l= unquote (l);
	scheme_tree r= cmds[i][1];
	if (is_atomic (r)) {
	  if (is_quoted (r->label)) r= unquote (r->label);
	  if ((N(cmds[i]) >= 3) && is_atomic (cmds[i][2])) {
	    string help= cmds[i][2]->label;
	    if (is_quoted (help)) help= unquote (help);
	    r= tuple (r, help);
	  }
	  else r= tuple (r, "");
	}
	else {
	  if (N(cmds[i]) >= 3)
	    r= join (tuple ("begin"), cmds[i] (1, N(cmds[i])));
	  r= tuple (r, "");
	}
	determine_sub_key_bindings (pred, l);
	insert_key_binding (pred, copy (l), copy (r));
      }
  }
}

void
tm_config_rep::remove_keymap (scheme_tree preds, scheme_tree cmds) {
  int i, j, m= arity (cmds), n= arity (preds);
  for (j=0; j<n; j++) {
    if (!is_atomic (preds[j])) continue;
    string pred= preds[j]->label;
    for (i=0; i<m; i++)
      if (is_atomic (cmds[i])) {
	string l= cmds[i]->label;
	if (is_quoted (l)) l= unquote (l);
	delete_key_binding (pred, l);
      }
  }
}

void
tm_config_rep::get_keycomb (
  string which, int& status, scheme_tree& keym, string& shorth, string& help)
{
  scheme_tree t= find_key_binding (which);
  if (!is_func (t, TUPLE, 2)) {
    status= 0;
    keym  = "";
    shorth= copy (which);
    help  = "";
  }
  else if (!is_atomic (t[0])) {
    status= 1;
    keym  = t[0];
    shorth= copy (which);
    help  = t[1]->label;
  }
  else {
    status= 2;
    keym  = "";
    shorth= t[0]->label;
    help  = t[1]->label;
  }
}

#endmodule // code_tm_config
