
/******************************************************************************
* MODULE     : tm_file.gen.h
* DESCRIPTION: Loading and saving TeXmacs files
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tm_buffer.gen.h>
#include <file.gen.h>
#include <dir.gen.h>
#include <convert.gen.h>

#module code_tm_file
#import tm_buffer
#import file
#import dir
#import convert

/******************************************************************************
* Global commands for loading files
******************************************************************************/

tree
load_file (string dir, string name, string fm) {
  tree fnf_error (ERROR, "file#" * name * "#not found");
  tree bdf_error (ERROR, "bad data format");
  if (!file_exists (dir, name)) return fnf_error;
  string unique_name= get_unique_file_name (dir, name);

  if ((fm == "TeXmacs") ||
      ((fm == "help") && (name (N(name)-3, N(name)) == ".tm"))) {
    tree doc;
    if (load_tree (dir, name, doc)) return fnf_error;
    if (doc == "error") return bdf_error;
    return doc;
  }

  else if (fm == "scheme") {
    string s;
    if (load_string (dir, name, s)) return fnf_error;
    else {
      tree doc= scheme_document_to_tree (s);
      if (doc == "error") return bdf_error;
      return doc;
    }
  }

  else if ((fm == "tex") || (fm == "latex") ||
	   ((fm == "help") && (name (N(name)-4, N(name)) == ".tex"))) {
    string s, style;
    if (load_string (dir, name, s)) return fnf_error;
    else {
      tree t= simplify_correct (latex_document_to_tree (string (s), style));
      if (fm == "help") style= string ("help");
      return tree (DOCUMENT,
		   tree (EXPAND, "body", t),
		   tree (EXPAND, "style", style));
    }
  }

  else if ((fm == "html") ||
	   ((fm == "help") && (name (N(name)-5, N(name)) == ".html"))) {
    string s, style;
    if (load_string (dir, name, s)) return fnf_error;
    else {
      tree t= simplify_correct (html_document_to_tree (string (s), style));
      if (fm == "help") style= string ("help");
      return tree (DOCUMENT,
		   tree (EXPAND, "body", t),
		   tree (EXPAND, "style", style));
    }
  }

  else if (fm == "verbatim") {
    string s;
    if (load_string (dir, name, s)) return fnf_error;
    else {
      tree t    = verbatim_to_tree (s);
      tree init = tree (COLLECTION,
			tree (ASSOCIATE, TEXT_LANGUAGE, "verbatim"),
			tree (ASSOCIATE, TEXT_FAMILY, "tt"),
			tree (ASSOCIATE, PAR_FIRST, "0cm"));
      return tree (DOCUMENT,
		   tree (EXPAND, "body", t),
		   tree (EXPAND, "initial", init));
    }
  }

  return bdf_error;
}

tree
extract_document (tree doc) {
  if (is_func (doc, ERROR)) return doc;
  tree body= extract (doc, "body");
  tree init= extract (doc, "initial");
  if (is_func (init, COLLECTION)) {
    tree w (WITH);
    int i, n= N(init);
    for (i=0; i<n; i++)
      if (is_func (init[i], ASSOCIATE, 2)) {
	tree l= init[i][0];
	tree r= init[i][1];
	if ((l == PAGE_MEDIUM) ||
	    (l == PAGE_TYPE) ||
	    (l == PAGE_ORIENTATION) ||
	    (l == PAGE_NR) ||
	    (l == PAGE_WIDTH) ||
	    (l == PAGE_HEIGHT) ||
	    (l == PAGE_ODD) ||
	    (l == PAGE_EVEN) ||
	    (l == PAGE_RIGHT) ||
	    (l == PAGE_TOP) ||
	    (l == PAGE_BOT) ||
	    (l == PAGE_REDUCE_LEFT) ||
	    (l == PAGE_REDUCE_RIGHT) ||
	    (l == PAGE_REDUCE_TOP) ||
	    (l == PAGE_REDUCE_BOT) ||
	    (l == PAGE_SHOW_HF)) continue;
	w << l << r;
      }
    if (N(w)>0) {
      w << body;
      body= w;
    }
  }
  return body;
}

static hashmap<string,tree> document_inclusions ("");

void
reset_inclusions () {
  document_inclusions= hashmap<string,tree> ("");
}

void
reset_inclusion (string base_file_name, string file_name) {
  document_inclusions->reset (base_file_name * ":" * file_name);
  string name= get_relative_file_name (base_file_name, file_name, FALSE);
  document_inclusions->reset (name);  
}

tree
load_inclusion (string base_file_name, string file_name) {
  if (document_inclusions->contains (base_file_name * ":" * file_name))
    return document_inclusions [base_file_name * ":" * file_name];
  string name= get_relative_file_name (base_file_name, file_name, FALSE);
  if (document_inclusions->contains (name))
    return document_inclusions [name];
  string fm= get_file_format (name);
  tree doc= extract_document (load_file (".", name, fm));
  if (!is_func (doc, ERROR)) {
    document_inclusions (base_file_name * ":" * file_name)= doc;
    document_inclusions (name)= doc;
  }
  return doc;
}

/******************************************************************************
* Loading files
******************************************************************************/

tree
tm_data_rep::file_to_tree (string dir, string name, string fm) {
  string action= "load " * fm * " file#" * name;
  tree doc= ::load_file (dir, name, fm);
  if (is_func (doc, ERROR, 1)) {
    set_message ("Error: " * as_string (doc[0]), action);
    return "error";
  }
  return doc;
}

void
tm_data_rep::load_buffer (
  string dir, string name, string fm, int where, bool autosave_flag)
{
  if ((fm == "help") &&
      (get_file_type (name) != "tex") &&
      (get_file_type (name) != "tm"))
    {
      string lan= get_output_language ();
      string dir=
	"$TEXMACS_DOC_PATH/" * lan * ":$TEXMACS_PATH/doc/" * lan *
	":$TEXMACS_DOC_PATH/english" * ":$TEXMACS_PATH/doc/english";
      if (file_exists (dir, name * ".tm"))
	load_buffer (dir, name * ".tm", fm, where);
      else if (file_exists (dir, name * ".tex"))
	load_buffer (dir, name * ".tex", fm, where);
      else {
	cerr << "TeXmacs] Help file '" << name << "' not found\n";
	if (N(bufs)!=0)
	  set_message ("Error: help file#'" * name * "'#not found",
		       "load help file");
      }
      return;
    }
  if (fm == "generic") fm= get_file_format (name);
  if (!file_exists (dir, name)) {
    cerr << "TeXmacs] File '" << name << "' not found\n";
    if (N(bufs)!=0)
      set_message ("Error: file#'" * name * "'#not found",
		   "load " * fm * " file");
    return;
  }

  string unique_name= get_unique_file_name (dir, name);
  if (autosave_flag) unique_name= unique_name (0, N(unique_name)-1);
  int nr= find_buffer (unique_name);
  tree doc= ((nr == -1)? file_to_tree (dir, name, fm): tree (DOCUMENT));
  if (doc == "error") return;
  switch (where) {
  case 0: new_buffer_in_this_window (unique_name, doc); break;
  case 1: new_buffer_in_new_window (unique_name, doc); break;
  case 2: new_buffer (unique_name, doc); break;
  default: fatal_error ("Bad value for 'where'", "load_buffer");
  }
  nr= find_buffer (unique_name);
  if (nr != -1) {
    tm_buffer buf= (tm_buffer) bufs[nr];
    buf->fm= fm;
    if ((fm == "help") || is_web_file (unique_name)) {
      tm_buffer buf= get_buffer ();
      buf->read_only= TRUE;
    }
  }
}

tm_buffer
tm_data_rep::load_passive_buffer (string name) {
  int nr= find_buffer (name);
  if (nr != -1) return (tm_buffer) bufs[nr];
  load_buffer (".", name, "TeXmacs", 2, FALSE);
  nr= find_buffer (name);
  if (nr != -1) return (tm_buffer) bufs[nr];
  return NULL;
}

/******************************************************************************
* Saving files
******************************************************************************/

tree
tm_data_rep::make_document (tm_view vw) {
  tree doc (DOCUMENT);
  doc << tree (EXPAND, "TeXmacs", TEXMACS_VERSION);
  if (vw->buf->project != "")
    doc << tree (EXPAND, "project", vw->buf->project);
  if (vw->ed->get_style() != tree (TUPLE))
    doc << tree (EXPAND, "style", copy (vw->ed->get_style()));
  if (vw->buf->t != tree (DOCUMENT, ""))
    doc << tree (EXPAND, "body", vw->buf->t);
  if (N (vw->ed->get_init()) != 0)
    doc << tree (EXPAND, "initial", ((tree) vw->ed->get_init()));
  if (N (vw->ed->get_fin()) != 0)
    doc << tree (EXPAND, "final", ((tree) vw->ed->get_fin()));
  if (N (vw->buf->ref) != 0)
    doc << tree (EXPAND, "references", ((tree) vw->buf->ref));
  if (N (vw->buf->aux) != 0)
    doc << tree (EXPAND, "auxiliary", ((tree) vw->buf->aux));
  return doc;
}

bool
tm_data_rep::save (string name, tm_view vw) {
  tree doc= make_document (vw);
  bool flag= save_tree (name, doc);
  if (name == "~/.TeXmacs-save.tm") return flag;
  if (flag) set_message ("Error: file " * name * " did not open",
			 "save TeXmacs file");
  else set_message ("saved " * name, "save TeXmacs file");
  return flag;
}

void
tm_data_rep::save_buffer (string name, string fm) {
  if (fm == "generic") fm= get_file_format (name);
  string action= "save " * fm * " file";

  tm_buffer buf= get_buffer ();

  if (buf->read_only && (name == buf->name)) {
    set_message ("Error: file is read only", action);
    return;
  }

  if ((fm == "TeXmacs") && (!buf->needs_to_be_saved ()))
    if (buf->name == name) {
      set_message ("No changes need to be saved", action);
      return;
    }

  if (fm == "TeXmacs") {
    tm_view vw= get_view ();
    if (!save (name, vw)) {
      set_name_buffer (get_unique_file_name ("", name));
      pretend_save_buffer ();
    }
  }

  else if (fm == "scheme") {
    tm_view vw= get_view ();
    if (save_string (name, tree_to_scheme_document (make_document (vw))))
      set_message ("Error: file did not open", action);
    else set_message ("saved " * name, action);
  }

  else if ((fm == "tex") || (fm == "latex") || (fm == "help")) {
    editor ed   = get_editor ();
    tree   style= ed->get_style ();
    string lan  = as_string (ed->get_init() [TEXT_LANGUAGE]);
    if (save_string (name, tree_to_latex_document (ed->et, style, lan)))
      set_message ("Error: file did not open", action);
    else set_message ("saved " * name, action);
  }

  else if (fm == "verbatim") {  
    if (save_string (name, tree_to_verbatim (get_editor()->et)))
      set_message ("Error: file did not open", action);
    else set_message ("saved " * name, action);
  }

  else set_message ("Error: unknown format", action);
}

/******************************************************************************
* Other routines concerning loading and saving buffers
******************************************************************************/

bool
tm_data_rep::no_name () {
  tm_buffer buf= get_buffer ();
  return has_no_name (buf->name);
}

bool
tm_data_rep::exists_unsaved_buffer () {
  bool flag= FALSE;
  int i, n= N(bufs);
  for (i=0; i<n; i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    flag = flag || buf->needs_to_be_saved ();
  }
  return flag;
}

void
tm_data_rep::pretend_save_buffer () {
  tm_buffer buf= get_buffer ();
  buf->mark_undo_block ();
  buf->need_save= buf->need_autosave= FALSE;
  buf->last_save= buf->last_autosave= buf->undo_depth- 1;
}

void
tm_data_rep::auto_save () {
  int i, n= N(bufs);
  for (i=0; i<n; i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    if ((buf->needs_to_be_autosaved () && (!buf->read_only))) {
      string name= buf->name * "~";
      if (has_no_name (buf->name))
	name= "$TEXMACS_HOME_PATH/system/autosave.tm";
      if (N(buf->vws)!=0) {
	if (!save (name, (tm_view) buf->vws[0])) {
	  buf->mark_undo_block ();
	  buf->need_autosave= FALSE;
	  buf->last_autosave= buf->undo_depth- 1;
	}
      }
    }
  }
  get_display()->delayed_message (get_meta()->get_this(), "auto save", 120000);
}

#endmodule // code_tm_buffer
