/* -*- mode: c; c-file-style: "gnu" -*-
 * network.c -- Network related routines
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file network.c
 * Network related routines.
 *
 * The commonly used low-level network stuff (like sendfile, recv,
 * send and a few others) are contained herein, within wrappers that
 * ease their use.
 *
 * There are even high-level functions, such as put_buffer(), put_file()
 * and put_file_tls(), which hide the gory details of networking, and
 * deal with any arising errors.
 */

#include "system.h"

#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <string.h>
#ifdef HAVE_SENDFILE_H
# include <sys/sendfile.h>
#endif
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>

#include "config.h"
#include "gzip.h"
#include "network.h"
#include "options.h"
#include "os.h"
#include "thy.h"
#include "tls.h"
#include "types.h"
#include "worker.h"

#if THY_OPTION_ZLIB
#include <zlib.h>
#endif

#if THY_OPTION_TLS
#include <gnutls/gnutls.h>
#endif

/** @internal Function used for debugging.
 * If THY_OPTION_DEBUG is specified, this function logs some
 * interesting details.
 *
 * @returns Always returns -1.
 */
static inline int
_network_borked_pipe (int infd, int outfd, size_t size,
		      off_t offset, int errn)
{
  bhc_debug ("Borked pipe: %d -> %d (" SIZET_FORMAT "/" SIZET_FORMAT "): "
	     "(%d) %s", infd, outfd, (size_t)offset, size, errn,
	     strerror (errno));
  return -1;
}

/** Push the contents of a file to the network.
 * Put as much from INFD to SESSION->fd as possible, starting at OFFSET.
 *
 * @param infd is the input file descriptor.
 * @param session is the session to work with.
 * @param size is the full size of the file.
 * @param offset is the offset to start pushing from.
 *
 * @returns Zero on success (which means that no error occurred, NOT
 * that all data was sent), -1 if an error occurred.
 *
 * @note As a side-effect, it also increments OFFSET.
 */
int
put_file (int infd, session_t *session, size_t size, off_t *offset)
{
  const config_t *config = config_get ();
  size_t count = size - *offset;
  ssize_t wcount;

  if (config->options.buffers && config->buffer_size < count)
    count = config->buffer_size;

  if ((wcount = thy_worker_check (infd, *offset, *offset + count)) == 0)
    return 0;

  if (wcount != -1)
    count = wcount;

  if (thy_sendfile (session->io.out, infd, offset, count) == -1)
    {
      if (errno != EWOULDBLOCK && errno != EAGAIN)
	return _network_borked_pipe (infd, session->io.out, size,
				     *offset, errno);
    }
  return 0;
}

/** Push the contents of a buffer to the network.
 * Put as much from BUF to SESSION->fd as possible, starting at OFFSET.
 *
 * @param session is the session to work with.
 * @param buf is the buffer itself.
 * @param size is its size.
 * @param offset is the offset to start pushing from.
 *
 * @returns Zero on success (which means that no error occurred, NOT
 * that all data was sent), -1 if an error occurred.
 *
 * @note As a side-effect, it also increments OFFSET.
 */
int
put_buffer (session_t *session, char *buf, size_t size, off_t *offset)
{
  off_t offs;

  if (!buf)
    return -1;

  offs = thy_send (session, &buf[*offset], size - *offset);
  if (offs < 0)
    {
      if (errno != EWOULDBLOCK && errno != EAGAIN)
	/* We aren't conting the length of the buffer. That would cost
	   way too much in non-debug mode */
	return _network_borked_pipe (-1, session->io.out, 0,
				     *offset, errno);
    }
  else
    *offset+=offs;

  return 0;
}

/** Push the contents of a buffer to the network.
 * Put the entire buffer out to the network.
 *
 * @param session is the session to work with.
 * @param buf is the buffer itself.
 * @param size is its size.
 * @param offset is the offset to start pushing from.
 */
void
push_buffer (session_t *session, char *buf, size_t size, off_t offset)
{
  int t;
  off_t offs = offset;

  do
    {
      t = put_buffer (session, buf, size - offs, &offs);
    } while (!t && size > (size_t)offs);
}

/** Wrapper around recv().
 * This is a wrapper around recv(), with optional TLS support.
 */
int
thy_recv (session_t *session, void *buf, size_t size)
{
#if THY_OPTION_TLS
  if (thy_tls_istls (session))
    return gnutls_record_recv (session->tls_session, buf, size);
  else
#endif
    return read (session->io.in, buf, size);
}

/** Wrapper around send().
 * This is a wrapper around send(), with optional TLS or gzip support.
 */
int
thy_send (session_t *session, void *buf, size_t size)
{
#if THY_OPTION_TLS
  if (thy_tls_istls (session))
    return gnutls_record_send (session->tls_session, buf, size);
  else
#endif
#if THY_OPTION_ZLIB
    if (session->compress.encoding == ENCODING_TYPE_DYNAMIC &&
	session->state == SESSION_STATE_OUTPUT_BODY)
      return thy_zlib_send (session, buf, size);
    else
#endif
      return thy_sendbuffer (session->io.out, buf, size);
}

/** Push the contents of a file to the network, using thy_send().
 * Put as much from INFD to SESSION->fd as possible, starting at OFFSET.
 *
 * @param infd is the input file descriptor.
 * @param session is the session to work with.
 * @param size is the full size of the file.
 * @param offset is the offset to start pushing from.
 *
 * @returns Zero on success (which means that no error occurred, NOT
 * that all data was sent), -1 if an error occurred.
 *
 * @note As a side-effect, it also increments OFFSET.
 */
int
put_file_simple (int infd, session_t *session, size_t size, off_t *offset)
{
#if defined(THY_OPTION_TLS) || defined (THY_OPTION_ZLIB)
  const config_t *config = config_get ();
  size_t count = size - *offset;

  if (config->options.buffers && config->buffer_size < count)
    count = config->buffer_size;

  if (thy_sendfile_simple (session, infd, offset, count) == -1)
    {
      if (errno != EWOULDBLOCK && errno != EAGAIN)
	return _network_borked_pipe (infd, session->io.out, size,
				     *offset, errno);
    }
  return 0;
#else
  bhc_log ("%s", "Unimplemented function called: put_file_simple()!");
  return -1;
#endif
}

/** Return a 32-bit, network byte ordered ipv4 or ipv6 address.
 * @param hostname is the address to resolve.
 * @param addr is where the result will go.
 *
 * @returns Zero on success or -1 if an error occurred.
 */
int
thy_addr_get (const char *hostname, struct sockaddr_storage *addr)
{
  struct addrinfo *res;
  struct addrinfo hints;
  size_t len;

  memset (&hints, 0, sizeof (hints));
  hints.ai_flags = AI_NUMERICHOST;

  if (getaddrinfo (hostname, NULL, &hints, &res) != 0)
    return -1;

  switch (res->ai_addr->sa_family)
    {
    case AF_INET:
      len = sizeof (struct sockaddr_in);
      break;
    case AF_INET6:
      len = sizeof (struct sockaddr_in6);
      break;
    default:
      freeaddrinfo (res);
      return -1;
    }

  if (len < res->ai_addrlen)
    {
      freeaddrinfo (res);
      return -1;
    }

  memcpy (addr, res->ai_addr, res->ai_addrlen);
  freeaddrinfo (res);

  return 0;
}

#if defined(THY_OPTION_TLS) || defined(THY_OPTION_ZLIB)
/** Push a file onto the network with thy_send().
 * @param session is the session to work with.
 * @param s is the input file descriptor.
 * @param offset is the offset to start sending from.
 * @param count is the number of bytes to push at most.
 *
 * @returns The number of bytes sent, or -1 on error.
 */
int
thy_sendfile_simple (session_t *session, int s, off_t *offset,
		     size_t count)
{
  char buffer[_THY_BUFSIZE];
  ssize_t rbytes;
  int sbytes;
  size_t len;

  if (count < sizeof (buffer))
    len = count;
  else
    len = sizeof (buffer);

  lseek (s, *offset, SEEK_SET);
  rbytes = read (s, buffer, len);
  sbytes = thy_send (session, buffer, rbytes);
  if (sbytes < 0)
    return -1;
  *offset += sbytes;
  return 0;
}
#endif

/** Get the port of a socket FD.
 * @param fd is the socket whose port we want.
 *
 * @returns The port number, or -1 on error.
 */
int
thy_sock_port_get (int fd)
{
  struct sockaddr_storage ss;
  socklen_t len = sizeof (struct sockaddr_storage);

  if (getsockname (fd, (struct sockaddr *)&ss, &len) < 0)
    return -1;

  switch (ss.ss_family)
    {
    case AF_INET6:
      return ntohs (((struct sockaddr_in6 *)(&ss))->sin6_port);
    case AF_INET:
      return ntohs (((struct sockaddr_in *)(&ss))->sin_port);
    }

  return -1;
}

/** Verify if a socket is still connected.
 * @param sock is the socket to verify.
 *
 * @returns zero on success (the socket is connected), -1 otherwise.
 */
int
thy_socket_is_alive (int sock)
{
  int r = getpeername (sock, NULL, NULL);

  if (r == -1 && errno == ENOTCONN)
    return -1;

  return 0;
}
