#include "PreProcessPane.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"
#include "wx/valtext.h"

using namespace indii::tint;
using namespace indii::tint::gui;

PreProcessPane::PreProcessPane(wxWindow* parent, ClusterModel* model) :
    TintPane(parent), model(model) {
  #ifdef __WXMAC__
  static const int PADDING = 2;
  #else
  static const int PADDING = 4;
  #endif
  watch(model);

  textMaxPixels = NULL;
  wxString maxPixels;
  maxPixels << model->getMaxPixels();

  /* options page */
  wxGridBagSizer* sizer = new wxGridBagSizer();
  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(1);

  wxStaticText *labelK, *labelReps, *labelMaxPixels, *labelSat;
  labelK = new wxStaticText(this, -1, _("Thumbs"));
  labelReps = new wxStaticText(this, -1, _("Repetitions"));
  labelMaxPixels = new wxStaticText(this, -1, _("Maximum pixels"));
  labelSat = new wxStaticText(this, -1, _("Saturation threshold"));

  spinK = new wxSpinCtrl(this, ID_SPIN_K,
      wxString::Format(_T("%d"), model->getNumClusters()),
      wxDefaultPosition,  wxDefaultSize, wxSP_ARROW_KEYS, 1, 16,
      model->getNumClusters());
  spinReps = new wxSpinCtrl(this, ID_SPIN_REPS, 
      wxString::Format(_T("%d"), model->getNumRepetitions()),
      wxDefaultPosition, wxDefaultSize, wxSP_ARROW_KEYS, 1, 20,
      model->getNumRepetitions());
  textMaxPixels = new wxTextCtrl(this, ID_TEXT_MAX_PIXELS,
      maxPixels, wxDefaultPosition, wxDefaultSize,
      wxTE_PROCESS_ENTER, wxTextValidator(wxFILTER_NUMERIC, &maxPixels));
  spinSat = new wxSpinCtrl(this, ID_SPIN_SAT,
      wxString::Format(_T("%d"), model->getSaturationThreshold()),
      wxDefaultPosition, wxDefaultSize, wxSP_ARROW_KEYS, 0, 255,
      model->getSaturationThreshold());

  sizer->Add(labelK, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxALL, PADDING);
  sizer->Add(labelReps, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxALL, PADDING);
  sizer->Add(labelMaxPixels, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxALL, PADDING);
  sizer->Add(labelSat, wxGBPosition(3,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxALL, PADDING);
      
  sizer->Add(spinK, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxALL, PADDING);
  sizer->Add(spinReps, wxGBPosition(1,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxALL, PADDING);
  sizer->Add(textMaxPixels, wxGBPosition(2,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxALL, PADDING);
  sizer->Add(spinSat, wxGBPosition(3,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxALL, PADDING);

  SetSizerAndFit(sizer);
}

void PreProcessPane::notifyNumClustersChange() {
  spinK->SetValue(model->getNumClusters());
}

void PreProcessPane::notifyNumRepetitionsChange() {
  spinReps->SetValue(model->getNumRepetitions());
}

void PreProcessPane::notifyMaxPixelsChange() {
  wxString maxPixels;
  maxPixels << model->getMaxPixels();

  textMaxPixels->SetValue(maxPixels);
}

void PreProcessPane::notifySaturationThresholdChange() {
  spinSat->SetValue(model->getSaturationThreshold());
}

void PreProcessPane::OnSpinK(wxSpinEvent& evt) {
  model->setNumClusters(evt.GetPosition());
}

void PreProcessPane::OnSpinReps(wxSpinEvent& evt) {
  model->setNumRepetitions(evt.GetPosition());
}

void PreProcessPane::OnSpinSat(wxSpinEvent& evt) {
  model->setSaturationThreshold(evt.GetPosition()/255.0f);
}

void PreProcessPane::OnTextMaxPixels(wxCommandEvent& evt) {
  if (textMaxPixels != NULL) {  
    unsigned long n;
    if (textMaxPixels->GetValue().ToULong(&n)) {
      model->setMaxPixels(n);
    }
  }
}

BEGIN_EVENT_TABLE(PreProcessPane, wxPanel)
EVT_SPINCTRL(ID_SPIN_K, PreProcessPane::OnSpinK)
EVT_SPINCTRL(ID_SPIN_REPS, PreProcessPane::OnSpinReps)
EVT_SPINCTRL(ID_SPIN_SAT, PreProcessPane::OnSpinSat)
EVT_TEXT(ID_TEXT_MAX_PIXELS, PreProcessPane::OnTextMaxPixels)
END_EVENT_TABLE()
