namespace eval wc {
    variable wc

    # Define the wc array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set wc {
	list.reset ""
	list.clear ""
	entry.file ""
	dialog ""
	file ""
	byte_count ""
	word_count ""
	line_count ""
	show_help ""
	show_version ""
    }
}

# wc::create --
#
#   Method to create the dialog box for the wc command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::create { } {
    global tkWorld
    variable wc

    # Put the focus on the wc dialog if it is already open.
    if [winfo exists $wc(dialog)] {
	switch -- [wm state $wc(dialog)] {
	    normal {
		raise $wc(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $wc(dialog)
	    }
	}
	focus $wc(dialog)
	return
    } else {
	set wc(dialog) [dialog::create .wc Wc]
    }

    # The first tab has file and regexp information, along with
    # the standard command line options.
    set tab1 [tabnotebook::page [dialog::interior $wc(dialog)] "Options"]

    # Use a frame to encapsulate the File and Regexp options so that
    # the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.file_button \
	    -text "File" \
	    -width 5 \
	    -command wc::open
    set wc(entry.file) [entry $f1.entry_file \
	    -width 35 \
	    -textvariable wc::wc(file)]
    grid $f1.file_button $f1.entry_file \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.byte_count \
	    -text "Byte Count" \
	    -variable wc::wc(byte_count) \
	    -onvalue "c" \
	    -offvalue ""
    checkbutton $f2.word_count \
	    -text "Word Count" \
	    -variable wc::wc(word_count) \
	    -onvalue "w" \
	    -offvalue ""
    checkbutton $f2.line_count \
	    -text "Line Count" \
	    -variable wc::wc(line_count) \
	    -onvalue "l" \
	    -offvalue ""
    checkbutton $f2.show_help \
	    -text "Display Help" \
	    -variable wc::wc(show_help) \
	    -onvalue "-help" \
	    -offvalue ""
    checkbutton $f2.show_version \
	    -text "Show Version" \
	    -variable wc::wc(show_version) \
	    -onvalue "-version" \
	    -offvalue ""

    grid $f2.byte_count $f2.word_count $f2.line_count \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.show_help $f2.show_version \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build the first tab.
    pack $f1 $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $wc(entry.file)

    # Define the lists for the reset and clear methods
    set wc(list.reset) "byte_count word_count line_count \
	    show_help show_version"
    set wc(list.clear) "file"
}

# wc::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::ok { } {
    global tkWorld
    variable wc

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    $tkWorld(cmd_center) insert insert "wc "

    # Build the command line argument.
    set cmd_arg ""
    foreach x $wc(list.reset) {
	if [string length $wc($x)] {
	    switch -- $x {
		show_help -
		show_version {
		    $tkWorld(cmd_center) insert insert "-$wc($x) "
		}
		default {
		    append cmd_arg $wc($x)
		}
	    }
	}
    }
    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert "-$cmd_arg "
    }
    if [string length $wc(file)] {
	$tkWorld(cmd_center) insert insert "$wc(file)"
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled

    # Change to the default background/foreground of the file entry.
    $wc(entry.file) configure -bg white -fg black
}

# wc::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::reset { } {
    variable wc

    # It is easier to list the elements to exclude rather than list
    # all of the values to reset.
    foreach x $wc(list.reset) {
	set wc($x) ""
    }
}

# wc::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::clear { } {
    variable wc

    # Reset the data structure elements and bg/fg.
    foreach x $wc(list.clear) {
	set wc($x) ""
	$wc(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $wc(entry.file)
}

# wc::help --
#
#   Method to invoke the Wc Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::help { } {
    global tkWorld

    help::create "help/wc.html" "Wc Command Help"
}

# wc::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::close { } {
    variable wc
    
    balloonhelp::cancel
    destroy $wc(dialog)
}

# wc::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wc::open { } {
    variable wc

    # Insert the file list and move the cursor.
    $wc(entry.file) insert insert [file::select]
    $wc(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $wc(entry.file)
}
